// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __project__
#define __project__

// The basic project is a set of source directories. It provides 
// methods for searching and writing files. 

#include "Puma/PathManager.h"
#include "Puma/UnitManager.h"
#include "Puma/SaveMode.h"
#include <fstream>
#include <iostream>
using namespace std;

namespace Puma {


class Project : public PathManager, public SaveMode {
  UnitManager  _unitManager; 
  const char  *_suffix;           // Current file name suffix.
  Mode         _mode;      // Current save mode.

  // Change to the directory, or create it if necessary.
  bool makeDir (const char *) const;
        
  // Create the directory hierarchy need for file 'path'.
  bool makeDirHierarchy (Filename path) const;

protected:
  // Write a unit on a file. Called by save().
  virtual void write (Unit *unit, ofstream &) const; 

  // Check/update the state of the given unit.
  bool checkState (const char *, Unit *) const;
  void updateState (const char *, Unit *) const;
        
public:
  Project (ErrorStream &);
  virtual ~Project ();

  // Set the mode for saving an unit.
  // OVERWRITE  -> Save the unit under its name and overwrite existing
  //                   files.
  // RENAME_OLD -> An existing file will be renamed. A suffix will be
  //                   added to the name of the file, like `main.cc.old'
  //                   for the file `main.cc' and the suffix `.old'.
  // NEW_SUFFIX -> The unit will be saved with a new suffix, like 
  //                   `main.cpp' for the unit `main.cc' and the suffix 
  //               `cpp'.
  void saveMode (Mode = OVERWRITE, const char *suffix = (const char*)0);

  // Save a unit using the name of the unit to get the full 
  // name of the file and path to save to. The save location
  // must not be protected by a protect pattern and the original
  // file must be located in one of the source directories.
  virtual void save (Unit *, bool = false) const;
        
  // Save a FileUnit (or all) to a file (or files).
  virtual void save (const char *file = (const char*)0,
                     bool only_modified = false, bool is_reg_ex = false) const;

  // Print an unit, regular expression for more units.
  // If no name is given print all units.
  void print (const char *name = (const char*)0, ostream &out = cout,
              bool is_reg_ex = false) const;
        
  // Close an unit, regular expression for more units.
  // If no name is given close all units. If destroy is true,
  // the unit(s) will be deleted. 
  void close (const char *name = (const char*)0, bool destroy = false,
              bool is_reg_ex = false) const;

  // Add a new file to the project.
  Unit *addFile (Filename);
  Unit *addFile (Filename, Filename);

  // Configure the project from the command line or a file.
  virtual void configure (const Config &);
        
  // Member access.
  UnitManager &unitManager ();
        
  // Return true if the given file in the source directory
  // (or at the unit manager) is newer than the corresponding
  // file in the destination directory.
  bool isNewer (const char *) const;

  // Returns true if the given file is in or below the project dirs
  virtual bool isBelow (const char *) const;
  virtual bool isBelow (Unit *) const;
};

inline UnitManager &Project::unitManager () 
 {  return _unitManager; }

inline bool Project::isBelow (const char *file) const
 { return PathManager::isBelow (file); }
 
 
} // namespace Puma

#endif /* __project__ */
