//caps.h:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2012
 *
 *  This file is part of aroarfw, a RoarAudio framework for
 *  embedded systems (Controlers).
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  or (at your option) any later version as published by
 *  the Free Software Foundation.
 *
 *  aroarfw is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#ifndef _AROARFW_CAPS_H_
#define _AROARFW_CAPS_H_

#include <stdint.h>
#include <aroarfw/vendor.h>

#ifdef __GNUC__
#define __RPACKED__ __attribute__((packed))
#else
#define __RPACKED__
#warn Don't know how to pack a struct with your compiler correctly
#endif

//! This is the base type for storing of standard IDs.
typedef union {
 //! 32 bit unsigned int representation in network byte order
 uint32_t u32;
 //! Structure of standard ID splitted in parts
 struct {
  //! Standard Vendor
  uint8_t vendor;
  //! Upper 8 bits of standard number
  uint8_t standardmsb;
  //! Lower 8 bit of standard number
  uint8_t standardlsb;
  //! Standard version
  uint8_t version;
 } __RPACKED__ subm;
 //! Standard ID as 4 byte array
 uint8_t u8array[4];
} rstandard_t;

/** \brief Macro to initialize a constant array of standard IDs
 \param _vendor   The Vendor ID (of type rstdvendor_t).
 \param _standard The standard number (in native byte order uint16_t)
 \param _version  The standard version (as uint8_t)
 \return          The Standard ID suitable for array initialization
*/
#define RSTANDARD_INIT(_vendor,_standard,_version) {.subm = { \
                                                  .vendor      = (uint8_t)(rstdvendor_t)(_vendor), \
                                                  .standardmsb = (uint8_t)(uint16_t)(((_standard) & (uint16_t)0xFF00) >> (uint16_t)8), \
                                                  .standardlsb = (uint8_t)(uint16_t)(((_standard) & (uint16_t)0x00FF)), \
                                                  .version     = (uint8_t)(_version) \
                                                  } \
                                                 }
/** \brief  Macro to access the Vendor ID from a standard ID
 \param x The Standard ID
 \return  The Vendor ID
*/
#define rstandard_getvendor(x) (((rstandard_t)(x)).subm.vendor)
/** \brief Macro to access the standard version from a standard ID
 \param x The sStandard ID
 \return  The standard version
*/
#define rstandard_getversion(x) (((rstandard_t)(x)).subm.version)
/** \brief Macro to access the standard number from a standard ID in host native format
 \param x The Standard ID
 \return  The standard number
*/
#define rstandard_getstandard(x) (((uint16_t)(((rstandard_t)(x)).subm.standardmsb) << (uint16_t)8) | \
                                  ((uint16_t)(((rstandard_t)(x)).subm.standardlsb)))
/** \brief Macro to convert a standard ID into a host native 32 bit unsigned integer
 \param x The Standard ID
 \return  The Standard ID in native format
*/
#define rstandard_getu32hbo(x)   RBO_NB2H32(&(x))

#endif

//ll
