/* $Id: SkyDome.cpp,v 1.17 2003/03/18 18:23:54 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2003 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef WIN32
 #include <windows.h>
#endif

#include <math.h>

#include <GL/gl.h>

#include <Ark/ArkRenderer.h>
#include <Ark/ArkPrimBlock.h>
#include <Modules/HeightField/SkyDome.h>

namespace Ark
{
	class OctahedronGeosphereBuilder
	{
	public:
		OctahedronGeosphereBuilder() : 
			subdivisions(0),
			vertexPerEdge(2),
			vertexPerFace(3),
			indexPerFace(5),
			data(0)
		{}

	private:
		int subdivisions;
		int vertexPerEdge;
		int vertexPerFace;
		int indexPerFace;
		VertexBuffer* data;
		PrimitiveBlock* indices;

		/**
		 * Returns the index of a vertex in a face.
		 *
		 * (x,y) are in [0, vertexPerEdge[
		 */
		inline int FaceVertexIndex(int x, int y) const
		{
			const int yOffset = vertexPerEdge - y;
			const int yIndex = vertexPerFace - (yOffset * (yOffset+1)) / 2;
			const int index = x + yIndex;
			return index;
		}

		void SubDivideRecurseOnly(int faceIndex_, int x0_, int y0_, int offset_)
		{
			// Termination test
			if (offset_ <= 1)
				return;

			// In this case, we have to subtract offset_
			const int mid = offset_/2;

			const int xA = x0_;
			const int yA = y0_;
			const int xAB = xA - mid;
			const int yAB = yA;

			const int xAC = xA;
			const int yAC = yA - mid;

			const int xBC = xA - mid;
			const int yBC = yA - mid;

			// Always SubDivide before SubDivideRecurseOnly
			SubDivide(faceIndex_, xBC, yBC, mid);
			SubDivideRecurseOnly(faceIndex_, xA, yA, mid);
			SubDivideRecurseOnly(faceIndex_, xAB, yAB, mid);
			SubDivideRecurseOnly(faceIndex_, xAC, yAC, mid);
		}
		
		void SubDivide(int faceIndex_, int x0_, int y0_, int offset_)
		{
			// Termination test
			if (offset_ <= 1)
				return;

			const int mid = offset_/2;

			const int xA = x0_;
			const int yA = y0_;
			const int pA = faceIndex_ + FaceVertexIndex(xA, yA);
			const Vector3& vA = data->Coord(pA);

			const int xB = xA + offset_;
			const int yB = yA;
			const int pB = faceIndex_ + FaceVertexIndex(xB, yB);
			const Vector3& vB = data->Coord(pB);

			const int xC = xA;
			const int yC = yA + offset_;
			const int pC = faceIndex_ + FaceVertexIndex(xC, yC);
			const Vector3& vC = data->Coord(pC);

			const int xAB = xA + mid;
			const int yAB = yA;
			const int pAB = faceIndex_ + FaceVertexIndex(xAB, yAB);
			Vector3& vAB = data->Coord(pAB);
			vAB = vA + vB;
			vAB.Normalize();

			const int xAC = xA;
			const int yAC = yA + mid;
			const int pAC = faceIndex_ + FaceVertexIndex(xAC, yAC);
			Vector3& vAC = data->Coord(pAC);
			vAC = vA + vC;
			vAC.Normalize();

			const int xBC = xA + mid;
			const int yBC = yA + mid;
			const int pBC = faceIndex_ + FaceVertexIndex(xBC, yBC);
			Vector3& vBC = data->Coord(pBC);
			vBC = vB + vC;
			vBC.Normalize();

			// Always SubDivide before SubDivideRecurseOnly
			SubDivide(faceIndex_, xA, yA, mid);
			SubDivide(faceIndex_, xAB, yAB, mid);
			SubDivide(faceIndex_, xAC, yAC, mid);
			SubDivideRecurseOnly(faceIndex_, xBC, yBC, mid);
		}
		
		void BuildFace(int faceNumber_, const Vector3& vA, const Vector3& vB, const Vector3& vC)
		{
			const int vbIndex = faceNumber_ * vertexPerFace;
			const int offset = vertexPerEdge - 1;

			const int pA = vbIndex + FaceVertexIndex(0, 0);
			data->Coord(pA) = vA;

			const int pB = vbIndex + FaceVertexIndex(offset, 0);
			data->Coord(pB) = vB;

			const int pC = vbIndex + FaceVertexIndex(0, offset);
			data->Coord(pC) = vC;

			// Generate the vertex position data
			SubDivide(vbIndex, 0, 0, offset);

			// Now strip the triangles
			PrimitiveBlock& pb = *indices;
			int pbIndex = faceNumber_ * indexPerFace;

			for (int line=0 ; line<offset ; ++line)
			{
				const int basePoint = vbIndex + FaceVertexIndex(0, line);
				const int upPoint = vbIndex + FaceVertexIndex(0, line+1);
				const int triangleInLine = upPoint - basePoint - 1;

				// stitch first point
				pb[pbIndex] = basePoint;
				++pbIndex;

				// add triangles
				for (int triangle=0 ; triangle<triangleInLine ; ++triangle)
				{
					const int baseVertex = basePoint + triangle;
					pb[pbIndex+0] = baseVertex;
					
					const int upVertex = upPoint + triangle;
					pb[pbIndex+1] = upVertex;

					pbIndex += 2;
				}

				// stitch last point
				const int lastVertex = upPoint - 1;
				pb[pbIndex+0] = lastVertex;
				pb[pbIndex+1] = lastVertex;
				pbIndex += 2;
			}
			
		}
		

	public:
		inline int GetVertexCount() const 
		{
			return vertexPerFace * 8;
		}
		
		void SetSubdivisionLevel(int sub_)
		{
			subdivisions = sub_;
			vertexPerEdge = (1 << sub_) + 1;
			vertexPerFace = (vertexPerEdge * (vertexPerEdge+1)) / 2;
			// Due to stitching, we need two more indices per line
			indexPerFace = (vertexPerEdge+3) * (vertexPerEdge-1);
		}

		void Build(VertexBuffer& vb_, PrimitiveBlock& pb_)
		{
			data = &vb_;
			indices = &pb_;

			Vector3 p[6];
			p[0] = Vector3( 0.f,  0.f,  1.f); // Z+
			p[1] = Vector3( 0.f,  0.f, -1.f); // Z-
			p[2] = Vector3( 0.f, -1.f,  0.f); // Y-
			p[3] = Vector3( 1.f,  0.f,  0.f); // X+
			p[4] = Vector3( 0.f,  1.f,  0.f); // Y+
			p[5] = Vector3(-1.f,  0.f,  0.f); // X-

			const int vertexCount = 8 * vertexPerFace;
			data->Resize(vertexCount);

			const int indexCount = 8 * indexPerFace;
			indices->SetType(PRIM_TRIANGLE_STRIP);
			indices->Resize(indexCount);
			indices->SetEnabledSize(indexCount);

			// Build the eight sides
			BuildFace(0, p[0], p[3], p[4]);
			BuildFace(1, p[0], p[4], p[5]);
			BuildFace(2, p[0], p[5], p[2]);
			BuildFace(3, p[0], p[2], p[3]);
			BuildFace(4, p[1], p[4], p[3]);
			BuildFace(5, p[1], p[5], p[4]);
			BuildFace(6, p[1], p[2], p[5]);
			BuildFace(7, p[1], p[3], p[2]);
		}

	};
	
    /**
     *  Create a triangular facet approximation to a sphere
     *  Return the number of facets created.
     *  The number of facets will be (4^iterations) * 8
     *
     *  iteration   facets
     *  1               32
     *  2              128
     *  3              512
     *  4             2048
     *  5             8192
     *  
     **/
    void SkyDome::CreateSphere(int iterations)
    {
	m_Triangles.SetFormat(
		VertexBuffer::VB_HAS_COORD
		|VertexBuffer::VB_HAS_UV0
		|VertexBuffer::VB_HAS_UV1);

	OctahedronGeosphereBuilder sphereBuilder;
	sphereBuilder.SetSubdivisionLevel(iterations);
	sphereBuilder.Build(m_Triangles, m_TriangleIndices);

	const int count = sphereBuilder.GetVertexCount();
	for (int idx=0 ; idx<count ; ++idx)
	{
	    ComputeTextureCoord(idx);
	    Vector3& pos = m_Triangles.Coord(idx);
	    pos = Vector3(m_Radius*pos.X, m_Radius*pos.Y, m_Radius*pos.Z);
	}
    }

    bool 
    SkyDome::LoadTexture(const char* name, TexturePtr& texmap)
    {
	String matname = m_Config.GetStr(name, String());

	if (matname.empty())
	{
	    return false;
	}

	return m_Cache->Get(V_TEXTURE, matname, texmap);
    }

    bool 
    SkyDome::LoadImage(const char* name, ImagePtr& texmap)
    {
	String matname = m_Config.GetStr(name, String());

	if (matname.empty())
	{
	    return false;
	}

	return m_Cache->Get(V_IMAGE, matname, texmap);
    }

    void
    SkyDome::Build()
    {
	// Resets materials to NULL
	Reset();

	if (!LoadImage("sky::AmbientMap", m_AmbientImage))
	{
	    // output warning
	    std::cerr << "Could not load sky ambient map" << std::endl;
	}
	
	TexturePtr toneTexture;
	if (LoadTexture("sky::ToneMap", toneTexture))
	{
	    // Reference image
	    m_ToneImage = ImagePtr(toneTexture->m_Image);

	    // Material is referenced by creation
	    m_ToneMap = MaterialPtr(new Material("ToneMap"), 0);
	    m_ToneMap->m_DecorName = "Sky Tone Map";
	    m_ToneMap->m_Flags = MATERIAL_HAS_PASS1 | MATERIAL_IS_DOUBLESIDED;

	    ShaderPass& tone = m_ToneMap->m_Passes[0];

	    tone.m_Flags |= PASS_HAS_BLENDING | PASS_HAS_TEXTURE
		| PASS_HAS_DEPTHFUNC | PASS_HAS_DEPTHWRITE;

	    tone.m_BlendSrc = BLEND_SRC_ALPHA;
	    tone.m_BlendDst = BLEND_ONE_MINUS_SRC_ALPHA;

	    tone.m_DepthTest = false;
	    tone.m_DepthWrite = false;
	    tone.m_RepeatMode = TEXTURE_REPEAT_CLAMP;

	    tone.m_Texture = toneTexture;
	    tone.m_Texture->m_RepeatMode = Image::CLAMP;

	    // Updload texture to renderer
	    toneTexture->Configure();
	}
	else
	{
	    // output warning
	    std::cerr << "Could not load sky tone map" << std::endl;
	}

	TexturePtr starTexture;
	if (LoadTexture("sky::StarMap", starTexture))
	{
	    // Material is referenced by creation
	    m_StarMap = MaterialPtr(new Material("StarMap"), 0);
	    m_StarMap->m_DecorName = "Sky Star Map";
	    m_StarMap->m_Flags = MATERIAL_HAS_PASS1 | MATERIAL_IS_DOUBLESIDED;

	    ShaderPass& star = m_StarMap->m_Passes[0];

	    star.m_Flags |= PASS_HAS_BLENDING | PASS_HAS_TEXTURE
		| PASS_HAS_DEPTHFUNC | PASS_HAS_DEPTHWRITE;

	    star.m_BlendSrc = BLEND_SRC_ALPHA;
	    star.m_BlendDst = BLEND_ONE_MINUS_SRC_ALPHA;

	    star.m_DepthTest = false;
	    star.m_DepthWrite = false;
	    star.m_RepeatMode = TEXTURE_REPEAT_REPEAT;

	    star.m_Texture = starTexture;
	    starTexture->m_RepeatMode = Image::REPEAT;

	    // Updload texture to renderer
	    starTexture->Configure();
	}
	else
	{
	    // output warning
	    std::cerr << "Could not load sky star map" << std::endl;
	}

	m_Radius = m_Config.GetScalar("sky::Size", 100.0f);

	const int ITERATIONS = 3;
	CreateSphere(ITERATIONS);
    }

    SkyDome::SkyDome (Cache *cache, Config& cfg) : 
		m_Cache(cache), 
		m_Config(cfg)
    {
    }

    void
    SkyDome::SetPosition(int n, const Vector3& pos)
    {
	m_Triangles.Coord(n) = pos;
    }

    void
    SkyDome::ComputeTextureCoord(int i)
    {
	const Vector3& pos = m_Triangles.Coord(i);

	// Angle from horizon to zenith
	const float angle = asinf(pos.Y);
	const float angleRatio = fabsf(2.f * angle / PI);

	// Direction
	const float direction = atan2f(pos.Z, pos.X);
	
	// Compute tonemap UV
	if (angle < 0.0f)
	{
	    m_Triangles.UV0(i) = Vector2(0.f, 1.f);
	}
	else
	{
	    const scalar lin = 1.f - angleRatio;
	    const scalar v = lin*lin;
	    m_Triangles.UV0(i) = Vector2(0.f, v);
	}

	// Compute starmap UV
	{
		const float distance = (1.f - angleRatio) / 2.f;
		const float c = distance * cosf(direction);
		const float s = distance * sinf(direction);

	    m_Triangles.UV1(i) = Vector2(s+.5f, c+.5f);
	}

    }


    void
    SkyDome::Reset()
    {
	m_AmbientImage = ImagePtr();
	m_ToneImage = ImagePtr();
	m_ToneMap = MaterialPtr();
	m_StarMap = MaterialPtr();
    }

    SkyDome::~SkyDome ()
    {
	Reset();
    }

    Color
    SkyDome::GetHorizonColor(float timeOfDay) const
    {
	if (!m_ToneImage)
	    return Color(1.f, 1.f, 1.f, 1.f);

	Color toneColor = m_ToneImage->GetColor(timeOfDay, 1.f);
	const float alpha = toneColor.A;

	// Alpha blend with black
	toneColor.R *= alpha;
	toneColor.G *= alpha;
	toneColor.B *= alpha;
	toneColor.A = 1.0f;

	return toneColor;
    }
	    
    Color
    SkyDome::GetAmbientColor(float timeOfDay) const
    {
	if (!m_AmbientImage)
	    return Color(1.f, 1.f, 1.f, 1.f);

	const Color ambient = m_AmbientImage->GetColor(timeOfDay, 0.f);
	return ambient;
    }
	    
    void
    SkyDome::Render (Renderer &renderer, const Vector3 &position, scalar timeOfDay)
    {
	//std::cerr << "Rendering skyDome at " << position << "\n";

	// Sets the modelview matrix to set the center at the camera position
	// like camera, we should smootly go to the next value
	const float angle = 40.f * timeOfDay;
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glRotatef(angle, 0.f, 1.f, 0.f);
	glTranslatef(position.X, position.Y, position.Z);

	// First render starmap
	if (m_StarMap)
	{
	    // Render skymap
	    m_Triangles.SetTextureCoordinateOffset(1);
	    renderer.SetActiveVB(m_Triangles);
	    renderer.LockVB(0, m_Triangles.Size());
	    renderer.RenderBlock(*m_StarMap, m_TriangleIndices);
	    renderer.UnlockVB();
	}

	// Then render tone map with blending
	if (m_ToneMap)
	{
	    glMatrixMode(GL_TEXTURE);
	    glPushMatrix();
	    glTranslatef(timeOfDay, 0.f, 0.f);

	    // Render skymap
	    m_Triangles.SetTextureCoordinateOffset(0);
	    renderer.SetActiveVB(m_Triangles);
	    renderer.LockVB(0, m_Triangles.Size());
	    renderer.RenderBlock(*m_ToneMap, m_TriangleIndices);
	    renderer.UnlockVB();

	    // restore texture matrix
	    glMatrixMode(GL_TEXTURE);
	    glPopMatrix();
	}

	// Restore the modelview matrix
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
    }

}
