///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
 *	OPCODE - Optimized Collision Detection
 *	Copyright (C) 2001 Pierre Terdiman
 *	Homepage: http://www.codercorner.com/Opcode.htm
 */
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/**
 *	Contains code for OPCODE models.
 *	\file		OPC_Model.h
 *	\author		Pierre Terdiman
 *	\date		March, 20, 2001
 */
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Include Guard
#ifndef __OPC_MODEL_H__
#define __OPC_MODEL_H__

	//! Model creation structure
	struct OPCODE_API OPCODECREATE
	{
		//! Constructor
							OPCODECREATE()
							{
								NbTris			= 0;
								NbVerts			= 0;
								Tris			= null;
								Verts			= null;
								Rules			= SPLIT_COMPLETE | SPLIT_LARGESTAXIS;
								NoLeaf			= true;
								Quantized		= true;
								KeepOriginal	= false;
							}

		udword				NbTris;			//!< Number of triangles in the input model
		udword				NbVerts;		//!< Number of vertices in the input model
		const udword*		Tris;			//!< List of indexed triangles
		const Point*		Verts;			//!< List of points
		udword				Rules;			//!< Splitting rules (SPLIT_COMPLETE is mandatory in OPCODE)
		bool				NoLeaf;			//!< true => discard leaf nodes (else use a normal tree)
		bool				Quantized;		//!< true => quantize the tree (else use a normal tree)
		bool				KeepOriginal;	//!< true => keep a copy of the original tree (debug purpose)
	};

	class OPCODE_API OPCODE_Model
	{
		public:
		// Constructor/Destructor
													OPCODE_Model();
													~OPCODE_Model();

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to build a collision model.
		 *	\param		create		[in] model creation structure
		 *	\return		true if success
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						bool						Build(const OPCODECREATE& create);

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to access the tree.
		 *	\return		the collision tree
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		__forceinline	const AABBOptimizedTree*	GetTree()		const	{ return mTree;					}

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to access the source tree.
		 *	\return		generic tree
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		__forceinline	const AABBTree*				GetSourceTree()	const	{ return mSource;				}

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to check whether the tree has leaf nodes or not.
		 *	\return		true if the tree has leaf nodes (normal tree), else false (optimized tree)
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		__forceinline	bool						HasLeafNodes()	const	{ return !mNoLeaf;				}

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to check whether the tree is quantized or not.
		 *	\return		true if the tree is quantized
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		__forceinline	bool						IsQuantized()	const	{ return mQuantized;			}

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to get the number of nodes in the tree.
		 *	Should be 2*N-1 for normal trees and N-1 for optimized ones.
		 *	\return		number of nodes
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						udword						GetNbNodes()	const	{ return mTree->GetNbNodes();	}

		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		/**
		 *	A method to get the number of bytes used by the tree.
		 *	\return		amount of bytes used
		 */
		///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
						udword						GetUsedBytes()	const	{ return mTree->GetUsedBytes();	}
		private:
						AABBTree*					mSource;		//!< Original source tree
						AABBOptimizedTree*			mTree;			//!< Optimized tree
						bool						mNoLeaf;		//!< Leaf/NoLeaf tree
						bool						mQuantized;		//!< Compressed/uncompressed tree
	};

#endif //__OPC_MODEL_H__
// END-OF-FILE\n
