/* $Id: GLClient.cpp,v 1.52 2003/03/23 20:09:25 mrq Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2003 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef WIN32
#include <windows.h>
#endif


#include <Ark/ArkTimer.h>
#include <Ark/ArkRenderer.h>
#include <Ark/ArkSystem.h>
#include <Ark/ArkFileSys.h>
#include <Client/GLClient.h>

#include <Client/TerrainView.h>
#include <Client/MenuView.h>
#include <Client/ScrollWidget.h>
#include <Client/SplashWidget.h>

#include <sstream>

#ifdef HAVE_MIXER
#include <SDL_mixer.h>
#endif

namespace Client
{
   GLClient::GLClient (int argc, char **argv) :
      Client (argc, argv),
      m_TerrainWidget (0),
      m_SplashWidget (0),
      m_MenuWidget (0)
   {
      Ark::Config *cfg = Ark::Sys()->Cfg();
      
      m_Width = Ark::Sys()->Cfg()->GetInt ("renderer::Width", 800);
      m_Height = Ark::Sys()->Cfg()->GetInt ("renderer::Height", 600);
      m_NearPlane = Ark::Sys()->Cfg()->GetScalar ("glrenderer::Near", 0.1f);
      m_FarPlane= Ark::Sys()->Cfg()->GetScalar ("glrenderer::Far", 1000.0f);
      
      //-------------------------------------------------------------
      m_Done = false;
  
      if (SDL_Init (SDL_INIT_VIDEO) < 0)
      {
	 Ark::Sys()->Fatal ("Couldn't initialize SDL: %s\n",
			    SDL_GetError ());
      }

      Ark::Sys()->AtExit ((Ark::ExitFunc) &SDL_Quit);
      const SDL_VideoInfo *vi = SDL_GetVideoInfo ();
      int bpp = Ark::Sys()->Cfg()->GetInt ("renderer::BPP",
					   vi->vfmt->BitsPerPixel);

      // Set the flags we want to use for setting the video mode
      int videoflags = SDL_OPENGL;//|SDL_FULLSCREEN;//|SDL_ANYFORMAT;

      // Check for Voodoo (Graphics or 2), which is always fullscreen
      if (getenv("MESA_GLX_FX") && getenv("MESA_GLX_FX")[0] == 'f')
	 videoflags |= SDL_FULLSCREEN;
      else
      {
	 // We should link to getopt/getlongopt, it would be cleaner...
	 // We have to get gettext/getopt for win32 anyway
	 for (int i = 1; argv [i]; ++i )
	 {
	    if (strcmp(argv[i], "--fullscreen") == 0)
	    {
	       videoflags |= SDL_FULLSCREEN;
	       break;
	    }

	    if (strcmp(argv[i], "--no-fullscreen") == 0)
	    {
	       videoflags &= ~SDL_FULLSCREEN;
	       break;
	    }
	 }
      }

      //--------------------------------------------------------------
      SDL_GL_SetAttribute (SDL_GL_DEPTH_SIZE, 16);
      SDL_GL_SetAttribute (SDL_GL_DOUBLEBUFFER, 1);

      if (SDL_SetVideoMode (m_Width, m_Height, bpp, videoflags) == NULL)
      {
	 Ark::Sys()->Fatal ("Couldn't set GL mode: %s", SDL_GetError());
      }

      // Set title
      Ark::String title = cfg->GetStr ("client::Title", "Ark Client");
      SDL_WM_SetCaption (title.c_str(), "arkclient");

      // Display some informations about OpenGL
      Ark::Sys()->Log ("Renderer informations: \n");
      Ark::Sys()->Log ("  Resolution: %dx%dx%d\n", m_Width, m_Height, bpp);
      Ark::Sys()->Log ("  GL_VENDOR: %s\n", glGetString(GL_VENDOR));
      Ark::Sys()->Log ("  GL_VERSION: %s\n", glGetString(GL_VERSION));
      Ark::Sys()->Log ("  GL_RENDERER: %s\n", glGetString(GL_RENDERER));
      Ark::Sys()->Log ("----------\n\n");

      m_Renderer = Ark::RendererFactory::CreateRenderer
	 ("ark::Renderer::OpenGL",0);
      m_UI = new UIRenderer (m_Renderer, 30, m_Width, m_Height);

      gamma_red = cfg->GetScalar ("client::Gamma",  1.0);
      gamma_green = cfg->GetScalar ("client::Gamma",  1.0);
      gamma_blue = cfg->GetScalar ("client::Gamma",  1.0);

      SDL_SetGamma(gamma_red,gamma_green,gamma_blue);
      Ark::Sys()->Log("gamma %f %f %f\n",gamma_red,gamma_green,gamma_blue );

      m_SplashWidget = WidgetPtr(new SplashWidget, 0);
      m_CurWidget = m_SplashWidget;
      m_Updater = NULL;
      Refresh();

#ifdef HAVE_MIXER
      Ark::Sys()->Log ("Audio informations: \n");
      Mix_OpenAudio (44100, AUDIO_S16, 1, 4096);
#endif
   }

   GLClient::~GLClient ()
   {
      if (m_UI)
	 delete m_UI;

#ifdef HAVE_MIXER
      Mix_CloseAudio();
#endif
   }

   bool
   GLClient::Refresh ()
   {
      glDepthRange (m_NearPlane, m_FarPlane);

      glClearDepth (m_FarPlane);
      glClearColor (0.0, 0.0, 0.0, 1.0);
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

      m_Renderer->SetViewport (0, 0, m_Width, m_Height);

      m_CurWidget->Render (m_UI);
      SDL_GL_SwapBuffers();

      return true;
   }

   void
   GLClient::HandleMenuString(const Ark::String &str)
   {
      if (str == "playsolo")
      {
	 if (!m_TerrainWidget)
	 {
	    m_CurWidget = m_SplashWidget;

	    Refresh();

	    Login(Ark::Sys()->Cfg()->GetStr("client::Login","player1"), "n");

#ifdef HAVE_MIXER
            Ark::String mus = Ark::Sys()->FS()->GetFileName
	       ("{game}/data/musics/ambient1.mp3", false);
	    Mix_Music *music = Mix_LoadMUS (mus.c_str());
	    Mix_PlayMusic (music, -1);
#endif
	    m_TerrainWidget = WidgetPtr(new TerrainView(m_UI, this), 0);
	 }

	 m_CurWidget = m_TerrainWidget;
      }
      else if (str == "credits")
      {
	 m_CurWidget = WidgetPtr(new ScrollWidget
				 (m_UI, "{game}/data/misc/credits.cfg"), 0);
      }
      else if (str == "tellstory")
      {
	 m_CurWidget = WidgetPtr(new ScrollWidget
				 (m_UI, "{game}/data/misc/tellstory.cfg"), 0);
      }
      else if (str == "quit")
      {
	 m_Done = true;
      }
   }

   bool
   GLClient::HandleEvent (SDL_Event *event)
   {
      switch (event->type)
      {
	 case SDL_ACTIVEEVENT:
	    break;

	 case SDL_KEYDOWN:
	    if (event->key.keysym.sym == SDLK_ESCAPE)
	    {
	       if (m_CurWidget == m_MenuWidget && m_TerrainWidget)
		  m_CurWidget = m_TerrainWidget;
	       else
		  m_CurWidget = m_MenuWidget;
	       return true;
	    }
	    else if (event->key.keysym.sym == SDLK_F10)
	    {
	       gamma_red   *= 1.02f;
	       gamma_green *= 1.02f;
	       gamma_blue  *= 1.02f;
	       SDL_SetGamma(gamma_red,gamma_green,gamma_blue);
	       Ark::Sys()->Log("Gamma INCreased : r=%f g=%f b=%f\n",gamma_red,gamma_green,gamma_blue);
	       return true;
	    }
	    else if (event->key.keysym.sym == SDLK_F11)
	    {
	       gamma_red   /= 1.02f;
	       gamma_green /= 1.02f;
	       gamma_blue  /= 1.02f;
	       SDL_SetGamma(gamma_red,gamma_green,gamma_blue);
	       Ark::Sys()->Log("Gamma DECreased : r=%f g=%f b=%f\n",gamma_red,gamma_green,gamma_blue);
	       return true;
	    }
	    else if (event->key.keysym.sym == SDLK_F12)
	    {
	       Ark::Image img ("screen", m_Width, m_Height,
			       Ark::Image::RGB_888);

	       /* read our image data from the frame buffer */
	       glReadPixels (0, 0,
			     m_Width, m_Height,
			     GL_RGB, GL_UNSIGNED_BYTE, img.m_Data);

	       Ark::String name;
	       int i = 0;

	       do
	       {
		   std::ostringstream os;
		   os << "{home}/screen" << i << ".tga";
		   name = os.str();
		   ++i;
	       } while (Ark::Sys()->FS()->IsFile (name));

	       img.SaveTGA (name);
	    }
	    m_CurWidget->HandleKey (true,
				    event->key.keysym.mod,
				    event->key.keysym.sym);
	    break;

	 case SDL_KEYUP:
	    m_CurWidget->HandleKey (false,
				    event->key.keysym.mod,
				    event->key.keysym.sym);
	    break;

	 case SDL_QUIT:
	    m_Done = true;
	    break;

	 case SDL_MOUSEBUTTONDOWN:
	    m_CurWidget->HandleMButton (true, event->button.button,
                                        event->button.x,
                                        event->button.y);
	    break;

	 case SDL_MOUSEBUTTONUP:
	    m_CurWidget->HandleMButton (false, event->button.button,
					event->button.x,
					event->button.y);
	    break;
      }

      return true;
   }

   bool
   GLClient::ProcessEvents ()
   {
      SDL_Event event;

      int x,y,state = SDL_GetMouseState (&x, &y);
      HandleMotion (state, x, y);

      // Check if there's a pending event.
      while (SDL_PollEvent (&event))
	 HandleEvent(&event);

      return true;
   }

   int
   GLClient::Loop ()
   {
      GLenum gl_error;
      char* sdl_error;
      
      m_MenuWidget = WidgetPtr(new MenuView(m_UI), 0);
      m_CurWidget = m_MenuWidget;

      scalar period = 1.0f/m_MaxFPS;
      scalar delta = 0.0f;
  
      while (!m_Done)
      {
	 Ark::Timer timer;

	 //-----------------------------------------------------------
	 if (m_Updater)
	    m_Updater->Update(delta);
	 Refresh ();
          
	 //-----------------------------------------------------------
	 // Check for SDL/OpenGL error
	 gl_error = glGetError ();
	 if (gl_error != GL_NO_ERROR)
	 {
	    Ark::Sys()->Log ("OpenGL error: %s (%x)\n",
			     gluErrorString(gl_error), gl_error);
	 }
     
	 sdl_error = SDL_GetError ();
	 if (sdl_error[0] != '\0')
	 {
	    Ark::Sys()->Log ("SDL error: %s\n", sdl_error );
	    SDL_ClearError ();
	 }
     
	 //-----------------------------------------------------------
	 ProcessEvents ();
	 delta = timer.GetDelta();


	 //----------------------------------------------------------
	 // sleep until we reach the target FPS.
	 if (0 && delta < period)
	 {
	    printf ("Sleeping a bit: %d msec\n", int((period - delta) * 1000));
	    SDL_Delay ((uint) ((period - delta) * 1000.0));
	 }
      }

      return 0;
   }


}
