/* $Id: ArkRefCounter.h,v 1.5 2002/10/11 01:10:04 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_REFCOUNTER_H
#define ARK_REFCOUNTER_H

#include <Ark/ArkTypes.h>

namespace Ark
{
   /**
    * A RefCounter is an object which has a reference counter. This reference
    * counter can be incremented/decremented using the Ref()/Unref() methods.
    * A refcounted object can have two interest : either the reference counter
    * is an hint for a cache managing several objects, either it allows a
    * single instance of an object to be used in several places, while ensuring
    * that this object gets destroyed when it is last unreferenced.
    * Remark: if you want this second kind of behaviour, you should probably
    * use the DRefCounter class.
    */
   class ARK_DLL_API RefCounter
   {
      public:
	 /**
	  * Constructs a "used-once" reference counter : the only referencer
	  * is the creator.
	  */
	 RefCounter () 
	 {m_RefCount = 1;}

	 /// Empty destructor.
	 virtual ~RefCounter ()
	 {}

	 /// Increments the reference counter
	 virtual void Ref ()
	 {m_RefCount++;}
	 
	 /**
	  * Decrements the reference counter. If it becomes nil, the
	  * refcounted object might be destroyed, as nobody seems to be
	  * using it any more. The behaviour depends on the Unref()
	  * implementation since this function is virtual. Anyway you shouldn't
	  * try to access an object you have unreferenced unless you really
	  * know what you are doing.
	  */
	 virtual void Unref ()
	 {
	    m_RefCount--;
	 }
	 
	 /**
	  * Get the value of the reference counter.
	  */
	 int RefCount () {return m_RefCount;}

	 /**
	  * Returns true if the object is currently in use (ie the refcounter
	  * is not nil), and false if it is not being used any more.
	  */
	 bool InUse () {return m_RefCount > 0;}

      private:

	 /// Number of references to this object.
	 int m_RefCount;
   };

   /**
    * A DRefCounter is a refcounted object which gets destroyed when the
    * last instance of the object is unreferenced. This is not the standard
    * behavior for the RefCounter class.
    */
   class ARK_DLL_API DRefCounter : public RefCounter
   {
      public:

	 /**
	  * When a DRefCounter is unreferenced, it gets destroyed if the
	  * number of references reaches 0.
	  */
	 virtual void Unref ()
	 {
	    RefCounter::Unref();
	    if (!InUse()) delete this;
	 }
   };
}

#endif
