#!/usr/bin/python
""" Arkose: Desktop application container """

import tempfile, os, subprocess, shutil, pwd, time, signal, netaddr, re

class ArkoseContainer:
    """
        Class generating a container and letting your un commands in it.
    """

    mounts = []
    workdir = None
    target = None
    name = None
    dbus_proxy = None
    xpra = None
    olddisplay = os.getenv("DISPLAY", None)
    pulse_module = None
    terminal_script = None
    path = ""
    files = []
    lxc_devices = []
    subnet = None
    config = None

    def __init__(self, path = None, fstype = "ext4", fssize = None, network = "none", xserver = "none", dbus = "none", dbusproxy = [], pulseaudio = False, devices = [], ctype = "cow", root = "/", bind = [], restrict = [], cow = []):
        """
            path: Base path used by Arkose to store the containers (default is ~/.arkose)
            fstype: ext4 or tmpfs
            fssize: size in MB. Defaults to 2GB for ext4 and 50% of available RAM for tmpfs
            network: Network access: "none", "direct" or "filtered"
            xserver: What kind of X access should be allowed: none, isolated, direct
            dbus: DBUS access: "none", "system", "session" or "both"
            dbusproxy: Configuration for DBUS proxy, these will be added line by line to the actual config
            pulseaudio: Enable pulseaudio access in the container
            devices: Allow the given devices in the container
            ctype: Container type, either cow (copy on write) or bind (full bind-mount of the FS)
            root: Root to use for the container (default is /)
            bind: List of paths to bind-mount in the container
            restrict: List of paths to empty in the container (bind-mounts an empty directory)
            cow: List of paths to mount as copy-on-write
        """

        if not os.geteuid() == 0:
            raise Exception("You must run this module as root")

        # Backward compatibility
        if type(network) == bool:
            if network:
                network = "direct"
            else:
                network = "none"

        # Input validation
        if path:
            if type(path) != str:
                raise Exception("path must be a string")
            else:
                if not os.path.exists(path):
                    raise Exception("path '%s' doesn't exist" % path)

        if fstype not in ("ext4", "tmpfs"):
            raise Exception("Unsupported fstype: %s" % fstype)

        if fssize and type(fssize) != int:
            raise Exception("fssize must be an integer")

        if network not in ("none", "direct", "filtered"):
            raise Exception("Unsupported network mode: %s" % network)

        if xserver not in ("none", "isolated", "direct"):
            raise Exception("Unsupported xserver mode: %s" % xserver)

        if dbus not in ("none", "system", "session", "both"):
            raise Exception("Unsupported dbus mode: %s" % dbus)

        if type(dbusproxy) != list:
            raise Exception("dbusproxy must be a list")

        if type(pulseaudio) != bool:
            raise Exception("pulseaudio must be a boolean")

        if type(devices) != list:
            raise Exception("devices must be a list")

        if ctype not in ("cow", "bind"):
            raise Exception("Unsupported container type: %s" % ctype)

        if root and type(root) != str:
            raise Exception("root must be a string")
        if not os.path.exists(root):
            raise Exception("root must be a valid path")

        if type(bind) != list:
            raise Exception("bind must be a list")

        if type(restrict) != list:
            raise Exception("restrict must be a list")

        if type(cow) != list:
            raise Exception("cow must be a list")


        # Deal with storage path
        if not path:
            if not os.path.exists(os.path.expanduser("~/.arkose/")):
                os.mkdir(os.path.expanduser("~/.arkose/"))
            path = os.path.expanduser("~/.arkose/")
        path = tempfile.mkdtemp(dir=path)
        os.chmod(path, 0o755)

        # Set default fssize if not set by the user
        if not fssize:
            if fstype == "ext4":
                fssize = 2000
            elif fstype == "tmpfs":
                fssize = self.__default_memory()

        # Export all the parameters as properties
        self.path = path
        self.fstype = fstype
        self.fssize = fssize
        self.network = network
        self.xserver = xserver
        self.dbus = dbus
        self.dbusproxy = dbusproxy
        self.pulseaudio = pulseaudio
        self.devices = devices
        self.ctype = ctype
        self.root = root
        self.bind = bind
        self.restrict = restrict
        self.cow = cow

        # Get some data on the user
        self.user = pwd.getpwuid(int(os.getenv("PKEXEC_UID", os.getenv("SUDO_UID", os.getenv("UID", 1000)))))

        # If anything fails, cleanup and then raise the exception
        try:
            self.__init_network()
            self.__init_fs()
            self.__init_pulseaudio()
            self.__init_xserver()
            self.__init_dbus()
            self.__init_devices()
            self.__generate_lxcconfig()
        except:
            self.cleanup()
            raise

    def __init_fs(self):
        """
            ONLY CALLED FROM __init__()
            Initializes the filesystem and sets self.path accordingly
        """

        self.name = "arkose-%s" % os.path.split(self.path)[1]
        workdir = os.path.join(self.path, "workdir")
        os.mkdir(workdir)

        # Create the workdir filesystem
        if self.fstype == "ext4":
            image = os.path.join(self.path, "root.img")

            # Create the partition
            sparse = open(image, "w+")
            sparse.truncate(1024*1024*self.fssize)
            sparse.close()

            # Create the filesystem
            cmd = ['mkfs.ext4', '-O', '^has_journal', '-F', '-q', image]
            mkfs = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            if mkfs.wait() != 0:
                print(mkfs.stdout.read())
                print(mkfs.stderr.read())
                raise Exception("Unable to create ext4 partition at: %s" % image)

            # Mount it
            self.__mount(image, workdir, "ext4", "loop")


        elif self.fstype == "tmpfs":
            # Mount it
            self.__mount("none", workdir, "tmpfs", "size=%s" % str(1024*1024*self.fssize))

        # Mount a cgroup filesystem
        cgroup = os.path.join(workdir, "cgroup")
        os.mkdir(cgroup)

        try:
            self.__mount("cgroup", cgroup, "cgroup", verbose=False)
        except Exception:
            os.rmdir(cgroup)

        # Create the target
        target = os.path.join(workdir, "target")
        os.mkdir(target)

        # Mount root FS
        if self.ctype == "cow":
            # Mount copy-on-write overlay
            root_cow = os.path.join(workdir, "root_cow")
            os.mkdir(root_cow)
            try:
                try:
                    self.__mount("none", target, "overlayfs", "upperdir=%s,lowerdir=%s" % (root_cow, self.root))
                except:
                    self.__mount("none", target, "aufs", "br=%s:%s" % (root_cow, self.root))
            except:
                raise Exception("No copy-on-write filesystem available")

        elif self.ctype == "bind":
            # Bind mount root device
            self.__mount(self.root, target, "none", "bind,ro")

        # Mount an empty directory on top of home unless it's already defined
        if not self.user.pw_dir in self.cow+self.bind+self.restrict:
            # Make sure the home directory exists
            homedir_target = os.path.join(target, self.user.pw_dir[1:])
            if not os.path.exists(homedir_target):
                os.makedirs(homedir_target, 0o755)

            # We can't just append it to self.restrict as we might override
            # other mounts (copy-on-write or bind) in sub-directories
            mount = self.user.pw_dir

            # Remove initial / if present
            if mount[0] == '/':
                mount = mount[1:]

            new_restrict = os.path.join(workdir, "%s_restrict" % mount.replace("/", "_"))
            os.mkdir(new_restrict)

            new_target = os.path.join(target, mount)
            self.__mount(new_restrict, new_target, "none", "bind")

            os.chown(homedir_target, self.user.pw_uid, self.user.pw_gid)
            os.chmod(homedir_target, 0o700)

        # Make DNS resolving work when using resolvconf
        if os.path.exists("/run/resolvconf/"):
            if self.ctype == "cow":
                self.cow.append("/run/resolvconf")
            elif self.ctype == "bind":
                self.bind.append("/run/resolvconf")

        # Setup other copy-on-write mounts
        for mount in self.cow:
            # If source doesn't exist, create it
            if not os.path.exists(mount):
                os.makedirs(mount, 0o755)
                os.chown(mount, self.user.pw_uid, self.user.pw_gid)

            # Remove initial / if present
            if mount[0] == '/':
                mount = mount[1:]

            new_cow = os.path.join(workdir, "%s_cow" % mount.replace("/", "_"))
            os.mkdir(new_cow)

            new_target = os.path.join(target, mount)
            if not os.path.exists(new_target):
                os.mkdir(new_target)
            try:
                try:
                    self.__mount("none", new_target, "overlayfs", "upperdir=%s,lowerdir=/%s" % (new_cow, mount))
                except:
                    self.__mount("none", new_target, "aufs", "br=%s:/%s" % (new_cow, mount))
            except:
                raise Exception("No copy-on-write filesystem available")

        # Setup other bind mounts
        for mount in self.bind:
            # If source doesn't exist, create it
            if not os.path.exists(mount):
                os.makedirs(mount, 0o755)
                os.chown(mount, self.user.pw_uid, self.user.pw_gid)

            # Remove initial / if present
            if mount[0] == '/':
                mount = mount[1:]

            new_target = os.path.join(target, mount)
            if not os.path.exists(new_target):
                if os.path.isdir("/%s" % mount):
                    os.mkdir(new_target)
                else:
                    if not os.path.exists(os.path.dirname(new_target)):
                        os.mkdir(os.path.dirname(new_target))
                    open(new_target, 'w').close()
            self.__mount("/%s" % mount, new_target, "none", "bind")

        # Setup other restricted mounts
        for mount in self.restrict:
            # Remove initial / if present
            if mount[0] == '/':
                mount = mount[1:]

            new_restrict = os.path.join(workdir, "%s_restrict" % mount.replace("/", "_"))
            os.mkdir(new_restrict)

            new_target = os.path.join(target, mount)
            if not os.path.exists(new_target):
                os.mkdir(new_target)
            self.__mount(new_restrict, new_target, "none", "bind")

        # Export some more parameters as properties
        self.workdir = workdir
        self.target = target

    def __init_xserver(self):
        """
            ONLY CALLED FROM __init__()
            Make X work in the container
        """
        if self.olddisplay:
            display = self.olddisplay.split(":")

        if self.xserver == "none":
            os.unsetenv("DISPLAY")
            return

        # Make sure we have some kind of X server
        if not display:
            raise Exception("No DISPLAY variable in environment")

        if self.xserver == "isolated":
            if not os.path.exists("/usr/bin/xpra"):
                raise Exception("Unable to find xpra")

            xdisplay = 0
            while 1:
                if not os.path.exists("/tmp/.X11-unix/X%s" % xdisplay):
                    break
                xdisplay += 1

            self.xpra = xdisplay

            # Start the X server
            cmd = ["xpra", "start", ":%s" % xdisplay]
            subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE).wait()

            # Wait for X to start as xpra returns before it's ready
            time.sleep(2)

            # Attach to it
            cmd = ["xpra", "attach", ":%s" % xdisplay]
            subprocess.Popen(cmd)

            # Set DISPLAY to new value
            os.environ["DISPLAY"] = ":%s" % xdisplay
            display = ["", str(xdisplay)]

            # Allow everyone
            cmd = ["xhost", "+"]
            subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE).wait()

        # If X server is accessed over the network, don't try to bind- mount the socket
        if display[0]:
            return

        socket_source = "/tmp/.X11-unix/X%s" % display[1].split(".")[0]
        if not os.path.exists(socket_source):
            raise Exception("Unable to find X socket: %s" % socket_source)

        # Check if the socket already exists (when using aufs or bind-mount)
        # if it doesn't, then "touch" it
        if not os.path.exists(os.path.join(self.target, "tmp/.X11-unix")):
            os.makedirs(os.path.join(self.target, "tmp/.X11-unix"), 0o755)

        socket_target = os.path.join(self.target, socket_source[1:])
        if not os.path.exists(socket_target):
            open(socket_target, 'w').close()

        # Mount-bind the socket
        self.__mount(socket_source, socket_target, "none", "bind")

    def __init_dbus(self):
        """
            ONLY CALLED FROM __init__()
            Make dbus work in the container
        """

        if self.dbus == "system" or self.dbus == "both":
            for system_bus in ("/run/dbus", "/var/run/dbus"):
                if not os.path.exists(system_bus):
                    continue

                # Create the target if missing
                dbus_target = os.path.join(self.target, system_bus[1:])
                if not os.path.exists(dbus_target):
                    os.mkdir(dbus_target)

                # Mount the dbus directory in the target
                self.__mount(system_bus, dbus_target, "none", "bind")

                break

            else:
                # Unable to find a system bus
                raise Exception("Unable to find a DBUS system bus.")

        if self.dbus == "session" or self.dbus == "both":
            # Get the dbus information from the environment
            current_dbus = os.getenv("DBUS_SESSION_BUS_ADDRESS", None)
            if not current_dbus:
                raise Exception("No valid DBUS_SESSION_BUS_ADDRESS in environment")

            # Create the target if missing
            dbus_target = os.path.join(self.target, "tmp/%s-dbus_session" % self.name)
            self.files.append(dbus_target)
            if os.path.exists(dbus_target):
                os.remove(dbus_target)

            # Generate the config
            proxy_config = os.path.join(self.path, "dbusproxy_config")
            config = open(proxy_config,"w+")
            for rule in self.dbusproxy:
                if re.match("^.*;.*;.*;.*$",rule):
                    config.write("%s\n" % rule)
            config.close()

            cmd = ["su", str(self.user.pw_name), "-c"]

            # Start the socket proxy between the current socket and the target socket
            if os.path.exists("dbus-proxy/arkose-dbus-proxy"):
                cmd.append("dbus-proxy/arkose-dbus-proxy %s %s" % (dbus_target, proxy_config))
            else:
                cmd.append("arkose-dbus-proxy %s %s" % (dbus_target, proxy_config))
            self.dbus_proxy = subprocess.Popen(cmd)

            # Generate the new environment variable
            os.environ["DBUS_SESSION_BUS_ADDRESS"] = "unix:path=/tmp/%s-dbus_session" % self.name

            # Check if we are currently using a dbus keyring
            keyring_current = os.path.join(self.user.pw_dir, ".dbus-keyrings")
            if not os.path.exists(keyring_current):
                # No keyring? Let's hope it's because we don't need it
                return

            # Create the target if missing
            keyring_target = os.path.join(self.target, self.user.pw_dir[1:], ".dbus-keyrings")
            if not os.path.exists(keyring_target):
                os.mkdir(keyring_target)

            # Mount the keyring and set the permissions
            self.__mount(keyring_current, keyring_target, "none", "bind")
            os.chmod(keyring_target, 0o700)
            os.chown(keyring_target, self.user.pw_uid, self.user.pw_gid)

    def __init_pulseaudio(self):
        """
            ONLY CALLED FROM __init__()
            Make pulseaudio work in the container
        """

        if self.pulseaudio == False:
            return

        # If PULSE_SERVER is in the environment, just send it as is
        if os.getenv("PULSE_SERVER", None):
            if self.network == "none":
                raise Exception("Pulseaudio is enabled with a remote server but no network.")
            return


        pulse_target = os.path.join(self.target, "tmp/%s-pulse" % self.name)
        self.files.append(pulse_target)
        cmd = ["su", str(self.user.pw_name), "-c", "pactl load-module module-native-protocol-unix socket=%s auth-cookie-enabled=0" % pulse_target]
        pactl = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        pactl.wait()
        self.pulse_module = pactl.stdout.read().strip()

        config_path = os.path.join(self.path, "pulse-config")
        config_target = os.path.join(self.target, "etc/pulse/client.conf")
        config_targetdir = os.path.join(self.target, "etc/pulse")
        config = open(config_path, "w")
        config.write("disable-shm=yes\n")
        config.close()

        if not os.path.exists(config_targetdir):
            os.mkdir(config_targetdir)

        self.__mount(config_path, config_target, "none", "bind")

        os.environ["PULSE_SERVER"] = "/tmp/%s-pulse" % self.name

    def __init_devices(self):
        """
            ONLY CALLED FROM __init__()
            Make specific devices work in the container
        """

        for device in self.devices:
            target = os.path.join(self.target, device[1:])
            if not os.path.exists(target):
                open(target, "w").close()
            self.__mount(device, target, "none", "bind")
            os.chmod(target, 0o666)
            stat = os.stat(device)
            lxcdev = "c %s:%s rwm" % (stat.st_rdev / 256, stat.st_rdev % 256)
            self.lxc_devices.append(lxcdev)

    def __init_network(self):
        """
            ONLY CALLED FROM __init__()
            Configure the network
        """

        if self.network != "filtered":
            return

        # Get a new subnet
        self.subnet = self.__get_subnet()

        # Immediately route it to loopback to avoid race conditions
        cmd = ['ip', 'route', 'add', "%s/31" % self.subnet[2], 'dev', 'lo']
        route = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        route.wait()

    def __mount(self, src, dst, fstype, fsoptions=None, verbose=True):
        """ Wrapper around mount adding mountpoints to the list of FS to unmount """

        if fsoptions:
            cmd = ['mount', '-t', fstype, src, dst, "-o", fsoptions]
        else:
            cmd = ['mount', '-t', fstype, src, dst]

        mount = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        if mount.wait() != 0:
            if verbose:
                print(mount.stdout.read())
                print(mount.stderr.read())
            raise Exception("Unable to mount: %s" % dst)

        self.mounts.append(dst)


    def __get_subnet(self):
        """
            Return the two IPs of a non-routed /31 subnet in 169.254.0.0/16 
            and the subnet itself
        """

        def ip_to_hex(ipaddr):
            """ Convert an IP to its hex representation """
            ip_parts = ipaddr.split(".")
            hexip = ""
            for part in reversed(ip_parts):
                hexpart = str(hex(int(part))).replace("0x", "")
                if len(hexpart) == 1:
                    hexpart = "0%s" % hexpart
                hexip += hexpart
            return hexip.replace("0x", "").upper()

        networks = []
        for line in open("/proc/net/route").readlines():
            fields = line.split()
            networks.append(fields[1])

        # Iterate through link-local
        for netrange in range(1, 254):
            for subnet in netaddr.IPNetwork("169.254.%s.0/24" % netrange).subnet(31):
                net = str(subnet.network)
                if ip_to_hex(net) not in networks:
                    dst = str(netaddr.IPAddress(subnet.first+1))
                    src = str(netaddr.IPAddress(subnet.last-1))
                    return (src, dst, net)

        return False

    def __generate_lxcconfig(self):
        """
            ONLY CALLED FROM __init__()
            Generates LXC's configuration
        """

        self.config = os.path.join(self.path, "config")

        config = open(self.config, "w+")
        config.write("""## Base config
lxc.utsname = %s
lxc.tty = 4
lxc.pts = 1024
lxc.rootfs = %s
lxc.console = none

## /dev filtering
lxc.cgroup.devices.deny = a
# /dev/null and zero
lxc.cgroup.devices.allow = c 1:3 rwm
lxc.cgroup.devices.allow = c 1:5 rwm
# /dev/{,u}random
lxc.cgroup.devices.allow = c 1:9 rwm
lxc.cgroup.devices.allow = c 1:8 rwm
# /dev/ptmx
lxc.cgroup.devices.allow = c 5:2 rwm

# /dev/tty
lxc.cgroup.devices.allow = c 5:0 rwm

# /dev/pts/*
lxc.cgroup.devices.allow = c 136:* rwm
""" % (self.name, self.target))
        if self.lxc_devices:
            config.write("# additional devices\n")
        for device in self.lxc_devices:
            config.write("lxc.cgroup.devices.allow = %s\n" % device)

        if self.network == "none":
            config.write("""
## Networking
lxc.network.type = empty
""")
        elif self.network == "filtered":
            interface = os.path.split(self.path)[1]

            # Create the network up script
            netup_path = os.path.join(self.path, "network-up")
            netup = open(netup_path, "w+")
            netup.write("""#!/bin/sh
ip addr add %s/31 dev %s
ip route del %s/31 dev lo
echo 1 > /proc/sys/net/ipv4/ip_forward
iptables -t nat -A POSTROUTING -s %s -j MASQUERADE
""" % (self.subnet[0], interface, self.subnet[2], self.subnet[1]))
            netup.close()
            os.chmod(netup_path, 0o755)

            # Create the network down script
            netdown_path = os.path.join(self.path, "network-down")
            netdown = open(netdown_path, "w+")
            netdown.write("""#!/bin/sh
ip route add %s/31 dev lo
iptables -t nat -D POSTROUTING -s %s -j MASQUERADE
""" % (self.subnet[2], self.subnet[1]))
            netdown.close()
            os.chmod(netdown_path, 0o755)

            config.write("""
## Networking
lxc.network.type = veth
lxc.network.flags = up
lxc.network.veth.pair = %s
lxc.network.ipv4 = %s/31
lxc.network.script.up = %s
""" % (interface, self.subnet[1], netup_path))

        config.close()

    def __default_memory(self):
        """ Returns 50% of free RAM """

        meminfo = open("/proc/meminfo", "r")
        meminfo_dict = {}
        for line in meminfo.readlines():
            values = line.split(":")
            meminfo_dict[values[0]] = values[1].split()[0].strip()
        meminfo.close()
        return ((int(meminfo_dict['MemFree']) + int(meminfo_dict['Buffers']) + int(meminfo_dict['Cached'])) / 1024 / 2)

    def run_command(self, command):
        """ Run "cmd" (list) in the container """

        # Avoid python going into the background
        signal.signal(signal.SIGTTOU, signal.SIG_IGN)

        # Generate terminal initialization script
        self.terminal_script = os.path.join(self.target, "tmp/%s-terminal" % self.name)
        self.files.append(self.terminal_script)
        script = open(self.terminal_script, "w+")

        if self.network == "filtered":
            network = "ip route add default via %s" % self.subnet[0]
        else:
            network = ""

        script.write("""#!/bin/sh
[ -d "%s" ] && cd %s
%s
reset -I
%s
""" % (os.getcwd(), os.getcwd(), network, command))
        script.close()

        cmd = ["lxc-execute", "-n", self.name, "-f", self.config]
        cmd += ["sh", "/tmp/%s-terminal" % self.name]

        subprocess.Popen(cmd).wait()

        # Drop the network configuration
        if self.network == "filtered":
            subprocess.Popen(os.path.join(self.path, "network-down")).wait()

        # Kill any remaining trace of the container
        cmd = ["lxc-stop", "-n", self.name]
        subprocess.Popen(cmd).wait()

    def cleanup(self):
        """ Unmount everything and remove temporary work dir """

        # Restore DISPLAY, must happen before disabling pulse
        if self.olddisplay:
            os.environ["DISPLAY"] = self.olddisplay
        else:
            if "DISPLAY" in os.environ:
                os.environ.pop("DISPLAY")

        if self.pulse_module:
            os.environ.pop("PULSE_SERVER")
            cmd = ["su", str(self.user.pw_name), "-c", "pactl unload-module %s" % self.pulse_module]
            pactl = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            pactl.wait()

            # Apparently in some cases race conditions can appear ...
            time.sleep(2)

        if self.dbus_proxy:
            self.dbus_proxy.terminate()
            self.dbus_proxy.wait()

        # Cleanup any file we created and that won't be removed automatically
        for path in self.files:
            if os.path.exists(path):
                os.remove(path)

        # Start unmounting everything, hopefully in the right order
        for path in reversed(self.mounts):
            cmd = ['umount', path]
            umount = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            if umount.wait() != 0:
                print(umount.stdout.read())
                print(umount.stderr.read())
                raise Exception("Unable to umount partition: %s" % path)
            self.mounts.remove(path)

        if os.path.exists(self.path):
            shutil.rmtree(self.path)

        if self.xpra:
            cmd = ["xpra", "stop", ":%s" % self.xpra]
            subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE).wait()

        if self.network == "filtered":
            # Just in case it's still there, Remove the firewall rules
            cmd = ["iptables", "-t", "nat", "-D", "POSTROUTING", "-s", self.subnet[1], "-j", "MASQUERADE"]
            subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE).wait()

            # Remove the temporary route
            cmd = ['ip', 'route', 'del', "%s/31" % self.subnet[2], 'dev', 'lo']
            subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE).wait()
