
//#define LOCAL_DEBUG
#include "debug.h"

#include "config.h"
#include "fileitem.h"
#include "header.h"
#include "acfg.h"
#include "acbuf.h"
#include "fileio.h"
#include "cleaner.h"

#include <errno.h>
#include <algorithm>

using namespace MYSTD;

#define MAXTEMPDELAY 27

header const & fileitem::GetHeaderUnlocked()
{
	return m_head;
}

string fileitem::GetHttpMsg()
{
	setLockGuard;
	if(m_head.frontLine.length()>9)
		return m_head.frontLine.substr(9);
	return m_head.frontLine;
}

time_t fileitem::m_nEarliestExpiration(0);

fileitem::fileitem(MYSTD::string sPath) :
	condition(),
	m_nIncommingCount(0),
	m_nSizeSeen(0),
	m_sPath(acfg::cachedir+SZPATHSEP+sPath),
	m_sKey(sPath),
	m_bCheckFreshness(true),
	m_bHeadOnly(false),
	m_bAllowStoreData(true),
	m_nSizeChecked(0),
	m_filefd(-1),
	m_nDlRefsCount(0),
	status(FIST_FRESH),
	m_nTimeExpireAt(0)
{

	// not using hexmap, just converting only few allowed chars

	for (int i = 0; i < int(m_sPath.length()) - 3; i++)
	{
		if (m_sPath[i] == '%')
		{
			switch (m_sPath[i + 1])
			{
			case '7':
				switch (m_sPath[i + 2])
				{
				case 'E':
				case 'e':
					m_sPath.replace(i, 3, "~", 1);
					break;
				}
				break;
			case '5':
				switch (m_sPath[i + 2])
				{
				case 'B':
				case 'b':
					m_sPath.replace(i, 3, "[", 1);
					break;
				case 'D':
				case 'd':
					m_sPath.replace(i, 3, "]", 1);
					break;
				}
				break;
			default:
				continue;
			}
		}
	}
	//ldbg("item created, m_sPath: "<< m_sPath);
}

void fileitem::IncDlRefCount()
{
	setLockGuard;
	m_nDlRefsCount++;
}

void fileitem::DecDlRefCount(const string &sReason)
{
	setLockGuard;
	
	notifyAll();

	m_nDlRefsCount--;
	if(m_nDlRefsCount>0)
		return; // someone will care...
	
	// ... otherwise: the last downloader disappeared, need to tell observers

	if (status<FIST_COMPLETE)
	{
		status=FIST_ERROR;
		m_head.clear();
		m_head.frontLine=string("HTTP/1.1 ")+sReason;
		m_head.type=header::ANSWER;

		if (acfg::debug>1)
			aclog::misc(string("Download of ")+m_sKey+" aborted");
	}
	checkforceclose(m_filefd);
}

uint64_t fileitem::GetTransferCount()
{
	setLockGuard;
	uint64_t ret=m_nIncommingCount;
	m_nIncommingCount=0;
	return ret;
}

int fileitem::GetFileFd() {
	LOGSTART("fileitem::GetFileFd");
	setLockGuard;
	if(status<FIST_DLGOTHEAD)
		return -1;
	ldbg("Opening " << m_sPath);
	int fd=open(m_sPath.c_str(), O_RDONLY);
	
#ifdef HAVE_FADVISE
	// optional, experimental
	if(status==FIST_COMPLETE)
		posix_fadvise(fd, 0, m_nSizeChecked, POSIX_FADV_SEQUENTIAL);
#endif

	return fd;
}

off_t GetFileSize(cmstring & path, off_t defret)
{
	struct stat stbuf;
	return (0==::stat(path.c_str(), &stbuf)) ? stbuf.st_size : defret;
}


void fileitem::ResetCacheState()
{
	setLockGuard;
	m_nSizeSeen = 0;
	m_nSizeChecked = 0;
	status = FIST_FRESH;
	m_bAllowStoreData = true;
	m_head.clear();
}

FiStatus fileitem::Setup(bool bCheckFreshness)
{
	LOGSTART2("fileitem::Setup", bCheckFreshness);

	setLockGuard;

	if(status>FIST_FRESH)
		return status;
	status=FIST_INITED;
	m_bCheckFreshness = bCheckFreshness;
	
	if(m_head.LoadFromFile(m_sPath+".head") >0 && m_head.type==header::ANSWER )
	{
		if(200 != m_head.getStatus())
			goto error_clean;
		
		LOG("good head");

		m_nSizeSeen=GetFileSize(m_sPath, 0);

		// some plausibility checks
		if(m_bCheckFreshness)
		{
			const char *p=m_head.h[header::LAST_MODIFIED];
			if(!p)
				goto error_clean; // suspicious, cannot use it
			LOG("check freshness, last modified: " << p );
		}
		else
		{
			// non-volatile files, so could accept the length, do some checks first
			const char *pContLen=m_head.h[header::CONTENT_LENGTH];
			if(pContLen)
			{
				off_t nContLen=atoofft(pContLen); // if it's 0 then we assume it's 0
				
				// file larger than it could ever be?
				if(nContLen < m_nSizeSeen)
					goto error_clean;
				
				LOG("Content-Length has a sane range");

				// is it complete? and 0 value also looks weird, try to verify
				if(m_nSizeSeen == nContLen && nContLen>0)
				{
					status=FIST_COMPLETE;
					m_nSizeChecked=m_nSizeSeen;
				}
			}
			else
			{
				// no content length known, assume it's ok
				m_nSizeChecked=m_nSizeSeen;
			}				
		}
	}
	else // -> no .head file
	{
		// maybe there is some left-over without head file?
		// Don't thrust volatile data, but otherwise try to reuse?
		if(!bCheckFreshness)
			m_nSizeSeen=GetFileSize(m_sPath, 0);
	}
	LOG("resulting status: " << status);
	return status;

	error_clean:
			unlink((m_sPath+".head").c_str());
			m_head.clear();
			m_nSizeSeen=0;
			status=FIST_INITED;
			return status; // unuseable, to be redownloaded
}

inline void _LogWithErrno(const char *msg, const string & sFile)
{
	aclog::errnoFmter f;
	aclog::err( (sFile+" storage error ["+msg+"]: "+f).c_str());
}

bool fileitem::DownloadStartedStoreHeader(const header & h, const char *pNextData)
{
	LOGSTART("fileitem::DownloadStartedStoreHeader");

	setLockGuard;

	USRDBG(5, "Download started, storeHeader for " << m_sKey << ", current status: " << status);
	
	if(status > FIST_DLPENDING) // already started? error? whatever
		return false;
	
	if(status<FIST_DLGOTHEAD)
		status=FIST_DLGOTHEAD; // assume that for now, may become error state...
	
	if(m_bCheckFreshness)
		m_nTimeExpireAt = time(0) + MAXTEMPDELAY;

	m_nIncommingCount+=h.m_nEstimLength;

	// optional optimization: hints for the filesystem resp. kernel
	off_t hint_start(0), hint_length(0);
	
	// status will change, most likely... ie. BOUNCE action
	notifyAll();

#define SETERROR(x) { m_bAllowStoreData=false; m_head.frontLine="HTTP/1.1 "; \
	m_head.frontLine+=x; status=FIST_ERROR; _LogWithErrno(x, m_sPath); }
#define SETERRORKILLFILE(x) { SETERROR(x); goto kill_file; }
#define BOUNCE(x) { SETERROR(x); return false; }

	m_head.set(header::XORIG, h.h[header::XORIG]);

	string sHeadPath=m_sPath+".head";
	
	switch(h.getStatus())
	{
	case 200:
	{
		m_nSizeChecked=0;
		m_head=h;
		const char *p=h.h[header::CONTENT_LENGTH];
		hint_length=p ? atoofft(p) : 0;
		break;
	}
	case 206:
	{
		if(m_nSizeSeen<=0)
		{
			// wtf? Cannot have requested partial content
			BOUNCE("500 Unexpected Partial Response");
		}
		/*
		 * Range: bytes=453291-
		 * ...
		 * Content-Length: 7271829
		 * Content-Range: bytes 453291-7725119/7725120
		 */
		const char *p=h.h[header::CONTENT_RANGE];
		if(!p)
			BOUNCE("500 Missing Content-Range in Partial Response");
		off_t myfrom, myto, mylen;
		int n=sscanf(p, "bytes " OFF_T_FMT "-" OFF_T_FMT "/" OFF_T_FMT, &myfrom, &myto, &mylen);
		if(n<=0)
			n=sscanf(p, "bytes=" OFF_T_FMT "-" OFF_T_FMT "/" OFF_T_FMT, &myfrom, &myto, &mylen);
		
		ldbg("resuming? n: "<< n << " und myfrom: " <<myfrom << 
				" und myto: " << myto << " und mylen: " << mylen);
		if(n!=3  // check for nonsense
				|| myfrom != m_nSizeSeen-1
				|| myfrom<0 || mylen<0)
			BOUNCE("500 Server reports illegal range");
	
		m_nSizeChecked=myfrom;
		
		hint_start=myfrom;
		hint_length=mylen;

		m_head=h;
		m_head.frontLine="HTTP/1.1 200 OK";
		m_head.del(header::CONTENT_RANGE);
		m_head.set(header::CONTENT_LENGTH, mylen);

		// special optimization; if "-1 trick" was used then maybe don't reopen that file for writing later
		if(m_bCheckFreshness && pNextData && m_nSizeSeen == mylen && m_nSizeChecked == mylen-1)
		{
			int fd=open(m_sPath.c_str(), O_RDONLY);
			if(fd>=0)
			{
				if(m_nSizeChecked==lseek(fd, m_nSizeChecked, SEEK_SET))
				{
					char c;
					if(1 == read(fd, &c, 1) && c == *pNextData)
					{
						aclog::err("known data hit, don't write to...");
						aclog::err(m_sPath);
						m_bAllowStoreData=false;
						m_nSizeChecked=mylen;
					}
				}
				forceclose(fd);
			}
		}
		break;
	}
	case 416:
		// that's always bad; it cannot have been complete before (the -1 trick)
		// -> kill cached file ASAP
		m_bAllowStoreData=false;
		SETERRORKILLFILE("503 Server disagrees on file size, cleaning up");
		break;
// use the Location header and be sure the server does not mess around with Location and other codes
	case 301:
	case 302:
	case 303:
	case 307:
		m_head.set(header::LOCATION, h.h[header::LOCATION]);
		// fall-through
	default:
		m_bAllowStoreData=false;
		// have a clean header with just the error message
		m_head.frontLine=h.frontLine;
		m_head.type=header::ANSWER;
		m_head.set(header::CONTENT_LENGTH, "0");
		if(status>FIST_DLGOTHEAD)
		{
			// oh shit. Client may have already started sending it. Prevent such trouble in future.
			unlink(sHeadPath.c_str());
		}
	}

	if(acfg::debug>1)
		aclog::misc(string("Download of ")+m_sKey+" started");

	if(m_bAllowStoreData)
	{

		// using adaptive Delete-Or-Replace-Or-CopyOnWrite strategy
		
		// First opening the file first to be sure that it can be written. Header storage is the critical point,
		// every error after that leads to full cleanup to not risk inconsistent file contents 
		
		int flags = O_WRONLY | O_CREAT | O_BINARY;
		struct stat stbuf;
				
		mkbasedir(m_sPath);
		m_filefd=open(m_sPath.c_str(), flags, acfg::fileperms);
		ldbg("file opened?! returned: " << m_filefd);
		
		// self-recovery from cache poisoned with files with wrong permissions
		if (m_filefd<0)
		{
			if(m_nSizeChecked>0) // OOOH CRAP! CANNOT APPEND HERE! Do what's still possible.
			{
				string temp=m_sPath+".tmp";
				if(FileCopy(m_sPath, temp) && 0==unlink(m_sPath.c_str()) )
				{
					if(0!=rename(temp.c_str(), m_sPath.c_str()))
						BOUNCE("503 Cannot rename files");
					
					// be sure about that
					if(0!=stat(m_sPath.c_str(), &stbuf) || stbuf.st_size!=m_nSizeSeen)
						BOUNCE("503 Cannot copy file parts, filesystem full?");
					
					m_filefd=open(m_sPath.c_str(), flags, acfg::fileperms);
					ldbg("file opened after copying around: ");
				}
				else
					BOUNCE((tSS()<<"503 Cannot store or remove files in "
							<< GetDirPart(m_sPath)).c_str());
			}
			else
			{
				unlink(m_sPath.c_str());
				m_filefd=open(m_sPath.c_str(), flags, acfg::fileperms);
				ldbg("file force-opened?! returned: " << m_filefd);
			}
		}
		
		if (m_filefd<0)
		{
			aclog::errnoFmter efmt("503 Cache storage error - ");
#ifdef DEBUG
			BOUNCE((efmt+m_sPath).c_str());
#else
			BOUNCE(efmt.c_str());
#endif
		}
		
		if(0!=fstat(m_filefd, &stbuf) || !S_ISREG(stbuf.st_mode))
			SETERRORKILLFILE("503 Not a regular file");
		
		// crop, but only if the new size is smaller. MUST NEVER become larger (would fill with zeros)
		if(m_nSizeChecked <= m_nSizeSeen)
		{
			if(0==ftruncate(m_filefd, m_nSizeChecked))
				fdatasync(m_filefd);
			else
				SETERRORKILLFILE("503 Cannot change file size");
		}
		else if(m_nSizeChecked>m_nSizeSeen) // should never happen and caught by the checks above
			SETERRORKILLFILE("503 Internal error on size checking");
		// else... nothing to fix since the expectation==reality

		falloc_helper(m_filefd, hint_start, hint_length);
		
		ldbg("Storing header as "+sHeadPath);
		int count=m_head.StoreToFile(sHeadPath);
		
		// unlink and retry
		if(count<0)
		{
			unlink(sHeadPath.c_str());
			count=m_head.StoreToFile(sHeadPath);
		}

		if(count<0)
			SETERRORKILLFILE( (-count!=ENOSPC ? "503 Cache storage error" : "503 OUT OF DISK SPACE"));
			
		// double-check the sane state
		if(0!=fstat(m_filefd, &stbuf) || stbuf.st_size!=m_nSizeChecked)
			SETERRORKILLFILE("503 Inconsistent file state");
			
		if(m_nSizeChecked!=lseek(m_filefd, m_nSizeChecked, SEEK_SET))
			SETERRORKILLFILE("503 IO error, positioning");
	}
	
	status=FIST_DLGOTHEAD;
	return true;

	kill_file:
	if(m_filefd>=0)
	{
#if _POSIX_SYNCHRONIZED_IO > 0
		fsync(m_filefd);
#endif
		forceclose(m_filefd);
	}

	LOG("Deleting " << m_sPath);
	unlink(m_sPath.c_str());
	unlink(sHeadPath.c_str());
	
	status=FIST_ERROR;
	return false;
}

bool fileitem::StoreFileData(const char *data, unsigned int size)
{

	setLockGuard;

	LOGSTART2("fileitem::StoreFileData", "status: " << status << ", size: " << size);

	// something might care, most likely... also about BOUNCE action
	notifyAll();
	
	m_nIncommingCount+=size;
	
	if(status >= FIST_ERROR || status < FIST_DLGOTHEAD)
		return false;
	
	if (size==0)
	{
		if(FIST_COMPLETE == status)
		{
			LOG("already completed");
			return true;
		}

		status=FIST_COMPLETE;

		if(acfg::debug>1)
			aclog::misc(tSS()<<"Download of "<<m_sKey<<" finished");
		
		// we are done! Fix header from chunked transfers?
		if (m_filefd>=0 && ! m_head.h[header::CONTENT_LENGTH])
		{
			m_head.set(header::CONTENT_LENGTH, m_nSizeChecked);
			m_head.StoreToFile(m_sPath+".head");
		}
	}
	else
	{
		status = FIST_DLRECEIVING;

		if (!m_bAllowStoreData)
			return true;

		if (m_filefd>=0)
		{
			while(size>0)
			{
				int r=write(m_filefd, data, size);
				if(r<0)
				{
					if(EINTR==errno || EAGAIN==errno)
						continue;
					aclog::errnoFmter efmt("503 ");
					BOUNCE(efmt.c_str());
				}
				m_nSizeChecked+=r;
				size-=r;
				data+=r;
			}
			
		}
	}
	return true;
}

fileitem::~fileitem()
{
	setLockGuard;
	m_head.clear();
	checkforceclose(m_filefd);
}


struct tFileRefEntry
{
	int nRefCount;
	tFileItemPtr ref;
	bool m_bNeedsDelayedExpiration;
	tFileRefEntry() : nRefCount(0), m_bNeedsDelayedExpiration(false) { };
};
static map<string, tFileRefEntry> mapItems;
lockable mapLck;

void fileitem::Unreg()
{
	LOGSTART("fileitem::Unreg");

   lockguard managementLock(mapLck); 
	
   std::map<string,tFileRefEntry>::iterator entry = mapItems.find(m_sKey);
   if(mapItems.end() == entry)
   {
	   aclog::err("INTERNAL ERROR, attempt to unregister non-existing download item");
	   return;
   }
   if(entry->second.ref.get() != this)
   {
	   aclog::err("INTERNAL ERROR, doppelganger running amok");
	   return;
   }
   entry->second.nRefCount--;
   if(entry->second.nRefCount<=0)
   {
	   LOG("*this is last entry, deleting dl/fi mapping");
	  setLockGuard;
	  status=FIST_ERRNOUSER;
	  notifyAll();
	  mapItems.erase(entry);
   }
}

tFileItemPtr fileitem::GetFileItem(MYSTD::string sPathKey)
{
	LOGSTART("fileitem::GetFileItem");

	tFileItemPtr p;
	MYTRY
	{
		lockguard lockGlobalMap(mapLck);

		map<string, tFileRefEntry>::iterator it=mapItems.find(sPathKey);
		if(it!=mapItems.end())
		{
			it->second.nRefCount++;
			LOG("Sharing existing file item");
			return it->second.ref;
		}
		p.reset(new fileitem(sPathKey));
		LOG("Created new file item");
		tFileRefEntry & entry = mapItems[sPathKey];
		LOG("Registering new file item...");
		if(!entry.ref)
		{
			entry.ref=p;
			entry.nRefCount++;
			LOG("New reference count: " << entry.nRefCount);
		}
		return entry.ref;
	}
	MYCATCH(MYSTD::bad_alloc&)
	{
		if(p)
			p.reset();
		return p;
	}
	return p;
}


// Opposite of GetFileItem, unregisters existing file item now or as soon as possible.
void fileitem::UnregOneASAP(tFileItemPtr pItem)
{
	if (!pItem)
		return;
	// download never started or no delayed expiration needed -> release now. Or expired long ago.
	if (!pItem->m_nTimeExpireAt || time(0) >= pItem->m_nTimeExpireAt)
	{
		pItem->Unreg();
		return;
	}
	lockguard lockGlobalMap(mapLck);
	std::map<string, tFileRefEntry>::iterator entry = mapItems.find(pItem->m_sKey);
	// not existing? -> not for us. Still enough references? -> let this one go
	if (mapItems.end() == entry || entry->second.nRefCount>1)
	{
		lockGlobalMap.unLock();
		pItem->Unreg();
		return;
	}
	// now the item's refCount is by one higher then it should be, set the flag instead of --
	entry->second.m_bNeedsDelayedExpiration=true;

// this will work as long as the delay is equal for all
	m_nEarliestExpiration=pItem->m_nTimeExpireAt;

}

// this method is supposed to be awaken periodically and detect items with ref count manipulated by
// the request storm prevention mechanism. Items shall be be dropped after some time if no other
// thread but us is using them.
time_t fileitem::DoDelayedUnregAndCheck()
{
	lockguard lockGlobalMap(mapLck);
	std::map<string, tFileRefEntry>::iterator it, here;

	time_t now=time(0);

	if(now<m_nEarliestExpiration) // unreachable anyhow
		return m_nEarliestExpiration;

	m_nEarliestExpiration = cleaner::never; // the correct time will be found soon

	for(it=mapItems.begin(); it!=mapItems.end();)
	{
		here=it++;
		if(!here->second.m_bNeedsDelayedExpiration)
			continue;
		// still active or young item? skip, but find the earliest time to expire
		if(now < here->second.ref->m_nTimeExpireAt || here->second.nRefCount>1)
		{
			m_nEarliestExpiration=min(here->second.ref->m_nTimeExpireAt, m_nEarliestExpiration);
			continue;
		}
		// ok, unused and delay is over. Destroy with the same sequence as Unreg... does.
		lockguard g(here->second.ref.get());
		here->second.ref->status=FIST_ERRNOUSER;
		here->second.ref->notifyAll();
		g.unLock();
		mapItems.erase(here);
	}
	return m_nEarliestExpiration;
}

#ifdef DEBUG

void DumpItems()
{
/*	lockguard lockGlobalMap(mapLck);

	map<string, tFileRefEntry>::iterator it=mapItems.begin();
	cout << "Remaining file items:\n";
	for(;it!=mapItems.end(); it++)
	{
		cout << it->second.ref->status << ": " << it->first <<endl;
	}
	*/
}
#endif
