
#define LOCAL_DEBUG
#include "debug.h"

#include "config.h"
#include "acfg.h"
#include "dlcon.h"

#include "fileitem.h"
#include <errno.h>
#include "dljob.h"

#define MAX_RETRIES 3

using namespace MYSTD;



dlcon::dlcon(bool bManualExecution, string *xff) :
m_bSingleRun(bManualExecution)
{
	ldbg("Creating dlcon");
	if (0==pipe(m_wakepipe))
	{
		set_nb(m_wakepipe[0]);
		set_nb(m_wakepipe[1]);
	}
	else
		m_wakepipe[0]=m_wakepipe[1]=-1;
	
	if(xff)
		m_sXForwardedFor=*xff;
}

void dlcon::EnqJob(tDlJob *todo)
{
	setLockGuard;
	
	m_qToReceive.push_back(todo);
			
	if (m_wakepipe[1]>=0)
		POKE(m_wakepipe[1]);
}
void dlcon::AddJob(tFileItemPtr m_pItem, 
		acfg::tHostiVec *pBackends, const MYSTD::string & sPatSuffix)
{
	//lockguard g(m_pItem.get());
	EnqJob(new tDlJob(this, m_pItem, pBackends, sPatSuffix));
}
void dlcon::AddJob(tFileItemPtr m_pItem, tHttpUrl hi)
{
	//lockguard g(m_pItem.get());
	EnqJob(new tDlJob(this, m_pItem, hi.sHost, hi.sPath));
}

void dlcon::SignalStop()
{
	setLockGuard;
	
	// stop all activity as far as possible
	m_bSingleRun=true;
	
	/* forget all but the first, those has to be fetched by the own downloader. If first
	 * item is being downloaded it will bail out RSN because of getting into NOMOREUSERS state. */
	while(m_qToReceive.size()>1)
	{
		delete m_qToReceive.back();
		m_qToReceive.pop_back();
	}
	POKE(m_wakepipe[1]);
}

dlcon::~dlcon()
{
	ldbg("Destroying dlcon");
	
	if (m_wakepipe[0]>=0)
		forceclose(m_wakepipe[0]);

	if (m_wakepipe[1]>=0)
		forceclose(m_wakepipe[1]);
		
	ldbg("destroying dlcon done. Any postponed tasks to do? ");
}
	
void dlcon::WorkLoop()
{

    string sCurrentlyConnectedHost, sSendBuf, sErrorMsg;
    
    // tmp stuff
    tDlJob *pj=NULL;
    int r, nTolErrorCount(0);
	  
	if (!m_InBuf.init(acfg::dlbufsize))
	{
		aclog::err("500 Out of memory");
		return;
	}

	if(m_wakepipe[0]<0 || m_wakepipe[1]<0)
	{
		aclog::err("Error creating pipe file descriptors");
		return;
	}

	fd_set rfds, wfds;
	struct timeval tv;
	       
	bool bStopRequesting=false; 
  
	while (true)
	{
		int nMaxFd=m_wakepipe[0];
		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
        FD_SET(m_wakepipe[0], &rfds);
		
		{
			setLockGuard;
			
			pj=NULL;
			
			dbgline;
			if(m_qToReceive.empty())
			{
				if(m_bSingleRun)
					return;
				
				dbgline;
				goto do_select; // wakepipe is in the fds set, just wait for parent there
			}

			pj = m_qToReceive.front();

			// found host change flag, prepare reconnection
			if(pj->bSuggestReconnect)
			{
				pj->bSuggestReconnect=false;
				nTolErrorCount=0; // new host, new luck...
				sCurrentlyConnectedHost.clear();
				dbgline;
			}
			if(sCurrentlyConnectedHost.empty())
			{
				dbgline;
				// set this right now... no matter how the rest looks, current
				// requests will be resend now anyways
				//
				m_InBuf.clear();
				sSendBuf.clear();
				bStopRequesting=false;
				// just be sure about that, not only in _Reconnect
				_Disconnect();
								
				/* fresh state and/or going for reconnection. Pick up a working hostname and
			    * connect to it.
			    * */
    
				if(!pj->FindConfig()) // bad, no usable host in the first item...
				{
					pj->MarkFailed(sErrorMsg);
					dbgline;
					
					delete pj;
					m_qToReceive.pop_front();
					continue;
				}
				
				const string sTargetHost = pj->GetPeerName();
				__lockguard.unLock();
				// might override it with proxy stuff
				bool bOk=_Connect(sTargetHost, sErrorMsg);
				
                if(bOk)
					sCurrentlyConnectedHost=sTargetHost;
				
                dbgline;
                
				__lockguard.reLock();
				dbgline;
				
                	
				if(!bOk)
					pj->BlacklistBackend();
				
				// weed out those with no way out, either in error state already, or 
				// ran out of backends (and notify users in this case)
				dljIter it=m_qToReceive.begin();
				bool bPrimaryFailed=false;
				while(it!=m_qToReceive.end())
				{
					tDlJob *p=*it;
					p->bSuggestReconnect=p->bRequestPrepared=false;
					dbgline;
					
					bool bDropIt=p->HasBrokenStorage();
					if(!bDropIt)
					{
						dbgline;
						
						if( ! p->FindConfig())
						{
							dbgline;
							bDropIt=true;
							p->MarkFailed(sErrorMsg); // sensible only for the first... whatever
						}
					}
					// ok, drop it for sure
					if(bDropIt)
					{
						if(pj == p)
							bPrimaryFailed=true;
						
						delete *it;
						dljIter itmp=it++;
						m_qToReceive.erase(itmp);
						
						dbgline;
						continue;
					}
					it++;
				}
				if(bPrimaryFailed)
				{
					dbgline;
					sCurrentlyConnectedHost.clear();
				}
				
				for(it=m_qToReceive.begin(); it!=m_qToReceive.end(); it++)
				{
					dbgline;
					if(! (*it)->AppendRequest(sTargetHost, sSendBuf))
					{
						// refused -> needs to reconnect later
						(*it)->bSuggestReconnect=true;
						bStopRequesting=true;  // MUST set this
						dbgline;
						break;
					}
				}
				
				if(!bOk) // failed somewhere above, recheck
					continue;
			} // (re)connect done

			dbgline;
			// connected!
          
		} // end of locked section	
		
					
		if(m_conFd>=0)
		{
			FD_SET(m_conFd, &rfds);
			nMaxFd=std::max(m_conFd, nMaxFd);
			
			if(!sSendBuf.empty())
			{
				ldbg("Needs to send " << sSendBuf.size() << " bytes")
				FD_SET(m_conFd, &wfds);
				nMaxFd=std::max(m_conFd, nMaxFd);
			}
		}

		do_select:
		
		ldbg("select dlcon");
		tv.tv_sec = acfg::nettimeout;
		tv.tv_usec = 0;
		
		r=select(nMaxFd+1, &rfds, &wfds, NULL, &tv);
		
		if (r<0)
		{
			if (EINTR == errno)
				continue;
			
			// catch errno code and format message
			aclog::errnoFmter fer;
			            
			ldbg("FAILURE: select, " << fer.msg);
		    sErrorMsg=string("500 Internal malfunction, ")+fer.msg;
		    goto drop_and_restart_stream;
		}
		else if(r==0)
		{
			if(!sCurrentlyConnectedHost.empty()) 
            {

				aclog::errnoFmter fer;
				
                ldbg("Select timeout for " << sCurrentlyConnectedHost << ". Trigger reconnection..., m_sConnectedHost.clear()");
                sCurrentlyConnectedHost.clear();
                
                if(pj && pj->HasStarted())
                {
                	aclog::err("Warning, disconnected during package download");
                	sErrorMsg=string("500 Connection abort, ")+fer.msg;
                	goto drop_and_restart_stream;
                }
            }
			continue;
		}

		if (FD_ISSET(m_wakepipe[0], &rfds))
		{
			for(int tmp; read(m_wakepipe[0], &tmp, 1) > 0; ) ;
			
			setLockGuard;
			// got new stuff? and needs to prepare requests? and queue is not empty?
			// walk around and send requests for them, if possible
			if(	!bStopRequesting &&
				!sCurrentlyConnectedHost.empty() &&
				!m_qToReceive.empty())
			{
				ldbg("Preparing requests for new stuff...");
				
				// walks backward from the end to find a good position where previous request-sending stoped,
				// instead of just starting from the beginning.
				
				// WARNING!!! Optimistic code, relies on condition checks above
				
				dljIter it=m_qToReceive.end();
				for(it--; it!=m_qToReceive.begin(); it--)
					if( (*it)->bRequestPrepared )
						break;
				
				for(; it!=m_qToReceive.end(); it++)
				{
					tDlJob *pjn=*it;
					// one position to far in backwalk
					if(pjn->bRequestPrepared)
						continue;
					
					// found the first unseen, do stuff
					
					pjn->bRequestPrepared=true;
					if(!pjn->FindConfig() || !pjn->AppendRequest(sCurrentlyConnectedHost, sSendBuf)) 
					{
						// bad stuff, check later
						bStopRequesting=true;
						pjn->bSuggestReconnect=true;
						break;
					}
				}
			}
		}

		if (m_conFd>=0 && !sSendBuf.empty() && FD_ISSET(m_conFd, &wfds))
		{
			FD_CLR(m_conFd, &wfds);
			
			ldbg("Sending data...\n" << sSendBuf);
			int s=::send(m_conFd, sSendBuf.data(), sSendBuf.length(), MSG_NOSIGNAL);
			ldbg("Sent " << s << " bytes from " << sSendBuf.length());
			if (s<0)
			{
				if(errno!=EAGAIN && errno!=EINTR)
                {
                    sErrorMsg="502 Peer communication error";
                    goto drop_and_restart_stream;
                }
                // else retry later...
			}
            else 
                sSendBuf.erase(0, s);
		}
	    
		if (m_conFd>=0 && FD_ISSET(m_conFd, &rfds))
		{
			if(!pj) // huh, sends something without request? Maybe disconnect hing (zero read)? 
			{
				// just disconnect, and maybe reconnect
				sCurrentlyConnectedHost.clear();
				continue;
			}
			int r = m_InBuf.sysread(m_conFd);
			if (r <= 0)
			{
				if(r==-EAGAIN)
					continue; // should never happen, though.
				
				// pickup the error code and make sure it's closed ASAP
				aclog::errnoFmter f;
				_Disconnect();
				
				if(pj)
				{
					/*
					 * disconnected? Maybe because of end of Keep-Alive sequence, or
					 * having hit the timeout at the moment of resuming... 
					 * Trying to work around (n retries) if possible
					 */
					
					if(++nTolErrorCount < MAX_RETRIES)
					{
						
						if( ! pj->HasStarted())
						{
							sCurrentlyConnectedHost.clear();
							continue;
						}
					}
					else
					{   // too many retries for this host :-(
						pj->BlacklistBackend();
						nTolErrorCount=0;
					}
				}
				
				sErrorMsg=string("502 ")+f.msg;
                goto drop_and_restart_stream;
			}
        }

		while( ! m_InBuf.empty() && pj)
		{
			ldbg("Processing input of " << pj->RemoteUri() );
			tDlJob::tDlResult res=pj->ProcessIncomming(m_InBuf, sErrorMsg);
			ldbg("... input processing result: " << res);
			switch(res)
			{
				case(tDlJob::R_MOREINPUT):
				case(tDlJob::R_NOOP):
					goto leave_recv_loop; // will get more in the next loop
				case(tDlJob::R_SKIPITEM): // will restart the stream
				// item was ours but download aborted... might do more fine recovery in the future
				case(tDlJob::R_ERROR_LOCAL):
				case(tDlJob::R_ERROR_REMOTE):
				case(tDlJob::R_NEXTSTATE): // this one should never appear, but who knows...
					goto drop_and_restart_stream;
									
				case(tDlJob::R_DONE):
				{
					nTolErrorCount=0; // well done and can sleep now, should be resumed carefully later
					
					setLockGuard;
					delete pj;
					m_qToReceive.pop_front();
					pj = m_qToReceive.empty() ? NULL : m_qToReceive.front();
					ldbg("Remaining dlitems: " << m_qToReceive.size());
					continue;
				}
				break;
			}
        }
		leave_recv_loop:
		
		continue;
		
		drop_and_restart_stream:
		if(pj)
		{
			pj->MarkFailed(sErrorMsg);
			sCurrentlyConnectedHost.clear();
			setLockGuard;
			delete pj;
			m_qToReceive.pop_front();
			pj=NULL;
			continue;
		}
	}
}

