//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008,2009 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>
#ifdef __FreeBSD__
	#include <signal.h>
#endif
#include <glib.h>

#include "common.h"
#include "conf.h"
#include "dispatcher.h"
#include "executor.h"
#include "pr_frontend.h"
#include "utils.h"

extern void freeDisplay();	// from xemulate.h

#define ALARM_CLEAN 	0
#define ALARM_CHECK 	1

extern CONF	conf;
extern int  	serverMode;
extern int      state;
extern gboolean dispIsJoinable;

char tmp	    [MAXMAXLEN];

GMainLoop	*loop 	 = NULL;
GAsyncQueue     *q2disp  = NULL;
GAsyncQueue     *q2fe 	 = NULL;
GAsyncQueue     *q2exec	 = NULL;
GAsyncQueue     *q2main	 = NULL;

GThread 	*dispThread = NULL;
GThread 	*execThread = NULL;

gboolean        stillRun = TRUE;

void aboutToExit() 
{
	logger("DBG", "[ML]: aboutToExit");
	g_main_loop_quit (loop);

	stillRun = FALSE;

	if (getFrontEnd() > 0) {
		disconnectFrontEnd();
	}
	
	logger("DBG", "[ML]: aboutToExit: join executor");
	sendEventToExecutor(ID_EVT_EXIT);
	g_thread_join(execThread);
	
	if (dispIsJoinable) {	// dispatcher can be blocked in accept()
		logger("DBG", "[ML]: aboutToExit: join dispatcher");
		dMessage* dm = malloc(sizeof(dMessage));
		dm->size  = 0;
		dm->value = (void*) NULL;
		dm->type  = DM_EVENT_EXIT;
		sendToDispatcher(dm);
		g_thread_join(dispThread);
 	}
		
	logger("DBG", "[ML]: aboutToExit: release main loop");
	g_main_loop_unref (loop);
	
	freeDisplay();
	freeTimers();
	
	logger("DBG", "[ML]: aboutToExit: EXIT");
	exit(0);
}

// handle signals only once (it is enough to send disconnect message to java client)
void sig_int() {
	signal(SIGINT, SIG_DFL);   
	printf("\nProcess %d: got SIGINT\n",getpid());
	//sig_handle();
	aboutToExit();
}

void sig_quit() {
	signal(SIGQUIT, SIG_DFL);
	printf("\nProcess %d: got SIGQUIT\n",getpid());
	//sig_handle();
	aboutToExit();
}

void sig_kill() {
	signal(SIGKILL, SIG_DFL);
	printf("\nProcess %d: got SIGKILL\n",getpid());
	//sig_handle();
	aboutToExit();
}

// mode: 0 - clear alarms
//       1 - if fired - process, then clear alarms
void manageAlarms(int mode)
{
        char    *pm;
        struct stat buf;
            
	char    *pf = getFirstAlarm(&pm);
        while (pf != NULL) {
                //sprintf(tmp, "[ML]: check alarm %s", pf);
                //logger("DBG", tmp);
        
                // If file exists
                int fs = stat(pf, &buf);
                if (fs == 0) {
                	if (mode == ALARM_CLEAN) {
                        	logger("DBG","[ML]: Remove file");
                        } else {            
                        	INFO2("[ML]: Alarm %s fired", pf);
				
				eMessage* em = malloc(sizeof(eMessage));
				em->type  = EM_ALARM;
				em->value = strdup(pm);
			
				sendToExecutor(em);
                        }
                        remove(pf);
                }
                pf = getNextAlarm(&pm);
        }
        return;
}

gboolean alarmTimeout(gpointer data)
{
	//logger("DBG","[ML]: alarmTimeout");
	manageAlarms(ALARM_CHECK);
	return TRUE;
}

gboolean adminRoutine(gpointer data)
{
	int *mi = (int*) g_async_queue_try_pop(q2main);
	
	if (mi != NULL) {
	
		int m = *mi;
		free(mi);
		
		if (m == M_ABORT) {
			logger("DBG","[ML]: got exit request");
			aboutToExit();	// exit here
			
		} else if (m == M_DISCONNECT) {
			logger("DBG","[ML]: got disconnect request");
			
			if (!autoConnect() &&
			    (serverMode == CLIENT_AT || 
			     serverMode == CLIENT_ILIRC)) {
			        logger("DBG","[ML]: Exit because of disconnect");
				aboutToExit();	// exit here 
			}
		}
	}	
	return TRUE;
}

int main(int argc, char *argv[])
{
	int i;
        for (i=1; i<argc; i++) {
        	if (strcmp(argv[i],"-f")==0 && i+1<argc) {
                	break;
                }
        }
	
        if (argv[1] && (strcmp(argv[1],"-h")==0 || strcmp(argv[1],"--help")==0)) {
		 //Just print help and exit
		 printHelp();
	         exit(0);
        } else if (argv[1] && (strcmp(argv[1],"-v")==0 || strcmp(argv[1],"--version")==0)) {
		printf("anyremote v%s\n", PACKAGE_VERSION); 
		exit(0);
        }

        createDataDir();
	
        if (init_cfg(i<argc ? argv[++i] : NULL) != EXIT_OK) {
        	printf("Can not initialize configuration. Exiting.\n");
		
		printHelp();
                
		freeCfg();
                exit(1);
        }
	
	// needs to be after init_cfg()
	addInternalVars();
                    
        if (parse_opts(argc, argv)!=1 ) {
        	printf("Invalid command line option used. Exiting.\n");
                freeCfg();
                exit(1);
        }

	g_thread_init(NULL);	// need to call it before initLog()
	
        initLog();
	manageAlarms(ALARM_CLEAN);
        
        // Not to core dump if connection will close
        signal(SIGPIPE, SIG_IGN);   
        signal(SIGCHLD, SIG_IGN);   

    	signal(SIGINT, sig_int);
    	signal(SIGQUIT,sig_quit);
   	signal(SIGKILL,sig_kill);
		
	logger("DBG","g_main_loop_new");
	
	loop = g_main_loop_new(NULL, FALSE);
	
	q2main = g_async_queue_new();

	q2disp = g_async_queue_new();
	dispThread = g_thread_create (dispatcherRoutine, NULL, TRUE, NULL);

	q2exec  = g_async_queue_new();
	execThread = g_thread_create (executorRoutine, NULL, TRUE, NULL);
	
	if (getFrontEnd() > 0) {
		q2fe = g_async_queue_new();
        	g_timeout_add(200, frontendRoutine, NULL); 	// 1/5 of second
	} else {
		logger("DBG","Do not setup front-end connection");
	}
	
	g_timeout_add(100,   adminRoutine, NULL); 		// 1/10 of second
	g_timeout_add(15000, alarmTimeout, NULL);
	
	logger("DBG","g_main_loop_run");
	g_main_loop_run(loop);

        return 0;
}
