/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
/****************************************************************************
 ** ui.h extension file, included from the uic-generated form implementation.
 **
 ** If you wish to add, delete or rename slots use Qt Designer which will
 ** update this file, preserving your code. Create an init() slot in place of
 ** a constructor, and a destroy() slot in place of a destructor.
 *****************************************************************************/

extern magic_t magic;

void ImportMealMasterDialog::init()
{
  databaseWizard = false;
}


void ImportMealMasterDialog::openErrorFile()
{
  // Select file for saving erroneous Mealmaster recipes to.
  QString fileName =
    checkOverWrite
    ( KFileDialog::getSaveFileName
      ( ":export",
        i18n( "*.mm *.mmf|UTF-8 Mealmaster (*.mm *.mmf)\n"
              "*|All Files (*)" ),
        this,
        i18n( "Save Erroneous Mealmaster Recipes" ) ), this );

  if ( fileName != QString::null ) {
    errorFileNameEdit->setText( fileName );
    updateEnableState();
  };  
}

void ImportMealMasterDialog::openMealMasterFile()
{

  std::string startAt;
  if ( databaseWizard ) {
    std::string fileName( (const char *)i18n( "scripts/english.utf8.mmf" ) );
    startAt = findAnyMealFile( "appdata", fileName.c_str() );
    if ( startAt == fileName )
      startAt = ":import";
  } else
    startAt = ":import";

  QStringList fileNames =
    KFileDialog::getOpenFileNames( startAt,
                                   i18n( "*.mm *.mmf|Mealmaster (*.mm *.mmf)\n"
                                         "*|All Files (*)" ),
                                   this,
                                   i18n( "Import Mealmaster Files" ) );
  
  if ( !fileNames.empty() ) {
    for ( int i=0; i<(signed)fileNames.size(); i++ )
      new KListViewItem( mealMasterListView, fileNames[ i ] );
    startTimer( 0 );
    updateEnableState();
  };
}


void ImportMealMasterDialog::timerEvent( QTimerEvent * )
{
  assert( magic != NULL );
  killTimers();

  for ( QListViewItem *item = mealMasterListView->firstChild();
        item != NULL; item = item->nextSibling() ) {

    if ( item->text( 1 ).isEmpty() ) {

      // DisplayWaitCursor w;
      const char *descr = magic_file( magic, item->text( 0 ) );
      if ( descr == NULL ) descr = magic_error( magic );

      if ( descr == NULL )
        item->setText
          ( 1, i18n( "Error retrieving error-message from libmagic." ) );
      else
        item->setText( 1, descr );

      startTimer( 0 );
      break;

    };

  };
  
}


void ImportMealMasterDialog::updateEnableState()
{
  bool okButtonEnable =
    mealMasterListView->firstChild() != NULL &&
    ( !storeErroneousRadio->isChecked() ||
      !errorFileNameEdit->text().isEmpty() ) &&
    encodingCombo->currentItem() != 0;
  okButton->setEnabled( okButtonEnable );
}


void ImportMealMasterDialog::selectionChanged()
{
  bool removeButtonEnable = false;
  for ( QListViewItem *item = mealMasterListView->firstChild();
        item != NULL; item = item->nextSibling() )
    if ( item->isSelected() ) {
      removeButtonEnable = true;
      break;
    };
  removeButton->setEnabled( removeButtonEnable );
}


void ImportMealMasterDialog::removeSelected()
{
  QListViewItem *item = mealMasterListView->firstChild();
  while ( item != NULL ) {
    QListViewItem *nextItem = item->nextSibling();
    if ( item->isSelected() )
      delete item;
    item = nextItem;
  };
  removeButton->setEnabled( false );
  updateEnableState();
}




void ImportMealMasterDialog::accept()
{
  try {

    assert( cookBook );

    DisplayWaitCursor w;

    boost::shared_ptr< std::ostream > stream;
    MealMasterParseErrorHandlerPtr parseErrorHandler;

    if ( storeErroneousRadio->isChecked() ) {

      stream = boost::shared_ptr< std::ostream >
        ( new std::ofstream( errorFileNameEdit->text(),
                             std::ios::binary ) );
      parseErrorHandler = MealMasterParseErrorHandlerPtr
        ( new MealMasterStoreErroneousHandler
          ( stream.get(), errorFileNameEdit->text() ) );

    } else if ( abortOnErrorRadio->isChecked() )
      parseErrorHandler = MealMasterParseErrorHandlerPtr( new MealMasterParseErrorHandler );
    else
      parseErrorHandler = MealMasterParseErrorHandlerPtr( new MealMasterIgnoreErrorHandler );
      
    // Compute overall number of characters to be parsed.
    std::list< int > fileSize;
    int numCharacters = 0;
    for ( QListViewItem *item = mealMasterListView->firstChild();
          item != NULL; item = item->nextSibling() ) {
      std::string fileName( (const char *)item->text( 0 ) );
      std::ifstream inputStream( fileName.c_str(), std::ios::binary );
      ERRORMACRO( inputStream.seekg( 0, std::ios::end ), Error, ,
                  i18n( "Error opening file \"%1\" ..." ).
                  arg( fileName.c_str() ) );
      fileSize.push_back( inputStream.tellg() );
      numCharacters += inputStream.tellg();
    };

    // Open progress dialog.
    QProgressDialog progress( i18n( "Importing Mealmaster files ... " ),
                              i18n( "&Cancel" ), numCharacters, this,
                              "progress", true );

    // Overall progress is zero.
    int currPos = 0;

    // Start reading in files.
    for ( QListViewItem *item = mealMasterListView->firstChild();
          item != NULL; item = item->nextSibling() ) {

      const int count = 50;
      // Start parsing new file in steps of 'count' recipes.
      // Constructor reset's counter for line-number!
      CompilerPtr mealMasterCompiler
        ( new MealMasterCompiler( count, parseErrorHandler ) );

      // Open the next file.
      std::string fileName( (const char *)item->text( 0 ) );
      std::ifstream inputFile( fileName.c_str(), std::ios::binary );

      std::istream *inputStream;
      boost::shared_ptr< std::stringstream > stringStream;

      // Recode file.
      assert( encodingCombo->count() == 3 );
      assert( encodingCombo->currentItem() != 0 );
      assert( encodingCombo->text( 1 ) == i18n( "ISO-8859-1" ) );
      assert( encodingCombo->text( 2 ) == i18n( "UTF-8" ) );

      if ( encodingCombo->currentItem() == 1 ) {

        progress.setLabelText( i18n( "Recoding file \"%1\" ..." ).
                               arg( fileName.c_str() ) );
        stringStream =
          boost::shared_ptr< std::stringstream >( new std::stringstream );
        inputStream = stringStream.get();
        Recoder( "latin1..utf8" ).translate( inputFile, *stringStream );
        
      } else {
        assert( encodingCombo->currentItem() == 2 );
        inputStream = &inputFile;
      };

      // Display info.
      progress.setLabelText( i18n( "Importing file \"%1\" ..." ).
                             arg( fileName.c_str() ) );

      while ( !inputStream->eof() ) {

        // Update overall progress.
        progress.setProgress( currPos + inputStream->tellg() );
        ERRORMACRO( !progress.wasCanceled(), Error, ,
                    i18n( "Import aborted." ) );

        // Parse a couple of recipes.
        std::ofstream nullDevice( "/dev/null" );
        ChainedCompiler
          ( mealMasterCompiler,
            cookBook->getXMLLayer() ).translate( *inputStream, nullDevice );

      };

      // Increment offset of progress.
      currPos += fileSize.front();
      fileSize.pop_front();

    };
    
    QDialog::accept();
 
  } catch ( Error &e ) {

    KMessageBox::error( this, e.what() );
    QDialog::reject();

  };
}

