// ---------------------------------------------------------------------------
// - Rmatrix.hpp                                                             -
// - afnix:mth module - real matrix definitions                              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2011 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_RMATRIX_HPP
#define  AFNIX_RMATRIX_HPP

#ifndef  AFNIX_RMI_HPP
#include "Rmi.hpp"
#endif
 
#ifndef  AFNIX_RVECTOR_HPP
#include "Rvector.hpp"
#endif
 
namespace afnix {

  /// This Rmatrix class is the default implementation of the real matrix
  /// interface. Internally, the matrix is represented as an array of rows.
  /// Each row is represented as an array of t_real.
  /// @author amaury darsch

  class Rmatrix : public Rmi {
  public:
    /// generate a random matrix by size
    /// @param size the matrix size
    /// @param ddom the diagonally dominant flag
    /// @param rmin the minimum value
    /// @param rmax the maximum value
    static Rmatrix random (const t_long size, const bool   ddom,
			   const t_real rmin, const t_real rmax);

    /// add a matrix with another one
    /// @param mx the matrix argument
    /// @param my the matrix argument
    friend Rmatrix operator + (const Rmatrix& mx, const Rmatrix& my);

    /// substract a matrix with another one
    /// @param mx the matrix argument
    /// @param my the matrix argument
    friend Rmatrix operator - (const Rmatrix& mx, const Rmatrix& my);

    /// multiply a vector with a matrix
    /// @param m the matrix argument
    /// @param x the vector argument
    friend Rvector operator * (const Rmatrix& m, const Rvector& x);

    /// multiply two matrices
    /// @param mx the matrix argument
    /// @param my the matrix argument
    friend Rmatrix operator * (const Rmatrix& mx, const Rmatrix& my);
    
  private:
    /// the row array
    t_real** p_rtab;

  public:
    /// create a null matrix
    Rmatrix (void);

    /// create a square matrix by size
    /// @param size the matrix size
    Rmatrix (const t_long size);

    /// create a matrix by size
    /// @param rsiz the row size
    /// @param csiz the column size
    Rmatrix (const t_long rsiz, const t_long csiz);

    /// copy construct this matrix
    /// @param that the matrix to copy
    Rmatrix (const Rmatrix& that);

    /// destroy this matrix
    ~Rmatrix (void);

    /// @return the class name
    String repr (void) const;

    /// @return a clone of this object
    Object* clone (void) const;

    /// assign a matrix to this one
    /// @param that the matrix to assign
    Rmatrix& operator = (const Rmatrix& that);

    /// set a matrix by position
    /// @param row the row position
    /// @param col the column position
    /// @param val the value to set
    void set (const t_long row, const t_long col, const t_real val);

    /// get a matrix value by position
    /// @param row the row position
    /// @param col the column position
    t_real get (const t_long row, const t_long col) const;
    
    /// positive multiply a matrix with a vector
    /// @param r the result vector
    /// @param x the vector argument
    Rvi& pmul (Rvi& r, const Rvi& x) const;

    /// negatively multiply a matrix with a vector
    /// @param r the result vector
    /// @param x the vector argument
    Rvi& nmul (Rvi& r, const Rvi& x) const;

    /// map a matrix row as a sparse object
    /// @param row the row to map as a sparse object
    Rsi* torso (const t_long row) const;

    /// extract a real cofactor by row and column
    /// @param row the cofactor row
    /// @param col the cofactor column
    Rmi* getcfm (const t_long row, const t_long col) const;

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// operate this object with another object
    /// @param type   the operator type
    /// @param object the operand object
    Object* oper (t_oper type, Object* object);
  };
}

#endif
