# C runtime object models for cross module analysis.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/metric"
require "adlint/util"

module AdLint #:nodoc:
module Ld #:nodoc:

  class Variable
    def initialize(variable_definition_record)
      @metric_record = variable_definition_record
    end

    def location
      @metric_record.location
    end

    def name
      @metric_record.variable_name
    end

    def type
      @metric_record.type_rep
    end

    def extern?
      @metric_record.var_linkage_type == "X"
    end

    def eql?(rhs)
      name == rhs.name && location == rhs.location
    end

    alias :== :eql?

    def hash
      "#{name} #{location}".hash
    end
  end

  class VariableDeclaration
    def initialize(global_variable_declaration_record)
      @metric_record = global_variable_declaration_record
    end

    def location
      @metric_record.location
    end

    def name
      @metric_record.variable_name
    end

    def type
      @metric_record.type_rep
    end

    def extern?
      true
    end

    def eql?(rhs)
      name == rhs.name && location == rhs.location
    end

    alias :== :eql?

    def hash
      "#{name} #{location}".hash
    end
  end

  class VariableMapping
    def initialize
      @variable_index = Hash.new { |hash, key| hash[key] = Set.new }
      @declaration_index = Hash.new { |hash, key| hash[key] = Set.new }
    end

    def add_variable(variable)
      @variable_index[variable.name].add(variable)
    end

    def add_variable_declaration(variable_declaration)
      @declaration_index[variable_declaration.name].add(variable_declaration)
    end

    def all_variables
      @variable_index.values.reduce(Set.new) { |all, variables|
        all + variables
      }.to_a
    end

    def all_variable_declarations
      @declaration_index.values.reduce(Set.new) { |all, declarations|
        all + declarations
      }.to_a
    end

    def lookup_variables(variable_name)
      @variable_index[variable_name].to_a
    end

    def lookup_variable_declarations(variable_name)
      @declaration_index[variable_name].to_a
    end
  end

  class VariableMapper
    def initialize
      @result = VariableMapping.new
    end

    attr_reader :result

    def execute(metric_fpath)
      CSV.foreach(metric_fpath) do |row|
        record = MetricRecord.of(row)
        case
        when record.variable_definition?
          if record.var_linkage_type == "X"
            @result.add_variable(Variable.new(record))
          end
        when record.global_variable_declaration?
          @result.add_variable_declaration(VariableDeclaration.new(record))
        end
      end
    end
  end

  class VariableReference
    def initialize(function, variable, location)
      @function = function
      @variable = variable
      @location = location
    end

    attr_reader :function
    attr_reader :variable
    attr_reader :location

    def eql?(rhs)
      @function == rhs.function && @variable == rhs.variable &&
        @location == rhs.location
    end

    alias :== :eql?

    def hash
      [@function, @variable, @location].hash
    end
  end

  class VariableReferenceGraph
    extend Memoizable

    def initialize(function_call_graph)
      @function_call_graph = function_call_graph
      @referrer_index = Hash.new { |hash, key| hash[key] = Set.new }
      @variable_index = Hash.new { |hash, key| hash[key] = Set.new }
    end

    def add(variable_reference)
      @referrer_index[variable_reference.function].add(variable_reference)
      @variable_index[variable_reference.variable].add(variable_reference)
    end

    def all_referrers_of(variable)
      direct_referrers_of(variable) + indirect_referrers_of(variable)
    end

    def direct_referrers_of(variable)
      @variable_index[variable].map { |varref| varref.function }.to_set
    end

    def indirect_referrers_of(variable)
      direct_referrers = direct_referrers_of(variable)
      direct_referrers.reduce(Set.new) do |result, function|
        result + @function_call_graph.all_callers_of(function)
      end
    end
  end

  class VariableReferenceGraphBuilder
    def initialize(variable_mapping, function_mapping, function_call_graph)
      @variable_mapping = variable_mapping
      @function_mapping = function_mapping
      @result = VariableReferenceGraph.new(function_call_graph)
    end

    attr_reader :result

    def execute(metric_fpath)
      CSV.foreach(metric_fpath) do |row|
        record = MetricRecord.of(row)
        case
        when record.variable_xref?
          function = @function_mapping.lookup_functions(
            record.accessor_function.name).first

          variable = @variable_mapping.lookup_variables(
            record.accessee_variable).first

          if function && variable
            @result.add(VariableReference.new(
              function, variable, record.location))
          end
        end
      end
    end
  end

  class VariableTraversal
    def initialize(variable_mapping)
      @variable_mapping = variable_mapping
    end

    extend Pluggable

    def_plugin :on_declaration
    def_plugin :on_definition

    def execute
      @variable_mapping.all_variable_declarations.each do |declaration|
        on_declaration.invoke(declaration)
      end

      @variable_mapping.all_variables.each do |definition|
        on_definition.invoke(definition)
      end
    end
  end

  class Function
    def initialize(function_definition_record)
      @metric_record = function_definition_record
    end

    def location
      @metric_record.location
    end

    def signature
      @metric_record.function_identifier.signature
    end

    def name
      @metric_record.function_identifier.name
    end

    def extern?
      @metric_record.function_linkage_type == "X"
    end

    def eql?(rhs)
      signature == rhs.signature && location == rhs.location
    end

    alias :== :eql?

    def hash
      "#{signature} #{location}".hash
    end
  end

  class FunctionDeclaration
    def initialize(function_declaration_record)
      @metric_record = function_declaration_record
    end

    def location
      @metric_record.location
    end

    def signature
      @metric_record.function_identifier.signature
    end

    def name
      @metric_record.function_identifier.name
    end

    def extern?
      @metric_record.function_linkage_type == "X"
    end

    def eql?(rhs)
      signature == rhs.signature && location == rhs.location
    end

    alias :== :eql?

    def hash
      "#{signature} #{location}".hash
    end
  end

  class FunctionMapping
    def initialize
      @function_index = Hash.new { |hash, key| hash[key] = Set.new }
      @declaration_index = Hash.new { |hash, key| hash[key] = Set.new }
    end

    def add_function(function)
      @function_index[function.name].add(function)
    end

    def add_function_declaration(function_declaration)
      @declaration_index[function_declaration.name].add(function_declaration)
    end

    def all_functions
      @function_index.values.reduce(Set.new) { |all, functions|
        all + functions
      }.to_a
    end

    def all_function_declarations
      @declaration_index.values.reduce(Set.new) { |all, declarations|
        all + declarations
      }.to_a
    end

    def lookup_functions(function_name)
      @function_index[function_name].to_a
    end

    def lookup_function_declarations(function_name)
      @declaration_index[function_name].to_a
    end
  end

  class FunctionMapper
    def initialize
      @result = FunctionMapping.new
    end

    attr_reader :result

    def execute(metric_fpath)
      CSV.foreach(metric_fpath) do |row|
        record = MetricRecord.of(row)
        case
        when record.function_definition?
          @result.add_function(Function.new(record))
        when record.function_declaration?
          @result.add_function_declaration(FunctionDeclaration.new(record))
        end
      end
    end
  end

  class FunctionCall
    def initialize(caller_function, callee_function)
      @caller_function = caller_function
      @callee_function = callee_function
    end

    attr_reader :caller_function
    attr_reader :callee_function

    def eql?(rhs)
      @caller_function == rhs.caller_function &&
        @callee_function == rhs.callee_function
    end

    alias :== :eql?

    def hash
      [@caller_function, @callee_function].hash
    end
  end

  class FunctionCallGraph
    extend Memoizable

    def initialize
      @caller_index = Hash.new { |hash, key| hash[key] = Set.new }
      @callee_index = Hash.new { |hash, key| hash[key] = Set.new }
    end

    def add(function_call)
      @caller_index[function_call.caller_function].add(function_call)
      @callee_index[function_call.callee_function].add(function_call)
    end

    def all_callers_of(callee_function)
      direct_callers_of(callee_function) + indirect_callers_of(callee_function)
    end
    memoize :all_callers_of

    def direct_callers_of(callee_function)
      @callee_index[callee_function].map { |funcall|
        funcall.caller_function
      }.to_set
    end
    memoize :direct_callers_of

    def indirect_callers_of(callee_function)
      direct_callers = direct_callers_of(callee_function)
      direct_callers.reduce(Set.new) do |all_callers, function|
        all_callers + collect_callers_of(function, all_callers)
      end
    end
    memoize :indirect_callers_of

    private
    def collect_callers_of(callee_function, exclusion_list)
      direct_callers = direct_callers_of(callee_function)

      direct_callers.reduce(Set.new) do |all_callers, function|
        if exclusion_list.include?(function)
          all_callers.add(function)
        else
          all_callers.add(function) +
            collect_callers_of(function, exclusion_list + all_callers)
        end
      end
    end
    memoize :collect_callers_of, 0
  end

  class FunctionCallGraphBuilder
    def initialize(function_mapping)
      @function_mapping = function_mapping
      @result = FunctionCallGraph.new
    end

    attr_reader :result

    def execute(metric_fpath)
      CSV.foreach(metric_fpath) do |row|
        record = MetricRecord.of(row)
        case
        when record.function_call?
          caller_function, callee_function = lookup_functions_by_call(record)
          if caller_function && callee_function
            @result.add(FunctionCall.new(caller_function, callee_function))
          end
        when record.function_xref?
          caller_function, callee_function = lookup_functions_by_xref(record)
          if caller_function && callee_function
            @result.add(FunctionCall.new(caller_function, callee_function))
          end
        end
      end
    end

    def lookup_functions_by_call(function_call_record)
      caller_function = @function_mapping.lookup_functions(
        function_call_record.caller_function.name).find { |function|
          function.location.fpath == function_call_record.location.fpath
      }
      return nil, nil unless caller_function

      callee_functions = @function_mapping.lookup_functions(
        function_call_record.callee_function.name)

      callee_function = callee_functions.first
      callee_functions.each do |function|
        if function.location.fpath == caller_function.location.fpath
          callee_function = function
          break
        end
      end

      return caller_function, callee_function
    end

    def lookup_functions_by_xref(function_xref_record)
      caller_function = @function_mapping.lookup_functions(
        function_xref_record.accessor_function.name).find { |function|
          function.location.fpath == function_xref_record.location.fpath
      }
      return nil, nil unless caller_function

      callee_functions = @function_mapping.lookup_functions(
        function_xref_record.accessee_function.name)

      callee_function = callee_functions.first
      callee_functions.each do |function|
        if function.location.fpath == caller_function.location.fpath
          callee_function = function
          break
        end
      end

      return caller_function, callee_function
    end
  end

  class FunctionTraversal
    def initialize(function_mapping)
      @function_mapping = function_mapping
    end

    extend Pluggable

    def_plugin :on_declaration
    def_plugin :on_definition

    def execute
      @function_mapping.all_function_declarations.each do |declaration|
        on_declaration.invoke(declaration)
      end

      @function_mapping.all_functions.each do |definition|
        on_definition.invoke(definition)
      end
    end
  end

end
end
