# Message detection classes for C language.
#
# Author::    Rie Shima <mailto:rkakuuchi@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Rie Shima <mailto:rkakuuchi@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/report"
require "adlint/message"

module AdLint #:nodoc:
module C #:nodoc:

  class W0573 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*scanf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          if cs.scanset && cs.scanset.include?("-")
            W(:W0573, format.location)
            break
          end
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return ScanfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0606 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_union_type_declaration += method(:check)
    end

    private
    def check(node)
      node.struct_declarations.each do |struct_declaration|
        struct_declaration.items.each do |member_decl|
          if member_decl.type.scalar? && member_decl.type.floating?
            W(:W0606, node.location)
            return
          end
        end
      end
    end
  end

  class W0685 < W0573
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*scanf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each do |cs|
          next unless cs.scanset

          cs.scanset.scan(/(.)-(.)/).each do |lhs, rhs|
            W(:W0685, format.location) if lhs.ord > rhs.ord
          end
        end
      end
    end
  end

  class W0686 < W0573
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*scanf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each do |cs|
          next unless cs.scanset

          unless cs.valid_scanset?
            W(:W0686, format.location)
          end
        end
      end
    end
  end

  class W0698 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:start_function)
      visitor.enter_kandr_function_definition += method(:start_function)
      visitor.leave_ansi_function_definition += method(:end_function)
      visitor.leave_kandr_function_definition += method(:end_function)
      visitor.enter_return_statement += method(:check)
      @current_function = nil
    end

    private
    def start_function(function_definition)
      @current_function = function_definition
    end

    def end_function(function_definition)
      @current_function = nil
    end

    def check(return_statement)
      return unless @current_function.explicitly_typed?

      if return_type = @current_function.type.return_type
        if !return_type.void? && return_statement.expression.nil?
          W(:W0698, return_statement.location,
            @current_function.identifier.value)
        end
      end
    end
  end

  class W0699 < W0698
    private
    def check(return_statement)
      return unless @current_function.implicitly_typed?

      if return_statement.expression.nil?
        W(:W0699, return_statement.location,
          @current_function.identifier.value)
      end
    end
  end

  class W0711 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_relational_expression += method(:check)
    end

    private
    def check(expression)
      if !expression.lhs_operand.logical? && expression.rhs_operand.logical?
        W(:W0711, expression.rhs_operand.location)
      end
    end
  end

  class W0712 < W0711
    private
    def check(expression)
      if expression.lhs_operand.logical? && !expression.rhs_operand.logical?
        W(:W0712, expression.lhs_operand.location)
      end
    end
  end

  class W0713 < W0711
    private
    def check(expression)
      if expression.lhs_operand.logical? && expression.rhs_operand.logical?
        W(:W0713, expression.location)
      end
    end
  end

  class W0714 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_and_expression += method(:check)
    end

    private
    def check(expression)
      if expression.lhs_operand.logical? && expression.rhs_operand.logical?
        W(:W0714, expression.location)
      end
    end
  end

  class W0715 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_inclusive_or_expression += method(:check)
    end

    private
    def check(expression)
      if expression.lhs_operand.logical? && expression.rhs_operand.logical?
        W(:W0715, expression.location)
      end
    end
  end

  class W0716 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_additive_expression += method(:check)
      visitor.enter_multiplicative_expression += method(:check)
      visitor.enter_shift_expression += method(:check)
      visitor.enter_and_expression += method(:check)
      visitor.enter_exclusive_or_expression += method(:check)
      visitor.enter_inclusive_or_expression += method(:check)
      visitor.enter_compound_assignment_expression += lambda { |expr|
        case expr.operator.type
        when "+=", "-=", "*=", "/=", "%=", "<<=", ">>=", "&=", "^=", "|="
          check(expr)
        end
      }
    end

    private
    def check(expression)
      if expression.lhs_operand.logical? && expression.rhs_operand.logical?
        W(:W0716, expression.location)
      end
    end
  end

  class W0717 < W0716
    private
    def check(expression)
      if expression.lhs_operand.logical? && !expression.rhs_operand.logical?
        W(:W0717, expression.lhs_operand.location)
      end
    end
  end

  class W0718 < W0716
    private
    def check(expression)
      if !expression.lhs_operand.logical? && expression.rhs_operand.logical?
        W(:W0718, expression.rhs_operand.location)
      end
    end
  end

  class W0726 < W0698
    private
    def check(return_statement)
      return unless return_statement.expression

      if return_type = @current_function.type.return_type
        if return_type.void? && (return_type.const? || return_type.volatile?)
          W(:W0726, return_statement.location,
            @current_function.identifier.value)
        end
      end
    end
  end

  class W0781 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_switch_statement += method(:enter_switch_statement)
      visitor.leave_switch_statement += method(:check)
      visitor.enter_case_labeled_statement += method(:add_exec_path)
      visitor.enter_default_labeled_statement += method(:add_exec_path)
      @exec_path_nums = []
    end

    private
    def enter_switch_statement(node)
      @exec_path_nums.push(0)
    end

    def check(node)
      if exec_path_num = @exec_path_nums.last and exec_path_num < 2
        W(:W0781, node.location)
      end
      @exec_path_nums.pop
    end

    def add_exec_path(node)
      @exec_path_nums[-1] += 1 if node.executed?
    end
  end

  class W0801 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_struct_type_declaration += method(:check)
      visitor.enter_union_type_declaration += method(:check)
    end

    private
    def check(node)
      W(:W0801, node.location) if node.type.members.empty?
    end
  end

  class W0809 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_function_declaration += method(:check_function_name)
      visitor.enter_variable_declaration += method(:check_variable_name)
      visitor.enter_variable_definition += method(:check_variable_name)
      visitor.enter_parameter_definition += method(:check_variable_name)
      visitor.enter_typedef_declaration += method(:check_typedef_name)
      visitor.enter_struct_type_declaration += method(:check_tag_name)
      visitor.enter_union_type_declaration += method(:check_tag_name)
      visitor.enter_enum_type_declaration += method(:check_tag_name)
      visitor.enter_enumerator += method(:check_enumerator_name)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      @function_def_level = 0
    end

    private
    def check_function_name(function_decl)
      check_object_name(function_decl)
    end

    def check_variable_name(variable_decl_or_def)
      check_object_name(variable_decl_or_def)
    end

    def check_typedef_name(typedef_decl)
      if typedef_decl.identifier
        case name = typedef_decl.identifier.value
        when /\A__/, /\A_[A-Z]/, /\A_/
          W(:W0809, typedef_decl.location, name)
        end
      end
    end

    def check_tag_name(type_decl)
      if type_decl.identifier
        case name = type_decl.identifier.value
        when /\A__adlint/
          # NOTE: To ignore AdLint internal tag names.
        when /\A__/, /\A_[A-Z]/, /\A_/
          W(:W0809, type_decl.location, name)
        end
      end
    end

    def check_enumerator_name(enumerator)
      if enumerator.identifier
        case name = enumerator.identifier.value
        when /\A__/, /\A_[A-Z]/, /\A_/
          W(:W0809, enumerator.location, name)
        end
      end
    end

    def enter_function(node)
      check_object_name(node)
      @function_def_level += 1
    end

    def leave_function(node)
      @function_def_level -= 1
    end

    def check_object_name(decl_or_def)
      if decl_or_def.identifier
        case name = decl_or_def.identifier.value
        when /\A__/, /\A_[A-Z]/
          W(:W0809, decl_or_def.location, name)
        when /\A_/
          check_filelocal_object_name(name, decl_or_def)
        end
      end
    end

    def check_filelocal_object_name(name, decl_or_def)
      if @function_def_level == 0
        if scs = decl_or_def.storage_class_specifier and scs.type == :STATIC
          W(:W0809, decl_or_def.location, name)
        end
      end
    end
  end

  class W1030 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_generic_labeled_statement += method(:check)
      visitor.enter_ansi_function_definition += method(:enter_function)
      visitor.leave_ansi_function_definition += method(:leave_function)
      visitor.enter_kandr_function_definition += method(:enter_function)
      visitor.leave_kandr_function_definition += method(:leave_function)
      @labels = nil
    end

    private
    def check(generic_labeled_statement)
      return unless @labels

      label = generic_labeled_statement.label
      if @labels.include?(label.value)
        W(:W1030, label.location, label.value)
      else
        @labels.add(label.value)
      end
    end

    def enter_function(node)
      @labels = Set.new
    end

    def leave_function(node)
      @labels = nil
    end
  end

  class W1033 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:check)
      visitor.enter_kandr_function_definition += method(:check)
      visitor.enter_function_declaration += method(:check)
    end

    private
    def check(definition_or_declaration)
      if return_type = definition_or_declaration.type.return_type
        if return_type.const? || return_type.volatile?
          W(:W1033, definition_or_declaration.location)
        end
      end
    end
  end

end
end
