# Domain of values bound to variables.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/traits"
require "adlint/util"

module AdLint #:nodoc:
module C #:nodoc:

  module ValueDomainFactory
    extend Memoizable

    # NOTE: To avoid huge composite value-domain creation in the interp phase.
    COMPOSITE_MAX_COMPLEXITY = 16
    private_constant :COMPOSITE_MAX_COMPLEXITY

    def equal_to(numeric)
      EqualToValueDomain.new(numeric)
    end
    memoize :equal_to

    def not_equal_to(numeric)
      equal_to(numeric).inversion
    end
    memoize :not_equal_to

    def less_than(numeric)
      LessThanValueDomain.new(numeric)
    end
    memoize :less_than

    def greater_than(numeric)
      GreaterThanValueDomain.new(numeric)
    end
    memoize :greater_than

    def less_than_or_equal_to(numeric)
      less_than(numeric).union(equal_to(numeric))
    end
    memoize :less_than_or_equal_to

    def greater_than_or_equal_to(numeric)
      greater_than(numeric).union(equal_to(numeric))
    end
    memoize :greater_than_or_equal_to

    def of_true
      not_equal_to(0)
    end
    memoize :of_true

    def of_false
      equal_to(0)
    end
    memoize :of_false

    def of_unlimited
      UnlimitedValueDomain.new
    end
    memoize :of_unlimited

    def of_nil
      NilValueDomain.new
    end
    memoize :of_nil

    def of_nan
      NaN.new
    end
    memoize :of_nan

    def of_intersection(lhs_value_domain, rhs_value_domain)
      case lhs_value_domain
      when UndefinedValueDomain
        lhs_value_domain = lhs_value_domain.domain
        undefined = true
      end

      case rhs_value_domain
      when UndefinedValueDomain
        rhs_value_domain = rhs_value_domain.domain
        undefined = true
      end

      case
      when lhs_value_domain.empty?
        intersection = lhs_value_domain
      when rhs_value_domain.empty?
        intersection = rhs_value_domain
      when lhs_value_domain.contain?(rhs_value_domain)
        intersection = rhs_value_domain
      when rhs_value_domain.contain?(lhs_value_domain)
        intersection = lhs_value_domain
      when lhs_value_domain.intersect?(rhs_value_domain)
        intersection = _create_intersection(lhs_value_domain, rhs_value_domain)
      else
        intersection = of_nil
      end

      undefined ? of_undefined(intersection) : intersection
    end
    memoize :of_intersection

    def of_union(lhs_value_domain, rhs_value_domain)
      case lhs_value_domain
      when UndefinedValueDomain
        lhs_value_domain = lhs_value_domain.domain
        undefined = true
      end

      case rhs_value_domain
      when UndefinedValueDomain
        rhs_value_domain = rhs_value_domain.domain
        undefined = true
      end

      case
      when lhs_value_domain.empty?
        union = rhs_value_domain
      when rhs_value_domain.empty?
        union = lhs_value_domain
      when lhs_value_domain.contain?(rhs_value_domain)
        union = lhs_value_domain
      when rhs_value_domain.contain?(lhs_value_domain)
        union = rhs_value_domain
      else
        union = _create_union(lhs_value_domain, rhs_value_domain)
      end

      undefined ? of_undefined(union) : union
    end
    memoize :of_union

    def of_undefined(range_or_value_domain)
      case range_or_value_domain
      when Range
        UndefinedValueDomain.new(range_or_value_domain)
      when ValueDomain
        if range_or_value_domain.undefined?
          range_or_value_domain
        else
          UndefinedValueDomain.new(range_or_value_domain)
        end
      end
    end
    memoize :of_undefined

    def of_ambiguous(undefined)
      AmbiguousValueDomain.new(undefined)
    end
    memoize :of_ambiguous

    def _create_intersection(lhs_value_domain, rhs_value_domain)
      expected = lhs_value_domain.complexity + rhs_value_domain.complexity
      if expected < COMPOSITE_MAX_COMPLEXITY
        IntersectionValueDomain.new(lhs_value_domain, rhs_value_domain)
      else
        of_ambiguous(
          lhs_value_domain.undefined? || rhs_value_domain.undefined?)
      end
    end
    memoize :_create_intersection

    def _create_union(lhs_value_domain, rhs_value_domain)
      expected = lhs_value_domain.complexity + rhs_value_domain.complexity
      if expected < COMPOSITE_MAX_COMPLEXITY
        return UnionValueDomain.new(lhs_value_domain, rhs_value_domain)
      else
        of_ambiguous(
          lhs_value_domain.undefined? || rhs_value_domain.undefined?)
      end
    end
    memoize :_create_union

    def clear_memo
      clear_equal_to_memo
      clear_not_equal_to_memo
      clear_less_than_memo
      clear_greater_than_memo
      clear_less_than_or_equal_to_memo
      clear_greater_than_or_equal_to_memo
      clear_of_true_memo
      clear_of_false_memo
      clear_of_unlimited_memo
      clear_of_nil_memo
      clear_of_nan_memo
      clear_of_intersection_memo
      clear_of_union_memo
      clear_of_undefined_memo
      clear_of_ambiguous_memo
      clear__create_intersection_memo
      clear__create_union_memo
    end
  end

  # == DESCRIPTION
  # === ValueDomain class hierarchy
  #  ValueDomain <-----------------------------+
  #    <--- NilValueDomain                     |
  #    <--- UnlimitedValueDomain               |
  #           <--- NaN                         |
  #    <--- EqualToValueDomain                 |
  #    <--- LessThanValueDomain                |
  #    <--- GreaterThanValueDomain             |
  #    <--- CompositeValueDomain <>------------+
  #           <--- IntersectionValueDomain <------+
  #           <--- UnionValueDomain               |
  #    <--- UndefinedValueDomain <>---------------+
  #    <--- AmbiguousValueDomain
  class ValueDomain
    # NOTE: Instances of ValueDomain class are immutable.  It is safe to share
    #       the instance of the ValueDomain class of the same value.

    extend Memoizable
    extend ValueDomainFactory

    def empty?
      subclass_responsibility
    end

    def nan?
      subclass_responsibility
    end

    def undefined?
      subclass_responsibility
    end

    def ambiguous?
      subclass_responsibility
    end

    def contain?(value_domain_or_numeric)
      case value_domain_or_numeric
      when ValueDomain
        contain_value_domain?(value_domain_or_numeric)
      when Numeric
        contain_value_domain?(ValueDomain.equal_to(value_domain_or_numeric))
      else
        raise TypeError, "`#{value_domain_or_numeric.inspect}' " +
          "must be kind of ValueDomain or Numeric."
      end
    end

    def contain_value_domain?(rhs_domain)
      subclass_responsibility
    end

    def _contain_nil?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _contain_unlimited?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _contain_equal_to?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _contain_less_than?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _contain_greater_than?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _contain_intersection?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _contain_union?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def intersect?(value_domain)
      subclass_responsibility
    end

    def _intersect_nil?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _intersect_unlimited?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _intersect_equal_to?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _intersect_less_than?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _intersect_greater_than?(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def narrow(operator_symbol, operand_value_domain)
      case operator_symbol
      when :==
        _narrow_by_eq(operand_value_domain)
      when :!=
        _narrow_by_ne(operand_value_domain)
      when :<
        _narrow_by_lt(operand_value_domain)
      when :>
        _narrow_by_gt(operand_value_domain)
      when :<=
        _narrow_by_le(operand_value_domain)
      when :>=
        _narrow_by_ge(operand_value_domain)
      else
        self
      end
    end

    def _narrow_by_eq(rhs_domain, lhs_domain = self)
      subclass_responsibility
    end

    def _narrow_nil_by_eq(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_unlimited_by_eq(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_equal_to_by_eq(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_less_than_by_eq(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_greater_than_by_eq(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_by_ne(rhs_domain, lhs_domain = self)
      subclass_responsibility
    end

    def _narrow_nil_by_ne(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_unlimited_by_ne(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_equal_to_by_ne(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_less_than_by_ne(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_greater_than_by_ne(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_by_lt(rhs_domain, lhs_domain = self)
      subclass_responsibility
    end

    def _narrow_nil_by_lt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_unlimited_by_lt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_equal_to_by_lt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_less_than_by_lt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_greater_than_by_lt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_by_gt(rhs_domain, lhs_domain = self)
      subclass_responsibility
    end

    def _narrow_nil_by_gt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_unlimited_by_gt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_equal_to_by_gt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_less_than_by_gt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_greater_than_by_gt(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _narrow_by_le(rhs_domain, lhs_domain = self)
      _narrow_by_lt(rhs_domain).union(_narrow_by_eq(rhs_domain))
    end

    def _narrow_by_ge(rhs_domain, lhs_domain = self)
      _narrow_by_gt(rhs_domain).union(_narrow_by_eq(rhs_domain))
    end

    def widen(operator_symbol, operand_value_domain)
      case operator_symbol
      when :==
        _widen_by_eq(operand_value_domain)
      when :!=
        _widen_by_ne(operand_value_domain)
      when :<
        _widen_by_lt(operand_value_domain)
      when :>
        _widen_by_gt(operand_value_domain)
      when :<=
        _widen_by_le(operand_value_domain)
      when :>=
        _widen_by_ge(operand_value_domain)
      else
        self
      end
    end

    def _widen_by_eq(rhs_domain, lhs_domain = self)
      lhs_domain.union(rhs_domain)
    end

    def _widen_by_ne(rhs_domain, lhs_domain = self)
      lhs_domain.union(rhs_domain.inversion)
    end

    def _widen_by_lt(rhs_domain, lhs_domain = self)
      lhs_domain.union(ValueDomain.of_unlimited.narrow(:<, rhs_domain))
    end

    def _widen_by_gt(rhs_domain, lhs_domain = self)
      lhs_domain.union(ValueDomain.of_unlimited.narrow(:>, rhs_domain))
    end

    def _widen_by_le(rhs_domain, lhs_domain = self)
      _widen_by_lt(rhs_domain).union(_widen_by_eq(rhs_domain))
    end

    def _widen_by_ge(rhs_domain, lhs_domain = self)
      _widen_by_gt(rhs_domain).union(_widen_by_eq(rhs_domain))
    end

    def inversion
      subclass_responsibility
    end

    def ~
      subclass_responsibility
    end

    def +@
      subclass_responsibility
    end

    def -@
      subclass_responsibility
    end

    def +(rhs_domain)
      subclass_responsibility
    end

    def _add_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _add_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _add_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _add_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _add_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def -(rhs_domain)
      self + -rhs_domain
    end

    def *(rhs_domain)
      # NOTE: Operator * cannot be defined by `LHS / (1.0 / RHS)'.
      #       Because `1.0 / RHS' will make NaN, when the value domain of the
      #       right hand side contains 0.
      subclass_responsibility
    end

    def _mul_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _mul_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _mul_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _mul_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _mul_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def /(rhs_domain)
      subclass_responsibility
    end

    def _div_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _div_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _div_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _div_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _div_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def %(rhs_domain)
      self - rhs_domain * (self / rhs_domain).coerce_to_integer
    end

    def &(rhs_domain)
      subclass_responsibility
    end

    def _and_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _and_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _and_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _and_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _and_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def |(rhs_domain)
      subclass_responsibility
    end

    def _or_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _or_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _or_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _or_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _or_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def ^(rhs_domain)
      subclass_responsibility
    end

    def _xor_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _xor_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _xor_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _xor_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _xor_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def <<(rhs_domain)
      subclass_responsibility
    end

    def _shl_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shl_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shl_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shl_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shl_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def >>(rhs_domain)
      subclass_responsibility
    end

    def _shr_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shr_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shr_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shr_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shr_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def !
      subclass_responsibility
    end

    def <(rhs_domain)
      subclass_responsibility
    end

    def _less_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _less_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _less_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _less_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _less_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def >(rhs_domain)
      rhs_domain < self
    end

    def ==(rhs_domain)
      # NOTE: Operator == cannot be defined by `!(LHS < RHS || LHS > RHS)'.
      #       When the value domain of the left hand side is (--<===>-<===>--),
      #       and the value domain of the right hand side is (-------|-------).
      #       `LHS < RHS' should make `true or false' because values in the
      #       left hand side may be less than or greater than the value in
      #       the right hand side.
      #       `LHS > RHS' should make `true or false', too.
      #       So, `!(LHS < RHS) && !(LHS > RHS)' will make `true or false'.
      subclass_responsibility
    end

    def _equal_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _equal_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _equal_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _equal_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _equal_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def !=(rhs_domain)
      # NOTE: Operator != cannot be defined by `!(LHS == RHS)'.
      #       When the value domain of the left hand side or the right hand
      #       side is NilValueDomain, `LHS == RHS' should make `false'.
      #       But `LHS != RHS' should make `false', too.
      subclass_responsibility
    end

    def _not_equal_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _not_equal_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _not_equal_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _not_equal_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _not_equal_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def <=(rhs_domain)
      (self < rhs_domain).logical_or(self == rhs_domain)
    end

    def >=(rhs_domain)
      (self > rhs_domain).logical_or(self == rhs_domain)
    end

    def logical_and(rhs_domain)
      # NOTE: Operator && cannot be defined as a method in Ruby.
      subclass_responsibility
    end

    def _logical_and_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _logical_and_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _logical_and_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _logical_and_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _logical_and_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def logical_or(rhs_domain)
      # NOTE: Operator || cannot be defined as a method in Ruby.
      subclass_responsibility
    end

    def _logical_or_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _logical_or_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _logical_or_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _logical_or_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _logical_or_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def intersection(rhs_domain)
      subclass_responsibility
    end

    def _intersection_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _intersection_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _intersection_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _intersection_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _intersection_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def union(rhs_domain)
      subclass_responsibility
    end

    def _union_nil(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _union_unlimited(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _union_equal_to(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _union_less_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _union_greater_than(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def coerce_to_integer
      subclass_responsibility
    end

    def coerce_to_real
      subclass_responsibility
    end

    def min_value
      subclass_responsibility
    end

    def max_value
      subclass_responsibility
    end

    def each_sample
      subclass_responsibility
    end

    def to_defined_domain
      subclass_responsibility
    end

    def <=>(rhs_domain)
      to_s <=> rhs_domain.to_s
    end

    def eql?(rhs_domain)
      to_s.eql?(rhs_domain.to_s)
    end

    def hash
      to_s.hash
    end
    memoize :hash

    def to_s
      subclass_responsibility
    end

    def complexity
      subclass_responsibility
    end

    private
    def right_shift(lhs_numeric, rhs_numeric)
      if Traits.instance.of_compiler.arithmetic.logical_right_shift
        lhs_numeric.to_i.logical_right_shift(rhs_numeric.to_i)
      else
        lhs_numeric.to_i.arithmetic_right_shift(rhs_numeric.to_i)
      end
    end

    def left_shift(lhs_numeric, rhs_numeric)
      lhs_numeric.to_i.left_shift(rhs_numeric.to_i)
    end
  end

  class NilValueDomain < ValueDomain
    extend Memoizable

    def empty?
      true
    end

    def nan?
      false
    end

    def undefined?
      false
    end

    def ambiguous?
      false
    end

    def contain_value_domain?(rhs_domain)
      rhs_domain._contain_nil?(self)
    end

    def _contain_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_unlimited?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_equal_to?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_less_than?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_greater_than?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_intersection?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_union?(lhs_domain, rhs_domain = self)
      true
    end

    def intersect?(value_domain)
      value_domain._intersect_nil?(self)
    end

    def _intersect_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _intersect_unlimited?(lhs_domain, rhs_domain = self)
      false
    end

    def _intersect_equal_to?(lhs_domain, rhs_domain = self)
      false
    end

    def _intersect_less_than?(lhs_domain, rhs_domain = self)
      false
    end

    def _intersect_greater_than?(lhs_domain, rhs_domain = self)
      false
    end

    def _narrow_by_eq(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_nil_by_eq(lhs_domain)
    end

    def _narrow_nil_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `== NilValueDomain' makes
      #       NilValueDomain.
      lhs_domain
    end

    def _narrow_unlimited_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `== NilValueDomain' makes
      #       NilValueDomain.
      rhs_domain
    end

    def _narrow_equal_to_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `== NilValueDomain' makes
      #       NilValueDomain.
      rhs_domain
    end

    def _narrow_less_than_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `== NilValueDomain' makes
      #       NilValueDomain.
      rhs_domain
    end

    def _narrow_greater_than_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `== NilValueDomain' makes
      #       NilValueDomain.
      rhs_domain
    end

    def _narrow_by_ne(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_nil_by_ne(lhs_domain)
    end

    def _narrow_nil_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `!= NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `!= NilValueDomain' makes
      #       no effect to the target value-domain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       UnlimitedValueDomain should be narrowed to be
      #       UnlimitedValueDomain, and NaN should be narrowed to be NaN.
      lhs_domain
    end

    def _narrow_equal_to_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `!= NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_less_than_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `!= NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_greater_than_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `!= NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_by_lt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_nil_by_lt(lhs_domain)
    end

    def _narrow_nil_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `< NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `< NilValueDomain' makes
      #       no effect to the target value-domain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       UnlimitedValueDomain should be narrowed to be
      #       UnlimitedValueDomain, and NaN should be narrowed to be NaN.
      lhs_domain
    end

    def _narrow_equal_to_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `< NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_less_than_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `< NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_greater_than_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `< NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_by_gt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_nil_by_gt(lhs_domain)
    end

    def _narrow_nil_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `> NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `> NilValueDomain' makes
      #       no effect to the target value-domain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       UnlimitedValueDomain should be narrowed to be
      #       UnlimitedValueDomain, and NaN should be narrowed to be NaN.
      lhs_domain
    end

    def _narrow_equal_to_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `> NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_less_than_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `> NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_greater_than_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing any value-domain by `> NilValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def inversion
      ValueDomain.of_unlimited
    end

    def ~
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      self
    end

    def +@
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      self
    end

    def -@
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      self
    end

    def +(rhs_domain)
      rhs_domain._add_nil(self)
    end

    def _add_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _add_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      rhs_domain
    end

    def _add_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _add_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _add_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def *(rhs_domain)
      rhs_domain._mul_nil(self)
    end

    def _mul_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _mul_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      rhs_domain
    end

    def _mul_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _mul_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _mul_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def /(rhs_domain)
      rhs_domain._div_nil(self)
    end

    def _div_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _div_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _div_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _div_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _div_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def &(rhs_domain)
      rhs_domain.coerce_to_integer._add_nil(coerce_to_integer)
    end

    def _and_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _and_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def |(rhs_domain)
      rhs_domain.coerce_to_integer._or_nil(coerce_to_integer)
    end

    def _or_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _or_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def ^(rhs_domain)
      rhs_domain.coerce_to_integer._xor_nil(coerce_to_integer)
    end

    def _xor_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _xor_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _xor_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _xor_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _xor_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def <<(rhs_domain)
      rhs_domain.coerce_to_integer._shl_nil(coerce_to_integer)
    end

    def _shl_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shl_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _shl_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _shl_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _shl_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def >>(rhs_domain)
      rhs_domain.coerce_to_integer._shr_nil(coerce_to_integer)
    end

    def _shr_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shr_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _shr_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shr_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def _shr_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      rhs_domain
    end

    def !
      # NOTE: NilValueDomain contains no values.
      #       So, logical negation of NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def <(rhs_domain)
      rhs_domain._less_nil(self)
    end

    def _less_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _less_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _less_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _less_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _less_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def ==(rhs_domain)
      rhs_domain._equal_nil(self)
    end

    def _equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _equal_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _equal_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def !=(rhs_domain)
      rhs_domain._not_equal_nil(self)
    end

    def _not_equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _not_equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _not_equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _not_equal_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _not_equal_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def logical_and(rhs_domain)
      rhs_domain._logical_and_nil(self)
    end

    def _logical_and_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only zero value, the logical AND makes false.
      ValueDomain.of_unlimited
    end

    def _logical_and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only zero value, the logical AND makes false.
      ValueDomain.of_unlimited
    end

    def _logical_and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only zero value, the logical AND makes false.
      lhs_domain.value == 0 ? ValueDomain.of_false : ValueDomain.of_unlimited
    end

    def _logical_and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only zero value, the logical AND makes false.
      ValueDomain.of_unlimited
    end

    def _logical_and_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only zero value, the logical AND makes false.
      ValueDomain.of_unlimited
    end

    def logical_or(rhs_domain)
      rhs_domain._logical_or_nil(self)
    end

    def _logical_or_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only non-zero values, the logical OR makes true.
      ValueDomain.of_unlimited
    end

    def _logical_or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only non-zero values, the logical OR makes true.
      ValueDomain.of_unlimited
    end

    def _logical_or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only non-zero values, the logical OR makes true.
      lhs_domain.value == 0 ? ValueDomain.of_unlimited : ValueDomain.of_true
    end

    def _logical_or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only non-zero values, the logical OR makes true.
      lhs_domain.max_value >= 0 ?
        ValueDomain.of_unlimited : ValueDomain.of_true
    end

    def _logical_or_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       If the value-domain of the other side of NilValueDomain contains
      #       only non-zero values, the logical OR makes true.
      lhs_domain.min_value <= 0 ?
        ValueDomain.of_unlimited : ValueDomain.of_true
    end

    def intersection(rhs_domain)
      rhs_domain._intersection_nil(self)
    end

    def _intersection_nil(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _intersection_unlimited(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _intersection_equal_to(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _intersection_less_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _intersection_greater_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def union(rhs_domain)
      rhs_domain._union_nil(self)
    end

    def _union_nil(lhs_domain, rhs_domain = self)
      lhs_domain
    end

    def _union_unlimited(lhs_domain, rhs_domain = self)
      lhs_domain
    end

    def _union_equal_to(lhs_domain, rhs_domain = self)
      lhs_domain
    end

    def _union_less_than(lhs_domain, rhs_domain = self)
      lhs_domain
    end

    def _union_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain
    end

    def coerce_to_integer
      self
    end

    def coerce_to_real
      self
    end

    def min_value
      nil
    end

    def max_value
      nil
    end

    def each_sample
      return ::Enumerator.new(self, :each_sample) unless block_given?
      self
    end

    def to_defined_domain
      self
    end

    def to_s
      "(== Nil)"
    end
    memoize :to_s

    def complexity
      1
    end
  end

  class UnlimitedValueDomain < ValueDomain
    extend Memoizable

    def empty?
      false
    end

    def nan?
      false
    end

    def undefined?
      false
    end

    def ambiguous?
      false
    end

    def contain_value_domain?(rhs_domain)
      rhs_domain._contain_unlimited?(self)
    end

    def _contain_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_unlimited?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_equal_to?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_less_than?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_greater_than?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_intersection?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_union?(lhs_domain, rhs_domain = self)
      false
    end

    def intersect?(value_domain)
      value_domain._intersect_unlimited?(self)
    end

    def _intersect_nil?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes NilValueDomain#_intersect_unlimited?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_unlimited?(lhs_domain, rhs_domain = self)
      true
    end

    def _intersect_equal_to?(lhs_domain, rhs_domain = self)
      true
    end

    def _intersect_less_than?(lhs_domain, rhs_domain = self)
      true
    end

    def _intersect_greater_than?(lhs_domain, rhs_domain = self)
      true
    end

    def _narrow_by_eq(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_unlimited_by_eq(lhs_domain)
    end

    def _narrow_nil_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `== UnlimitedValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `== UnlimitedValueDomain' makes
      #       no effect to the target value-domain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       UnlimitedValueDomain should be narrowed to be
      #       UnlimitedValueDomain, and NaN should be narrowed to be NaN.
      lhs_domain
    end

    def _narrow_equal_to_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `== UnlimitedValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_less_than_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `== UnlimitedValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_greater_than_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `== UnlimitedValueDomain' makes
      #       no effect to the target value-domain.
      lhs_domain
    end

    def _narrow_by_ne(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_unlimited_by_ne(lhs_domain)
    end

    def _narrow_nil_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `!= UnlimitedValueDomain' makes
      #       NilValueDomain.
      lhs_domain
    end

    def _narrow_unlimited_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `!= UnlimitedValueDomain' makes
      #       NilValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       UnlimitedValueDomain should be narrowed to be
      #       UnlimitedValueDomain, and NaN should be narrowed to be NaN.
      ValueDomain.of_nil
    end

    def _narrow_equal_to_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `!= UnlimitedValueDomain' makes
      #       NilValueDomain.
      ValueDomain.of_nil
    end

    def _narrow_less_than_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `!= UnlimitedValueDomain' makes
      #       NilValueDomain.
      ValueDomain.of_nil
    end

    def _narrow_greater_than_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `!= UnlimitedValueDomain' makes
      #       NilValueDomain.
      ValueDomain.of_nil
    end

    def _narrow_by_lt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_unlimited_by_lt(lhs_domain)
    end

    def _narrow_nil_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `< UnlimitedValueDomain' makes
      #       NilValueDomain.
      lhs_domain
    end

    def _narrow_unlimited_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `< UnlimitedValueDomain' makes
      #       NilValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       UnlimitedValueDomain should be narrowed to be
      #       UnlimitedValueDomain, and NaN should be narrowed to be NaN.
      ValueDomain.of_nil
    end

    def _narrow_equal_to_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `< UnlimitedValueDomain' makes
      #       NilValueDomain.
      ValueDomain.of_nil
    end

    def _narrow_less_than_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `< UnlimitedValueDomain' makes
      #       NilValueDomain.
      ValueDomain.of_nil
    end

    def _narrow_greater_than_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `< UnlimitedValueDomain' makes
      #       NilValueDomain.
      ValueDomain.of_nil
    end

    def _narrow_by_gt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_unlimited_by_gt(lhs_domain)
    end

    def _narrow_nil_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `> UnlimitedValueDomain' makes
      #       NilValueDomain.
      lhs_domain
    end

    def _narrow_unlimited_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `> UnlimitedValueDomain' makes
      #       NilValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       UnlimitedValueDomain should be narrowed to be
      #       UnlimitedValueDomain, and NaN should be narrowed to be NaN.
      ValueDomain.of_nil
    end

    def _narrow_equal_to_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `> UnlimitedValueDomain' makes
      #       NilValueDomain.
      ValueDomain.of_nil
    end

    def _narrow_less_than_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `> UnlimitedValueDomain' makes
      #       NilValueDomain.
      ValueDomain.of_nil
    end

    def _narrow_greater_than_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, narrowing any value-domain by `> UnlimitedValueDomain' makes
      #       NilValueDomain.
      ValueDomain.of_nil
    end

    def inversion
      ValueDomain.of_nil
    end

    def ~
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      self
    end

    def +@
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      self
    end

    def -@
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      self
    end

    def +(rhs_domain)
      rhs_domain._add_unlimited(self)
    end

    def _add_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes NilValueDomain#_add_unlimited.
      rhs_domain + lhs_domain
    end

    def _add_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain.nan? ? lhs_domain : rhs_domain
    end

    def _add_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _add_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _add_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def *(rhs_domain)
      rhs_domain._mul_unlimited(self)
    end

    def _mul_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes NilValueDomain#_mul_unlimited.
      rhs_domain * lhs_domain
    end

    def _mul_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain.nan? ? lhs_domain : rhs_domain
    end

    def _mul_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _mul_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _mul_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def /(rhs_domain)
      rhs_domain._div_unlimited(self)
    end

    def _div_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _div_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain.nan? ? lhs_domain : rhs_domain
    end

    def _div_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _div_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _div_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def &(rhs_domain)
      rhs_domain.coerce_to_integer._and_unlimited(coerce_to_integer)
    end

    def _and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes NilValueDomain#_and_unlimited.
      rhs_domain & lhs_domain
    end

    def _and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain.nan? ? lhs_domain : rhs_domain
    end

    def _and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _and_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def |(rhs_domain)
      rhs_domain.coerce_to_integer._or_unlimited(coerce_to_integer)
    end

    def _or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes NilValueDomain#_or_unlimited.
      rhs_domain | lhs_domain
    end

    def _or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain.nan? ? lhs_domain : rhs_domain
    end

    def _or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _or_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def ^(rhs_domain)
      rhs_domain.coerce_to_integer._xor_unlimited(coerce_to_integer)
    end

    def _xor_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes NilValueDomain#_or_unlimited.
      rhs_domain ^ lhs_domain
    end

    def _xor_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain.nan? ? lhs_domain : rhs_domain
    end

    def _xor_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _xor_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def _xor_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      rhs_domain
    end

    def <<(rhs_domain)
      rhs_domain.coerce_to_integer._shl_unlimited(coerce_to_integer)
    end

    def _shl_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shl_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain.nan? ? lhs_domain : rhs_domain
    end

    def _shl_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      rhs_domain
    end

    def _shl_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      rhs_domain
    end

    def _shl_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      rhs_domain
    end

    def >>(rhs_domain)
      rhs_domain.coerce_to_integer._shr_unlimited(coerce_to_integer)
    end

    def _shr_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shr_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-underflow.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain.nan? ? lhs_domain : rhs_domain
    end

    def _shr_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-underflow.
      rhs_domain
    end

    def _shr_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-underflow.
      rhs_domain
    end

    def _shr_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-underflow.
      rhs_domain
    end

    def !
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, logical negation of UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def <(rhs_domain)
      rhs_domain._less_unlimited(self)
    end

    def _less_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _less_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _less_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _less_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _less_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def ==(rhs_domain)
      rhs_domain._equal_unlimited(self)
    end

    def _equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes NilValueDomain#_equal_unlimited.
      rhs_domain == lhs_domain
    end

    def _equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _equal_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _equal_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def !=(rhs_domain)
      rhs_domain._not_equal_unlimited(self)
    end

    def _not_equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes NilValueDomain#_not_equal_nil.
      rhs_domain != lhs_domain
    end

    def _not_equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _not_equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _not_equal_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _not_equal_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def logical_and(rhs_domain)
      rhs_domain._logical_and_unlimited(self)
    end

    def _logical_and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes NilValueDomain#_logical_and_unlimited.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       If the value-domain of the other side of UnlimitedValueDomain
      #       contains only zero value, the logical AND makes false.
      ValueDomain.of_unlimited
    end

    def _logical_and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       If the value-domain of the other side of UnlimitedValueDomain
      #       contains only zero value, the logical AND makes false.
      lhs_domain.value == 0 ? ValueDomain.of_false : ValueDomain.of_unlimited
    end

    def _logical_and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       If the value-domain of the other side of UnlimitedValueDomain
      #       contains only zero value, the logical AND makes false.
      ValueDomain.of_unlimited
    end

    def _logical_and_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       If the value-domain of the other side of UnlimitedValueDomain
      #       contains only zero value, the logical AND makes false.
      ValueDomain.of_unlimited
    end

    def logical_or(rhs_domain)
      rhs_domain._logical_or_unlimited(self)
    end

    def _logical_or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes NilValueDomain#_logical_or_unlimited.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       If the value-domain of the other side of UnlimitedValueDomain
      #       contains only non-zero values, the logical OR makes true.
      ValueDomain.of_unlimited
    end

    def _logical_or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       If the value-domain of the other side of UnlimitedValueDomain
      #       contains only non-zero values, the logical OR makes true.
      lhs_domain.value == 0 ? ValueDomain.of_unlimited : ValueDomain.of_true
    end

    def _logical_or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       If the value-domain of the other side of UnlimitedValueDomain
      #       contains only non-zero values, the logical OR makes true.
      lhs_domain.max_value >= 0 ?
        ValueDomain.of_unlimited : ValueDomain.of_true
    end

    def _logical_or_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       If the value-domain of the other side of UnlimitedValueDomain
      #       contains only non-zero values, the logical OR makes true.
      lhs_domain.min_value <= 0 ?
        ValueDomain.of_unlimited : ValueDomain.of_true
    end

    def intersection(rhs_domain)
      rhs_domain._intersection_unlimited(self)
    end

    def _intersection_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes NilValueDomain#_intersection_unlimited.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_unlimited(lhs_domain, rhs_domain = self)
      lhs_domain
    end

    def _intersection_equal_to(lhs_domain, rhs_domain = self)
      lhs_domain
    end

    def _intersection_less_than(lhs_domain, rhs_domain = self)
      lhs_domain
    end

    def _intersection_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain
    end

    def union(rhs_domain)
      rhs_domain._union_unlimited(self)
    end

    def _union_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes NilValueDomain#_union_unlimited.
      rhs_domain.union(lhs_domain)
    end

    def _union_unlimited(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _union_equal_to(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _union_less_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _union_greater_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def coerce_to_integer
      self
    end

    def coerce_to_real
      self
    end

    def min_value
      nil
    end

    def max_value
      nil
    end

    def each_sample
      return ::Enumerator.new(self, :each_sample) unless block_given?
      yield(0)
      self
    end

    def to_defined_domain
      self
    end

    def to_s
      "(== Unlimited)"
    end
    memoize :to_s

    def complexity
      1
    end
  end

  # NOTE: To increase the coverage of the analysis, NaN should not be derived
  #       from NilValueDomain but UnlimitedValueDomain.
  class NaN < UnlimitedValueDomain
    def nan?
      true
    end

    extend Memoizable

    def to_s
      "(== NaN)"
    end
    memoize :to_s
  end

  class EqualToValueDomain < ValueDomain
    extend Memoizable

    def initialize(value)
      if value
        @value = value
      else
        raise ArgumentError, "equal to nil?"
      end
    end

    attr_reader :value

    def empty?
      false
    end

    def nan?
      false
    end

    def undefined?
      false
    end

    def ambiguous?
      false
    end

    def contain_value_domain?(rhs_domain)
      rhs_domain._contain_equal_to?(self)
    end

    def _contain_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_unlimited?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_equal_to?(lhs_domain, rhs_domain = self)
      lhs_domain.value == rhs_domain.value
    end

    def _contain_less_than?(lhs_domain, rhs_domain = self)
      lhs_domain.max_value >= rhs_domain.value
    end

    def _contain_greater_than?(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.value
    end

    def _contain_intersection?(lhs_domain, rhs_domain = self)
      lhs_domain.domain_pair.all? do |lhs|
        lhs.contain_value_domain?(rhs_domain)
      end
    end

    def _contain_union?(lhs_domain, rhs_domain = self)
      lhs_domain.domain_pair.any? do |lhs|
        lhs.contain_value_domain?(rhs_domain)
      end
    end

    def intersect?(value_domain)
      value_domain._intersect_equal_to?(self)
    end

    def _intersect_nil?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes NilValueDomain#_intersect_equal_to?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_unlimited?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes
      #       UnlimitedValueDomain#_intersect_equal_to?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_equal_to?(lhs_domain, rhs_domain = self)
      lhs_domain.value == rhs_domain.value
    end

    def _intersect_less_than?(lhs_domain, rhs_domain = self)
      lhs_domain.max_value >= rhs_domain.value
    end

    def _intersect_greater_than?(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.value
    end

    def _narrow_by_eq(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_equal_to_by_eq(lhs_domain)
    end

    def _narrow_nil_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_eq(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _narrow_equal_to_by_eq(lhs_domain, rhs_domain = self)
      if lhs_domain.value == rhs_domain.value
        # NOTE: Narrowing `------|------' by `== ------|-----' makes
        #       `------|------'.
        lhs_domain
      else
        # NOTE: Narrowing `---|---------' by `== --------|---' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_less_than_by_eq(lhs_domain, rhs_domain = self)
      if lhs_domain.max_value >= rhs_domain.value
        # NOTE: Narrowing `=========>---' by `== ---|--------' makes
        #       `---|---------'.
        rhs_domain
      else
        # NOTE: Narrowing `===>---------' by `== --------|---' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_greater_than_by_eq(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value <= rhs_domain.value
        # NOTE: Narrowing `---<=========' by `== --------|---' makes
        #       `---------|---'.
        rhs_domain
      else
        # NOTE: Narrowing `---------<===' by `== ---|--------' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_by_ne(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_equal_to_by_ne(lhs_domain)
    end

    def _narrow_nil_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_ne(lhs_domain, rhs_domain = self)
      rhs_domain.inversion
    end

    def _narrow_equal_to_by_ne(lhs_domain, rhs_domain = self)
      if lhs_domain.value == rhs_domain.value
        # NOTE: Narrowing `------|------' by `!= ------|-----' makes
        #       `-------------'.
        ValueDomain.of_nil
      else
        # NOTE: Narrowing `---|---------' by `!= --------|---' makes
        #       `---|---------'.
        lhs_domain
      end
    end

    def _narrow_less_than_by_ne(lhs_domain, rhs_domain = self)
      if lhs_domain.max_value >= rhs_domain.value
        # NOTE: Narrowing `=========>---' by `!= ------|------' makes
        #       `=====>-<=>---'.
        lhs_domain.intersection(rhs_domain.inversion)
      else
        # NOTE: Narrowing `===>---------' by `!= ------|------' makes
        #       `===>---------'.
        lhs_domain
      end
    end

    def _narrow_greater_than_by_ne(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value <= rhs_domain.value
        # NOTE: Narrowing `---<=========' by `!= ------|------' makes
        #       `---<=>-<====='.
        lhs_domain.intersection(rhs_domain.inversion)
      else
        lhs_domain
      end
    end

    def _narrow_by_lt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_equal_to_by_lt(lhs_domain)
    end

    def _narrow_nil_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_lt(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(rhs_domain.value)
    end

    def _narrow_equal_to_by_lt(lhs_domain, rhs_domain = self)
      lhs_domain.value < rhs_domain.value ? lhs_domain : ValueDomain.of_nil
    end

    def _narrow_less_than_by_lt(lhs_domain, rhs_domain = self)
      if lhs_domain.max_value >= rhs_domain.value
        # NOTE: Narrowing `=========>---' by `< ------|------' makes
        #       `=====>-------'.
        ValueDomain.less_than(rhs_domain.value)
      else
        lhs_domain
      end
    end

    def _narrow_greater_than_by_lt(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value <= rhs_domain.value
        # NOTE: Narrowing `---<=========' by `< ------|------' makes
        #       `---<=>-------'.
        lhs_domain.intersection(ValueDomain.less_than(rhs_domain.value))
      else
        ValueDomain.of_nil
      end
    end

    def _narrow_by_gt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_equal_to_by_gt(lhs_domain)
    end

    def _narrow_nil_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_gt(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(rhs_domain.value)
    end

    def _narrow_equal_to_by_gt(lhs_domain, rhs_domain = self)
      if lhs_domain.value > rhs_domain.value
        # NOTE: Narrowing `---------|---' by `> ---|---------' makes
        #       `---------|---'.
        lhs_domain
      else
        # NOTE: Narrowing `---|---------' by `> ---------|---' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_less_than_by_gt(lhs_domain, rhs_domain = self)
      if lhs_domain.max_value >= rhs_domain.value
        # NOTE: Narrowing `=========>---' by `> ---|---------' makes
        #       `---<=====>---'.
        ValueDomain.greater_than(rhs_domain.value).intersection(lhs_domain)
      else
        # NOTE: Narrowing `===>---------' by `> ------|------' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_greater_than_by_gt(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value <= rhs_domain.value
        # NOTE: Narrowing `---<=========' by `> ------|------' makes
        #       `------<======'.
        ValueDomain.greater_than(rhs_domain.value)
      else
        # NOTE: Narrowing `---------<===' by `> ---|---------' makes
        #       `---------<==='.
        lhs_domain
      end
    end

    def inversion
      ValueDomain.less_than(@value).union(ValueDomain.greater_than(@value))
    end

    def ~
      ValueDomain.equal_to(~coerce_to_integer.value)
    end

    def +@
      self
    end

    def -@
      ValueDomain.equal_to(-@value)
    end

    def +(rhs_domain)
      rhs_domain._add_equal_to(self)
    end

    def _add_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes NilValueDomain#_add_equal_to.
      rhs_domain + lhs_domain
    end

    def _add_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes UnlimitedValueDomain#_add_equal_to.
      rhs_domain + lhs_domain
    end

    def _add_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.equal_to(lhs_domain.value + rhs_domain.value)
    end

    def _add_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(lhs_domain.value + rhs_domain.value)
    end

    def _add_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(lhs_domain.value + rhs_domain.value)
    end

    def *(rhs_domain)
      rhs_domain._mul_equal_to(self)
    end

    def _mul_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes NilValueDomain#_mul_equal_to.
      rhs_domain * lhs_domain
    end

    def _mul_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes UnlimitedValueDomain#_mul_equal_to.
      rhs_domain * lhs_domain
    end

    def _mul_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.equal_to(lhs_domain.value * rhs_domain.value)
    end

    def _mul_less_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value == 0
        ValueDomain.equal_to(0)
      else
        case
        when lhs_domain.value <= 0 && rhs_domain.value < 0
          ValueDomain.greater_than(lhs_domain.value * rhs_domain.value)
        when lhs_domain.value > 0 && rhs_domain.value < 0
          ValueDomain.of_unlimited
        when lhs_domain.value <= 0 && rhs_domain.value > 0
          ValueDomain.less_than(lhs_domain.value * rhs_domain.value)
        when lhs_domain.value > 0 && rhs_domain.value > 0
          ValueDomain.of_unlimited
        end
      end
    end

    def _mul_greater_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value == 0
        ValueDomain.equal_to(0)
      else
        case
        when lhs_domain.value >= 0 && rhs_domain.value < 0
          ValueDomain.less_than(lhs_domain.value * rhs_domain.value)
        when lhs_domain.value < 0 && rhs_domain.value < 0
          ValueDomain.of_unlimited
        when lhs_domain.value >= 0 && rhs_domain.value > 0
          ValueDomain.greater_than(lhs_domain.value * rhs_domain.value)
        when lhs_domain.value < 0 && rhs_domain.value > 0
          ValueDomain.of_unlimited
        end
      end
    end

    def /(rhs_domain)
      rhs_domain._div_equal_to(self)
    end

    def _div_nil(lhs_domain, rhs_domain = self)
      rhs_domain.value == 0 ? ValueDomain.of_nan : lhs_domain
    end

    def _div_unlimited(lhs_domain, rhs_domain = self)
      rhs_domain.value == 0 ? ValueDomain.of_nan : lhs_domain
    end

    def _div_equal_to(lhs_domain, rhs_domain = self)
      if rhs_domain.value == 0
        ValueDomain.of_nan
      else
        ValueDomain.equal_to(lhs_domain.value / rhs_domain.value)
      end
    end

    def _div_less_than(lhs_domain, rhs_domain = self)
      case
      when rhs_domain.value < 0
        ValueDomain.greater_than(lhs_domain.value / rhs_domain.value)
      when rhs_domain.value == 0
        ValueDomain.of_nan
      when rhs_domain.value > 0
        ValueDomain.less_than(lhs_domain.value / rhs_domain.value)
      end
    end

    def _div_greater_than(lhs_domain, rhs_domain = self)
      case
      when rhs_domain.value < 0
        ValueDomain.less_than(lhs_domain.value / rhs_domain.value)
      when rhs_domain.value == 0
        ValueDomain.of_nan
      when rhs_domain.value > 0
        ValueDomain.greater_than(lhs_domain.value / rhs_domain.value)
      end
    end

    def &(rhs_domain)
      rhs_domain.coerce_to_integer._and_equal_to(coerce_to_integer)
    end

    def _and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes NilValueDomain#_and_equal_to.
      rhs_domain & lhs_domain
    end

    def _and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes UnlimitedValueDomain#_and_equal_to.
      rhs_domain & lhs_domain
    end

    def _and_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.equal_to(lhs_domain.value & rhs_domain.value)
    end

    def _and_less_than(lhs_domain, rhs_domain = self)
      case
      when rhs_domain.value < 0
        ValueDomain.less_than(0)
      when rhs_domain.value == 0
        ValueDomain.equal_to(0)
      when rhs_domain.value > 0
        ValueDomain.greater_than(0)
      end
    end

    def _and_greater_than(lhs_domain, rhs_domain = self)
      case
      when rhs_domain.value < 0
        ValueDomain.less_than(0)
      when rhs_domain.value == 0
        ValueDomain.equal_to(0)
      when rhs_domain.value > 0
        ValueDomain.greater_than(0)
      end
    end

    def |(rhs_domain)
      rhs_domain.coerce_to_integer._or_equal_to(coerce_to_integer)
    end

    def _or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes NilValueDomain#_or_equal_to.
      rhs_domain | lhs_domain
    end

    def _or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes UnlimitedValueDomain#_or_equal_to.
      rhs_domain | lhs_domain
    end

    def _or_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.equal_to(lhs_domain.value | rhs_domain.value)
    end

    def _or_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(lhs_domain.value | rhs_domain.value)
    end

    def _or_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(lhs_domain.value | rhs_domain.value)
    end

    def ^(rhs_domain)
      rhs_domain.coerce_to_integer._xor_equal_to(coerce_to_integer)
    end

    def _xor_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes NilValueDomain#_xor_equal_to.
      rhs_domain ^ lhs_domain
    end

    def _xor_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes UnlimitedValueDomain#_xor_equal_to.
      rhs_domain ^ lhs_domain
    end

    def _xor_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.equal_to(lhs_domain.value ^ rhs_domain.value)
    end

    def _xor_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(lhs_domain.value ^ rhs_domain.value)
    end

    def _xor_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(lhs_domain.value ^ rhs_domain.value)
    end

    def <<(rhs_domain)
      rhs_domain.coerce_to_integer._shl_equal_to(coerce_to_integer)
    end

    def _shl_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shl_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain
    end

    def _shl_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.equal_to(left_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shl_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(left_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shl_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(left_shift(lhs_domain.value, rhs_domain.value))
    end

    def >>(rhs_domain)
      rhs_domain.coerce_to_integer._shr_equal_to(coerce_to_integer)
    end

    def _shr_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shr_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain
    end

    def _shr_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.equal_to(right_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shr_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(right_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shr_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(right_shift(lhs_domain.value, rhs_domain.value))
    end

    def !
      @value == 0 ? ValueDomain.of_true : ValueDomain.of_false
    end

    def <(rhs_domain)
      rhs_domain._less_equal_to(self)
    end

    def _less_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _less_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _less_equal_to(lhs_domain, rhs_domain = self)
      lhs_domain.value < rhs_domain.value ?
        ValueDomain.of_true : ValueDomain.of_false
    end

    def _less_less_than(lhs_domain, rhs_domain = self)
      lhs_domain.max_value < rhs_domain.value ?
        ValueDomain.of_true : ValueDomain.of_unlimited
    end

    def _less_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value > rhs_domain.value ?
        ValueDomain.of_false : ValueDomain.of_unlimited
    end

    def ==(rhs_domain)
      rhs_domain._equal_equal_to(self)
    end

    def _equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes NilValueDomain#_equal_equal_to.
      rhs_domain == lhs_domain
    end

    def _equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes UnlimitedValueDomain#_equal_equal_to.
      rhs_domain == lhs_domain
    end

    def _equal_equal_to(lhs_domain, rhs_domain = self)
      lhs_domain.value == rhs_domain.value ?
        ValueDomain.of_true : ValueDomain.of_false
    end

    def _equal_less_than(lhs_domain, rhs_domain = self)
      lhs_domain.max_value >= rhs_domain.value ?
        ValueDomain.of_unlimited : ValueDomain.of_false
    end

    def _equal_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.value ?
        ValueDomain.of_unlimited : ValueDomain.of_false
    end

    def !=(rhs_domain)
      rhs_domain._not_equal_equal_to(self)
    end

    def _not_equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes NilValueDomain#_not_equal_equal_to.
      rhs_domain != lhs_domain
    end

    def _not_equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes UnlimitedValueDomain#_not_equal_equal_to.
      rhs_domain != lhs_domain
    end

    def _not_equal_equal_to(lhs_domain, rhs_domain = self)
      lhs_domain.value != rhs_domain.value ?
        ValueDomain.of_true : ValueDomain.of_false
    end

    def _not_equal_less_than(lhs_domain, rhs_domain = self)
      lhs_domain.max_value < rhs_domain.value ?
        ValueDomain.of_true : ValueDomain.of_unlimited
    end

    def _not_equal_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value > rhs_domain.value ?
        ValueDomain.of_true : ValueDomain.of_unlimited
    end

    def logical_and(rhs_domain)
      rhs_domain._logical_and_equal_to(self)
    end

    def _logical_and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes NilValueDomain#_logical_and_equal_to.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes UnlimitedValueDomain#_logical_and_equal_to.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_equal_to(lhs_domain, rhs_domain = self)
      if lhs_domain.value == 0 || rhs_domain.value == 0
        ValueDomain.of_false
      else
        ValueDomain.of_true
      end
    end

    def _logical_and_less_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value == 0
        ValueDomain.of_false
      else
        lhs_domain.value < 0 ? ValueDomain.of_true : ValueDomain.of_unlimited
      end
    end

    def _logical_and_greater_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value == 0
        ValueDomain.of_false
      else
        lhs_domain.value > 0 ? ValueDomain.of_true : ValueDomain.of_unlimited
      end
    end

    def logical_or(rhs_domain)
      rhs_domain._logical_or_equal_to(self)
    end

    def _logical_or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes NilValueDomain#_logical_or_equal_to.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes UnlimitedValueDomain#_logical_or_equal_to.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_equal_to(lhs_domain, rhs_domain = self)
      if lhs_domain.value == 0 && rhs_domain.value == 0
        ValueDomain.of_false
      else
        ValueDomain.of_true
      end
    end

    def _logical_or_less_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value == 0
        lhs_domain.value < 0 ? ValueDomain.of_true : ValueDomain.of_unlimited
      else
        ValueDomain.of_true
      end
    end

    def _logical_or_greater_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value == 0
        lhs_domain.value > 0 ? ValueDomain.of_true : ValueDomain.of_unlimited
      else
        ValueDomain.of_true
      end
    end

    def intersection(rhs_domain)
      rhs_domain._intersection_equal_to(self)
    end

    def _intersection_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes NilValueDomain#_intersection_equal_to.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes UnlimitedValueDomain#_intersection_equal_to.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_equal_to(lhs_domain, rhs_domain = self)
      lhs_domain.value == rhs_domain.value ? lhs_domain : ValueDomain.of_nil
    end

    def _intersection_less_than(lhs_domain, rhs_domain = self)
      lhs_domain.max_value >= rhs_domain.value ?
        rhs_domain : ValueDomain.of_nil
    end

    def _intersection_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.value ?
        rhs_domain : ValueDomain.of_nil
    end

    def union(rhs_domain)
      rhs_domain._union_equal_to(self)
    end

    def _union_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes NilValueDomain#_union_equal_to.
      rhs_domain.union(lhs_domain)
    end

    def _union_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes UnlimitedValueDomain#_union_equal_to.
      rhs_domain.union(lhs_domain)
    end

    def _union_equal_to(lhs_domain, rhs_domain = self)
      lhs_domain.value == rhs_domain.value ?
        lhs_domain : ValueDomain._create_union(lhs_domain, rhs_domain)
    end

    def _union_less_than(lhs_domain, rhs_domain = self)
      lhs_domain.max_value >= rhs_domain.value ?
        lhs_domain : ValueDomain._create_union(lhs_domain, rhs_domain)
    end

    def _union_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.value ?
        lhs_domain : ValueDomain._create_union(lhs_domain, rhs_domain)
    end

    def coerce_to_integer
      @value.integer? ? self : ValueDomain.equal_to(@value.to_i)
    end

    def coerce_to_real
      @value.real? ? self : ValueDomain.equal_to(@value.to_f)
    end

    def min_value
      @value
    end

    def max_value
      @value
    end

    def each_sample
      return ::Enumerator.new(self, :each_sample) unless block_given?
      yield(@value)
      self
    end

    def to_defined_domain
      self
    end

    def to_s
      "(== #{@value})"
    end
    memoize :to_s

    def complexity
      1
    end
  end

  class LessThanValueDomain < ValueDomain
    extend Memoizable

    def initialize(value)
      if value
        @value = value
      else
        raise ArgumentError, "less than nil?"
      end
    end

    attr_reader :value

    def empty?
      false
    end

    def nan?
      false
    end

    def undefined?
      false
    end

    def ambiguous?
      false
    end

    def contain_value_domain?(rhs_domain)
      rhs_domain._contain_less_than?(self)
    end

    def _contain_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_unlimited?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_equal_to?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_less_than?(lhs_domain, rhs_domain = self)
      lhs_domain.value >= rhs_domain.value
    end

    def _contain_greater_than?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_intersection?(lhs_domain, rhs_domain = self)
      lhs_domain.domain_pair.all? do |lhs|
        lhs.contain_value_domain?(rhs_domain)
      end
    end

    def _contain_union?(lhs_domain, rhs_domain = self)
      lhs_domain.domain_pair.any? do |lhs|
        lhs.contain_value_domain?(rhs_domain)
      end
    end

    def intersect?(value_domain)
      value_domain._intersect_less_than?(self)
    end

    def _intersect_nil?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes NilValueDomain#_intersect_less_than?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_unlimited?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes
      #       UnlimitedValueDomain#_intersect_less_than?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_equal_to?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes EqualToValueDomain#_intersect_less_than?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_less_than?(lhs_domain, rhs_domain = self)
      true
    end

    def _intersect_greater_than?(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.max_value
    end

    def _narrow_by_eq(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_less_than_by_eq(lhs_domain)
    end

    def _narrow_nil_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_eq(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _narrow_equal_to_by_eq(lhs_domain, rhs_domain = self)
      if lhs_domain.value < rhs_domain.value
        # NOTE: Narrowing `------|------' by `== =========>---' makes
        #       `------|------'.
        lhs_domain
      else
        # NOTE: Narrowing `---------|---' by `== ===>---------' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_less_than_by_eq(lhs_domain, rhs_domain = self)
      if lhs_domain.value < rhs_domain.value
        # NOTE: Narrowing `===>---------' by `== =========>---' makes
        #       `===>---------'.
        lhs_domain
      else
        # NOTE: Narrowing `=========>---' by `== ===>---------' makes
        #       `===>---------'.
        rhs_domain
      end
    end

    def _narrow_greater_than_by_eq(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value > rhs_domain.max_value
        ValueDomain.of_nil
      else
        # NOTE: Narrowing `---<=========' by `== =========>---' makes
        #       `---<=====>---'.
        lhs_domain.intersection(rhs_domain)
      end
    end

    def _narrow_by_ne(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_less_than_by_ne(lhs_domain)
    end

    def _narrow_nil_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_ne(lhs_domain, rhs_domain = self)
      rhs_domain.inversion
    end

    def _narrow_equal_to_by_ne(lhs_domain, rhs_domain = self)
      if lhs_domain.value <= rhs_domain.max_value
        # NOTE: Narrowing `---|---------' by `!= =========>---' makes
        #       `-------------'.
        ValueDomain.of_nil
      else
        # NOTE: Narrowing `---------|---' by `!= ===>---------' makes
        #       `---------|---'.
        lhs_domain
      end
    end

    def _narrow_less_than_by_ne(lhs_domain, rhs_domain = self)
      if lhs_domain.max_value <= rhs_domain.max_value
        # NOTE: Narrowing `===>---------' by `!= =========>---' makes
        #       `-------------'.
        ValueDomain.of_nil
      else
        # NOTE: Narrowing `=========>---' by `!= ===>---------' makes
        #       `---<=====>---'.
        lhs_domain.intersection(rhs_domain.inversion)
      end
    end

    def _narrow_greater_than_by_ne(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value <= rhs_domain.max_value
        # NOTE: Narrowing `---<=========' by `!= =========>---' makes
        #       `---------<==='.
        rhs_domain.inversion
      else
        # NOTE: Narrowing `---------<===' by `!= ===>---------' makes
        #       `---------<==='.
        lhs_domain
      end
    end

    def _narrow_by_lt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_less_than_by_lt(lhs_domain)
    end

    def _narrow_nil_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `=============' by `< ======>------' makes
      #       `-------------'.
      ValueDomain.of_nil
    end

    def _narrow_equal_to_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `------|------' by `< ======>------' makes
      #       `-------------'.
      ValueDomain.of_nil
    end

    def _narrow_less_than_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `=========>---' by `< =======>-----' makes
      #       `-------------'.
      ValueDomain.of_nil
    end

    def _narrow_greater_than_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `---<=========' by `< =========>---' makes
      #       `-------------'.
      ValueDomain.of_nil
    end

    def _narrow_by_gt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_less_than_by_gt(lhs_domain)
    end

    def _narrow_nil_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `=============' by `> ======>------' makes
      #       `------<======'.
      rhs_domain.inversion
    end

    def _narrow_equal_to_by_gt(lhs_domain, rhs_domain = self)
      if lhs_domain.value > rhs_domain.max_value
        # NOTE: Narrowing `---------|---' by `> ===>---------' makes
        #       `---------|---'.
        lhs_domain
      else
        # NOTE: Narrowing `---|---------' by `> =========>---' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_less_than_by_gt(lhs_domain, rhs_domain = self)
      if lhs_domain.max_value > rhs_domain.max_value
        # NOTE: Narrowing `=========>---' by `> ===>---------' makes
        #       `---<=====>---'.
        rhs_domain.inversion.intersection(lhs_domain)
      else
        # NOTE: Narrowing `===>---------' by `> =========>---' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_greater_than_by_gt(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value < rhs_domain.max_value
        # NOTE: Narrowing `---<=========' by `> =========>---' makes
        #       `---------<==='.
        rhs_domain.inversion
      else
        # NOTE: Narrowing `---------<===' by `> ===>---------' makes
        #       `---------<==='.
        lhs_domain
      end
    end

    def inversion
      ValueDomain.greater_than_or_equal_to(@value)
    end

    def ~
      ValueDomain.less_than(~coerce_to_integer.value)
    end

    def +@
      self
    end

    def -@
      ValueDomain.greater_than(-@value)
    end

    def +(rhs_domain)
      rhs_domain._add_less_than(self)
    end

    def _add_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes NilValueDomain#_add_less_than.
      rhs_domain + lhs_domain
    end

    def _add_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes UnlimitedValueDomain#_add_less_than.
      rhs_domain + lhs_domain
    end

    def _add_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes EqualToValueDomain#_add_less_than.
      rhs_domain + lhs_domain
    end

    def _add_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(lhs_domain.value + rhs_domain.value)
    end

    def _add_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def *(rhs_domain)
      rhs_domain._mul_less_than(self)
    end

    def _mul_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes NilValueDomain#_mul_less_than.
      rhs_domain * lhs_domain
    end

    def _mul_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes UnlimitedValueDomain#_mul_less_than.
      rhs_domain * lhs_domain
    end

    def _mul_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes EqualToValueDomain#_mul_less_than.
      rhs_domain * lhs_domain
    end

    def _mul_less_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value > 0
        ValueDomain.of_unlimited
      else
        case
        when lhs_domain.value <= 0
          ValueDomain.greater_than(0)
        when lhs_domain.value > 0
          ValueDomain.less_than(0)
        end
      end
    end

    def _mul_greater_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value > 0
        ValueDomain.of_unlimited
      else
        case
        when lhs_domain.value >= 0
          ValueDomain.less_than(0)
        when lhs_domain.value < 0
          ValueDomain.greater_than(0)
        end
      end
    end

    def /(rhs_domain)
      rhs_domain._div_less_than(self)
    end

    def _div_nil(lhs_domain, rhs_domain = self)
      rhs_domain.max_value >= 0 ? ValueDomain.of_nan : lhs_domain
    end

    def _div_unlimited(lhs_domain, rhs_domain = self)
      rhs_domain.max_value >= 0 ? ValueDomain.of_nan : lhs_domain
    end

    def _div_equal_to(lhs_domain, rhs_domain = self)
      if rhs_domain.value >= 0
        ValueDomain.of_nan
      else
        case
        when lhs_domain.value < 0
          ValueDomain.greater_than(0).intersection(
            ValueDomain.less_than(lhs_domain.value / rhs_domain.value))
        when lhs_domain.value == 0
          ValueDomain.equal_to(0)
        when lhs_domain.value > 0
          ValueDomain.greater_than(
            lhs_domain.value / rhs_domain.value).intersection(
              ValueDomain.less_than(0))
        end
      end
    end

    def _div_less_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value > 0
        ValueDomain.of_nan
      else
        case
        when lhs_domain.value <= 0
          ValueDomain.greater_than(0)
        when lhs_domain.value > 0
          ValueDomain.less_than(0)
        end
      end
    end

    def _div_greater_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value > 0
        ValueDomain.of_nan
      else
        case
        when lhs_domain.value >= 0
          ValueDomain.less_than(0)
        when lhs_domain.value < 0
          ValueDomain.greater_than(0)
        end
      end
    end

    def &(rhs_domain)
      rhs_domain.coerce_to_integer._and_less_than(coerce_to_integer)
    end

    def _and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes NilValueDomain#_and_less_than.
      rhs_domain & lhs_domain
    end

    def _and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes UnlimitedValueDomain#_and_less_than.
      rhs_domain & lhs_domain
    end

    def _and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes EqualToValueDomain#_and_less_than.
      rhs_domain & lhs_domain
    end

    def _and_less_than(lhs_domain, rhs_domain = self)
      if lhs_domain.value < 0 && rhs_domain.value < 0
        ValueDomain.less_than(lhs_domain.value & rhs_domain.value)
      else
        ValueDomain.of_unlimited
      end
    end

    def _and_greater_than(lhs_domain, rhs_domain = self)
      if lhs_domain.value > 0 && rhs_domain.value < 0
        ValueDomain.less_than(lhs_domain.value & rhs_domain.value)
      else
        ValueDomain.of_unlimited
      end
    end

    def |(rhs_domain)
      rhs_domain.coerce_to_integer._or_less_than(coerce_to_integer)
    end

    def _or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes NilValueDomain#_or_less_than.
      rhs_domain | lhs_domain
    end

    def _or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes UnlimitedValueDomain#_or_less_than.
      rhs_domain | lhs_domain
    end

    def _or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes EqualToValueDomain#_or_less_than.
      rhs_domain | lhs_domain
    end

    def _or_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(lhs_domain.value | rhs_domain.value)
    end

    def _or_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def ^(rhs_domain)
      rhs_domain.coerce_to_integer._xor_less_than(coerce_to_integer)
    end

    def _xor_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes NilValueDomain#_xor_less_than.
      rhs_domain ^ lhs_domain
    end

    def _xor_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes UnlimitedValueDomain#_xor_less_than.
      rhs_domain ^ lhs_domain
    end

    def _xor_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes EqualToValueDomain#_xor_less_than.
      rhs_domain ^ lhs_domain
    end

    def _xor_less_than(lhs_domain, rhs_domain = self)
      case
      when lhs_domain.value < 0 && rhs_domain.value < 0
        ValueDomain.greater_than(0)
      when lhs_domain.value < 0 || rhs_domain.value < 0
        ValueDomain.less_than(0)
      else
        ValueDomain.greater_than(0)
      end
    end

    def _xor_greater_than(lhs_domain, rhs_domain = self)
      case
      when lhs_domain.value < 0 && rhs_domain.value < 0
        ValueDomain.greater_than(0)
      when lhs_domain.value < 0 || rhs_domain.value < 0
        ValueDomain.less_than(0)
      else
        ValueDomain.greater_than(0)
      end
    end

    def <<(rhs_domain)
      rhs_domain.coerce_to_integer._shl_less_than(coerce_to_integer)
    end

    def _shl_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shl_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain
    end

    def _shl_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(left_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shl_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(left_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shl_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(left_shift(lhs_domain.value, rhs_domain.value))
    end

    def >>(rhs_domain)
      rhs_domain.coerce_to_integer._shr_less_than(coerce_to_integer)
    end

    def _shr_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shr_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain
    end

    def _shr_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(right_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shr_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(right_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shr_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(right_shift(lhs_domain.value, rhs_domain.value))
    end

    def !
      @value < 0 ? ValueDomain.of_false : ValueDomain.of_unlimited
    end

    def <(rhs_domain)
      rhs_domain._less_less_than(self)
    end

    def _less_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _less_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _less_equal_to(lhs_domain, rhs_domain = self)
      lhs_domain.value >= rhs_domain.max_value ?
        ValueDomain.of_false : ValueDomain.of_unlimited
    end

    def _less_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def _less_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value >= rhs_domain.max_value ?
        ValueDomain.of_false : ValueDomain.of_unlimited
    end

    def ==(rhs_domain)
      rhs_domain._equal_less_than(self)
    end

    def _equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes NilValueDomain#_equal_less_than.
      rhs_domain == lhs_domain
    end

    def _equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes UnlimitedValueDomain#_equal_less_than.
      rhs_domain == lhs_domain
    end

    def _equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes EqualToValueDomain#_equal_less_than.
      rhs_domain == lhs_domain
    end

    def _equal_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def _equal_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.max_value ?
        ValueDomain.of_unlimited : ValueDomain.of_false
    end

    def !=(rhs_domain)
      rhs_domain._not_equal_less_than(self)
    end

    def _not_equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes NilValueDomain#_not_equal_less_than.
      rhs_domain != lhs_domain
    end

    def _not_equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes UnlimitedValueDomain#_not_equal_less_than.
      rhs_domain != lhs_domain
    end

    def _not_equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes EqualToValueDomain#_not_equal_less_than.
      rhs_domain != lhs_domain
    end

    def _not_equal_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def _not_equal_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.max_value ?
        ValueDomain.of_unlimited : ValueDomain.of_true
    end

    def logical_and(rhs_domain)
      rhs_domain._logical_and_less_than(self)
    end

    def _logical_and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes NilValueDomain#_logical_and_less_than.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes UnlimitedValueDomain#_logical_and_less_than.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes EqualToValueDomain#_logical_and_less_than.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_less_than(lhs_domain, rhs_domain = self)
      if lhs_domain.max_value >= 0 || rhs_domain.max_value >= 0
        ValueDomain.of_unlimited
      else
        ValueDomain.of_true
      end
    end

    def _logical_and_greater_than(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value <= 0 || rhs_domain.max_value >= 0
        ValueDomain.of_unlimited
      else
        ValueDomain.of_true
      end
    end

    def logical_or(rhs_domain)
      rhs_domain._logical_or_less_than(self)
    end

    def _logical_or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes NilValueDomain#_logical_or_less_than.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes UnlimitedValueDomain#_logical_or_less_than.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes EqualToValueDomain#_logical_or_less_than.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_less_than(lhs_domain, rhs_domain = self)
      if lhs_domain.value < 0 || rhs_domain.value < 0
        ValueDomain.of_true
      else
        ValueDomain.of_unlimited
      end
    end

    def _logical_or_greater_than(lhs_domain, rhs_domain = self)
      if lhs_domain.value > 0 || rhs_domain.value < 0
        ValueDomain.of_true
      else
        ValueDomain.of_unlimited
      end
    end

    def intersection(rhs_domain)
      rhs_domain._intersection_less_than(self)
    end

    def _intersection_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes NilValueDomain#_intersection_less_than.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes UnlimitedValueDomain#_intersection_less_than.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes EqualToValueDomain#_intersection_less_than.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_less_than(lhs_domain, rhs_domain = self)
      lhs_domain.max_value <= rhs_domain.max_value ? lhs_domain : rhs_domain
    end

    def _intersection_greater_than(lhs_domain, rhs_domain = self)
      case
      when lhs_domain.min_value < rhs_domain.max_value
        ValueDomain._create_intersection(lhs_domain, rhs_domain)
      when lhs_domain.min_value == rhs_domain.max_value
        ValueDomain.equal_to(lhs_domain.min_value)
      when lhs_domain.min_value > rhs_domain.max_value
        ValueDomain.of_nil
      end
    end

    def union(rhs_domain)
      rhs_domain._union_less_than(self)
    end

    def _union_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes NilValueDomain#_union_less_than.
      rhs_domain.union(lhs_domain)
    end

    def _union_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes UnlimitedValueDomain#_union_less_than.
      rhs_domain.union(lhs_domain)
    end

    def _union_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes EqualToValueDomain#_union_less_than.
      rhs_domain.union(lhs_domain)
    end

    def _union_less_than(lhs_domain, rhs_domain = self)
      lhs_domain.max_value <= rhs_domain.max_value ? rhs_domain : lhs_domain
    end

    def _union_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.max_value ?
        ValueDomain.of_unlimited :
        ValueDomain._create_union(lhs_domain, rhs_domain)
    end

    def coerce_to_integer
      @value.integer? ? self : ValueDomain.less_than(@value.to_i)
    end

    def coerce_to_real
      @value.real? ? self : ValueDomain.less_than(@value.to_f)
    end

    def min_value
      nil
    end

    def max_value
      @value.integer? ? @value - 1 : @value - Float::EPSILON
    end

    def each_sample
      return ::Enumerator.new(self, :each_sample) unless block_given?
      yield(max_value)
      self
    end

    def to_defined_domain
      self
    end

    def to_s
      "(< #{@value})"
    end
    memoize :to_s

    def complexity
      1
    end
  end

  class GreaterThanValueDomain < ValueDomain
    extend Memoizable

    def initialize(value)
      if value
        @value = value
      else
        raise ArgumentError, "greater than nil?"
      end
    end

    attr_reader :value

    def empty?
      false
    end

    def nan?
      false
    end

    def undefined?
      false
    end

    def ambiguous?
      false
    end

    def contain_value_domain?(rhs_domain)
      rhs_domain._contain_greater_than?(self)
    end

    def _contain_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_unlimited?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_equal_to?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_less_than?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_greater_than?(lhs_domain, rhs_domain = self)
      lhs_domain.value <= rhs_domain.value
    end

    def _contain_intersection?(lhs_domain, rhs_domain = self)
      lhs_domain.domain_pair.all? do |lhs|
        lhs.contain_value_domain?(rhs_domain)
      end
    end

    def _contain_union?(lhs_domain, rhs_domain = self)
      lhs_domain.domain_pair.any? do |lhs|
        lhs.contain_value_domain?(rhs_domain)
      end
    end

    def intersect?(value_domain)
      value_domain._intersect_greater_than?(self)
    end

    def _intersect_nil?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes NilValueDomain#_intersect_greater_than?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_unlimited?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes
      #       UnlimitedValueDomain#_intersect_greater_than?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_equal_to?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes
      #       EqualToValueDomain#_intersect_greater_than?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_less_than?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes
      #       LessThanValueDomain#_intersect_greater_than?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_greater_than?(lhs_domain, rhs_domain = self)
      true
    end

    def _narrow_by_eq(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_greater_than_by_eq(lhs_domain)
    end

    def _narrow_nil_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_eq(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _narrow_equal_to_by_eq(lhs_domain, rhs_domain = self)
      if lhs_domain.value > rhs_domain.value
        # NOTE: Narrowing `---------|---' by `== ---<=========' makes
        #       `---------|---'.
        lhs_domain
      else
        # NOTE: Narrowing `---|---------' by `== ---------<===' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_less_than_by_eq(lhs_domain, rhs_domain = self)
      if lhs_domain.value > rhs_domain.value
        # NOTE: Narrowing `=========>---' by `== ---<=========' makes
        #       `---<=====>---'.
        lhs_domain.intersection(rhs_domain)
      else
        # NOTE: Narrowing `===>---------' by `== ---------<===' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_greater_than_by_eq(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value >= rhs_domain.min_value
        # NOTE: Narrowing `---------<===' by `== ---<=========' makes
        #       `---------<==='.
        lhs_domain
      else
        # NOTE: Narrowing `---<=========' by `== ---------<===' makes
        #       `---------<==='.
        rhs_domain
      end
    end

    def _narrow_by_ne(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_greater_than_by_ne(lhs_domain)
    end

    def _narrow_nil_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_ne(lhs_domain, rhs_domain = self)
      rhs_domain.inversion
    end

    def _narrow_equal_to_by_ne(lhs_domain, rhs_domain = self)
      if lhs_domain.value >= rhs_domain.min_value
        # NOTE: Narrowing `---|---------' by `!= ---<=========' makes
        #       `-------------'.
        ValueDomain.of_nil
      else
        # NOTE: Narrowing `---|---------' by `!= ---------<===' makes
        #       `---|---------'.
        lhs_domain
      end
    end

    def _narrow_less_than_by_ne(lhs_domain, rhs_domain = self)
      if lhs_domain.max_value >= rhs_domain.min_value
        # NOTE: Narrowing `=========>---' by `!= ---<=========' makes
        #       `===>---------'.
        rhs_domain.inversion
      else
        # NOTE: Narrowing `===>---------' by `!= ---------<===' makes
        #       `===>---------'.
        lhs_domain
      end
    end

    def _narrow_greater_than_by_ne(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value >= rhs_domain.min_value
        # NOTE: Narrowing `---------<===' by `!= ---<=========' makes
        #       `-------------'.
        ValueDomain.of_nil
      else
        # NOTE: Narrowing `---<=========' by `!= ---------<===' makes
        #       `---<=====>---'.
        lhs_domain.intersection(rhs_domain.inversion)
      end
    end

    def _narrow_by_lt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_greater_than_by_lt(lhs_domain)
    end

    def _narrow_nil_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `=============' by `< ------<======' makes
      #       `======>------'.
      rhs_domain.inversion
    end

    def _narrow_equal_to_by_lt(lhs_domain, rhs_domain = self)
      if lhs_domain.value >= rhs_domain.min_value
        # NOTE: Narrowing `---------|---' by `< ---<=========' makes
        #       `-------------'.
        ValueDomain.of_nil
      else
        # NOTE: Narrowing `---|---------' by `< ---------<===' makes
        #       `---|---------'.
        lhs_domain
      end
    end

    def _narrow_less_than_by_lt(lhs_domain, rhs_domain = self)
      if lhs_domain.value > rhs_domain.value
        # NOTE: Narrowing `=========>---' by `< ---<=========' makes
        #       `===>---------'.
        rhs_domain.inversion
      else
        # NOTE: Narrowing `===>---------' by `< ---------<===' makes
        #       `===>---------'.
        lhs_domain
      end
    end

    def _narrow_greater_than_by_lt(lhs_domain, rhs_domain = self)
      if lhs_domain.value < rhs_domain.value
        # NOTE: Narrowing `---<=========' by `< ---------<===' makes
        #       `---<=====>---'.
        lhs_domain.intersect(rhs_domain.inversion)
      else
        # NOTE: Narrowing `---------<===' by `< ---<=========' makes
        #       `-------------'.
        ValueDomain.of_nil
      end
    end

    def _narrow_by_gt(rhs_domain, lhs_domain = self)
      rhs_domain._narrow_greater_than_by_gt(lhs_domain)
    end

    def _narrow_nil_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `=============' by `> ------<======' makes
      #       `-------------'.
      ValueDomain.of_nil
    end

    def _narrow_equal_to_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `------|------' by `> ------<======' makes
      #       `-------------'.
      ValueDomain.of_nil
    end

    def _narrow_less_than_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `======>------' by `> ------<======' makes
      #       `-------------'.
      ValueDomain.of_nil
    end

    def _narrow_greater_than_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: Narrowing `------<======' by `> ------<======' makes
      #       `-------------'.
      ValueDomain.of_nil
    end

    def inversion
      ValueDomain.less_than_or_equal_to(@value)
    end

    def ~
      ValueDomain.greater_than(~coerce_to_integer.value)
    end

    def +@
      self
    end

    def -@
      ValueDomain.less_than(-@value)
    end

    def +(rhs_domain)
      rhs_domain._add_greater_than(self)
    end

    def _add_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes NilValueDomain#_add_greater_than.
      rhs_domain + lhs_domain
    end

    def _add_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes UnlimitedValueDomain#_add_greater_than.
      rhs_domain + lhs_domain
    end

    def _add_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes EqualToValueDomain#_add_greater_than.
      rhs_domain + lhs_domain
    end

    def _add_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes LessThanValueDomain#_add_greater_than.
      rhs_domain + lhs_domain
    end

    def _add_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(lhs_domain.value + rhs_domain.value)
    end

    def *(rhs_domain)
      rhs_domain._mul_greater_than(self)
    end

    def _mul_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes NilValueDomain#_mul_greater_than.
      rhs_domain * lhs_domain
    end

    def _mul_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes UnlimitedValueDomain#_mul_greater_than.
      rhs_domain * lhs_domain
    end

    def _mul_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes EqualToValueDomain#_mul_greater_than.
      rhs_domain * lhs_domain
    end

    def _mul_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes LessThanValueDomain#_mul_greater_than.
      rhs_domain * lhs_domain
    end

    def _mul_greater_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value < 0
        ValueDomain.of_unlimited
      else
        case
        when lhs_domain.value >= 0
          ValueDomain.greater_than(0)
        when lhs_domain.value < 0
          ValueDomain.less_than(0)
        end
      end
    end

    def /(rhs_domain)
      rhs_domain._div_greater_than(self)
    end

    def _div_nil(lhs_domain, rhs_domain = self)
      rhs_domain.min_value <= 0 ? ValueDomain.of_nan : lhs_domain
    end

    def _div_unlimited(lhs_domain, rhs_domain = self)
      rhs_domain.min_value <= 0 ? ValueDomain.of_nan : lhs_domain
    end

    def _div_equal_to(lhs_domain, rhs_domain = self)
      if rhs_domain.value <= 0
        ValueDomain.of_nan
      else
        case
        when lhs_domain.value < 0
          ValueDomain.greater_than(
            lhs_domain.value / rhs_domain.value).intersection(
              ValueDomain.less_than(0))
        when lhs_domain.value == 0
          ValueDomain.equal_to(0)
        when lhs_domain.value > 0
          ValueDomain.greater_than(0).intersection(
            ValueDomain.less_than(lhs_domain.value / rhs_domain.value))
        end
      end
    end

    def _div_less_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value < 0
        ValueDomain.of_nan
      else
        case
        when lhs_domain.value >= 0
          ValueDomain.greater_than(0)
        when lhs_domain.value < 0
          ValueDomain.less_than(0)
        end
      end
    end

    def _div_greater_than(lhs_domain, rhs_domain = self)
      if rhs_domain.value < 0
        ValueDomain.of_nan
      else
        case
        when lhs_domain.value <= 0
          ValueDomain.less_than(0)
        when lhs_domain.value > 0
          ValueDomain.greater_than(0)
        end
      end
    end

    def &(rhs_domain)
      rhs_domain.coerce_to_integer._and_greater_than(coerce_to_integer)
    end

    def _and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes NilValueDomain#_and_greater_than.
      rhs_domain & lhs_domain
    end

    def _and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes UnlimitedValueDomain#_and_greater_than.
      rhs_domain & lhs_domain
    end

    def _and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes EqualToValueDomain#_and_greater_than.
      rhs_domain & lhs_domain
    end

    def _and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes LessThanValueDomain#_and_greater_than.
      rhs_domain & lhs_domain
    end

    def _and_greater_than(lhs_domain, rhs_domain = self)
      if lhs_domain.value > 0 && rhs_domain.value > 0
        ValueDomain.greater_than(lhs_domain.value & rhs_domain.value)
      else
        ValueDomain.of_unlimited
      end
    end

    def |(rhs_domain)
      rhs_domain.coerce_to_integer._or_greater_than(coerce_to_integer)
    end

    def _or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes NilValueDomain#_or_greater_than.
      rhs_domain | lhs_domain
    end

    def _or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes UnlimitedValueDomain#_or_greater_than.
      rhs_domain | lhs_domain
    end

    def _or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes EqualToValueDomain#_or_greater_than.
      rhs_domain | lhs_domain
    end

    def _or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes LessThanValueDomain#_or_greater_than.
      rhs_domain | lhs_domain
    end

    def _or_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def ^(rhs_domain)
      rhs_domain.coerce_to_integer._xor_greater_than(coerce_to_integer)
    end

    def _xor_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes NilValueDomain#_xor_greater_than.
      rhs_domain ^ lhs_domain
    end

    def _xor_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes UnlimitedValueDomain#_xor_greater_than.
      rhs_domain ^ lhs_domain
    end

    def _xor_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes EqualToValueDomain#_xor_greater_than.
      rhs_domain ^ lhs_domain
    end

    def _xor_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes LessThanValueDomain#_xor_greater_than.
      rhs_domain ^ lhs_domain
    end

    def _xor_greater_than(lhs_domain, rhs_domain = self)
      case
      when lhs_domain.value > 0 && rhs_domain.value > 0
        ValueDomain.greater_than(0)
      when lhs_domain.value > 0 || rhs_domain.value > 0
        ValueDomain.less_than(0)
      else
        ValueDomain.greater_than(0)
      end
    end

    def <<(rhs_domain)
      rhs_domain.coerce_to_integer._shl_greater_than(coerce_to_integer)
    end

    def _shl_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shl_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain
    end

    def _shl_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(left_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shl_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(left_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shl_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.greater_than(left_shift(lhs_domain.value, rhs_domain.value))
    end

    def >>(rhs_domain)
      rhs_domain.coerce_to_integer._shr_greater_than(coerce_to_integer)
    end

    def _shr_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any arithmetic operation with NilValueDomain makes
      #       NilValueDomain.
      lhs_domain
    end

    def _shr_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, this arithmetic operation with UnlimitedValueDomain makes
      #       UnlimitedValueDomain because of the bit-overflow.
      # NOTE: NaN is a subclass of UnlimitedValueDomain.
      #       Arithmetic operation with UnlimitedValueDomain should make
      #       UnlimitedValueDomain, and with NaN should make NaN.
      lhs_domain
    end

    def _shr_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(right_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shr_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(right_shift(lhs_domain.value, rhs_domain.value))
    end

    def _shr_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.less_than(right_shift(lhs_domain.value, rhs_domain.value))
    end

    def !
      @value > 0 ? ValueDomain.of_false : ValueDomain.of_unlimited
    end

    def <(rhs_domain)
      rhs_domain._less_greater_than(self)
    end

    def _less_nil(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, any comparison with NilValueDomain makes no sense.
      ValueDomain.of_nil
    end

    def _less_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: UnlimitedValueDomain contains everything.
      #       So, any comparison with UnlimitedValueDomain makes
      #       UnlimitedValueDomain.
      ValueDomain.of_unlimited
    end

    def _less_equal_to(lhs_domain, rhs_domain = self)
      lhs_domain.value < rhs_domain.min_value ?
        ValueDomain.of_true : ValueDomain.of_unlimited
    end

    def _less_less_than(lhs_domain, rhs_domain = self)
      lhs_domain.max_value < rhs_domain.min_value ?
        ValueDomain.of_true : ValueDomain.of_unlimited
    end

    def _less_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def ==(rhs_domain)
      rhs_domain._equal_greater_than(self)
    end

    def _equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes NilValueDomain#_equal_greater_than.
      rhs_domain == lhs_domain
    end

    def _equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes UnlimitedValueDomain#_equal_greater_than.
      rhs_domain == lhs_domain
    end

    def _equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes EqualToValueDomain#_equal_greater_than.
      rhs_domain == lhs_domain
    end

    def _equal_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes LessThanValueDomain#_equal_greater_than.
      rhs_domain == lhs_domain
    end

    def _equal_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def !=(rhs_domain)
      rhs_domain._not_equal_greater_than(self)
    end

    def _not_equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes NilValueDomain#_not_equal_greater_than.
      rhs_domain != lhs_domain
    end

    def _not_equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes UnlimitedValueDomain#_not_equal_greater_than.
      rhs_domain != lhs_domain
    end

    def _not_equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes EqualToValueDomain#_not_equal_greater_than.
      rhs_domain != lhs_domain
    end

    def _not_equal_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes LessThanValueDomain#_not_equal_greater_than.
      rhs_domain != lhs_domain
    end

    def _not_equal_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def logical_and(rhs_domain)
      rhs_domain._logical_and_greater_than(self)
    end

    def _logical_and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes NilValueDomain#_logical_and_greater_than.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes
      #       UnlimitedValueDomain#_logical_and_greater_than.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes EqualToValueDomain#_logical_and_greater_than.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes
      #       LessThanValueDomain#_logical_and_greater_than.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_greater_than(lhs_domain, rhs_domain = self)
      if lhs_domain.min_value <= 0 || rhs_domain.min_value <= 0
        ValueDomain.of_unlimited
      else
        ValueDomain.of_true
      end
    end

    def logical_or(rhs_domain)
      rhs_domain._logical_or_greater_than(self)
    end

    def _logical_or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes NilValueDomain#_logical_or_greater_than.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes
      #       UnlimitedValueDomain#_logical_or_greater_than.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes EqualToValueDomain#_logical_or_greater_than.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes LessThanValueDomain#_logical_or_greater_than.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_greater_than(lhs_domain, rhs_domain = self)
      if lhs_domain.value > 0 || rhs_domain.value > 0
        ValueDomain.of_true
      else
        ValueDomain.of_unlimited
      end
    end

    def intersection(rhs_domain)
      rhs_domain._intersection_greater_than(self)
    end

    def _intersection_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes NilValueDomain#_intersection_greater_than.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes
      #       UnlimitedValueDomain#_intersection_greater_than.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes
      #       EqualToValueDomain#_intersection_greater_than.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes
      #       LessThanValueDomain#_intersection_greater_than.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.min_value ? rhs_domain : lhs_domain
    end

    def union(rhs_domain)
      rhs_domain._union_greater_than(self)
    end

    def _union_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes NilValueDomain#_union_greater_than.
      rhs_domain.union(lhs_domain)
    end

    def _union_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes UnlimitedValueDomain#_union_greater_than.
      rhs_domain.union(lhs_domain)
    end

    def _union_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes EqualToValueDomain#_union_greater_than.
      rhs_domain.union(lhs_domain)
    end

    def _union_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes LessThanValueDomain#_union_greater_than.
      rhs_domain.union(lhs_domain)
    end

    def _union_greater_than(lhs_domain, rhs_domain = self)
      lhs_domain.min_value <= rhs_domain.min_value ? lhs_domain : rhs_domain
    end

    def coerce_to_integer
      @value.integer? ? self : ValueDomain.greater_than(@value.to_i)
    end

    def coerce_to_real
      @value.real? ? self : ValueDomain.greater_than(@value.to_f)
    end

    def min_value
      @value.integer? ? @value + 1 : @value + Float::EPSILON
    end

    def max_value
      nil
    end

    def each_sample
      return ::Enumerator.new(self, :each_sample) unless block_given?
      yield(min_value)
      self
    end

    def to_defined_domain
      self
    end

    def to_s
      "(> #{@value})"
    end
    memoize :to_s

    def complexity
      1
    end
  end

  class CompositeValueDomain < ValueDomain
    def initialize(lhs_value_domain, rhs_value_domain)
      @domain_pair = [lhs_value_domain, rhs_value_domain].sort
    end

    attr_reader :domain_pair

    def empty?
      false
    end

    def nan?
      false
    end

    def undefined?
      false
    end

    def ambiguous?
      false
    end

    def _intersect_nil?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes CompositeValueDomain#intersect?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_unlimited?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes CompositeValueDomain#intersect?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_equal_to?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes CompositeValueDomain#intersect?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_less_than?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes CompositeValueDomain#intersect?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_greater_than?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes CompositeValueDomain#intersect?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _narrow_by_eq(rhs_domain, lhs_domain = self)
      lhs_domain.intersection(rhs_domain)
    end

    def _narrow_nil_by_eq(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_eq(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _narrow_equal_to_by_eq(lhs_domain, rhs_domain = self)
      lhs_domain.intersection(rhs_domain)
    end

    def _narrow_less_than_by_eq(lhs_domain, rhs_domain = self)
      lhs_domain.intersection(rhs_domain)
    end

    def _narrow_greater_than_by_eq(lhs_domain, rhs_domain = self)
      lhs_domain.intersection(rhs_domain)
    end

    def _narrow_by_ne(rhs_domain, lhs_domain = self)
      lhs_domain.intersection(rhs_domain.inversion)
    end

    def _narrow_nil_by_ne(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_ne(lhs_domain, rhs_domain = self)
      rhs_domain.inversion
    end

    def _narrow_equal_to_by_ne(lhs_domain, rhs_domain = self)
      lhs_domain.intersection(rhs_domain.inversion)
    end

    def _narrow_less_than_by_ne(lhs_domain, rhs_domain = self)
      lhs_domain.intersection(rhs_domain.inversion)
    end

    def _narrow_greater_than_by_ne(lhs_domain, rhs_domain = self)
      lhs_domain.intersection(rhs_domain.inversion)
    end

    def _narrow_by_lt(rhs_domain, lhs_domain = self)
      lhs_domain.intersection(ValueDomain.of_unlimited.narrow(:<, rhs_domain))
    end

    def _narrow_nil_by_lt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_lt(lhs_domain, rhs_domain = self)
      if rhs_max_value = rhs_domain.max_value
        ValueDomain.less_than(rhs_max_value)
      else
        lhs_domain
      end
    end

    def _narrow_equal_to_by_lt(lhs_domain, rhs_domain = self)
      if rhs_max_value = rhs_domain.max_value and
          lhs_domain.value > rhs_max_value
        ValueDomain.of_nil
      else
        lhs_domain
      end
    end

    def _narrow_less_than_by_lt(lhs_domain, rhs_domain = self)
      if rhs_max_value = rhs_domain.max_value and
          lhs_domain.max_value > rhs_max_value
        ValueDomain.less_than(rhs_max_value)
      else
        lhs_domain
      end
    end

    def _narrow_greater_than_by_lt(lhs_domain, rhs_domain = self)
      if rhs_max_value = rhs_domain.max_value and
          lhs_domain.min_value > rhs_max_value
        ValueDomain.of_nil
      else
        lhs_domain
      end
    end

    def _narrow_by_gt(rhs_domain, lhs_domain = self)
      lhs_domain.intersection(ValueDomain.of_unlimited.narrow(:>, rhs_domain))
    end

    def _narrow_nil_by_gt(lhs_domain, rhs_domain = self)
      # NOTE: NilValueDomain contains no values.
      #       So, narrowing NilValueDomain by anything makes no effect to the
      #       target value-domain.
      lhs_domain
    end

    def _narrow_unlimited_by_gt(lhs_domain, rhs_domain = self)
      if rhs_min_value = rhs_domain.min_value
        ValueDomain.greater_than(rhs_min_value)
      else
        lhs_domain
      end
    end

    def _narrow_equal_to_by_gt(lhs_domain, rhs_domain = self)
      if rhs_min_value = rhs_domain.min_value and
          lhs_domain.value < rhs_min_value
        ValueDomain.of_nil
      else
        lhs_domain
      end
    end

    def _narrow_less_than_by_gt(lhs_domain, rhs_domain = self)
      if rhs_min_value = rhs_domain.min_value and
          lhs_domain.max_value < rhs_min_value
        ValueDomain.of_nil
      else
        lhs_domain
      end
    end

    def _narrow_greater_than_by_gt(lhs_domain, rhs_domain = self)
      if rhs_min_value = rhs_domain.min_value and
          lhs_domain.min_value < rhs_min_value
        ValueDomain.greater_than(rhs_min_value)
      else
        lhs_domain
      end
    end

    def _add_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes CompositeValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes CompositeValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes CompositeValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes CompositeValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes CompositeValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _mul_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes CompositeValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes CompositeValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes CompositeValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes CompositeValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes CompositeValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _div_nil(lhs_domain, rhs_domain = self)
      _div(lhs_domain, rhs_domain)
    end

    def _div_unlimited(lhs_domain, rhs_domain = self)
      _div(lhs_domain, rhs_domain)
    end

    def _div_equal_to(lhs_domain, rhs_domain = self)
      _div(lhs_domain, rhs_domain)
    end

    def _div_less_than(lhs_domain, rhs_domain = self)
      _div(lhs_domain, rhs_domain)
    end

    def _div_greater_than(lhs_domain, rhs_domain = self)
      _div(lhs_domain, rhs_domain)
    end

    def _and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes CompositeValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes CompositeValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes CompositeValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes CompositeValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes CompositeValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes CompositeValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes CompositeValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes CompositeValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes CompositeValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes CompositeValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _xor_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes CompositeValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes CompositeValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes CompositeValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes CompositeValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes CompositeValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _shl_nil(lhs_domain, rhs_domain = self)
      _shl(lhs_domain, rhs_domain)
    end

    def _shl_unlimited(lhs_domain, rhs_domain = self)
      _shl(lhs_domain, rhs_domain)
    end

    def _shl_equal_to(lhs_domain, rhs_domain = self)
      _shl(lhs_domain, rhs_domain)
    end

    def _shl_less_than(lhs_domain, rhs_domain = self)
      _shl(lhs_domain, rhs_domain)
    end

    def _shl_greater_than(lhs_domain, rhs_domain = self)
      _shl(lhs_domain, rhs_domain)
    end

    def _shr_nil(lhs_domain, rhs_domain = self)
      _shr(lhs_domain, rhs_domain)
    end

    def _shr_unlimited(lhs_domain, rhs_domain = self)
      _shr(lhs_domain, rhs_domain)
    end

    def _shr_equal_to(lhs_domain, rhs_domain = self)
      _shr(lhs_domain, rhs_domain)
    end

    def _shr_less_than(lhs_domain, rhs_domain = self)
      _shr(lhs_domain, rhs_domain)
    end

    def _shr_greater_than(lhs_domain, rhs_domain = self)
      _shr(lhs_domain, rhs_domain)
    end

    def _less_nil(lhs_domain, rhs_domain = self)
      _less(lhs_domain, rhs_domain)
    end

    def _less_unlimited(lhs_domain, rhs_domain = self)
      _less(lhs_domain, rhs_domain)
    end

    def _less_equal_to(lhs_domain, rhs_domain = self)
      _less(lhs_domain, rhs_domain)
    end

    def _less_less_than(lhs_domain, rhs_domain = self)
      _less(lhs_domain, rhs_domain)
    end

    def _less_greater_than(lhs_domain, rhs_domain = self)
      _less(lhs_domain, rhs_domain)
    end

    def ==(rhs_domain)
      _equal(rhs_domain, self)
    end

    def _equal_nil(lhs_domain, rhs_domain = self)
      _equal(lhs_domain, rhs_domain)
    end

    def _equal_unlimited(lhs_domain, rhs_domain = self)
      _equal(lhs_domain, rhs_domain)
    end

    def _equal_equal_to(lhs_domain, rhs_domain = self)
      _equal(lhs_domain, rhs_domain)
    end

    def _equal_less_than(lhs_domain, rhs_domain = self)
      _equal(lhs_domain, rhs_domain)
    end

    def _equal_greater_than(lhs_domain, rhs_domain = self)
      _equal(lhs_domain, rhs_domain)
    end

    def !=(rhs_domain)
      _not_equal(rhs_domain, self)
    end

    def _not_equal_nil(lhs_domain, rhs_domain = self)
      _not_equal(lhs_domain, rhs_domain)
    end

    def _not_equal_unlimited(lhs_domain, rhs_domain = self)
      _not_equal(lhs_domain, rhs_domain)
    end

    def _not_equal_equal_to(lhs_domain, rhs_domain = self)
      _not_equal(lhs_domain, rhs_domain)
    end

    def _not_equal_less_than(lhs_domain, rhs_domain = self)
      _not_equal(lhs_domain, rhs_domain)
    end

    def _not_equal_greater_than(lhs_domain, rhs_domain = self)
      _not_equal(lhs_domain, rhs_domain)
    end

    def _logical_and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes CompositeValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes CompositeValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes CompositeValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes CompositeValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes CompositeValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes CompositeValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes CompositeValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes CompositeValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes CompositeValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes CompositeValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def _intersection_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes CompositeValueDomain#intersection which
      #       should be overriden by IntersectionValueDomain and
      #       UnionValueDomain.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes CompositeValueDomain#intersection which
      #       should be overriden by IntersectionValueDomain and
      #       UnionValueDomain.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes CompositeValueDomain#intersection which
      #       should be overriden by IntersectionValueDomain and
      #       UnionValueDomain.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes CompositeValueDomain#intersection which
      #       should be overriden by IntersectionValueDomain and
      #       UnionValueDomain.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes CompositeValueDomain#intersection which
      #       should be overriden by IntersectionValueDomain and
      #       UnionValueDomain.
      rhs_domain.intersection(lhs_domain)
    end

    def _union_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes CompositeValueDomain#union which should be
      #       overriden by IntersectionValueDomain and UnionValueDomain.
      rhs_domain.union(lhs_domain)
    end

    def _union_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes CompositeValueDomain#union which should be
      #       overriden by IntersectionValueDomain and UnionValueDomain.
      rhs_domain.union(lhs_domain)
    end

    def _union_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes CompositeValueDomain#union which should be
      #       overriden by IntersectionValueDomain and UnionValueDomain.
      rhs_domain.union(lhs_domain)
    end

    def _union_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes CompositeValueDomain#union which should be
      #       overriden by IntersectionValueDomain and UnionValueDomain.
      rhs_domain.union(lhs_domain)
    end

    def _union_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes CompositeValueDomain#union which should be
      #       overriden by IntersectionValueDomain and UnionValueDomain.
      rhs_domain.union(lhs_domain)
    end

    def to_defined_domain
      self
    end

    def complexity
      domain_pair.map { |domain| domain.complexity + 1 }.max
    end

    private
    def _div(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shl(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _shr(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _less(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _equal(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end

    def _not_equal(lhs_domain, rhs_domain = self)
      subclass_responsibility
    end
  end

  class IntersectionValueDomain < CompositeValueDomain
    extend Memoizable

    def contain_value_domain?(rhs_domain)
      rhs_domain._contain_intersection?(self)
    end

    def _contain_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_unlimited?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_equal_to?(lhs_domain, rhs_domain = self)
      rhs_domain.domain_pair.any? do |rhs|
        lhs_domain.contain_value_domain?(rhs)
      end
    end

    def _contain_less_than?(lhs_domain, rhs_domain = self)
      rhs_domain.domain_pair.any? do |rhs|
        lhs_domain.contain_value_domain?(rhs)
      end
    end

    def _contain_greater_than?(lhs_domain, rhs_domain = self)
      rhs_domain.domain_pair.any? do |rhs|
        lhs_domain.contain_value_domain?(rhs)
      end
    end

    def _contain_intersection?(lhs_domain, rhs_domain = self)
      lhs_first, lhs_second = lhs_domain.domain_pair
      rhs_first, rhs_second = rhs_domain.domain_pair
      case
      when lhs_first.contain_value_domain?(rhs_first) &&
           lhs_second.contain_value_domain?(rhs_second)
        true
      when lhs_first.contain_value_domain?(rhs_second) &&
           lhs_second.contain_value_domain?(rhs_first)
        true
      else
        false
      end
    end

    def _contain_union?(lhs_domain, rhs_domain = self)
      lhs_domain.domain_pair.any? do |lhs|
        lhs.contain_value_domain?(rhs_domain)
      end
    end

    def intersect?(value_domain)
      domain_pair.all? { |lhs| lhs.intersect?(value_domain) }
    end

    def inversion
      new_sub_domains = domain_pair.map { |domain| domain.inversion }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def ~
      new_sub_domains = domain_pair.map { |domain| ~domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def +@
      new_sub_domains = domain_pair.map { |domain| +domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def -@
      new_sub_domains = domain_pair.map { |domain| -domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def +(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs + rhs_domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def *(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs * rhs_domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def /(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs / rhs_domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def &(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs & rhs_domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def |(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs | rhs_domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def ^(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs ^ rhs_domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def <<(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs << rhs_domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def >>(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs >> rhs_domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def !
      new_sub_domains = domain_pair.map { |domain| !domain }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def <(rhs_domain)
      # NOTE: The intersection value domain must be a close-domain (--<===>--).
      if intersect?(rhs_domain)
        if rhs_domain.max_value && rhs_domain.max_value == min_value
          ValueDomain.of_false
        else
          ValueDomain.of_unlimited
        end
      else
        # NOTE: When value domains are not intersected, the RHS value domain is
        #       in the left or right of the LHS intersection value domain.
        if rhs_domain.min_value && max_value < rhs_domain.min_value
          # NOTE: The RHS value domain is in the right of the LHS intersection
          #       value domain.
          ValueDomain.of_true
        else
          # NOTE: The RHS value domain is in the left of the LHS intersection
          #       value domain.
          ValueDomain.of_false
        end
      end
    end

    def logical_and(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs.logical_and(rhs_domain) }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def logical_or(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs.logical_or(rhs_domain) }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def intersection(rhs_domain)
      if rhs_domain.kind_of?(UnionValueDomain)
        return rhs_domain.intersection(self)
      end

      case
      when contain_value_domain?(rhs_domain)
        rhs_domain
      when rhs_domain.contain_value_domain?(self)
        self
      else
        new_sub_domains =
          domain_pair.map { |lhs| lhs.intersection(rhs_domain) }
        ValueDomain.of_intersection(*new_sub_domains)
      end
    end

    def union(rhs_domain)
      case
      when contain_value_domain?(rhs_domain)
        self
      when rhs_domain.contain_value_domain?(self)
        rhs_domain
      else
        ValueDomain.of_union(self, rhs_domain)
      end
    end

    def coerce_to_integer
      new_sub_domains = domain_pair.map { |domain| domain.coerce_to_integer }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def coerce_to_real
      new_sub_domains = domain_pair.map { |domain| domain.coerce_to_real }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def min_value
      # NOTE: Intersection-value-domain must be a close-domain (---<=====>---).
      #       So, min-value is defined by the lower greater-than value domain.
      domain_pair.map { |domain| domain.min_value }.compact.min
    end
    memoize :min_value

    def max_value
      # NOTE: Intersection-value-domain must be a close-domain (---<=====>---).
      #       So, max-value is defined by the higher lower-than value domain.
      domain_pair.map { |domain| domain.max_value }.compact.max
    end
    memoize :max_value

    def each_sample
      return ::Enumerator.new(self, :each_sample) unless block_given?

      samples = domain_pair.map { |domain|
        domain.each_sample.to_a
      }.flatten.uniq

      samples.each do |sample|
        yield(sample) if contain?(sample)
      end
    end

    def to_s
      "(#{domain_pair.first.to_s} && #{domain_pair.last.to_s})"
    end
    memoize :to_s

    private
    def _div(lhs_domain, rhs_domain = self)
      new_sub_domains = rhs_domain.domain_pair.map { |rhs| lhs_domain / rhs }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def _shl(lhs_domain, rhs_domain = self)
      new_sub_domains = rhs_domain.domain_pair.map { |rhs| lhs_domain << rhs }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def _shr(lhs_domain, rhs_domain = self)
      new_sub_domains = rhs_domain.domain_pair.map { |rhs| lhs_domain >> rhs }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def _less(lhs_domain, rhs_domain = self)
      comp_result = rhs_domain.domain_pair.map { |rhs| lhs_domain < rhs }
      if comp_result.any? { |domain| domain.eql?(ValueDomain.of_false) }
        ValueDomain.of_false
      else
        comp_result.first.intersection(comp_result.last)
      end
    end

    def _equal(lhs_domain, rhs_domain = self)
      # NOTE: The intersection value domain must be a close-domain (--<===>--).
      #       If one of the sub domains is not equal to LHS, result should be
      #       false.
      comp_result = rhs_domain.domain_pair.map { |rhs| lhs_domain == rhs }
      if comp_result.any? { |domain| domain.eql?(ValueDomain.of_false) }
        ValueDomain.of_false
      else
        comp_result.first.intersection(comp_result.last)
      end
    end

    def _not_equal(lhs_domain, rhs_domain = self)
      # NOTE: The intersection value domain must be a close-domain (--<===>--).
      #       If one of the sub domains is not equal to LHS, result should be
      #       true.
      comp_result = rhs_domain.domain_pair.map { |rhs| lhs_domain != rhs }
      if comp_result.any? { |domain| domain.eql?(ValueDomain.of_false) }
        ValueDomain.of_false
      else
        comp_result.first.intersection(comp_result.last)
      end
    end
  end

  class UnionValueDomain < CompositeValueDomain
    extend Memoizable

    def contain_value_domain?(rhs_domain)
      rhs_domain._contain_union?(self)
    end

    def _contain_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_unlimited?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_equal_to?(lhs_domain, rhs_domain = self)
      rhs_domain.domain_pair.all? do |rhs|
        lhs_domain.contain_value_domain?(rhs)
      end
    end

    def _contain_less_than?(lhs_domain, rhs_domain = self)
      rhs_domain.domain_pair.all? do |rhs|
        lhs_domain.contain_value_domain?(rhs)
      end
    end

    def _contain_greater_than?(lhs_domain, rhs_domain = self)
      rhs_domain.domain_pair.all? do |rhs|
        lhs_domain.contain_value_domain?(rhs)
      end
    end

    def _contain_intersection?(lhs_domain, rhs_domain = self)
      rhs_domain.domain_pair.all? do |rhs|
        lhs_domain.contain_value_domain?(rhs)
      end
    end

    def _contain_union?(lhs_domain, rhs_domain = self)
      lhs_first, lhs_second = lhs_domain.domain_pair
      rhs_domain_pair = rhs_domain.domain_pair
      case
      when rhs_domain_pair.all? { |rhs| lhs_first.contain_value_domain?(rhs) }
        true
      when rhs_domain_pair.all? { |rhs| lhs_second.contain_value_domain?(rhs) }
        true
      else
        rhs_first, rhs_second = rhs_domain.domain_pair
        case
        when lhs_first.contain_value_domain?(rhs_first) &&
             lhs_second.contain_value_domain?(rhs_second)
          true
        when lhs_first.contain_value_domain?(rhs_second) &&
             lhs_second.contain_value_domain?(rhs_first)
          true
        else
          false
        end
      end
    end

    def intersect?(value_domain)
      domain_pair.any? { |lhs| lhs.intersect?(value_domain) }
    end

    def inversion
      new_sub_domains = domain_pair.map { |domain| domain.inversion }
      new_sub_domains.first.intersection(new_sub_domains.last)
    end

    def ~
      new_sub_domains = domain_pair.map { |domain| ~domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def +@
      new_sub_domains = domain_pair.map { |domain| +domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def -@
      new_sub_domains = domain_pair.map { |domain| -domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def +(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs + rhs_domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def *(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs * rhs_domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def /(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs / rhs_domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def &(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs & rhs_domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def |(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs | rhs_domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def ^(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs ^ rhs_domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def <<(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs << rhs_domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def >>(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs >> rhs_domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def !
      new_sub_domains = domain_pair.map { |domain| !domain }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def <(rhs_domain)
      comp_result = domain_pair.map { |lhs| lhs < rhs_domain }
      comp_result.first.union(comp_result.last)
    end

    def logical_and(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs.logical_and(rhs_domain) }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def logical_or(rhs_domain)
      new_sub_domains = domain_pair.map { |lhs| lhs.logical_or(rhs_domain) }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def intersection(rhs_domain)
      case
      when contain_value_domain?(rhs_domain)
        rhs_domain
      when rhs_domain.contain_value_domain?(self)
        self
      else
        new_sub_domains =
          domain_pair.map { |lhs| lhs.intersection(rhs_domain) }
        ValueDomain.of_union(*new_sub_domains)
      end
    end

    def union(rhs_domain)
      case
      when contain_value_domain?(rhs_domain)
        self
      when rhs_domain.contain_value_domain?(self)
        rhs_domain
      else
        new_sub_domains = domain_pair.map { |lhs| lhs.union(rhs_domain) }
        ValueDomain.of_union(*new_sub_domains)
      end
    end

    def coerce_to_integer
      new_sub_domains = domain_pair.map { |domain| domain.coerce_to_integer }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def coerce_to_real
      new_sub_domains = domain_pair.map { |domain| domain.coerce_to_real }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def min_value
      # NOTE: The union value domain may be a open-domain (==>---<==),
      #       a half-open-domain (==>--<==>-) or (-<==>--<==), or
      #       a close-domain (-<==>-<==>-).
      #       When the union value domain is a open-domain, the min-value is
      #       -infinite.
      #       When the union value domain is a half-open-domain and lower sub
      #       domain is less-than value domain, the min-value is -infinite.
      #       When the union value domain is a half-open-domain and higher sub
      #       domain is greater-than value domain, the min-value is defined by
      #       the lower value domain.
      #       When the union value domain is a close-domain, the min-value is
      #       defined by the lower value domain.
      min_values = domain_pair.map { |domain| domain.min_value }
      min_values.include?(nil) ? nil : min_values.min
    end
    memoize :min_value

    def max_value
      # NOTE: If this is an "open-domain" (===>---<===), max-value is
      #       undefined.
      #       But, when the domain is (===>--<===>--), max-value is defined.
      max_values = domain_pair.map { |domain| domain.max_value }
      max_values.include?(nil) ? nil : max_values.max
    end
    memoize :max_value

    def each_sample
      return ::Enumerator.new(self, :each_sample) unless block_given?

      samples = domain_pair.map { |domain|
        domain.each_sample.to_a
      }.flatten.uniq

      samples.each do |sample|
        yield(sample)
      end
    end

    def to_s
      "(#{domain_pair.first.to_s} || #{domain_pair.last.to_s})"
    end
    memoize :to_s

    private
    def _div(lhs_domain, rhs_domain = self)
      new_sub_domains = rhs_domain.domain_pair.map { |rhs| lhs_domain / rhs }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def _shl(lhs_domain, rhs_domain = self)
      new_sub_domains = rhs_domain.domain_pair.map { |rhs| lhs_domain << rhs }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def _shr(lhs_domain, rhs_domain = self)
      new_sub_domains = rhs_domain.domain_pair.map { |rhs| lhs_domain >> rhs }
      new_sub_domains.first.union(new_sub_domains.last)
    end

    def _less(lhs_domain, rhs_domain = self)
      comp_result = rhs_domain.domain_pair.map { |rhs| lhs_domain < rhs }
      comp_result.first.union(comp_result.last)
    end

    def _equal(lhs_domain, rhs_domain = self)
      # NOTE: The union value domain may be a open-domain (==>---<==),
      #       a half-open-domain (==>--<==>-) or (-<==>--<==), or
      #       a close-domain (-<==>-<==>-).
      #       If one of the sub domains is equal to LHS, result should be true.
      comp_result = rhs_domain.domain_pair.map { |rhs| lhs_domain == rhs }
      comp_result.first.union(comp_result.last)
    end

    def _not_equal(lhs_domain, rhs_domain = self)
      # NOTE: The union value domain may be a open-domain (==>---<==),
      #       a half-open-domain (==>--<==>-) or (-<==>--<==), or
      #       a close-domain (-<==>-<==>-).
      #       If one of the sub domains is equal to LHS, result should be
      #       false.
      comp_result = rhs_domain.domain_pair.map { |rhs| lhs_domain != rhs }
      comp_result.first.union(comp_result.last)
    end
  end

  class UndefinedValueDomain < ValueDomain
    extend Memoizable
    extend Forwardable

    def initialize(range_or_value_domain)
      case range_or_value_domain
      when Range
        @domain = ValueDomain.of_intersection(
          ValueDomain.greater_than_or_equal_to(range_or_value_domain.first),
          ValueDomain.less_than_or_equal_to(range_or_value_domain.last))
      when ValueDomain
        @domain = range_or_value_domain
      end
    end

    attr_reader :domain

    def_delegator :@domain, :empty?

    def nan?
      false
    end

    def undefined?
      true
    end

    def ambiguous?
      false
    end

    def_delegator :@domain, :contain?
    def_delegator :@domain, :contain_value_domain?
    def_delegator :@domain, :_contain_nil?
    def_delegator :@domain, :_contain_unlimited?
    def_delegator :@domain, :_contain_equal_to?
    def_delegator :@domain, :_contain_less_than?
    def_delegator :@domain, :_contain_greater_than?
    def_delegator :@domain, :_contain_intersection?
    def_delegator :@domain, :_contain_union?
    def_delegator :@domain, :intersect?
    def_delegator :@domain, :_intersect_nil?
    def_delegator :@domain, :_intersect_unlimited?
    def_delegator :@domain, :_intersect_equal_to?
    def_delegator :@domain, :_intersect_less_than?
    def_delegator :@domain, :_intersect_greater_than?

    def _narrow_by_eq(rhs_domain, lhs_domain = self)
      ValueDomain.of_undefined(lhs_domain.domain._narrow_by_eq(rhs_domain))
    end

    def_delegator :@domain, :_narrow_nil_by_eq
    def_delegator :@domain, :_narrow_unlimited_by_eq
    def_delegator :@domain, :_narrow_equal_to_by_eq
    def_delegator :@domain, :_narrow_less_than_by_eq
    def_delegator :@domain, :_narrow_greater_than_by_eq

    def _narrow_by_ne(rhs_domain, lhs_domain = self)
      ValueDomain.of_undefined(lhs_domain.domain._narrow_by_ne(rhs_domain))
    end

    def_delegator :@domain, :_narrow_nil_by_ne
    def_delegator :@domain, :_narrow_unlimited_by_ne
    def_delegator :@domain, :_narrow_equal_to_by_ne
    def_delegator :@domain, :_narrow_less_than_by_ne
    def_delegator :@domain, :_narrow_greater_than_by_ne

    def _narrow_by_lt(rhs_domain, lhs_domain = self)
      ValueDomain.of_undefined(lhs_domain.domain._narrow_by_lt(rhs_domain))
    end

    def_delegator :@domain, :_narrow_nil_by_lt
    def_delegator :@domain, :_narrow_unlimited_by_lt
    def_delegator :@domain, :_narrow_equal_to_by_lt
    def_delegator :@domain, :_narrow_less_than_by_lt
    def_delegator :@domain, :_narrow_greater_than_by_lt

    def _narrow_by_gt(rhs_domain, lhs_domain = self)
      ValueDomain.of_undefined(lhs_domain.domain._narrow_by_gt(rhs_domain))
    end

    def_delegator :@domain, :_narrow_nil_by_gt
    def_delegator :@domain, :_narrow_unlimited_by_gt
    def_delegator :@domain, :_narrow_equal_to_by_gt
    def_delegator :@domain, :_narrow_less_than_by_gt
    def_delegator :@domain, :_narrow_greater_than_by_gt

    def_delegator :@domain, :_widen_by_eq
    def_delegator :@domain, :_widen_by_ne
    def_delegator :@domain, :_widen_by_lt
    def_delegator :@domain, :_widen_by_gt
    def_delegator :@domain, :_widen_by_le
    def_delegator :@domain, :_widen_by_ge

    def inversion
      ValueDomain.of_undefined(@domain.inversion)
    end

    def ~
      ValueDomain.of_undefined(~@domain)
    end

    def +@
      ValueDomain.of_undefined(+@domain)
    end

    def -@
      ValueDomain.of_undefined(-@domain)
    end

    def +(rhs_domain)
      ValueDomain.of_undefined(@domain + rhs_domain)
    end

    def _add_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes UndefinedValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes UndefinedValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes UndefinedValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes UndefinedValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes UndefinedValueDomain#+.
      rhs_domain + lhs_domain
    end

    def *(rhs_domain)
      ValueDomain.of_undefined(@domain * rhs_domain)
    end

    def _mul_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes UndefinedValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes UndefinedValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes UndefinedValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes UndefinedValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes UndefinedValueDomain#*.
      rhs_domain * lhs_domain
    end

    def /(rhs_domain)
      ValueDomain.of_undefined(@domain / rhs_domain)
    end

    def _div_nil(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._div_nil(lhs_domain))
    end

    def _div_unlimited(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._div_unlimited(lhs_domain))
    end

    def _div_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._div_equal_to(lhs_domain))
    end

    def _div_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._div_less_than(lhs_domain))
    end

    def _div_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._div_greater_than(lhs_domain))
    end

    def &(rhs_domain)
      ValueDomain.of_undefined(@domain & rhs_domain)
    end

    def _and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes UndefinedValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes UndefinedValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes UndefinedValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes UndefinedValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes UndefinedValueDomain#&.
      rhs_domain & lhs_domain
    end

    def |(rhs_domain)
      ValueDomain.of_undefined(@domain | rhs_domain)
    end

    def _or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes UndefinedValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes UndefinedValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes UndefinedValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes UndefinedValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes UndefinedValueDomain#|.
      rhs_domain | lhs_domain
    end

    def ^(rhs_domain)
      ValueDomain.of_undefined(@domain ^ rhs_domain)
    end

    def _xor_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes UndefinedValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes UndefinedValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes UndefinedValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes UndefinedValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes UndefinedValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def <<(rhs_domain)
      ValueDomain.of_undefined(@domain << rhs_domain)
    end

    def _shl_nil(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shl_nil(lhs_domain))
    end

    def _shl_unlimited(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shl_unlimited(lhs_domain))
    end

    def _shl_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shl_equal_to(lhs_domain))
    end

    def _shl_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shl_less_than(lhs_domain))
    end

    def _shl_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shl_greater_than(lhs_domain))
    end

    def >>(rhs_domain)
      ValueDomain.of_undefined(@domain >> rhs_domain)
    end

    def _shr_nil(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shr_nil(lhs_domain))
    end

    def _shr_unlimited(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shr_unlimited(lhs_domain))
    end

    def _shr_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shr_equal_to(lhs_domain))
    end

    def _shr_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shr_less_than(lhs_domain))
    end

    def _shr_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_undefined(rhs_domain.domain._shr_greater_than(lhs_domain))
    end

    def_delegator :@domain, :!
    def_delegator :@domain, :<
    def_delegator :@domain, :_less_nil
    def_delegator :@domain, :_less_unlimited
    def_delegator :@domain, :_less_equal_to
    def_delegator :@domain, :_less_less_than
    def_delegator :@domain, :_less_greater_than
    def_delegator :@domain, :==
    def_delegator :@domain, :_equal_nil
    def_delegator :@domain, :_equal_unlimited
    def_delegator :@domain, :_equal_equal_to
    def_delegator :@domain, :_equal_less_than
    def_delegator :@domain, :_equal_greater_than
    def_delegator :@domain, :!=
    def_delegator :@domain, :_not_equal_nil
    def_delegator :@domain, :_not_equal_unlimited
    def_delegator :@domain, :_not_equal_equal_to
    def_delegator :@domain, :_not_equal_less_than
    def_delegator :@domain, :_not_equal_greater_than
    def_delegator :@domain, :logical_and
    def_delegator :@domain, :_logical_and_nil
    def_delegator :@domain, :_logical_and_unlimited
    def_delegator :@domain, :_logical_and_equal_to
    def_delegator :@domain, :_logical_and_less_than
    def_delegator :@domain, :_logical_and_greater_than
    def_delegator :@domain, :logical_or
    def_delegator :@domain, :_logical_or_nil
    def_delegator :@domain, :_logical_or_unlimited
    def_delegator :@domain, :_logical_or_equal_to
    def_delegator :@domain, :_logical_or_less_than
    def_delegator :@domain, :_logical_or_greater_than

    def intersection(rhs_domain)
      ValueDomain.of_undefined(@domain.intersection(rhs_domain))
    end

    def _intersection_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes UndefinedValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes UndefinedValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes UndefinedValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes UndefinedValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes UndefinedValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def union(rhs_domain)
      ValueDomain.of_undefined(@domain.union(rhs_domain))
    end

    def _union_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes UndefinedValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def _union_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes UndefinedValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def _union_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes UndefinedValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def _union_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes UndefinedValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def _union_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes UndefinedValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def coerce_to_integer
      ValueDomain.of_undefined(@domain.coerce_to_integer)
    end

    def coerce_to_real
      ValueDomain.of_undefined(@domain.coerce_to_real)
    end

    def_delegator :@domain, :min_value
    def_delegator :@domain, :max_value
    def_delegator :@domain, :each_sample

    def to_defined_domain
      @domain
    end

    def to_s
      "(== Undefined[#{@domain.to_s}])"
    end
    memoize :to_s

    def_delegator :@domain, :complexity
  end

  class AmbiguousValueDomain < ValueDomain
    extend Memoizable

    def initialize(undefined)
      @undefined = undefined
    end

    def empty?
      false
    end

    def nan?
      false
    end

    def undefined?
      @undefined
    end

    def ambiguous?
      true
    end

    def contain_value_domain?(rhs_domain)
      true
    end

    def _contain_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_unlimited?(lhs_domain, rhs_domain = self)
      true
    end

    def _contain_equal_to?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_less_than?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_greater_than?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_intersection?(lhs_domain, rhs_domain = self)
      false
    end

    def _contain_union?(lhs_domain, rhs_domain = self)
      false
    end

    def intersect?(value_domain)
      true
    end

    def _intersect_nil?(lhs_domain, rhs_domain = self)
      false
    end

    def _intersect_unlimited?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes AmbiguousValueDomain#intersect?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_equal_to?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes AmbiguousValueDomain#intersect?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_less_than?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes AmbiguousValueDomain#intersect?.
      rhs_domain.intersect?(lhs_domain)
    end

    def _intersect_greater_than?(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersect? RHS' equals to `RHS intersect? LHS'.
      #       This method invokes AmbiguousValueDomain#intersect?.
      rhs_domain.intersect?(lhs_domain)
    end

    def narrow(operator_symbol, operand_value_domain)
      self
    end

    def _narrow_by_eq(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_nil_by_eq(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_unlimited_by_eq(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_equal_to_by_eq(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_less_than_by_eq(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_greater_than_by_eq(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_by_ne(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_nil_by_ne(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_unlimited_by_ne(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_equal_to_by_ne(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_less_than_by_ne(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_greater_than_by_ne(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_by_lt(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_nil_by_lt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_unlimited_by_lt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_equal_to_by_lt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_less_than_by_lt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_greater_than_by_lt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_by_gt(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_nil_by_gt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_unlimited_by_gt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_equal_to_by_gt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_less_than_by_gt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _narrow_greater_than_by_gt(lhs_domain, rhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def widen(operator_symbol, operand_value_domain)
      self
    end

    def _widen_by_eq(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _widen_by_ne(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _widen_by_lt(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _widen_by_gt(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _widen_by_le(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def _widen_by_ge(rhs_domain, lhs_domain = self)
      ValueDomain.of_ambiguous(@undefined)
    end

    def inversion
      self
    end

    def ~
      self
    end

    def +@
      self
    end

    def -@
      self
    end

    def +(rhs_domain)
      self
    end

    def _add_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes AmbiguousValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes AmbiguousValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes AmbiguousValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes AmbiguousValueDomain#+.
      rhs_domain + lhs_domain
    end

    def _add_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS + RHS' equals to `RHS + LHS'.
      #       This method invokes AmbiguousValueDomain#+.
      rhs_domain + lhs_domain
    end

    def *(rhs_domain)
      self
    end

    def _mul_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes AmbiguousValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes AmbiguousValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes AmbiguousValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes AmbiguousValueDomain#*.
      rhs_domain * lhs_domain
    end

    def _mul_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS * RHS' equals to `RHS * LHS'.
      #       This method invokes AmbiguousValueDomain#*.
      rhs_domain * lhs_domain
    end

    def /(rhs_domain)
      self
    end

    def _div_nil(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _div_unlimited(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _div_equal_to(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _div_less_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _div_greater_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def &(rhs_domain)
      self
    end

    def _and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes AmbiguousValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes AmbiguousValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes AmbiguousValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes AmbiguousValueDomain#&.
      rhs_domain & lhs_domain
    end

    def _and_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS & RHS' equals to `RHS & LHS'.
      #       This method invokes AmbiguousValueDomain#&.
      rhs_domain & lhs_domain
    end

    def |(rhs_domain)
      self
    end

    def _or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes AmbiguousValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes AmbiguousValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes AmbiguousValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes AmbiguousValueDomain#|.
      rhs_domain | lhs_domain
    end

    def _or_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS | RHS' equals to `RHS | LHS'.
      #       This method invokes AmbiguousValueDomain#|.
      rhs_domain | lhs_domain
    end

    def ^(rhs_domain)
      self
    end

    def _xor_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes AmbiguousValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes AmbiguousValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes AmbiguousValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes AmbiguousValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def _xor_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS ^ RHS' equals to `RHS ^ LHS'.
      #       This method invokes AmbiguousValueDomain#^.
      rhs_domain ^ lhs_domain
    end

    def <<(rhs_domain)
      self
    end

    def _shl_nil(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _shl_unlimited(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _shl_equal_to(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _shl_less_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _shl_greater_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def >>(rhs_domain)
      self
    end

    def _shr_nil(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _shr_unlimited(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _shr_equal_to(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _shr_less_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def _shr_greater_than(lhs_domain, rhs_domain = self)
      rhs_domain
    end

    def !
      ValueDomain.of_unlimited
    end

    def <(rhs_domain)
      ValueDomain.of_unlimited
    end

    def _less_nil(lhs_domain, rhs_domain = self)
      ValueDomain.of_nil
    end

    def _less_unlimited(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def _less_equal_to(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def _less_less_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def _less_greater_than(lhs_domain, rhs_domain = self)
      ValueDomain.of_unlimited
    end

    def ==(rhs_domain)
      ValueDomain.of_unlimited
    end

    def _equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes AmbiguousValueDomain#==.
      rhs_domain == lhs_domain
    end

    def _equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes AmbiguousValueDomain#==.
      rhs_domain == lhs_domain
    end

    def _equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes AmbiguousValueDomain#==.
      rhs_domain == lhs_domain
    end

    def _equal_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes AmbiguousValueDomain#==.
      rhs_domain == lhs_domain
    end

    def _equal_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS == RHS' equals to `RHS == LHS'.
      #       This method invokes AmbiguousValueDomain#==.
      rhs_domain == lhs_domain
    end

    def !=(rhs_domain)
      ValueDomain.of_unlimited
    end

    def _not_equal_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes AmbiguousValueDomain#!=.
      rhs_domain != lhs_domain
    end

    def _not_equal_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes AmbiguousValueDomain#!=.
      rhs_domain != lhs_domain
    end

    def _not_equal_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes AmbiguousValueDomain#!=.
      rhs_domain != lhs_domain
    end

    def _not_equal_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes AmbiguousValueDomain#!=.
      rhs_domain != lhs_domain
    end

    def _not_equal_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS != RHS' equals to `RHS != LHS'.
      #       This method invokes AmbiguousValueDomain#!=.
      rhs_domain != lhs_domain
    end

    def logical_and(rhs_domain)
      ValueDomain.of_unlimited
    end

    def _logical_and_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes AmbiguousValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes AmbiguousValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes AmbiguousValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes AmbiguousValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def _logical_and_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS && RHS' equals to `RHS && LHS'.
      #       This method invokes AmbiguousValueDomain#logical_and.
      rhs_domain.logical_and(lhs_domain)
    end

    def logical_or(rhs_domain)
      ValueDomain.of_unlimited
    end

    def _logical_or_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes AmbiguousValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes AmbiguousValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes AmbiguousValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes AmbiguousValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def _logical_or_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS || RHS' equals to `RHS || LHS'.
      #       This method invokes AmbiguousValueDomain#logical_or.
      rhs_domain.logical_or(lhs_domain)
    end

    def intersection(rhs_domain)
      self
    end

    def _intersection_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes AmbiguousValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes AmbiguousValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes AmbiguousValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes AmbiguousValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def _intersection_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS intersection RHS' equals to `RHS intersection LHS'.
      #       This method invokes AmbiguousValueDomain#intersection.
      rhs_domain.intersection(lhs_domain)
    end

    def union(rhs_domain)
      self
    end

    def _union_nil(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes AmbiguousValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def _union_unlimited(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes AmbiguousValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def _union_equal_to(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes AmbiguousValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def _union_less_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes AmbiguousValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def _union_greater_than(lhs_domain, rhs_domain = self)
      # NOTE: `LHS union RHS' equals to `RHS union LHS'.
      #       This method invokes AmbiguousValueDomain#union.
      rhs_domain.union(lhs_domain)
    end

    def coerce_to_integer
      self
    end

    def coerce_to_real
      self
    end

    def min_value
      nil
    end

    def max_value
      nil
    end

    def each_sample
      return ::Enumerator.new(self, :each_sample) unless block_given?
      yield(0)
      self
    end

    def to_defined_domain
      ValueDomain.of_ambiguous(false)
    end

    def to_s
      "(== Ambiguous)"
    end
    memoize :to_s

    def complexity
      1
    end
  end

end
end

if $0 == __FILE__
  include AdLint::C

  d1 = ValueDomain.equal_to(-2147483647)
  d2 = ValueDomain.equal_to(-2147483647)
  p d1.contain?(d2)

  puts "---"
  d1 = ValueDomain.of_undefined(-10..10)
  d2 = ValueDomain.equal_to(-5).union(ValueDomain.equal_to(5))
  p d1, d2, d1.contain?(d2), !d2.contain?(d1)

  puts "---"
  d2 = d2.intersection(ValueDomain.greater_than_or_equal_to(-10))
  d2 = d2.intersection(ValueDomain.less_than_or_equal_to(10))
  p d1, d2, d1.contain?(d2), !d2.contain?(d1)

  puts "---"
  d1 = d1.intersection(ValueDomain.less_than(0))
  p d1, d2, !d1.contain?(d2), !d2.contain?(d1)

  puts "---"
  d2 = d2.intersection(ValueDomain.less_than(0))
  p d1, d2, d1.contain?(d2), !d2.contain?(d1)

  puts "---"
  d1 = ValueDomain.greater_than(-2147483647)
  d1 = d1.intersection(ValueDomain.less_than(0))
  d1 = d1.union(ValueDomain.equal_to(0))
  d2 = ValueDomain.equal_to(-2147483647)
  d2 = d2.union(ValueDomain.equal_to(0))
  p d1, d2
  d1 = d1.union(d2)
  p d1

  puts "---"
  d1 = ValueDomain.equal_to(-2147483647).union(ValueDomain.equal_to(0))
  d2 = ValueDomain.equal_to(-2147483647).union(ValueDomain.equal_to(0))
  p d1, d2
  p d1.contain?(d2)
  p d1.contain?(ValueDomain.equal_to(-2147483647))
  p d1.contain?(ValueDomain.equal_to(0))

  puts "---"
  d1 = ValueDomain.greater_than(-2147483648)
  d1 = d1.intersection(ValueDomain.less_than(2147483647))
  d1 = d1.union(ValueDomain.equal_to(2147483647))
  d2 = d1.dup
  p d1, d2
  p d1.contain?(d2)
  p d1.intersection(d2)

  puts "---"
  d1 = ValueDomain.greater_than(1)
  d1 = d1.intersection(ValueDomain.less_than(2147483647))
  d1 = d1.union(ValueDomain.equal_to(1))
  d2 = ValueDomain.equal_to(2147483647)
  d2 = d2.union(ValueDomain.equal_to(1))
  p d1, d2
  p d1.union(d2)

  puts "---"
  d1 = ValueDomain.greater_than(0).intersection(ValueDomain.less_than(429))
  d1 = d1.union(ValueDomain.equal_to(429))
  d2 = ValueDomain.equal_to(0)
  p d1.intersection(d2)

  puts "---"
  d1 = UndefinedValueDomain.new(0..10)
  d2 = UndefinedValueDomain.new(0..10)
  p d1 + d2

  puts "---"
  d1 = ValueDomain.greater_than_or_equal_to(-10).intersection(
    ValueDomain.less_than_or_equal_to(10))
  d2 = ValueDomain.equal_to(0)
  p d1, d2
  p d1.intersect?(d2)
  p d1 < d2

  puts "---"
  d1 = ValueDomain.greater_than(1)
  d1 = d1.intersection(ValueDomain.less_than(11))
  d1 = d1.union(ValueDomain.equal_to(11))
  d2 = ValueDomain.greater_than(0)
  d2 = d2.intersection(ValueDomain.less_than(10))
  d2 = d2.union(ValueDomain.equal_to(0))
  d2 = d2.union(ValueDomain.equal_to(10))
  p d1, d2
  p d1.narrow(:==, d2)

  puts "---"
  d1 = ValueDomain.greater_than(-129)
  d1 = d1.intersection(ValueDomain.less_than(128))
  d2 = ValueDomain.equal_to(-128)
  p d1, d2
  p d1.intersect?(d2)
  p d1 < d2

  puts "---"
  d1 = ValueDomain.greater_than(-129)
  d1 = d1.intersection(ValueDomain.less_than(128))
  d2 = ValueDomain.equal_to(127)
  p d1, d2
  p d1.intersect?(d2)
  p d1 > d2

  puts "---"
  d1 = ValueDomain.greater_than(-129)
  d2 = ValueDomain.equal_to(127)
  p d1, d2
  p d1 < d2

  puts "---"
  d1 = ValueDomain.less_than(128)
  d2 = ValueDomain.equal_to(127)
  p d1, d2
  p d1 < d2

  puts "---"
  d1 = ValueDomain.greater_than_or_equal_to(-2147483648)
  d1 = d1.intersection(ValueDomain.less_than_or_equal_to(2147483647))
  d1 = ValueDomain.of_undefined(d1)
  d2 = ValueDomain.equal_to(1)
  p d1, d2
  p d1.narrow(:!=, d2)
end
