# Analyzer classes.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/version"
require "adlint/traits"
require "adlint/report"
require "adlint/metric"
require "adlint/phase"
require "adlint/lang"
require "adlint/source"
require "adlint/symbol"
require "adlint/monitor"
require "adlint/ld/phase"

module AdLint #:nodoc:

  class Analyzer
    def self.current
      @@current
    end

    def self.current=(analyzer)
      @@current = analyzer
    end

    def initialize(name, log_base_fpath, output_dpath = nil)
      @name = name
      @log_base_fpath = log_base_fpath
      @output_dpath = output_dpath
    end

    def run
      File.open(log_fpath, "w") do |log_io|
        @logger = Logger.new(log_io)
        @logger.progname = @name
        @logger.datetime_format = "%F %T "
        begin
          Analyzer.current = self
          execute
        rescue => ex
          fatal(ex)
          return false
        ensure
          Analyzer.current = nil
        end
      end
      true
    end

    extend Forwardable

    def_delegator :@logger, :fatal
    def_delegator :@logger, :error
    def_delegator :@logger, :warn
    def_delegator :@logger, :info
    def_delegator :@logger, :debug

    private
    def execute
      subclass_responsibility
    end

    def log_fpath
      if @output_dpath
        @output_dpath.join(@log_base_fpath).add_ext(".log")
      else
        @log_base_fpath.add_ext(".log")
      end
    end
  end

  # == Single module analysis driver.
  class SingleModuleAnalyzer < Analyzer
    def initialize(fpath, strip_num = 0, output_dpath = nil)
      super("SMA-#{SHORT_VERSION}", fpath.strip(strip_num), output_dpath)

      @fpath = fpath
      @strip_num = strip_num
    end

    private
    def execute
      lang = Language.of(@fpath)
      return unless lang

      ProgressMonitor.instance.reset(@fpath, lang.single_module_phases.size)

      Report.new(msg_fpath, met_fpath) do |report|
        context = PhaseContext.new(report, log_fpath)

        context[:sources] = [Source.new(@fpath)]
        context[:symbol_table] = SymbolTable.new

        begin
          lang.single_module_phases.each do |phase_class|
            phase_class.new(context).execute
          end
        ensure
          File.open(i_fpath, "w") do |io|
            io.set_encoding(Encoding.default_external)
            io.puts(context[:c_source].to_s)
          end
        end
      end
    rescue
      if verbose?
        $stderr.puts "An error was occurred while processing `#{@fpath}'."
        $stderr.puts "See `#{msg_fpath}' and `#{log_fpath}' for more details."
      end
      raise
    end

    def msg_fpath
      if @output_dpath
        @output_dpath.join(@fpath.strip(@strip_num)).add_ext(".msg.csv")
      else
        @fpath.strip(@strip_num).add_ext(".msg.csv")
      end
    end

    def met_fpath
      if @output_dpath
        @output_dpath.join(@fpath.strip(@strip_num)).add_ext(".met.csv")
      else
        @fpath.strip(@strip_num).add_ext(".met.csv")
      end
    end

    def i_fpath
      if @output_dpath
        @output_dpath.join(@fpath.strip(@strip_num)).sub_ext(".i")
      else
        @fpath.strip(@strip_num).sub_ext(".i")
      end
    end
  end

  class CrossModuleAnalyzer < Analyzer
    def initialize(fpaths, output_dpath = nil)
      @project_name = Traits.instance.of_project.project_name
      super("CMA-#{SHORT_VERSION}", Pathname.new(@project_name), output_dpath)

      @fpaths = fpaths
    end

    private
    PHASE_CLASSES = [
      AdLint::Ld::MapTypedefPhase,
      AdLint::Ld::MapFunctionPhase,
      AdLint::Ld::MapVariablePhase,
      AdLint::Ld::LinkFunctionPhase,
      AdLint::Ld::LinkVariablePhase,
      AdLint::Ld::PreparePhase,
      AdLint::Ld::TypedefReviewPhase,
      AdLint::Ld::FunctionReviewPhase,
      AdLint::Ld::VariableReviewPhase,
      AdLint::Ld::CommandPhase
    ].freeze
    private_constant :PHASE_CLASSES

    def execute
      ProgressMonitor.instance.reset(@project_name, PHASE_CLASSES.size)

      Report.new(msg_fpath, met_fpath) do |report|
        context = PhaseContext.new(report, log_fpath)

        context[:project_name] = @project_name
        context[:metric_fpaths] = @fpaths

        PHASE_CLASSES.each do |phase_class|
          phase_class.new(context).execute
        end
      end
    rescue
      if verbose?
        $stderr.puts "An error was occurred while executing " +
          "cross module analysis."
        $stderr.puts "See `#{log_fpath}' for more details."
      end
      raise
    end

    def msg_fpath
      if @output_dpath
        @output_dpath.join(Pathname.new(@project_name)).add_ext(".msg.csv")
      else
        Pathname.new(@project_name).add_ext(".msg.csv")
      end
    end

    def met_fpath
      if @output_dpath
        @output_dpath.join(Pathname.new(@project_name)).add_ext(".met.csv")
      else
        Pathname.new(@project_name).add_ext(".met.csv")
      end
    end
  end

  # == Configuration files validator.
  class ConfigurationValidator < Analyzer
    def initialize(fpath, strip_num = 0, output_dpath = nil)
      super("CHK-#{SHORT_VERSION}", fpath.strip(strip_num), output_dpath)

      @fpath = fpath
      @strip_num = strip_num
    end

    private
    def execute
      lang = Language.of(@fpath)
      return unless lang

      ProgressMonitor.instance.reset(@fpath, lang.check_phases.size)

      Report.new(msg_fpath, met_fpath) do |report|
        context = PhaseContext.new(report, log_fpath)

        context[:sources] = [Source.new(@fpath)]
        context[:symbol_table] = SymbolTable.new

        begin
          lang.check_phases.each do |phase_class|
            phase_class.new(context).execute
          end
        ensure
          File.open(i_fpath, "w") do |io|
            io.set_encoding(Encoding.default_external)
            io.puts(context[:c_source].to_s)
          end
        end
      end
    rescue
      if verbose?
        $stderr.puts "An error was occurred while processing `#{@fpath}'."
        $stderr.puts "See `#{msg_fpath}' and `#{log_fpath}' for more details."
      end
      raise
    end

    def msg_fpath
      if @output_dpath
        @output_dpath.join(@fpath.strip(@strip_num)).add_ext(".msg.csv")
      else
        @fpath.strip(@strip_num).add_ext(".msg.csv")
      end
    end

    def met_fpath
      if @output_dpath
        @output_dpath.join(@fpath.strip(@strip_num)).add_ext(".met.csv")
      else
        @fpath.strip(@strip_num).add_ext(".met.csv")
      end
    end

    def i_fpath
      if @output_dpath
        @output_dpath.join(@fpath.strip(@strip_num)).sub_ext(".i")
      else
        @fpath.strip(@strip_num).sub_ext(".i")
      end
    end
  end

end
