// -*- C++ -*- (c) 2008 Petr Rockai <me@mornfall.net>

#include <apt-pkg/debsystem.h>
#include <adept/guidpkgpm.h>

#include <dirent.h>

#ifndef ADEPT_PKGSYSTEM_H
#define ADEPT_PKGSYSTEM_H

namespace adept {

struct RecoveryNeededException : wibble::exception::Generic {
};

struct LockingFailureException : wibble::exception::System {
    LockingFailureException( std::string msg ) 
        : System( msg )
    {}
};

struct PkgSystem : public debSystem
{
    int m_lockFd;
    int m_lockCount;

    PkgSystem( AptDatabase &db ) : m_db( &db ), pkgs( db ) {
        m_lockCount = 0;
        m_lockFd = -1;
        std::cerr << "Constructing PkgSystem" << std::endl;
        Label = "adeptDPkgSystem";
    }

    virtual pkgPackageManager *CreatePM( pkgDepCache * ) const {
        std::cerr << "Creating GuiDPkgPM" << std::endl;
        return new GuiDPkgPM( pkgs );
    }

    virtual int Score( Configuration const &Cnf ) {
        std::cerr << "PkgSystem Score:"
                  << debSystem::Score( Cnf ) + 2 << std::endl;
        return debSystem::Score( Cnf ) + 2;
    }

    struct TemporaryUnlock {
        PkgSystem *p;
        int count;
        TemporaryUnlock( pkgSystem *_p ) : p( dynamic_cast< PkgSystem *> ( _p ) )
        {
            count = p->m_lockCount;
            for ( int i = 0; i < count; ++i )
                p->UnLock();
        }

        void dropLock() {
            count = 0;
        }

        ~TemporaryUnlock() {
            for ( int i = 0; i < count; ++i )
                p->Lock();
        }
    };

    /* The following 3 methods are copies of code in debsystem.cc,
     * which is kind of necessary, since it is private and we can't
     * extract lock failure reason from the API either. Silly. */
    bool Lock()
    {
        // Disable file locking
        if (_config->FindB("Debug::NoLocking",false) == true || m_lockCount > 1)
        {
            m_lockCount++;
            return true;
        }
        
        // Create the lockfile
        string AdminDir = flNotFile(_config->Find("Dir::State::status"));
        m_lockFd = GetLock(AdminDir + "lock");
        if (m_lockFd == -1)
        {
            throw LockingFailureException(
                "Unable to lock administrative directory " + AdminDir );
        }
        
        // See if we need to abort with a dirty journal
        if (checkUpdates() == true)
        {
            close(m_lockFd);
            m_lockFd = -1;
            throw RecoveryNeededException();
        }
        
        m_lockCount++;
        
        return true;
    }

    bool UnLock(bool NoErrors = true)
    {
        if (m_lockCount == 0 && NoErrors == true)
            return false;
        
        if (m_lockCount < 1)
            return _error->Error("Not locked");
        if (--m_lockCount == 0)
        {
            close(m_lockFd);
            m_lockCount = 0;
        }
        
        return true;
    }

    bool checkUpdates()
    {
        // Check for updates.. (dirty)
        string File = flNotFile(_config->Find("Dir::State::status")) + "updates/";
        DIR *DirP = opendir(File.c_str());
        if (DirP == 0)
            return false;
        
        /* We ignore any files that are not all digits, this skips .,.. and 
           some tmp files dpkg will leave behind.. */
        bool Damaged = false;
        for (struct dirent *Ent = readdir(DirP); Ent != 0; Ent = readdir(DirP))
        {
            Damaged = true;
            for (unsigned int I = 0; Ent->d_name[I] != 0; I++)
            {
                // Check if its not a digit..
                if (isdigit(Ent->d_name[I]) == 0)
                {
                    Damaged = false;
                    break;
                }
            }
            if (Damaged == true)
                break;
        }
        closedir(DirP);
        
        return Damaged;
    }

protected:
    AptDatabase *m_db;
    mutable package::Source pkgs;

    void statusChanged( int p, QString m );
};

}

#endif
