#include <adept/packagelist.h>

namespace adept {

// TODO code duplication
int PackageExtender::iconSize() const {
    int h = QFontMetrics( font() ).height();
    if ( desktop.valid() ) {
        if ( h > 32 ) // this is line height; we have 2 lines
            return 64;
        else
            return 32;
    } else {
        return h + 2;
    }
}

PackageExtender::PackageExtender( PackageData  &_d, ept::Token _t,
                                  QWidget *parent )
    : PackageInfo( _d, _t, parent )
{
    QSizePolicy pol( QSizePolicy::Expanding, QSizePolicy::Fixed );
    m_layout = new QVBoxLayout(this);
    QFontMetrics fm( font() );
    layout()->setContentsMargins( 12 + iconSize(), 5, 10, 10 );
    m_info = new QLabel( this );
    m_info->setWordWrap( true );
    m_info->setSizePolicy( pol );
    m_layout->addWidget(m_info);

    m_desc = new QLabel( this );
    m_desc->setWordWrap( true );
    m_desc->setSizePolicy( pol );
    // Text should be selectable/copyable
    m_desc->setTextInteractionFlags( Qt::TextBrowserInteraction );
    adjustFontSize( m_desc, -1 );
    m_layout->addWidget( m_desc );

    m_box = new KHBox( this );
    m_layout->addWidget(m_box);
    m_actBox = new KVBox( m_box );

    QWidget *filler = new QWidget( this ); // spacer
    m_layout->addWidget(filler);

    refresh();
}

void PackageExtender::refresh() {
    block = true;
    setUpdatesEnabled( false );

    QString info;
    PackageState s = d.pkgs.get< package::State >( t );
    ept::Token candidate = d.pkgs.get< package::CandidateVersion >( t ),
               installed = d.pkgs.get< package::InstalledVersion >( t );
    if ( s.hasNewVersion() ) {
        info = i18n( "Version <b>" )
               + u8( installed.version() )
               + i18n( "</b> is installed, version <b>" )
               + u8( candidate.version() )
               + i18n( "</b> is available." );
    } else if ( !s.installed() && d.pkgs.exists( candidate ) ) {
        info = i18n( "Version <b>" )
               + u8( candidate.version() )
               + i18n( "</b> is available." );
    } else if ( d.pkgs.exists( installed ) ) {
        info = i18n( "Version <b>" )
               + u8( installed.version() )
               + i18n( "</b> is installed." );
    } else {
        info = i18n( "No versions available." );
    }

    info += i18n( " Package is maintained by <i>" )
            + u8( d.recs.get< record::Maintainer >( t ) ) + i18n( "</i>" );
            
    m_desc->setText(
        formatLongDescription(
            u8( d.recs.get< record::LongDescription >( t ) ) )
        + i18n( "<a href=\"foo\">more...</a>" ) );
    connect( m_desc, SIGNAL( linkActivated( const QString & ) ),
             this, SIGNAL( detailsRequested() ) );
    m_info->setText( info );

    refreshCheckboxes( m_actBox, s );
    setUpdatesEnabled( true );
    update();
}

QWidget *PackageDelegate::createEditorWidget( QWidget *parent,
                                              const QStyleOptionViewItem &,
                                              const QModelIndex &idx ) const
{
    PackageExtender *e = new PackageExtender( d, token( idx ), parent );
    connect( e, SIGNAL( changed() ), this, SIGNAL( changed() ) );
    connect( this, SIGNAL( changed() ), e, SLOT( refresh() ) );
    connect( this, SIGNAL( externallyChanged() ), e, SLOT( refresh() ) );
    connect( e, SIGNAL( detailsRequested() ),
             this, SLOT( requestDetails() ) );
    return e;
}

void PackageDelegate::requestDetails() {
    ept::Token t = dynamic_cast< PackageExtender * >( sender() )->token();
    detailsRequested( t );
}

int PackageDelegate::iconSize( const QStyleOptionViewItem &opt,
                               const QModelIndex &idx ) const {
    int h = QFontMetrics( opt.font ).height();
    if ( token( idx ).isDesktop() ) {
        if ( h > 32 ) // this is line height; we have 2 lines
            return 64;
        else
            return 32;
    } else {
        return h + 2;
    }
}

QSize PackageDelegate::retractedSizeHint( const QStyleOptionViewItem &opt,
                                     const QModelIndex & ) const
{
    int h = QFontMetrics( opt.font ).height() + 2;
    return QSize( 200, 2 * h + 2 );
}

/** Paints the short/retracted package description */
void PackageDelegate::paint( QPainter *p, const QStyleOptionViewItem &_opt,
                             const QModelIndex &idx ) const
{
    QStyleOptionViewItemV2 opt =
        static_cast< QStyleOptionViewItemV2 >( _opt );
    bool alternate = opt.features & QStyleOptionViewItemV2::Alternate;
    int width = opt.rect.width(),
       height = opt.rect.height();

    if ( width == 0 )
        return;

    QPalette pal = opt.palette;
    QColor ct = pal.color( QPalette::Current, QPalette::Text );
        
    QFont fb = opt.font;
    fb.setBold( true );
    QFontMetrics fm( opt.font );
    QFontMetrics fmb( fb );
    QBrush base = pal.brush( 
        QPalette::Current,
        alternate ? QPalette::AlternateBase : QPalette::Base );

    int size = iconSize( opt, idx );
    int off = size + 4;

    p->save();
    p->translate( opt.rect.left(), opt.rect.top() );
    p->setClipRect( 0, 0, width, height );
    p->setFont( opt.font );
    p->fillRect( 0, 0, width, height, QBrush( base ) );

    if ( opt.state & QStyle::State_Selected ) {
        p->fillRect( 0, 0, off, height, pal.color( QPalette::Current,
                                                   QPalette::Highlight ) );
    }

    off += 2;

    ept::Token t = token( idx ), desktop;
    if ( t.isDesktop() ) {
        desktop = t;
        t = d.desk.get< desktop::Package >( t );
    }

    PackageState s = d.pkgs.get< package::State >( t );

    QString name = u8( desktop.valid() ?
                       d.desk.get< desktop::Name >( desktop ) :
                       ( t.package() ) ) + ": ",
          status = statusString( s ),
            dash = u8( " — " ),
          action = actionString( s ),
            desc = u8( d.recs.get< record::ShortDescription >( t ) );
    QSize szname = fmb.size( 0, name ),
        szstatus = fm.size( 0, status ),
          szdash = fm.size( 0, dash ),
        szaction = fm.size( 0, action ),
          szdesc = fm.size( 0, desc );
    szdesc.setWidth( std::min( szdesc.width(), width - off - 2 ) );
    desc = fm.elidedText( desc, Qt::ElideRight, szdesc.width() );
        
    int baseline = szname.height() - 2;
        
    p->setFont( fb );
    p->drawText( off, baseline, name );
    p->setFont( opt.font );
    p->drawText( off + 1, baseline + szdesc.height(), desc );
    off += szname.width();
        
    p->setPen( statusColor( s ) );
    p->drawText( off, baseline, status );
    off += szstatus.width();
        
    if ( !s.keep() ) {
        p->setPen( ct );
        p->drawText( off, baseline, dash );
        off += szdash.width();
            
        p->setPen( actionColor( s ) );
        p->drawText( off, baseline, action );
    }

    QPixmap icon;
    if ( desktop.valid() ) {
        QString iname = u8( d.desk.get< desktop::Icon >( desktop ) );
        if ( !QPixmapCache::find( iname, icon ) ) {
            QString path = KIconLoader::global()->iconPath(
                iname, -size, true );
            if ( path.isNull() )
                path = KGlobal::dirs()->findResource( "desktopicon", iname );
            icon = KIconLoader::global()->loadIcon(
                path, KIconLoader::NoGroup, size );
            QPixmapCache::insert( iname, icon );
        }
    } else {
        icon = statusIcon( s, size );
    }
    int y = ( szname.height() + szdesc.height() + 4 - size ) / 2;
    p->drawPixmap( 2, y, icon );
    p->restore();
}

PackageListView *setupList( QObject *obj, QWidget *p,
                            TokenModel *m, PackageData &d )
{
    PackageDelegate *delegate;
    PackageListView *r = new PackageListView(
        delegate = new PackageDelegate( d ), p );
    QObject::connect( delegate, SIGNAL( changed() ), obj, SLOT( refresh() ) );
    r->setModel( m );
    r->header()->hide();
    return r;
}

}

