; ACL2 Version 3.3 -- A Computational Logic for Applicative Common
; Lisp Copyright (C) 2007 University of Texas at Austin

; This version of ACL2 is a descendent of ACL2 Version 1.9, Copyright
; (C) 1997 Computational Logic, Inc.  See the documentation topic
; NOTE-2-0.

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software
; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

; Regarding authorship of ACL2 in general:

; Written by:  Matt Kaufmann               and J Strother Moore
; email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
; Department of Computer Sciences
; University of Texas at Austin
; Austin, TX 78712-1188 U.S.A.

; The original version of this file was contributed by Bob Boyer and
; Warren A. Hunt, Jr.  The design of this system of Hash CONS,
; function memoization, and fast association lists (applicative hash
; tables) was initially implemented by Boyer and Hunt.  The code has
; been augmented by Matt Kaufmann, Sol Swords, and others.

(in-package "ACL2")

#+openmcl
(eval-when (load eval compile)
  ; (pushnew :sol *features*)
  (setq *compile-verbose* t)
  (setq *load-print* t)
  (setq *load-verbose* t))

;;;;;;;; UTILITIES ;;;;;;;; UTILITIES ;;;;;;;; UTILITIES
;;;;;;;; UTILITIES ;;;;;;;; UTILITIES ;;;;;;;; UTILITIES


; DEFGLOBAL

(defmacro defg (&rest r)

  "In OpenMCL, DEFG behaves exactly as DEFPARAMETER; however, its use
  includes three promises: (1) never to locally bind the variable,
  e.g., with LET or LAMBDA, (2) never to reference the variable when
  it is not set, which would be an error anyway, of course, and (3)
  never to test whether the variable is BOUNDP.  OpenMCL uses about
  ten fewer machine instructions to reference such a variable."

  #-Openmcl
  `(defparameter ,@r)
  #+Openmcl
  `(ccl::defglobal ,@r))

; PRINL

(defmacro prinl (&rest r)

  "PRINL is for debugging.  PRINL PRIN1s the members of r and their
  values.  For example, (prinl a b (+ a b)) might print:
    A 
    => 1
    B
     => 2
    (+ A B)
     => 3
  PRINL returns the principal value of the last member of r.  PRINL
  does not evaluate the members of r that are neither symbols nor
  conses, but it does PRINC those members."

  (let ((tem (make-symbol "TEM")))
    `(our-syntax-nice
      (let ((,tem nil)
            (*print-right-margin* 70)
            (*print-level* 3)
            (*print-length* 16))
        ,@(loop for x in r collect
                (cond ((or (consp x)
                           (symbolp x))
                       `(progn (ofd "~&~:s ~32t=>~40t" ',x)
                               (ofd "~:d" (setq ,tem ,x))))
                      (t `(ofd "~&~:d" (setq ,tem ',x)))))
        ,tem))))

; TIMING UTILITIES

#+openmcl
(eval-when (load eval compile)
  (if (fboundp 'ccl::rdtsc) (pushnew :rdtsc *features*)))

; *float-ticks/second* is set correctly by hons-init.
(defg *float-ticks/second* 1.0)
(declaim (float *float-ticks/second*))

(defmacro internal-real-time ()
  `(the fixnum
     (let ((n #+RDTSC (ccl::rdtsc)
              #-RDTSC (get-internal-real-time)))
       #+RDTSC
       n
       #-RDTSC
       (if (typep n 'fixnum)
           n
         (error "~&; Error: ** get-internal-real-time returned a ~
                 nonfixnum.")))))

(defun float-ticks/second-init ()
  (setq *float-ticks/second*
        #+RDTSC
        (let ((i1 (ccl::rdtsc64))
              (i2 (progn (sleep .01) (ccl::rdtsc))))
          (if (>= i2 i1)
              (* 100 (float (- i2 i1)))
            (ofe "~&; hons-init:  ** Error:  RDTSC nonsense.")))
        #-RDTSC
        (float internal-time-units-per-second))
  (check-type *float-ticks/second*
              (and float (satisfies plusp))))


; SAFE-INCF

(defmacro safe-incf (x inc &optional where)

  "SAFE-INCF is a raw Lisp macro that behaves the same as INCF when
  both X and INCF are nonnegative fixnums and their sum is a
  nonnegative FIXNUM.  In a call of (SAFE-INCF x inc), X must be a
  place that holds a FIXNUM.  INC must evaluate to a FIXNUM.  Both X
  and INC must evaluate without side effects, so that it is impossible
  to tell which was executed first or whether only one or both were
  executed.  If INC is not positive, no update takes place at all.
  Otherwise, if the sum of the values of X and INC is not a FIXNUM,
  which is tested without causing an error, a run-time error will be
  caused.  If the sum is a FIXNUM then, as with INCF, the place X will
  be set to hold the sum of the old value of that place and the value
  of INC.  The value returned by SAFE-INCF is NIL.  Caution:  INC may
  be evaluated first, which is why side effects are prohibited.

  An optional third parameter is merely to help with error location
  identification.

  In (SAFE-INCF (AREF A (FOO)) INC), (FOO) is only evaluted once.
  Same for SVREF."

  (cond ((integerp inc)
         (cond ((<= inc 0) nil)
               (t `(safe-incf-aux ,x ,inc ,where))))
        ((symbolp inc)
         `(cond ((>= 0 ,inc) nil)
                (t (safe-incf-aux ,x ,inc ,where))))
        (t (let ((incv (make-symbol "INCV")))
             `(let ((,incv ,inc))
                (declare (fixnum ,incv))
                (cond ((>= 0 ,incv) nil)
                      (t (safe-incf-aux ,x ,incv ,where))))))))

(defmacro safe-incf-aux (x inc where)
  (cond
   ((not (or (symbolp inc)
             (and (typep inc 'fixnum)
                  (> inc 0))))
    (error "~%; SAFE-INCF-AUX: ** Error: ~a."
           (list :x x :inc inc :where where)))
   ((and (true-listp x)
         (equal (len x) 3)
         (member (car x) '(aref svref))
         (symbolp (nth 1 x))
         (consp (nth 2 x)))
    (let ((idx (make-symbol "IDX")))
      `(let ((,idx (the fixnum ,(nth 2 x))))
         (declare (fixnum ,idx))
         (safe-incf (,(nth 0 x)
                     ,(nth 1 x)
                     ,idx)
                    ,inc
                    ',where))))
   (t
    (let ((v (make-symbol "V")))
      `(let ((,v ,x))
         (declare (fixnum ,v))
         (cond ((<= ,v (the fixnum (- most-positive-fixnum ,inc)))
                (setf ,x (the fixnum (+ ,v ,inc)))
                nil)
               (t (error "~%; SAFE-INCF-AUX: ** Error: Overflow in ~a"
                         (list :x ',x :inc ',inc :where ',where)))))))))

; PARALLEL

; Feel free any time to comment out the following instruction, namely
; (pushnew :parallel *features*) features.  Doing so will simply stop
; a lot of locking and binding that are of no value in a sequentially
; executing system.  We have attempted to make honsing, memoizing, and
; Emod-compilation 'thread safe', whatever in hell that means, but we
; have no idea what we are really doing and are simply coding based
; upon what we feel is intuitive common sense.  Very subtle stuff.

; (pushnew :parallel *features*)

#+parallel
(unless (member :openmcl *features*)
  (error "We use OpenMCL primitives for parallelism."))

; We limit our efforts at thread-safedness to (a) locking/unlocking
; some hash tables and (b) in extreme cases, checking that no more
; than 3 processes are running.

(declaim (special *hons-cdr-ht* *compiled-module-ht* *memoize-info-ht*))

; Important lock order.  To avoid deadlock, we always lock HONS before
; COMPILED and COMPILED before MEMOIZE and then unlock in the order
; MEMOIZE, COMPILED, HONS.

; Individual pons tables get locked and unlocked briefly running
; memoized functions, but only for ponsing together arguments of
; multiple argument functions.  It would be very bad if that ponsing
; activity could get out to any honsing, memoizing, or emod-compiling.

(defmacro our-lock-unlock-ht1 (ht &rest r)
  (declare (ignorable ht))
  #+parallel
  `(progn (ccl::lock-hash-table ,ht)
          (prog1 ,@r
                 (ccl::unlock-hash-table ,ht)))
  #-parallel `(prog1 ,@ r))

(defmacro our-lock-unlock-hons1 (&rest r)
  `(our-lock-unlock-ht1 *hons-cdr-ht* ,@r))

(defmacro our-lock-unlock-compile1 (&rest r)
  `(our-lock-unlock-hons1
    (our-lock-unlock-ht1 *compiled-module-ht* ,@r)))

(defmacro our-lock-unlock-memoize1 (&rest r)
  `(our-lock-unlock-compile1
    (our-lock-unlock-ht1 *memoize-info-ht* ,@r)))

(defmacro our-lock-unlock-htmv1 (ht &rest r)
  (declare (ignorable ht))
  #+parallel
  `(progn (ccl::lock-hash-table ,ht)
          (multiple-value-prog1 ,@r
                 (ccl::unlock-hash-table ,ht)))
  #-parallel `(multiple-value-prog1 ,@r))

(defmacro our-lock-unlock-honsmv1 (&rest r)
  `(our-lock-unlock-htmv1 *hons-cdr-ht* ,@r))

(defmacro our-lock-unlock-compilemv1 (&rest r)
  `(our-lock-unlock-honsmv1
    (our-lock-unlock-htmv1 *compiled-module-ht* ,@r)))

(defmacro our-lock-unlock-memoizemv1 (&rest r)
  `(our-lock-unlock-compilemv1
    (our-lock-unlock-htmv1 *memoize-info-ht* ,@r)))

(defmacro check-no-extra-processes ()
  #+parallel
  `(let ((watch (eval '(and (boundp '*watch-thread*)
                            (ccl::processp *watch-thread*)))))
     (when watch
       (ofd "~%; check-no-extra-processes:   Killing watch.")
       (funcall 'watch-kill))
     (when (> (length (ccl::all-processes)) 2)
       (ofe "~%; Too many processes running at this time for ~
           hons/memo/emod-compile sanity.")))
  #-parallel
  nil)

;  OUR-SYNTAX

(defg *print-pprint-dispatch-orig* *print-pprint-dispatch*)

(declaim (special *pause-lines*))

(defmacro our-syntax (&rest args)

  "OUR-SYNTAX is derived from Common Lisp's WITH-STANDARD-IO-SYNTAX;
  we note below with an asterisk lines that differ from
  WITH-STANDARD-IO-SYNTAX.

  These settings are oriented towards reliable, standard, vanilla,
  mechanical reading and printing, but not towards debugging or human
  interaction.  Please, before changing the following, consider
  existing uses of this macro insofar as the changes might impact
  reliable, standard, vanilla, mechanical printing.  Especially
  consider COMPACT-PRINT-FILE."

  `(let ((*package*                    *acl2-package*) ; *
         (*print-array*                t)
         (*print-base*                 10)
         (*print-case*                 :upcase)
         (*print-circle*               nil)
         (*print-escape*               t)
         (*print-gensym*               t)
         (*print-length*               nil)
         (*print-level*                nil)
         (*print-lines*                nil)
         (*pause-lines*                nil)            ; *
         (*print-pretty*               nil)
         (*print-radix*                nil)
         (*print-readably*             t)
         (*print-right-margin*         nil)
         (*print-pprint-dispatch*      *print-pprint-dispatch-orig*)
         (*read-base*                  10)
         (*read-default-float-format*  'single-float)
         (*print-miser-width*          nil)
         (*read-eval*                  nil)            ; *
         (*read-suppress*              nil)
         (*readtable*                  *acl2-readtable*))  ; *
     ,@args))

(defmacro our-syntax-nice (&rest args)

; for more pleasant human interaction

  `(let ((*package*                    *acl2-package*)
         (*print-array*                t)
         (*print-base*                 10)
         (*print-case*                 :downcase)
         (*print-circle*               nil)
         (*print-escape*               t)
         (*print-gensym*               t)
         (*print-length*               nil)
         (*print-level*                nil)
         (*print-lines*                nil)
         (*pause-lines*                nil)
         (*print-pretty*               t)
         (*print-radix*                nil)
         (*print-readably*             nil)
         (*print-right-margin*         70)
         (*print-pprint-dispatch*      *print-pprint-dispatch-orig*)
         (*read-base*                  10)
         (*read-default-float-format*  'single-float)
         (*print-miser-width*          40)
         (*read-eval*                  nil)
         (*read-suppress*              nil)
         (*readtable*                  *acl2-readtable*))
     ,@args))

(defg *hons-verbose* t)

(defmacro ofd (&rest r) ; For warnings.
  `(format *debug-io* ,@r))

(defmacro ofv (&rest r) ; For verbose info.
    `(when *hons-verbose*
       (format *debug-io* ,@r)))

(defmacro ofg (&rest r) ; For verbose gc info.
    `(when *hons-verbose*
       (format *debug-io* ,@r)))

(defun ofe (&rest r)  ; Causes an error.
  (our-syntax-nice
   (apply #'format *error-output* r)
   (error "")))

(defmacro ofn (&rest r) ; For forming strings.
  `(format nil ,@r))

(defun ofnum (n)
  (check-type n number)
  (if (= n 0) (setq n 0))
  (cond ((typep n '(integer -99 999))
         (format nil "~d" n))
        ((or (< -1000 n -1/100)
             (< 1/100 n 1000))
         (format nil "~,2f" n))
        (t (format nil "~,1e" n))))

(defmacro ofni (&rest r) ; For forming symbols.
  `(our-syntax (intern (format nil ,@r) "ACL2")))

(defmacro ofnm (&rest r) ; For forming uninterned symbols.
  `(our-syntax (make-symbol (format nil ,@r))))

(defmacro oft (&rest r) ; To *standard-output*.
  `(format t ,@r))

(defmacro oftr (&rest r) ; Fyi.
  `(format *trace-output* ,@r))

(defun suffix (str sym)
  (check-type str string)
  (check-type sym symbol)
  (let ((spkn (package-name (symbol-package sym)))
        (sn (symbol-name sym)))
    (ofn "~s,~s,~s" str spkn sn)))


;  PHYSICAL MEMORY

#-openmcl
(defun physical-memory ()
  (cond ((probe-file "/proc/meminfo")
         (let* (n kb
                  (key "MemTotal:")
                  (info (with-open-file (si "/proc/meminfo")
                          (with-output-to-string
                            (so)
                            (let (c)
                              (loop while (setq c (read-char
                                                   si nil nil))
                                    do (write-char c so))))))
                  (loc (search key info)))
           (our-syntax
            (with-input-from-string
             (s info :start (+ loc (length key)))
             (setq n (read s))
             (setq kb (read s))
             (cond ((and (integerp n) (equal kb 'kb))
                    (* n 1024)))))))))

#+openmcl
(with-standard-io-syntax
 (let ((*package* (find-package "CCL")))
   (eval (read-from-string "

   ;;; Work of Gary Byers.

   ;;; The #_ and #$ reader macros in the code below are part of
   ;;; OpenMCL's ffi; you'd basically need to hide this code in
   ;;; a file that's isolated from other implementations.
   (defun acl2::physical-memory ()
      #+darwin-target
      (rlet ((count :mach_msg_type_number_t #$HOST_BASIC_INFO_COUNT)
             (info :host_basic_info))
        (if (eql 0 (#_host_info (#_mach_host_self)
                                #$HOST_BASIC_INFO
                                info
                                count))
          (pref info :host_basic_info.max_mem)))
      #+freebsd-target
       (rlet ((ret :unsigned-doubleword 0)
              (mib (:array :uint 2))
              (oldsize :uint (ccl::record-length
                               :unsigned-doubleword)))
         (setf (paref mib (:* :uint) 0) #$CTL_HW
               (paref mib (:* :uint) 1) #$HW_PHYSMEM)
         (if (eql 0 (#_sysctl mib 2 ret oldsize (%null-ptr) 0))
           (pref ret unsigned-doubleword)))
      #+linux-target
      (rlet ((info :sysinfo))
        (if (eql 0 (#_sysinfo info))
          (pref info :sysinfo.totalram))))"))))


; NUMBER OF ARGS AND RETURN VALUES

(defun number-of-args (fn)

; For simplicity and speed, we only memoize functions that take a
; fixed number of arguments and return a fixed number of values.
; NUMBER-OF-ARGS returns an integer if the number of args of FN is
; fixed, and otherwise NIL.

  (assert (fboundp fn) (fn))
  (let* ((state *the-live-state*) (w (w state)))
    (and (symbolp fn)
         (fboundp fn)
         (not (macro-function fn))
         (not (special-operator-p fn))
         (let ((x (getprop fn 'stobjs-in t 'current-acl2-world w)))
           (cond
            ((or (eq x t) (member 'state x))
             #+openmcl
             (multiple-value-bind (a b c)

; No idea whether or why this really works.  Ask Gary Byers.
; Hope this can be done right in other Lisps.

                 (ccl::min-max-actual-args (symbol-function fn) 0)
               (declare (ignore c))
               (and (integerp a) (eql a b) a))
             #-openmcl nil)
            (t (length x)))))))

(defun number-of-return-values (fn)
  (assert (fboundp fn) (fn))
  (let* ((state *the-live-state*) (w (w state)))
    (let ((x (getprop fn 'stobjs-out t 'current-acl2-world w)))
      (cond ((not (eq x t))
             (length x))
            (t (case fn
                 ((macroexpand
                   macroexpand-1
                   find-symbol
                   intern
                   array-displacement
                   get-properties
                   gethash
                   pprint-dispatch)
                  2)
                 ((decode-float integer-decode-float)
                  3)
                 (apropos 0)
                 ((and
                   apply
                   call-next-method
                   invoke-restart-interactively
                   step
                   time
                   funcall
                   values
                   eval)
                  'number-of-return-values-varies)
                 (otherwise 1)))))))


(defun event-number (fn)
  (check-type fn symbol)
  (fgetprop fn 'absolute-event-number t (w *the-live-state*)))


;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS;;;;;;;; HONS
;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS ;;;;;;;; HONS;;;;;;;; HONS


; HONS VARIABLES, MACROS, AND DATA STRUCTURES

; Gary Byers recalls Lisp folklore that alists are faster than hash
; tables up to length 18.

(defconstant *start-car-ht-size*            18)

(defconstant *hons-acons-ht-threshold*      18)

(defconstant *small-ht-size*                60)

(defconstant *hons-cdr-ht-size*             (expt 2 20))

(defconstant *nil-ht-size*                  (expt 2 17))

(declaim (fixnum *start-car-ht-size* *hons-acons-ht-threshold*
                 *small-ht-size* *hons-cdr-ht-size* *nil-ht-size*))

(defmacro mht (&key (test (quote (function eql)))
                    (size *small-ht-size*)
                    (shared t)
                    (rehash-size 1.5)
                    (rehash-threshold 0.7)
                    (weak nil))
  (declare (ignorable shared weak))
  `(make-hash-table :test             ,test
                    :size             (max *small-ht-size* ,size)
                    :rehash-size ,rehash-size
                    :rehash-threshold ,rehash-threshold
                    #+openmcl :weak
                    #+openmcl ,weak
                    #+openmcl :shared
                    #+openmcl ,shared
                    ))

; To minimize metering overhead costs, one may set these two
; *COUNT-HONS-CALLS* and *COUNT-PONS-CALLS* to NIL before building.

(defg *count-hons-calls*                   t
  "If *COUNT-HONS-CALLS*, then each call of HONS increments
  *HONS-CALL-COUNTER* by 1, and each call of HONS that does not find
  the desired HONS to already exist increments *HONS-MISSES-COUNTER*
  by 1.")

(defg *count-pons-calls*                   t
  "If *COUNT-PONS-CALLS*, then each call of PONS increments
  *PONS-CALL-COUNTER* by 1, and each call of PONS that does find the
  desired PONS to already exist increments *PONS-HIT-COUNTER* by 1.")

(defg *break-honsp*                        nil)

(defg *hons-report-discipline-failure*     'break)

; The choice of :weak below deserves careful explanation. !!

(defg *hons-cdr-ht*        (mht :test #'eq :weak :key))

(defg *hons-cdr-ht-eql*    (mht))

(defg *nil-ht*             (mht :weak :value))

(defg *hons-acons-ht*      (mht :test #'eq :weak :key))

(defg *hons-str-ht*        (mht :test #'equal :weak :value))

(defg *hons-copy-aux-ht*   (mht :test #'eq))

(defg *rehons-culprit* nil)

(defg *init-hash-tables-done* nil)

(declaim (hash-table
          *hons-cdr-ht*
          *hons-cdr-ht-eql*
          *nil-ht* *hons-acons-ht*
          *hons-str-ht*
          *compact-print-file-ht*
          *compact-read-file-ht*))

(declaim (fixnum *hons-call-counter* *hons-misses-counter*
                 *pons-call-counter* *pons-hit-counter*))

(defg *hons-call-counter* 0)

(defg *hons-misses-counter* 0)

(defmacro hons-let (form)
  (let ((old-cdr-ht     (gensym "OLD-CDR-HT"))
        (old-cdr-ht-eql (gensym "OLD-CDR-HT-EQL"))
        (old-nil-ht     (gensym "OLD-NIL-HT")))
    `(our-lock-unlock-honsmv1
; This lock is crippling to parallel use of hons-let, but I
; don't yet see any alternative.
      (let ((,old-cdr-ht *hons-cdr-ht*)
            (,old-cdr-ht-eql *hons-cdr-ht-eql*)
            (,old-nil-ht *nil-ht*))
        (unwind-protect
            (progn (clear-hash-tables) ,form)
          (setq *hons-cdr-ht* ,old-cdr-ht)
          (setq *hons-cdr-ht-eql* ,old-cdr-ht-eql)
          (setq *nil-ht* ,old-nil-ht))))))

; Except at an init-hash-tables, a clear-hash-tables, or a
; maybe-shrink, our hash tables grow, so have a kind of monotonic
; truth.  If foo is the str-hash of bar now, then it still
; will be in a few minutes even if new strings are added.

; Definition. ***** means 'Do not call this function unless you are
; sure that a caller above locked *hons-cdr-ht*.'

(defun maybe-str-hash (x)

;  *****

  (cond ((typep x '(and array string))
         (cond ((gethash x *hons-str-ht*))
               (t (setf (gethash x *hons-str-ht*) x))))
        (t x)))

(defmacro maybe-break-honsp ()
  (cond (*break-honsp*
         `(break "~&; HONSP returned nil.")
         )))

(defmacro honsp (x)

; HONSP checks a cons see if it's in our hons tables.
; HONSP assumes x is a CONSP.

; If something is a honsp, then it stays a honsp.

; If something is not a honsp, then the only way it can become a honsp
; is via a copy-consume or hons-normed-with-suggestion, which is
; restricted to clear-hash-tables, or a file reader function.  In both
; cases, it is assumed that no one else is holding on to an opinion
; about any of those conses that they are not honses.

  `(cond ((let* ((x ,x)
                 (ax (car x))
                 (dx (cdr x))
                 (v (cond ((null dx) *nil-ht*)
                          ((consp dx) (gethash dx *hons-cdr-ht*))
                          (t (gethash dx *hons-cdr-ht-eql*)))))
            (if (listp v)
                (let ((av (car v)))
                  (if (typep ax '(or cons symbol (and array string)))
                      (loop (if (eq ax (car av)) (return (eq x av)))
                            (setq v (cdr v))
                            (if (null v) (return nil))
                            (setq av (car v)))
                    (loop (if (eql ax (car av)) (return (eq x av)))
                          (setq v (cdr v))
                          (if (null v) (return nil))
                          (setq av (car v)))))
              (eq x (gethash (car x) v)))))
         (t (maybe-break-honsp) nil)))

; We do not assure any sort of mutual excusion in the gathering
; of statistics, which makes them largely bogus. 

(defmacro maybe-count-hons-calls ()
  (and *count-hons-calls*
       '(safe-incf *hons-call-counter* 1 maybe-count-hons-calls)))

(defmacro maybe-count-hons-misses ()
  (and *count-hons-calls*
       '(safe-incf *hons-misses-counter* 1 maybe-count-hons-misses)))

(defmacro maybe-report-discipline-failure (fn args)
  (cond
   (*hons-report-discipline-failure*
    `(cond
      ((eq *hons-report-discipline-failure* 't)
       (ofd "~&; Warning: ~s discipline failure on args:~% ~s~%"
            ',fn ,args)
       nil)
      ((eq *hons-report-discipline-failure* 'break)
       (break "~&; Break: ~s discipline failure on args:~% ~s~%"
              ',fn ,args)
       nil)))))


; HONS INVARIANTS

; If A and B are consp+honsp, then (eq A B) iff (equal A B).  The car
; of a consp+honsp is an atom or a consp+honsp.  The cdr of a
; consp+honsp is an atom or a consp+honsp.  No consp+honsp is
; circular.  If a string occurs in any consp+honsp, then no other
; EQUAL string occurs in any consp+honsp.

; Here are some basic data structures for honsing and memoizing.  Some
; of these are significantly expanded in size later by hons-init, but
; there is no reason to clog up saved images with large empty versions
; of them.



; HONS FUNCTIONS

(defun assoc-no-error-at-end (x l)

; We assume that every element of L is CONSP.

  (if (typep x '(or cons symbol (and array string)))
      (loop (if (consp l)
                (let ((al (car l)))
                  (if (eq x (car al))
                      (return al)
                    (setq l (cdr l))))
              (return nil)))
    (loop (if (consp l)
              (let ((al (car l)))
                (if (eql x (car al))
                    (return al)
                  (setq l (cdr l))))
            (return nil)))))

(defun too-long (x n)
  (declare (fixnum n))
  (loop (cond ((atom x) (return nil))
              ((eql n 0) (return t))
              (t (setq x (cdr x))
                 (setq n (the fixnum (1- n)))))))

; Definition of NORMED.  An ACL2 Common Lisp object x is normed iff
; both (a) if x is a string, then it is in *HONS-STR-HT* and (b) if x
; is a CONSP, then it is also a HONSP.  HONS-COPY and HONS produce
; normed objects.  A non-HONSP cons whose car and cdr are normed is
; honsed up by putting it in the right place in the hash tables, after
; first checking that it can be legitimately placed there.

(defun hons-normed (x y)

;  *****

  (let* ((yt (if (consp y) *hons-cdr-ht* *hons-cdr-ht-eql*))
         (yval (if y (gethash y yt) *nil-ht*))
         (yp (listp yval)))
    (maybe-count-hons-calls)
    (cond ((if yp (assoc-no-error-at-end x yval) (gethash x yval)))
          (t (maybe-count-hons-misses)
             (let ((nhons (cons x y)))
               (cond
                (yp
                 (cond ((too-long yval *start-car-ht-size*)
                        (let ((tab (mht :weak :value)))
                          (loop for pair in yval do
                                (setf (gethash (car pair) tab)
                                      pair))
                          (setf (gethash (car nhons) tab) nhons)
                          (setf (gethash y yt) tab)
                          nhons))
                       (t (setf (gethash y yt) (cons nhons yval))
                          nhons)))
                (t (setf (gethash x yval) nhons))))))))

(defun hons-normed-top (x y)
  (our-lock-unlock-hons1 (hons-normed x y)))
   
; HONS-COPY

; In general, HONS-COPY has no justification for reusing, much less
; smashing, the conses it is passed.  However, during
; CLEAR-HASH-TABLES, in the rehons phase, such reuse is precisely what
; is needed and permitted.  If some day we learn that no cons in, say,
; (w state) will ever be RPLACAed or RPLACDed and that REPLACAing any
; cons in it with an EQUAL CAR value is ok, and same for RPLACD/CDR,
; then we could probably legitimately absorb the conses in (w state)
; as honses via HONS-COPY1-CONSUME.  HONS-COPY is partially and
; temporarily self-memoizing.

(defn uncopy (x)

; *****

  (unless (atom x)
    (when (remhash x *hons-copy-aux-ht*)
      (uncopy (car x))
      (uncopy (cdr x)))))       

(defun hons-normed-with-suggestion (x y nhons)

; *****

  (let* ((yt (if (consp y) *hons-cdr-ht* *hons-cdr-ht-eql*))
         (yval (if y (gethash y yt) *nil-ht*))
         (yp (listp yval)))
    (maybe-count-hons-calls)
    (cond ((if yp (assoc-no-error-at-end x yval) (gethash x yval)))
          (t (maybe-count-hons-misses)
             (cond (yp
                    (cond ((too-long yval *start-car-ht-size*)
                           (let ((tab (mht :weak :value)))
                             (loop for pair in yval do
                                   (setf (gethash (car pair) tab)
                                         pair))
                             (setf (gethash (car nhons) tab) nhons)
                             (setf (gethash y yt) tab)
                             nhons))
                          (t (setf (gethash y yt) (cons nhons yval))
                             nhons)))
                   (t (setf (gethash x yval) nhons)))))))

(defabbrev hons-copy2-consume (x)

; *****

  (let ((a (hons-copy3-consume (car x)))
        (d (hons-copy3-consume (cdr x))))
    (or (eql a (car x)) (rplaca x a))
    (or (eql d (cdr x)) (rplacd x d))
    (hons-normed-with-suggestion a d x)))

(defn hons-copy3-consume (x)

; *****

  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        ((gethash x *hons-copy-aux-ht*))
        (t (setf (gethash x *hons-copy-aux-ht*)
                 (hons-copy2-consume x)))))

(defn hons-copy1-consume (x)

; *****

    (cond ((atom x) (maybe-str-hash x))
          ((honsp x) x)
          (t (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
               ;; Only get here because of an error during a HONS-COPY.
               (setq *hons-copy-aux-ht* (mht :test #'eq)))
             (let ((ans (hons-copy2-consume x)))
               (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
                 (uncopy (car ans))
                 (uncopy (cdr ans)))
               ans))))

(defn hons-copy1-consume-top (x)

; This function should only be called when we are sure that no other
; threads are running, or when we are sure that no other threads have
; ever had access to the conses we are consuming.

  (our-lock-unlock-hons1 (hons-copy1-consume x)))
    
(defabbrev hons-copy2 (x)

; *****

  (let ((a (hons-copy3 (car x)))
        (d (hons-copy3 (cdr x))))
    (hons-normed a d)))

(defn hons-copy3 (x)

; *****

  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        ((gethash x *hons-copy-aux-ht*))
        (t (setf (gethash x *hons-copy-aux-ht*)
                 (hons-copy2 x)))))

(defn hons-copy1 (x)

; *****

  (cond ((atom x) (maybe-str-hash x))
        ((honsp x) x)
        (t (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
             ;; Only get here because of an error
             ;; during a hons-copy.
             (setq *hons-copy-aux-ht* (mht :test #'eq)))
           (let ((ans (hons-copy2 x)))
             (unless (eql 0 (hash-table-count *hons-copy-aux-ht*))
               (uncopy (car ans))
               (uncopy (cdr ans)))
             ans))))

(defn hons-copy1-top (x)
  (our-lock-unlock-hons1 (hons-copy1 x)))
  
(defun hons-copy (x)
  (hons-copy1-top x))

(defun hons-when-x-is-honsp (x y)
  (our-lock-unlock-hons1 (hons-normed x (hons-copy1 y))))

(defun hons-when-y-is-honsp (x y)
  (our-lock-unlock-hons1 (hons-normed (hons-copy1 x) y)))

(defun hons-equal (x y)

; Cf. hons.lisp.
  
  (cond ((eq x y))
        ((atom x)
         (cond ((consp y) nil)
               ((symbolp y) nil)
               ((typep y 'fixnum) (eql x y))
               (t (equal x y))))
        ((atom y) nil)
        ((honsp x)
         (cond ((honsp y) nil)
               (t (and (hons-equal-h1 (car x) (car y))
                       (hons-equal-h1 (cdr x) (cdr y))))))
        ((honsp y)
         (and (hons-equal-h1 (car y) (car x))
              (hons-equal-h1 (cdr y) (cdr x))))
        (t (and (hons-equal (car y) (car x))
                (hons-equal (cdr y) (cdr x))))))

; HONS-EQUAL-H1 is like HONS-EQUAL, but with the assumption that X has
; been normed.
(defun hons-equal-h1 (x y)
  (cond ((eq x y))
        ((atom x)
         (cond ((consp y) nil)
               ((symbolp y) nil)
               ((typep y 'fixnum) (eql x y))
               (t (equal x y))))
        ((atom y) nil)
        ((honsp y) nil)
        (t (and (hons-equal-h1 (car x) (car y))
                (hons-equal-h1 (cdr x) (cdr y))))))


; HONS

(defun hons (x y)

;  See also hons.lisp.
    (our-lock-unlock-hons1 (hons-normed (hons-copy1 x) (hons-copy1 y))))

; HONS-GET, HONS-ACONS and HONS-ACONS!

; HONS-ACONS and HONS-GET provide fast lookup in alists, with
; ASSOC-EQUAL semantics but with the speed of hash tables in some
; cases.  These operations permit one to reasonably efficiently work
; with extremely long alists in some cases.  Informally speaking, each
; HONS-ACONS operation steals the hash table associated with the alist
; that is being extended.  The key is always honsed before the hashing
; operation in HONS-ACONS, and HONS-ACONS!.  In order to take
; advantage of the hash table lookup speed, when coding one must be
; very conscious of which object is the most recent extension of an
; alist and use that extension exclusively.  This may require careful
; passing of the alist up and down through function calls, as with any
; single threaded object in an applicative setting.  There is no
; syntactic enforcement to force one to only use the most recent
; extension of an alist, as there is for single threaded objects.  The
; only penalty for a failure to keep track of the most recent
; extension is a loss of execution speed, not of correctness.  And
; perhaps the annoyance of some warning messages about 'discipline'.

; If we limit ourselves to alists that are recognized by ALISTP, a
; possible gotcha when using HONS-ACONS! is that it might "steal" a
; hash table without one's expecting it.  For example, if you start
; two alists with (HONS-ACONS! '1 '2 nil) and (HONS-ACONS! '1 '2 nil),
; then adding something to the first with HONS-ACONS! will "steal" the
; table associated with what you thought was the second, with the
; result that adding things to the second will result in slow access.
; One can get around this annoyance to some extent by putting
; something 'unique' at the end of the alist, e.g., the final cdr.
; Our (partial) fix for this is to permit the final NIL of an
; HONS-GET/HONS-ACONS! association list to be any symbol, effectively
; naming each association list and maybe preventing the "gotcha" just
; mentioned.

(defun-one-output hons-get-fn-do-not-hopy (key l)
  (when (atom l) (return-from hons-get-fn-do-not-hopy nil))
  (let (h)
    (when (or (and (consp key) (not (honsp key)))
              (null (setq h (gethash l *hons-acons-ht*))))
      (return-from hons-get-fn-do-not-hopy (hons-assoc-equal key l)))
    (let ((key (hons-copy1-top key)))
      (loop
       (cond ((typep h 'fixnum)
              (return (assoc-no-error-at-end key l)))
             (h (return (values (gethash key h))))
             (t (cond ((and (consp (car l))
                            (hons-equal-h1 key (caar l)))
                       (return (car l)))
                      (t (setq l (cdr l))
                         (when (atom l) (return nil))
                         (setq h (gethash (cdr l)
                                          *hons-acons-ht*))))))))))

(defun-one-output hons-get-fn-do-hopy (key l)
  (when (atom l) (return-from hons-get-fn-do-hopy nil))
  (let ((h (gethash l *hons-acons-ht*)))
    (when (null h)
      (maybe-report-discipline-failure
       'hons-get-fn-do-hopy (list key l)))
    (let ((key (hons-copy1-top key)))
      (loop
       (cond ((typep h 'fixnum)
              (return (assoc-no-error-at-end key l)))
             (h (return (values (gethash key h))))
             (t (cond ((and (consp (car l))
                            (hons-equal-h1 key (caar l)))
                       (return (car l)))
                      (t (setq l (cdr l))
                         (when (atom l) (return nil))
                         (setq h (gethash (cdr l)
                                          *hons-acons-ht*))))))))))

; Why do we want both HONS-ACONS and HONS-ACONS!, which is the HONSing
; version of HONS-ACONS?  On the one hand, since it is quite possible
; that one will not want to look further into the alist that is
; formed, given fast hashing lookup, one may not wish the overhead of
; HONSing it.  On the other hand, if the alist is going to be an
; argument to a function that is to be memoized, then the memoization
; process may hons it -- possibly over and over and over, which can be
; very time consuming if it is very long.

(defun hons-acons (key value l)
  (setq key (hons-copy1-top key))
  (our-lock-unlock-hons1
      (let ((ans (cons (cons key value) l)))
        (cond ((atom l)
               (setf (gethash ans *hons-acons-ht*) 0))
              (t (let ((tab (gethash l *hons-acons-ht*)))
                   (remhash l *hons-acons-ht*)
                   (cond
                    ((typep tab 'fixnum)
                     (cond ((< (the fixnum tab) *hons-acons-ht-threshold*)
                            (setf (gethash ans *hons-acons-ht*)
                                  (the fixnum (+ 1 tab))))
                           (t (let ((tab (mht :test #'eq :weak :key)))
; if you change this, see also fast-alist-len.
                                (loop for tail on ans
                                      unless (gethash (caar tail) tab)
                                      do (setf (gethash (caar tail) tab)
                                               (car tail)))
                                (setf (gethash ans *hons-acons-ht*)
                                      tab)))))
                    (tab
                     (setf (gethash key tab) (car ans))
                     (setf (gethash ans *hons-acons-ht*) tab))
                    (t (maybe-report-discipline-failure
                        'hons-acons (list key value l)))))))
        ans)))

(defun hons-acons! (key value l)
  (setq key (hons-copy1-top key))
  (our-lock-unlock-hons1
      (let ((ans (hons-when-x-is-honsp
                  (hons-when-x-is-honsp key value)
                  l)))
        (cond ((atom l)
               (setf (gethash ans *hons-acons-ht*) 0))
              (t (let ((tab (gethash l *hons-acons-ht*)))
                   (remhash l *hons-acons-ht*)
                   (cond
                    ((typep tab 'fixnum)
                     (cond ((< (the fixnum tab) *hons-acons-ht-threshold*)
                            (setf (gethash ans *hons-acons-ht*)
                                  (the fixnum (+ 1 tab))))
                           (t (let ((tab (mht :test #'eq :weak :key)))
                                (loop for tail on ans
                                      unless (gethash (caar tail) tab)
                                      do (setf (gethash (caar tail) tab)
                                               (car tail)))
                                (setf (gethash ans *hons-acons-ht*)
                                      tab)))))
                    (tab
                     (setf (gethash key tab) (car ans))
                     (setf (gethash ans *hons-acons-ht*) tab))
                    (t (maybe-report-discipline-failure
                        'hons-acons! (list key value l)))))))
        ans)))

(defn fast-alist-len (al)
  (our-lock-unlock-hons1 
   (cond ((atom al) 0)
            (t (let ((h (gethash al *hons-acons-ht*)))
                 (cond ((typep h 'fixnum)
                        ;; worry:  hons-acons vs. hons-acons!
                        (let ((tab (mht :test #'eq :weak :key)))
                          (loop for tail on al
                                unless (gethash (caar tail) tab)
                                do (setf (gethash (caar tail) tab)
                                         (car tail)))
                          (setf (gethash al *hons-acons-ht*) tab)
                          (hash-table-count tab)))
                       (h (hash-table-count h))
                       (t (maybe-report-discipline-failure
                           'fast-alist-len al)
                          (fast-alist-len-acc al nil))))))))

(defun hons-shrink-alist-orig (alcdr ans)
  (cond ((atom alcdr) ans)
        (t (let ((p (hons-get (car (car alcdr)) ans)))
             (cond (p (hons-shrink-alist-orig (cdr alcdr) ans))
                   (t (hons-shrink-alist-orig
                       (cdr alcdr)
                       (hons-acons (car (car alcdr))
                                   (cdr (car alcdr))
                                   ans))))))))

(defun hons-shrink-alist-help (alcdr ans tab)

;   *****

  (if (atom alcdr)
      (progn (setf (gethash ans *hons-acons-ht*) tab)
             ans)
    (let* ((pair (car alcdr))
           ;; (key (hons-copy (car pair)))
           (key (car pair)) ; We know (car pair) is HONS-NORMEDP
           (val (gethash key tab))
           (ans (if val ans (cons (cons key (cdr pair)) ans)))
           (tab (if val tab
                  (progn
                    (setf (gethash key tab) (car ans))
                    tab))))
      (hons-shrink-alist-help (cdr alcdr) ans tab))))

(defun hons-shrink-alist!-orig (alcdr ans)
  (cond ((atom alcdr) ans)
        (t (let ((p (hons-get (car (car alcdr)) ans)))
             (cond (p (hons-shrink-alist!-orig (cdr alcdr) ans))
                   (t (hons-shrink-alist!-orig
                       (cdr alcdr)
                       (hons-acons! (car (car alcdr))
                                    (cdr (car alcdr))
                                    ans))))))))

(defun hons-shrink-alist!-help (alcdr ans tab)

;   *****

  (if (atom alcdr)
      (progn (setf (gethash ans *hons-acons-ht*) tab)
             ans)
    (let* ((pair (car alcdr))
           ;; (key (hons-copy (car pair)))
           (key (car pair))
           (val (gethash key tab))
           (ans (if val ans (hons-normed
                             (hons-when-x-is-honsp key (cdr pair))
                             ans)))
           (tab (if val tab
                  (progn
                    (setf (gethash key tab) (car ans))
                    tab))))
      (hons-shrink-alist!-help (cdr alcdr) ans tab))))

(defun hons-shrink-alist!-help-honsp-alcdr (alcdr ans tab)

;   *****

  (if (atom alcdr)
      (progn (setf (gethash ans *hons-acons-ht*) tab)
             ans)
    (let* ((pair (car alcdr))
           ;; (key (hons-copy (car pair)))
           (key (car pair))
           (val (gethash key tab))
           (ans (if val ans (hons-normed pair ans)))
           (tab (if val tab
                  (progn
                    (setf (gethash key tab) (car ans))
                    tab))))
      (hons-shrink-alist!-help-honsp-alcdr (cdr alcdr) ans tab))))

(defun hons-shrink-alist (alcdr ans)

; fixed by Sol Swords

  (our-lock-unlock-hons1
      (if (atom alcdr)
          ans

; Question:  Why do we call (maybe-str-hash ans) in
; HONS-SHRINK-ALIST! but not in HONS-SHRINK-ALIST?

        (let* ((tab (gethash alcdr *hons-acons-ht*))
               (ans-size
                (if (and tab (not (integerp tab)))
                    #-openmcl (1+ (ceiling
                                   (hash-table-count tab)
                                   .7))
                    #+openmcl (1+ (hash-table-count tab))
                    nil)))
          (if (or (not ans-size) (consp ans))
              (hons-shrink-alist-orig alcdr ans)
            (let ((ans-tab (mht :test #'eq :size ans-size :weak :key)))
              (hons-shrink-alist-help alcdr ans ans-tab)))))))

(defun hons-shrink-alist! (alcdr ans)
  (our-lock-unlock-hons1
   (if (atom alcdr)
       ans
     (let* ((ans (maybe-str-hash ans))
            (tab (gethash alcdr *hons-acons-ht*))
            (ans-size
             (if (and tab (not (integerp tab)))
                 #-openmcl (1+ (ceiling
                                (hash-table-count tab)
                                .7))
                 #+openmcl (1+ (hash-table-count tab))
                 nil)))
       (if (or (not ans-size) (consp ans))
           (hons-shrink-alist!-orig alcdr ans)
         (let ((ans-tab (mht :test #'eq :size ans-size :weak :key)))
           (if (honsp alcdr)
               (hons-shrink-alist!-help-honsp-alcdr alcdr ans ans-tab)
             (hons-shrink-alist!-help alcdr ans ans-tab))))))))

; GC HACKS FOR HONS

#+openmcl
; *GC-MSG-FOR-OUR-GC* is set to T after a gc in OpenMCL.
(defg *gc-msg-for-our-gc* nil)
(defun our-gc ()
  #+openmcl
  (progn (setq *gc-msg-for-our-gc* nil)
         (ccl::gc)
         (loop (sleep .01)
               (when *gc-msg-for-our-gc*
                 (setq *gc-msg-for-our-gc* nil)
                 (return nil))))
  #+GCL
  (si::gbc t))

(defun maybe-shrink-some-hash-tables ()
  (check-no-extra-processes)
  (our-lock-unlock-hons1
   (labels ((needs-shrinking
            (h)
            (let ((size (hash-table-size h)))
              (declare (fixnum size))
              (and (> size (cond ((eq h *hons-cdr-ht*) *hons-cdr-ht-size*)
                                 ((eq h *nil-ht*) *nil-ht-size*)
                                 (t *small-ht-size*)))
                   (< (* 3 (hash-table-count h)) size))))
           (maybe-shrink-sub
            (h)
            (cond ((consp h)
                   (let ((nh (mht :size (length h) :weak :value)))
                     (loop for x in h do (setf (gethash (car x) nh) x))
                     nh))
                  ((needs-shrinking h)
                   (let ((nh (mht :size (* 3 (hash-table-count h))
                                  :weak :value)))
                     (maphash (lambda (k v) (setf (gethash k nh) v)) h)
                     nh))
                  (t h)))

; We sometimes replace lists with subsidiary hash tables even though
; short lists are faster to search.  By converting to hash tables, we
; permit the garbage collection of honses that are referenced by no
; one else, thanks to idea of 'weak' in OpenMCL.  We sometimes convert
; subsidiary hash tables back to lists when their counts are low
; enough because short lists are faster to search.

; Possible improvement: In maybe-shrink-..., it might be a better idea
; to move all the honses on lists to a single weak value hash table,
; instead of to many small hash tables, then, do the gc, and then
; rehash all those values (back to lists).  One downside to putting
; all those honses into a hash table is that they would all have to be
; rehashed to be placed back into the correct sub lists; if they are
; kept in separate small hashtables, one merely needs to maphash and
; create a list.

           (maybe-shrink-main
            (h)
            (maphash (lambda (k v)
                       (cond ((and (not (listp v))
                                   (eql 0 (hash-table-count v)))
                              (remhash k h))
                             (t (let ((nv (maybe-shrink-sub v)))
                                  (or (eq nv v) (setf (gethash k h) nv))))))
                     h)
            (cond ((needs-shrinking h)
                   (let ((nh (mht :test (hash-table-test h)
                                  :size (* 3 (hash-table-count h))
                                  #+openmcl :weak
                                  #+openmcl (ccl::hash-table-weak-p h))))
                     (maphash (lambda (k v) (setf (gethash k nh) v)) h)
                     nh))
                  (t h)))
           (ht-list
            (h)
            (maphash
             (lambda (k v)
               (if (not (listp v))
                   (let ((c (hash-table-count v)))
                     (declare (fixnum c))
                     (cond ((eql 0 c) (remhash k h))
                           ((< c *start-car-ht-size*)
                            (setf (gethash k h)
                                  (let (l)
                                    (maphash (lambda (k v) (declare (ignore k))
                                               (push v l))
                                             v)
                                    l)))))))
             h)))
    (setq *hons-cdr-ht*     (maybe-shrink-main *hons-cdr-ht*))
    (setq *hons-cdr-ht-eql* (maybe-shrink-main *hons-cdr-ht-eql*))
    (setq *nil-ht*     (maybe-shrink-sub  *nil-ht*))
    (our-gc)
    (ht-list *hons-cdr-ht*)
    (ht-list *hons-cdr-ht-eql*)
    (our-gc))))

; HONSP-CHECK

(defun honsp-check (x)
  (cond
   ((consp x)
    #+hons
    (when (not (honsp x))
      (er hard 'honsp-check
          "The value ~X01 is a consp but, contrary to expectation, ~
           not a honsp."
          x
          (list nil 3 4 nil)))
    t)
   ((typep x '(and array string))
    #+hons
    (when (not (gethash x *hons-str-ht*))
      (er hard 'honsp-check
          "The value ~X01 is a string or bignum but, contrary to ~
           expectation, not a string or bignum hashed for the HONS ~
           implementation."
          x
          (list nil 3 4 nil)))
    t)
   (t nil)))




;;;;;;;;;; MEMOIZE ;;;;;;;;;; MEMOIZE ;;;;;;;;;; MEMOIZE ;;;;;;;;;;
;;;;;;;;;; MEMOIZE ;;;;;;;;;; MEMOIZE ;;;;;;;;;; MEMOIZE ;;;;;;;;;;

;  MEMOIZE VARIABLES, MACROS, AND DATA STRUCTURES

(defparameter *never-profile-list*
  '(#+rdtsc ccl::rdtsc
            1+ + * - <= < > >= =
            aref arrayp atom apply
            car cdr clear-hash-tables cons consp
            eq eql error eval
            fixnum-to-symbol format funcall
            get-internal-real-time
            get-internal-run-time
            gethash
            list
            make-hash-table
            max memoize-fn watch-array-grow
            memoize-eval-compile
            mf-1st-warnings
            mf-2nd-warnings
            not null
            prin1 princ print print-object
            read
            stringp svref symbolp symbol-to-fixnum-create
            symbol-to-fixnum
            typep
            len length len1
            true-listp
            write-char write-byte write)
  
  "The members of *NEVER-PROFILE-LIST* are used in the run-time
  memoization machinery, so profiling or otherwise memoizing one of
  them can result in infinite recursion.")

(defg *memoize-debug* nil
  "If *MEMOIZE-DEBUG*, MEMOIZE-FN prints the new defun.")

;  recording vars

; To minimize metering overhead costs, one may set these "*RECORD-"
; variables to NIL before memoizing.

(defparameter *record-bytes*
  #+openmcl
  (> most-positive-fixnum (expt 2 32))
  #-openmcl
  nil
  "If *RECORD-BYTES* when a function is memoized, we keep track of
  heap bytes allocated during calls of that function.")

(defparameter *record-calls* t
  "If *RECORD-CALLS* when a function is memoized,
  we count all calls of the function.")

(defparameter *record-hits* t
  "If *RECORD-HITS* when a function is memoized, we count
  the number of times that a previously computed answer
  is used again.")

(defparameter *record-hons-calls* t
  "If *RECORD-HONS-CALLS* when a function is memoized,
  hons calls are counted.")

(defparameter *record-mht-calls* t
  "If *REPORT-HONS-CALLS*, then MEMOIZE-SUMMARY prints the number of
  times that a memo hash-table for the function was created.  This may
  be of interest to those who memoize functions that deal in changing
  stobjs; the memoization machinery sometimes 'forgets' an entire
  memoization hash table out of an abundance of caution, and then may
  later need to create it afresh.")

(defparameter *record-pons-calls* t
  "If *RECORD-PONS-CALLS* when a function is memoized,
  pons calls are counted.")

(defparameter *record-time* t
  "If *RECORD-TIME* when a function is memoized, we
  record the elapsed time for each outermost call of the function.")


;  reporting vars

(defg *report-bytes* t
  "If *REPORT-BYTES*, then MEMOIZE-SUMMARY prints the
  number of bytes allocated on the heap.")

(defg *report-calls* t
  "If *REPORT-CALLS*, MEMOIZE-SUMMARY prints the number of calls.")

(defg *report-calls-from* t
  "If *REPORT-CALLS-FROM*, MEMOIZE-SUMMARY prints which functions
  called a function, how many times, and and how long the calls took.")

(defg *report-hits* t
  "If *REPORT-HITS*, MEMOIZE-SUMMARY prints the number of times
  that a previously computed answer was reused.")

(defg *report-hons-calls* t
  "If *REPORT-HONS-CALLS*, then MEMOIZE-SUMMARY prints the number of
  times that hons was called.")

(defg *report-mht-calls* t
  "If *REPORT-MHT-CALLS*, then MEMOIZE-SUMMARY prints the number of
  times that a memo hash-table for the function was created.  This may
  be of interest to those who memoize functions that deal in changing
  stobjs; the memoization machinery sometimes 'forgets' an entire
  memoization hash table out of an abundance of caution, and then may
  later need to create it afresh.")

(defg *report-pons-calls* t
  "If *REPORT-PONS-CALLS*, MEMOIZE-SUMMARY prints the number of calls
  of PONS.")

(defg *report-time* t
  "If *REPORT-TIME*, MEMOIZE-SUMMARY prints the total time used to
   compute the outermost calls.")

(defg *report-on-hons-and-pons-tables* t
  "If *REPORT-ON-HONS-AND-PONS-TABLES*, MEMOIZE-SUMMARY prints
  information about memo tables and pons tables.")

; counters

(defg *pons-call-counter* 0)

(defg *pons-hit-counter* 0)

(defmacro maybe-count-pons-calls ()
  (and *count-pons-calls*
       '(safe-incf *pons-call-counter* 1 maybe-count-pons-calls)))

(defmacro maybe-count-pons-hits ()
  (and *count-pons-calls*
       '(safe-incf *pons-hit-counter* 1 maybe-count-pons-hits)))

; array and hash-tables

(defg *memoize-info-ht* (mht))

(defg *watch-array*
  (make-array 1 :element-type 'fixnum :initial-element 0)
  
  "*WATCH-ARRAY*, 'ma' for short, is used for storage of the
  monitoring information for memoized functions.  ma has as its length
  4 times the square of the maximum number of memoized functions.

  ma is initialized in MEMOIZE-INIT.  Think of ma as a two dimensional
  array with dimensions (twice the max number of memoized functions) x
  (twice the max number of memoized functions).  Each 'column'
  corresponds to info about a memoized function, but the first five
  columns are 'special'.  We count rows and columns starting at 0.
  Column 0 is used as scratch space by COMPUTE-CALLS-AND-TIMES for
  sums across all functions.  Columns 1, 2, and 3 are not currently
  used at all.  Column 4 is for the anonymous 'outside caller'.
  Column 5 is for the first memoized function.  In columns 5 and
  greater, row 0 is used to count 'bytes', 1 'hits', 2 MHT calls, 3
  HONS calls, and 4 PONS calls.

  The elements of an ma column starting at row 10 are for counting and
  timing info.  Suppose column 7 corresponds to the memoized function
  FOO and column 12 corresponds to the memoized function BAR.
  Whenever FOO calls BAR, element 2*12 of column 7 will be incremented
  by 1, and the total elapsed time for the call will be added to
  element 2*12+1 of column 7.

  Though ma may 'grow', it may not grow while any memoized function is
  running, and here is why: every memoized function has a cached
  opinion about the size of ma.  To avoid an abort during a call of
  MEMOIZE one may call (MEMOIZE-HERE-COME n) to assure that ma has
  room for at least n more memoized functions.")

(defg *compute-array* (make-array 0)
  
  "*COMPUTE-ARRAY*, ca for short, is an array of proper lists.  At the
  end of a call of COMPUTE-CALLS-AND-TIMES, which is called by
  MEMOIZE-SUMMARY, (aref ca n) will contain the numbers of the
  functions that have called the function numbered n.")

(declaim (type (simple-array t (*)) *compute-array*))

#+cmu
(declaim (type (simple-array fixnum (*)) *watch-array*))
#-cmu
(eval-when
 #-cltl2
 (load eval)
 #+cltl2
 (:load-toplevel :execute)
 (proclaim `(type (simple-array fixnum (*)) *watch-array*)))

(defg *initial-max-memoize-fns* 50)

(defg *2max-memoize-fns* (* 2 *initial-max-memoize-fns*))

(defconstant *ma-bytes-index*       0)

(defconstant *ma-hits-index*        1)

(defconstant *ma-mht-index*         2)

(defconstant *ma-hons-index*        3)

(defconstant *ma-pons-index*        4)

(defconstant *ma-initial-max-symbol-to-fixnum* 4)

(defg *max-symbol-to-fixnum* *ma-initial-max-symbol-to-fixnum*)

(declaim (fixnum *max-symbol-to-fixnum*
                 *initial-2max-memoize-fns*
                 *ma-initial-max-symbol-to-fixnum*
                 *2max-memoize-fns*))

; for debugging

(defg *memoize-hack-condition* nil)

(defg *memoize-hack-inline* nil)

(defg *memoize-hack-trace* nil)


; for initialization

(defg *memoize-init-done* nil)


; locals used in memoize-on and memoize-off

(defg *mo-f* (make-symbol "F"))

(defg *mo-h* (make-symbol "H"))

(defg *mo-o* (make-symbol "O"))


; locals used in functions generated by memoize-fn

(defg *mf-start-hons* (make-symbol "START-HONS"))

(defg *mf-start-pons* (make-symbol "START-PONS"))

(defg *mf-start-bytes* (make-symbol "START-BYTES"))

(defg *mf-ans* (make-symbol "ANS"))

(defg *mf-ans-p* (make-symbol "ANS-P"))

(defg *mf-ma* (make-symbol "MA"))

(defg *mf-old-caller* (make-symbol "OLD-CALLER"))

(defg *mf-args* (make-symbol "ARGS"))

(defg *mf-2mmf* (make-symbol "MF-2MMF"))

(defg *mf-2mmf-fnn* (make-symbol "MF-2MMF-FNN"))

(defg *mf-count-loc* (make-symbol "MF-COUNT-LOC"))

(defg *mf-cl-error-msg*
  "~%; Memoizing a function in the COMMON-LISP package is a ~
   violation of ~%; the rules of Common Lisp, and consequently ~
   this ACL2 session is ~%; unsound.  It is ok to do such ~
   memoizing for experimental reasons, ~%; just as one might trace ~
   such a function, which, strictly speaking, ~%; is also ~
   forbidden.")

(defparameter *caller* (* *ma-initial-max-symbol-to-fixnum*
                          *2max-memoize-fns*))
; *CALLER* is bound by each memoized function, so we use DEFPARAMETER
; rather than DEFG.

(declaim (fixnum *caller*))

; The :CONDITION parameter of MEMOIZE-FN can either be T, or a
; function symbol defined by the user within the ACL2 loop, or a LISTP
; (CONSP or NIL).  In the last case we think of the condition as an
; expression in the formals of FN.  If the :INLINE parameter T, then
; the body of FN is placed inline in the memoized definition;
; otherwise, a funcall of the original function is placed there.

(defun memoizeable-function-p (fn)
  (and (symbolp fn)
       (fboundp fn)
       (not (macro-function fn))
       (not (special-form-or-op-p fn))))

; memoize-flush 'forgets' all that was remembered for certain
; functions that use certain stobjs.  We must keep memoize-flush very
; fast in execution so as not to slow down stobj update or resize
; operations in general.  We 'forget' the pons table later.

(defmacro memoize-flush (st)
  (let ((s (st-lst st)))
    `(loop for sym in ,s do
           (let ((old (symbol-value (the symbol sym))))
             (unless (or (null old) (empty-ht-p old))
               (setf (symbol-value (the symbol sym)) nil))))))

(declaim (hash-table *memoize-info-ht*))

(defmacro pist (table &rest x)
  (cond ((atom x) nil)
        (t (list 'pons (car x)
                 (cons 'pist (cdr x)) table))))

(defmacro pist* (table &rest x)
  (cond ((atom x) x)
        ((atom (cdr x)) (car x))
        (t (list 'pons (car x)
                 (cons 'pist* (cons table (cdr x))) table))))

;  THE MEMO-INFO-HT-ENTRY DATA STRUCTURE

; *MEMOIZE-INFO-HT* maps each currently memoized function symbol, fn,
; to a DEFREC record of type MEMO-INFO-HT-ENTRY with 22 fields.

; fn             a symbol, the name of the function being memoized
; tablename      a symbol whose value is the memoize table for fn
; ponstablename  a symbol whose value is the pons table for fn
; old-fn         the old value of (symbol-function fn)
; memoized-fn    the new value of (symbol-function fn)
; condition      T or NIL. :condition arg as passed to memoize-fn
; inline         T or NIL. :inline arg as passed to memoize-fn
; num            an integer, unique to fn
; sts            the stobj memotable lists for fn
; trace          T or NIL. :trace arg as passed to memoize-fn
; start-time     a symbol whose val is the start time of the current,
;                   outermost call of fn, or -1 if no call of fn
;                   is in progress.
; cl-defun       the function body actually used, in the inline=t
;                case, as supplied (or as computed, if not supplied)
; formals        as supplied (or as computed, if not supplied)
; stobjs-in      as supplied (or as computed, if not supplied)
; stobjs-out     as supplied (or as computed, if not supplied)
; record-bytes   value as bound at the time MEMOIZE-FN is called
; record-calls            ''
; record-hits             ''
; record-hons-calls       ''
; record-mht-calls        ''
; record-pons-calls       ''
; record-time             ''

; *memoize-info-ht* also maps num back to the corresponding symbol.

(defrec memoize-info-ht-entry
  (start-time  ; vaguely ordered by most frequently referenced first
   num
   tablename
   ponstablename
   condition
   inline
   memoized-fn
   old-fn
   fn
   sts
   trace
   cl-defun
   formals
   stobjs-in
   stobjs-out
   record-bytes
   record-calls
   record-hits
   record-hons-calls
   record-mht-calls
   record-pons-calls
   record-time
   )
  t)

; MEMOIZE FUNCTIONS

(defmacro heap-bytes-allocated ()
  #+openmcl
  '(the fixnum (ccl::%heap-bytes-allocated))
  #-openmcl
  0)

(defn sync-watch-array ()

  ; To be called only if no other threads are running, and only by:
  ; 1. memoize-init.
  ; 2. watch-array-grow.  Aborts if a memoized function is running.

  ; Because memoized functions bind *CALLER* when running, if a
  ; function is running when we try to grow, the outermost *CALLER*
  ; value could not be reset correctly because it is BOUND.

  (let ((n1 (the fixnum (* *2max-memoize-fns* *2max-memoize-fns*)))
        (n2 (1+ *max-symbol-to-fixnum*)))
    (declare (fixnum n1 n2))
    (unless (eql n1 (length *watch-array*))
      (unless (eql 1 (length *watch-array*))
        (setq *watch-array*
              (make-array 1 :element-type 'fixnum
                          :initial-element 0))
        (gc$))
      (setq *watch-array*
            (make-array n1
                        :element-type 'fixnum
                        :initial-element 0)))
    (unless (eql n2 (length *compute-array*))
      (setq *compute-array*
            (make-array n2 :initial-element nil)))
    (setq *caller* (* *ma-initial-max-symbol-to-fixnum*
                      *2max-memoize-fns*))))

(defun watch-array-grow
  (&optional (2nmax (* 2 (ceiling (* 3/2 (/ *2max-memoize-fns* 2))))))
  (check-no-extra-processes)
  (check-type 2nmax (integer 100))
  (unless (evenp 2nmax)
    (ofe "~&; watch-array-grow: ** Error: 2nmax must be even."))
  (when (<= 2nmax *2max-memoize-fns*)
    (return-from watch-array-grow))
  (unless (<= (* 2nmax 2nmax) most-positive-fixnum)
    (ofe "~&; watch-array-grow: ** Error: most-positive-fixnum~%~
            exceeded.  Too many memoized functions."))
  (unless (<= (* 2nmax 2nmax) array-total-size-limit)
    (ofe "~&; watch-array-grow: ** Error: ARRAY-TOTAL-SIZE-LIMIT ~%~
            exceeded.  Too many memoized functions."))
  (unless (eql *caller* (the fixnum
                          (* *ma-initial-max-symbol-to-fixnum*
                             *2max-memoize-fns*)))
    (ofd "~%; A memoized function is running, namely ~s.~
          ~%; Quit to the top raw Lisp level, ~
              invoke (watch-array-grow ~s),
          ~%; and start again, e.g., with (lp)."
         (fixnum-to-symbol (/ *caller* *2max-memoize-fns*))
         2nmax)
    (break "watch-array-grow")
    (error "watch-array-grow"))
  (let ((watch #+openmcl (eval '(and (boundp '*watch-thread*)
                                     (ccl::processp *watch-thread*)))
               #-openmcl nil))
    (when watch
      (ofd "~%; watch-array-grow:  Killing watch.")
      (funcall 'watch-kill))
    (setq *watch-array*
          (make-array 1 :element-type 'fixnum
                      :initial-element 0))
    (setq *2max-memoize-fns* 2nmax)
    (sync-watch-array)
    (rememoize-all)
    (when watch
      (ofd "~%; watch-array-grow:  Restarting watch.")
      (funcall 'watch))))

(defun symbol-to-fixnum-create (s)
  (check-type s symbol)
  (let ((g (gethash s *memoize-info-ht*)))
    (if g (access memoize-info-ht-entry g :num)
      (let (new)
        (loop for i from
              (if (eql *caller*
                       (* *ma-initial-max-symbol-to-fixnum*
                          *2max-memoize-fns*))
                  (1+ *ma-initial-max-symbol-to-fixnum*)
                (1+ *max-symbol-to-fixnum*))
              below (the fixnum (floor *2max-memoize-fns* 2))
              do (unless (gethash i *memoize-info-ht*)
                   (setq new i)
                   (return)))
        (cond (new
               (setq *max-symbol-to-fixnum*
                     (max *max-symbol-to-fixnum* new))
               new)
              (t (watch-array-grow)
                 (safe-incf *max-symbol-to-fixnum*
                            1 symbol-to-fixnum-create)
                 *max-symbol-to-fixnum*))))))

(defun symbol-to-fixnum (s)
  (check-type s symbol)
  (let ((g (gethash s *memoize-info-ht*)))
    (if g (access memoize-info-ht-entry g :num)
      (ofe "~&; symbol-to-fixnum:  ** Error:  illegal symbol:  ~s."
           s))))

(defun fixnum-to-symbol (n)
  (check-type n fixnum)
  (or (gethash n *memoize-info-ht*)
      (ofe "~&; fixnum-to-symbol:  ** Error:  illegal number:  ~d."
           n)))

(defun coerce-index (x)
  (if (and (typep x 'fixnum)
           (>= x 0)
           (< x (length *watch-array*)))
      x
    (symbol-to-fixnum x)))

; This code has the 'feature' that if the condition causes an error,
; so will the memoized function.

; PONS differs from HONS in that it does not honsify its arguments and
; in that it takes a hash table as a third argument.  We use PONS in
; memoization.

; We use PONS instead of HONS in memoization because we could not
; afford to honsify (using hons-shrink-alist!) certain alists in
; certain biology tests.  About the same time, we (gratuitously)
; decided to stop hons'ifying the output of memoized functions.

(defun pons (x y ht)

; ***** pons *****

; If pons can create a hons, that will lead to a deadlock over locks!

  (let ((xval nil)
        (yval nil)
        (ans nil))
    (maybe-count-pons-calls)

; We have taken string normalization out of pons because there might
; be a chance of confusing a 'normal' string with a stobj.

; If x1, ..., xn is pointwise EQL to y1, ..., yn, then are we sure
; that (pist* x1 ... xn) is EQ to (pist* y1 ... yn)?

; If CONS exists, then return it.  Does CDR exist in hash table?

    (setq yval (gethash y ht))

; Does CAR exist in hash table?

    (cond (yval
           (cond ((not (consp yval))
                  (setq xval (gethash x yval))
                  (cond (xval
                         (maybe-count-pons-hits)
                         (setq ans xval))))
                 ((setq ans (assoc-no-error-at-end x yval))
                  (maybe-count-pons-hits)))))
    (cond

; If PONS found, then return previous CONS from hash table.
     (ans)

; Otherwise, maybe create new CONS and install in hash table.

     (t
      (setq yval (gethash y ht))
      (cond
       ((null yval)
        (setq ans (cons x y))
        (setf (gethash y ht) (list ans))
        ans)
       ((consp yval)
        (let ((ans (assoc-no-error-at-end x yval)))
            (cond
             (ans
              (maybe-count-pons-hits)
              ans)
             (t (let ((ans (cons (cons x y) yval)))
                  (cond
                   ((too-long ans *start-car-ht-size*)
                    (let ((tab (mht)))
                      (loop for pair in ans do
                            (setf (gethash (car pair) tab) pair))
                      (setf (gethash y ht) tab)
                      (car ans)))
                   (t (setf (gethash y ht) ans)
                      (car ans))))))))
       (t (setq xval (gethash x yval))
          (cond ((not xval)
                 (setf (gethash x yval)
                       (setq ans (cons x y))))
                (t (maybe-count-pons-hits)
                   (setq ans xval)))
          ans))))))

(defun memoize-eval-compile (defun)
  (let ((name (cadr defun)))
    (setf (symbol-function name)
          (eval
           `(lambda ,(caddr defun)
              (compile (eval ',defun))
              (,name ,@(caddr defun)))))
    name))

(defun memoizedp-raw (fn)
  (our-lock-unlock-memoize1
   (and (symbolp fn)
        (values (gethash fn *memoize-info-ht*)))))

(defg *hons-gentemp-counter* 0)
(declaim (fixnum *hons-gentemp-counter*))
(defun-one-output hons-gentemp (root)
  (check-type root string)
  (loop
   (safe-incf *hons-gentemp-counter* 1 hons-gentemp)
   (let ((name (ofn "HONS-G-~s,~s" root *hons-gentemp-counter*)))
     (multiple-value-bind (sym status)
         (intern name (find-package "ACL2_INVISIBLE"))
       (if (null status) (return sym))))))

(defun st-lst (st)

; ST-LST returns a symbol whose value is a list in which are saved the
; names of the memoize tables that will be set to nil whenever the
; stobj st is changed.

  (check-type st symbol)
  (multiple-value-bind (symbol status)
      (intern (ofn "HONS-S-~s,~s"
                   (package-name (symbol-package st))
                   (symbol-name st))
              (find-package "ACL2_INVISIBLE"))
    (or status (eval `(defg ,symbol nil)))
    symbol))

(defun dcls (l)
     (loop for dec in l nconc
           (let ((temp
                  (if (consp dec)
                      (loop for d in (cdr dec) nconc
                            (if (and (consp d) (eq (car d) 'ignore))
                                nil
                              (cons d nil))))))
             (if temp (cons (cons 'declare temp) nil)))))

(defun timer-error ()
  (ofe "~&; timer-error:  ** Error."))

; PRINE  - princ eviscerated

(defg *assoc-eq-hack-ht* (mht :test 'eql))

(defg *hons-vars-alist* nil)

(defn assoc-eq-hack (x y)
  (cond ((atom y) nil)
        (t (let ((h (gethash y *assoc-eq-hack-ht*)))
             (cond (h (gethash x h))
                   (t (setq h (mht :test 'eq))
                      (setf (gethash y *assoc-eq-hack-ht*)
                            h)
                      (loop for pair in y do
                            (setf (gethash (car pair)
                                           h)
                                  pair))
                      (gethash x h)))))))

(defn abbrev (x level length)
  (cond ((atom x) x)
        ((eql level 0) '?)
        ((eql length 0) '?)
        (t (let ((pair (assoc-eq-hack x (table-alist 'evisc-table (w *the-live-state*)))))
             (cond (pair (cdr pair))
                   (t (let ((a (abbrev (car x) (and level (1- level)) length))
                            (d (abbrev (cdr x) level (and length (1- length)))))
                        (cond ((and (eq a (car x))
                                    (eq d (cdr x)))
                               x)
                              ((and (eq a '?)
                                    (eq d '?))
                               '?)
                              (t (cons a d))))))))))
                                       
(defun prine (obj &optional stream)
  (let ((*print-pretty* nil))
    (princ (abbrev obj *print-level* *print-length*) stream)))

; MEMOIZE-FN

(defun memoize-fn (fn &key
                      (condition t)
                      (inline t)
                      (trace nil)
                      (cl-defun nil clp-p)
                      (formals t fop)
                      (stobjs-in t sip)
                      (stobjs-out t sop))
  (check-no-extra-processes)
  (unwind-protect
      (progn
        #+parallel (ccl::lock-hash-table *hons-cdr-ht*)
        #+parallel (ccl::lock-hash-table *compiled-module-ht*)
        #+parallel (ccl::lock-hash-table *memoize-info-ht*)
        (with-warnings-suppressed
         (when (mf-skip fn) (return-from memoize-fn nil))
         (mf-1st-warnings fn clp-p)
         (let* ((w (w *the-live-state*))
                (cl-defun (if clp-p cl-defun
                            (and inline (cltl-def-from-name fn nil w))))
                (formals (if fop formals
                           (getprop fn 'formals t 'current-acl2-world w)))
                (stobjs-in (if sip stobjs-in
                             (getprop fn 'stobjs-in t
                                      'current-acl2-world w)))
                (stobjs-out (if sop stobjs-out
                              (getprop fn 'stobjs-out t
                                       'current-acl2-world w))))
           (mf-2nd-warnings fn formals cl-defun stobjs-in stobjs-out inline)
           (let* ((fnn (symbol-to-fixnum-create fn))
                  (*acl2-unwind-protect-stack*
                   (cons nil *acl2-unwind-protect-stack*))
                  (old-fn (symbol-function fn))
                  (body (if inline (car (last cl-defun))
                          `(funcall (the (,(if (compiled-function-p old-fn)
                                               'compiled-function
                                             'function)
                                          ,(make-list (length formals)
                                                      :initial-element t)
                                          (values ,@(make-list
                                                     (length stobjs-out)
                                                     :initial-element t)))
                                      ,old-fn)
                                    ,@formals)))
                  (condition-body
                   (cond ((booleanp condition) condition)
                         ((symbolp condition)
                          (car (last (cltl-def-from-name condition nil w))))
                         (t condition)))
                  (dcls (dcls (cdddr (butlast cl-defun))))
                  (start-time (let ((v (hons-gentemp
                                        ;;   Must use defparameter here.
                                        (suffix "START-TIME-~s-" fn))))
                                (eval `(prog1 (defparameter ,v -1)
                                         (declaim (fixnum ,v))))))
                  (tablename (eval `(defg
                                      ,(hons-gentemp
                                        (suffix "MEMOIZE-HT-FOR-~s-" fn))
                                      nil)))
                  (ponstablename (eval `(defg
                                          ,(hons-gentemp
                                            (suffix "PONS-HT-FOR-~s-" fn))
                                          nil)))
                  (sts (loop for x in (union stobjs-in stobjs-out)
                             when x collect (st-lst x)))
                  defun
                  success)
             (let* ((body3
                     `(progn
                        ,@(if trace
                              `((oftr "~%(> ~s (" ',fn)
                                ,@(loop for v in formals
                                        append `((oftr "~& ~s = " ',v)
                                                 (prine ,v *trace-output*)))
                                (oftr "~& )")))

                        (,(if (cdr stobjs-out) 'multiple-value-prog1 'prog1)
                         (let (,*mf-ans* ,*mf-args* ,*mf-ans-p*)
                           (declare (ignorable ,*mf-ans* ,*mf-args*
                                               ,*mf-ans-p*))
                           (cond
                            ((not ,condition-body)
                             ,(if (not trace)
                                  body
                                (if (cdr stobjs-out)
                                    `(values-list
                                      (setq ,*mf-ans*
                                            (multiple-value-list ,body)))
                                  `(setq ,*mf-ans* ,body))))
                            (t (when (null ,tablename)
                                 ,@(if *record-mht-calls*
                                       `((safe-incf
                                          (aref ,*mf-ma*
                                                (the fixnum
                                                  (+ ,(* *2max-memoize-fns*
                                                         fnn)
                                                     *ma-mht-index*)))
                                          1
                                          ,fn)))
                                 (setq ,tablename (mht))
                                 ,@(if (cdr formals)
                                       `((setq ,ponstablename (mht)))))
                               #+parallel
                               ,@(if (cdr formals)
                                     `((ccl::lock-hash-table ,ponstablename)))
                               (setq ,*mf-args*
                                     (pist* ,ponstablename ,@formals))
                               #+parallel
                               ,@(if (cdr formals)
                                     `((ccl::unlock-hash-table ,ponstablename)))
                               (multiple-value-setq (,*mf-ans* ,*mf-ans-p*)
                                 (gethash ,*mf-args* ,tablename))
                               (cond
                                (,*mf-ans-p*
                                 ,@(when trace
                                     `((oftr "~% ~s remembered" ',fn)))
                                 ,@(when *record-hits*
                                     `((safe-incf
                                        (aref ,*mf-ma*
                                              (the fixnum
                                                (+ ,(* fnn
                                                       *2max-memoize-fns*)
                                                   *ma-hits-index*)))
                                        1
                                        ,fn)))
                                 ,(if (null (cdr stobjs-out))
                                      *mf-ans*
                                    (cons 'mv
                                          (nconc
                                           (loop for i below
                                                 (1- (length stobjs-out))
                                                 collect `(pop ,*mf-ans*))
                                           (list *mf-ans*)))))
                                (t
                                 ,(if (cdr stobjs-out)
                                      (let ((vars (loop
                                                   for i below
                                                   (length stobjs-out)
                                                   collect
                                                   (ofnm "O~a" i))))
                                        `(mv-let
                                          ,vars
                                          ,body
                                          (setf (gethash ,*mf-args*
                                                         ,tablename)
                                                (setq ,*mf-ans*
                                                      (list* ,@vars)))
                                          (mv ,@vars)))
                                    `(setf (gethash ,*mf-args* ,tablename)
                                           (setq ,*mf-ans* ,body)))))))
                           ,@(if trace `((oftr "~%< ~s " ',fn)
                                         (prine ,*mf-ans* *trace-output*)
                                         (oftr ")~%")))))))
                    (body2 `(let ((*caller* ,(* fnn *2max-memoize-fns*))
                                  (,start-time
                                   ,(if *record-time* '(internal-real-time)
                                      '0))
                                  ,@(and *record-bytes*
                                         `((,*mf-start-bytes*
                                            (heap-bytes-allocated))))
                                  ,@(and *record-hons-calls*
                                         `((,*mf-start-hons*
                                            *hons-call-counter*)))
                                  ,@(and *record-pons-calls*
                                         `((,*mf-start-pons*
                                            *pons-call-counter*))))
                              (declare (ignorable
                                        ,@(and *record-bytes*
                                               `(,*mf-start-bytes*))
                                        ,@(and *record-hons-calls*
                                               `(,*mf-start-hons*))
                                        ,@(and *record-pons-calls*
                                               `(,*mf-start-pons*)))
                                       (fixnum
                                        ,start-time
                                        ,@(and *record-hons-calls*
                                               `(,*mf-start-hons*))
                                        ,@(and *record-pons-calls*
                                               `(,*mf-start-pons*))
                                        ,@(and *record-bytes*
                                               `(,*mf-start-bytes*))))
                              (,(if (cdr stobjs-out)
                                    'multiple-value-prog1
                                  'prog1)
                               ,body3
                               ,@(and
                                  *record-hons-calls*
                                  `((safe-incf
                                     (aref ,*mf-ma*
                                           (the fixnum
                                             (+ *ma-hons-index*
                                                ,(* fnn
                                                    *2max-memoize-fns*))))
                                     (the fixnum
                                       (- *hons-call-counter*
                                          ,*mf-start-hons*))
                                     ,fn)))
                               ,@(and
                                  *record-pons-calls*
                                  `((safe-incf
                                     (aref
                                      ,*mf-ma*
                                      (the fixnum
                                        (+ *ma-pons-index*
                                           ,(* fnn *2max-memoize-fns*))))
                                     (the fixnum
                                       (- *pons-call-counter*
                                          ,*mf-start-pons*))
                                     ,fn)))
                               ,@(and
                                  *record-bytes*
                                  `((safe-incf
                                     (aref ,*mf-ma*
                                           (the fixnum
                                             (+ *ma-bytes-index*
                                                ,(* fnn
                                                    *2max-memoize-fns*))))
                                     (the fixnum
                                       (- (heap-bytes-allocated)
                                          ,*mf-start-bytes*))
                                     ,fn)))
                               ,@(and
                                  *record-time*
                                  `((safe-incf
                                     (aref ,*mf-ma*
                                           (the fixnum
                                             (1+ ,*mf-count-loc*)))
                                     (the fixnum (- (internal-real-time)
                                                    ,start-time))
                                     ,fn)))))))
               (setq
                defun
                `(defun ,fn ,formals ,@dcls
                   (declare (ignorable ,@formals))
                   (let* ((,*mf-count-loc* (the fixnum (+ *caller*
                                                          (* 2 ,fnn))))
                          (,*mf-ma* *watch-array*))
                     (declare (fixnum ,*mf-count-loc*)
                              (ignorable ,*mf-count-loc* ,*mf-ma*)
                              (type (simple-array fixnum (*)) ,*mf-ma*))
                     ,@(and *record-calls*
                            `((safe-incf (aref ,*mf-ma* ,*mf-count-loc*)
                                         1
                                         ,fn)))
                     (if (not (eql -1 ,start-time)) ,body3 ,body2)))))
             (when *memoize-debug*
               (let ((*standard-output* *debug-io*))
                 (our-syntax
                  (let ((*print-readably* nil)
                        (*print-pretty* t))
                    (fresh-line)
                    (prin1 defun)
                    (fresh-line)
                    (force-output)))))
             (unwind-protect
                 (progn
                   (let ((fnnma (the fixnum (* fnn *2max-memoize-fns*)))
                         (ma *watch-array*))
                     (declare (fixnum fnnma)
                              (type (simple-array fixnum (*)) ma))
                     (loop for i fixnum from fnnma
                           below (the fixnum (+ fnnma *2max-memoize-fns*))
                           unless (eql (aref ma i) 0)
                           do (setf (aref ma i) 0)))
                   (memoize-eval-compile defun)
                   (setf (gethash fn *memoize-info-ht*)
                         (make memoize-info-ht-entry
                               :fn fn
                               :tablename tablename
                               :ponstablename ponstablename
                               :old-fn old-fn
                               :memoized-fn (symbol-function fn)
                               :condition condition
                               :inline inline
                               :num fnn
                               :sts sts
                               :trace trace
                               :start-time start-time
                               :cl-defun cl-defun
                               :formals formals
                               :stobjs-in stobjs-in
                               :stobjs-out stobjs-out
                               :record-bytes      *record-bytes*
                               :record-calls      *record-calls*
                               :record-hits       *record-hits*
                               :record-hons-calls *record-hons-calls*
                               :record-mht-calls  *record-mht-calls*
                               :record-pons-calls *record-pons-calls*
                               :record-time       *record-time*))
                   (setf (gethash fnn *memoize-info-ht*) fn)
                   (and condition (loop for s in sts do
                                        (push tablename (symbol-value s))))
                   (setq success t)
                   fn)
               (unless success
                 (setf (symbol-function fn) old-fn)
                 (remhash fn *memoize-info-ht*)
                 (remhash fnn *memoize-info-ht*)
                 (and condition
                      (loop for s in sts
                            when (eq tablename
                                     (car (symbol-value (the symbol s))))
                            do (pop (symbol-value (the symbol s)))))
                 (ofd "~&; Memoize-fn:  Failed to memoize ~s." fn)))))))

    #+parallel (ccl::unlock-hash-table *memoize-info-ht*)
    #+parallel (ccl::unlock-hash-table *compiled-module-ht*)
    #+parallel (ccl::unlock-hash-table *hons-cdr-ht*)))

(defn mf-skip (fn)
  (cond ((memoizedp-raw fn)
         (ofd "~%; Memoize-fn: ** Warning: ~s is already memoized, ~
               so the current request to memoize it is simply ~
               being ignored." fn)
         t)
        ((member fn *never-profile-list*)
         (ofd "~%; Memoize-fn: ** Warning: Memoizing ~a is not ~
               allowed because it is a member of ~
               *NEVER-PROFILE-LIST*.  Ignoring." fn))))

(defn mf-1st-warnings (fn clp-p)
  (unless *memoize-init-done*
    (ofe "~%; Memoize-fn:  ** Error: *MEMOIZE-INIT-DONE* is nil."))
  (unless (or clp-p (memoizeable-function-p fn))
    (ofe "~%; Memoize-fn:  ** Error: ~s is not a function." fn))

  ; TRACE, UNTRACE, OLD-TRACE, and OLD-UNTRACE are macros that get
  ; redefined sometimes.  So we use EVAL in calling them.

  (when (member fn (eval '(trace)))
    (ofd "~%; Memoize-fn:  Untracing ~s before memoizing it." fn)
    (eval `(untrace ,fn)))
  (when (and (boundp 'old-trace)
             (member fn (eval '(old-trace))))
    (ofd "~%; Memoize-fn:  Old-untracing ~s before memoizing it." fn)
    (eval `(old-untrace ,fn))))


(defg *profile-reject* (mht :test 'eq)

  "The user may freely add to the hash table *PROFILE-REJECT-HT*, which
  inhibits the collection of functions into lists of functions to be
  profiled or memoized by PROFILE-UNMEMOIZED and MEMOIZE-UNMEMOIZED.

  Here are some reasons for adding a function fn to *profile-reject-ht*.

  1. A call of fn is normally so fast or fn is called so often that
  the extra instructions executed when a memoized version of fn is run
  will distort measurements excessively.  We tend not to profile any
  function that runs in under 6000 clock ticks or about 2
  microseconds.  The number of extra instructions seem to range
  between 20 and 100, depending upon what is being measured.
  Measuring only calls is relatively fast.  If one measures elapsed
  time, one might as well measure everything else too.  Or so it seems
  in 2007 on terlingua.csres.utexas.edu.

  2. fn is a subroutine of another function being profiled, and we
  wish to reduce the distortion that profiling fn will cause.

  3. fn is 'transparent', like EVAL.  Is EVAL fast or slow?  The
  answer, of course, is that it mostly depends upon what one is
  EVALing.

  4. fn's name ends in '1', meaning 'auxiliary' to some folks.

  5. fn is boring.")

(defn mf-2nd-warnings
  (fn formals cl-defun stobjs-in stobjs-out inline)
   (when (and (null cl-defun) inline)
     (ofe "~%; Memoize-fn: ** Error: ~s lacks a cltl-def-from-name ~
           property." fn))
   (when (eq t formals)
     (ofe "~%; Memoize-fn:  ** Error: ~s lacks a formals property."
          fn))
   (when (eq t stobjs-in)
     (ofe "~%; Memoize-fn:  ** Error: ~s lacks a stobjs-in property."
          fn))
   (when (eq t stobjs-out)
     (ofe "~%; Memoize-fn: ** Error: ~s lacks a stobjs-out property."
          fn))
   (when (or (member 'state stobjs-in) (member 'state stobjs-out))
     (ofe "~%; Memoize-fn: ** Error: ~s uses STATE." fn))
   (unless (member-equal (symbol-package-name fn) '("ACL2" "ACL2-PC" "U"))
     (ofd "~&; Memoize-fn: Warning.  ~s is not in the ACL2, ~
           ACL2-PC, or U packages."
          fn))
   (when (eq (symbol-package fn) *main-lisp-package*)
     (ofd "~&; Memoize-fn:  A warning about the memoization of ~s."
          fn)
     (ofd *mf-cl-error-msg*)
     (f-put-global 'certify-book-disabledp t *the-live-state*)))

(defun unmemoize-fn (fn)
  (check-no-extra-processes)
   (unwind-protect
       (progn
         #+parallel (ccl::lock-hash-table *hons-cdr-ht*)
         #+parallel (ccl::lock-hash-table *memoize-info-ht*)
         #+parallel (ccl::lock-hash-table *compiled-module-ht*)
         (eval `(maybe-untrace ,fn))
         (let* ((ma *watch-array*)
                (l (memoizedp-raw fn)))
           (declare (type (simple-array fixnum (*)) ma))
           (unless l (ofe "~&; Unmemoize-fn: ~s is not memoized." fn))
           (let* ((num (the fixnum (access memoize-info-ht-entry l :num)))
                  (tablename (and l (access memoize-info-ht-entry l
                                            :tablename)))
                  (n2 (* num *2max-memoize-fns*)))
             (declare (fixnum num n2))

; Note: condition is a first-class ACL2 function, not to be messed
; with here.

             (let (#+OpenMCL (ccl:*warn-if-redefine-kernel* nil))
               (setf (symbol-function (the symbol fn))
                     (access memoize-info-ht-entry l :old-fn)))
             (loop for i fixnum from n2
                   below (the fixnum (+ n2 *2max-memoize-fns*))
                   unless (eql (aref ma i) 0)
                   do (setf (aref ma i) 0))
             (remhash fn *memoize-info-ht*)
             (remhash num *memoize-info-ht*)
             (setf (symbol-value (the symbol tablename)) nil)
             (setf (symbol-value (the symbol (access memoize-info-ht-entry
                                                     l :ponstablename)))
                   nil)
             (loop for s in (access memoize-info-ht-entry l :sts) do
                   (setf (symbol-value (the symbol s))
                         (remove tablename (symbol-value (the symbol s)))))))
         fn)
     #+parallel (ccl::unlock-hash-table *hons-cdr-ht*)
     #+parallel (ccl::unlock-hash-table *memoize-info-ht*)
     #+parallel (ccl::unlock-hash-table *compiled-module-ht*)))

(defun maybe-unmemoize (fn)
  (when (memoizedp-raw fn) (unmemoize-fn fn)))

(defun memoized-functions ()
  (our-lock-unlock-memoize1
  (let (l)
    (maphash (lambda (fn v) (declare (ignore v))
               (when (symbolp fn) (push fn l))) *memoize-info-ht*)
             l)))

(defun length-memoized-functions ()
  (floor (1- (hash-table-count *memoize-info-ht*))
         2))

(defun unmemoize-all ()

  "(UNMEMOIZE-ALL) unmemoizes all currently memoized functions,
  including of course all profiled functions."

; A warning to would-be code improvers.  It would be a bad idea to
; redefine UNMEMOIZE-ALL to maphash over *memoize-info-ht* because of
; the ANSI rules concerning which hash table entries may be modified
; during a maphash.

   (loop for x in (memoized-functions) do (unmemoize-fn x))
   (memoize-init))

(defun rememoize-all ()
  (our-lock-unlock-memoize1
      (let (l)
        (maphash
         (lambda (k v)
           (when (symbolp k)
             (push
              (list (list (access memoize-info-ht-entry v :fn)
                          :condition
                          (access memoize-info-ht-entry v :condition)
                          :inline
                          (access memoize-info-ht-entry v :inline)
                          :trace
                          (access memoize-info-ht-entry v :trace)
                          :cl-defun
                          (access memoize-info-ht-entry v :cl-defun)
                          :formals
                          (access memoize-info-ht-entry v :formals)
                          :stobjs-in
                          (access memoize-info-ht-entry v :stobjs-in)
                          :stobjs-out
                          (access memoize-info-ht-entry v :stobjs-out))
                    (list
                     (access memoize-info-ht-entry v :record-bytes)
                     (access memoize-info-ht-entry v :record-calls)
                     (access memoize-info-ht-entry v :record-hits)
                     (access memoize-info-ht-entry v :record-hons-calls)
                     (access memoize-info-ht-entry v :record-mht-calls)
                     (access memoize-info-ht-entry v :record-pons-calls)
                     (access memoize-info-ht-entry v :record-time)))
              l)))
         *memoize-info-ht*)
        (loop for x in l do (unmemoize-fn (caar x)))
        (gc$)
        (setq *max-symbol-to-fixnum*
              *ma-initial-max-symbol-to-fixnum*)
        (loop for x in l do
              (progv '(*record-bytes*
                       *record-calls*
                       *record-hits*
                       *record-hons-calls*
                       *record-mht-calls*
                       *record-pons-calls*
                       *record-time*)
                  (cadr x)
                (apply 'memoize-fn (car x)))))))

(defun compliant-and-ideal ()
  (let* ((logic-fns
          (eval '(let ((world (w *the-live-state*)))
                   (strip-cadrs (set-difference-theories
                                 (function-theory :here)
                                 (universal-theory 'ground-zero))))))
         (ideal-fns (collect-non-common-lisp-compliants
                     logic-fns (w *the-live-state*))))
    (mv (set-difference-eq logic-fns ideal-fns) ideal-fns)))

(defun uses-state (fn)
  (let* ((w (w *the-live-state*))
         (stobjs-in (getprop fn 'stobjs-in t 'current-acl2-world w))
         (stobjs-out (getprop fn 'stobjs-out t
                              'current-acl2-world w)))
    (or (and (consp stobjs-in) (member 'state stobjs-in))
        (and (consp stobjs-out) (member 'state stobjs-out)))))

(defn memoize-here-come (n)
  (let ((m (ceiling
            (+ 100 (* 1.1 (- n (- (/ *2max-memoize-fns* 2)
                                  (floor
                                   (/ (hash-table-count
                                       *memoize-info-ht*)
                                      2)))))))))
    (when (posp m) (watch-array-grow (* 2 m)))))

(defun profile (fn &key
  (number-of-args (number-of-args fn))
  (number-of-return-values (number-of-return-values fn))
  (condition nil)
  (inline nil)
  (trace nil))

  "PROFILE is a raw Lisp function.  PROFILE can never be part of ACL2
  proper, but can be useful in debugging.

  Please call PROFILE only from the top-level of raw Lisp.  Exit
  ACL2, if appropriate, with :q, before calling PROFILE, and later
  reenter ACL2 with (lp).

  Unlike MEMOIZE, PROFILE is for experimental use only, and cannot be
  called from with ACL2.

  (PROFILE fn) calls MEMOIZE-FN to memoize the function fn.  PROFILE
  can even be called on functions not known to ACL2.  PROFILE may lie
  to MEMOIZE-FN by asserting that fn does not receive or return ACL2's
  STATE.
  
  Profile takes the following keyword paramters.

  :NUMBER-OF-ARGS and :NUMBER-OF-RETURN-VALUES default to the correct
  values for ACL2 functions, but otherwise may be Lisp-implementation
  dependent incorrect guesses.

  :CONDITION defaults to NIL.

  :INLINE defaults to NIL.

  :TRACE defaults to NIL.

  Using a :CONDITION other than NIL may be risky unless a function has
  been accepted as Common Lisp compliant by ACL2.  For one thing, it
  might make no sense to bypass, by memoization, calls to a function
  that alters STATE.

  It is possible to call MEMOIZE-FN, from within raw Lisp in such a
  way as to force the memoization of an ACL2 function that has STATE
  as an explicit parameter using fraudulent FORMALS, STOBJS-IN, and
  STOBJS-OUT parameters; this could be experimentally useful, but
  could be unsound, too."

  (check-type number-of-args (integer 0))
  (check-type number-of-return-values (integer 0))
  (memoize-fn fn
              :condition condition
              :inline inline
              :trace trace
              :formals
              (loop for i below number-of-args
                    collect (ofni "X-~s" i))
              :stobjs-in  (make-list number-of-args)
              :stobjs-out (make-list number-of-return-values)
              ))

(defun unmemoize-profiled ()

  "UNMEMOIZE-PROFILED is a raw Lisp function.  (UNMEMOIZE-PROFILED)
  unmemoized all functions currently memoized with :CONDITION=NIL and
  :INLINE=NIL."

  (let (l)
    (maphash
     (lambda (k v)
       (if (and (symbolp k)
                (null (access memoize-info-ht-entry
                              v :condition))
                (null (access memoize-info-ht-entry
                              v :inline)))
           (push k l)))
     *memoize-info-ht*)
    (loop for x in l do (unmemoize-fn x))))

(defun profile-unmemoized (&key (start 0) (condition nil) (inline nil)
                                (trace nil))

  "(PROFILE-UNMEMOIZED :START fn) profiles most all ACL2 functions
  starting with the event named (or numbered) fn, which defaults to 0.
  Not profiled by PROFILE-UNMEMOIZED are functions
     for which (hons-get fn *profile-reject*) is non-NIL,
     already memoized,
     in *NEVER-PROFILE-LIST*,
     with an unknown number of arguments or number of return values, or
     in package COMMON-LISP."

  (let ((*record-bytes* t)
        (*record-calls* t)
        (*record-hits* t)
        (*record-hons-calls* t)
        (*record-mht-calls* t)
        (*record-pons-calls* t)
        (*record-time* t))
    (when (symbolp start) (setq start (event-number start)))
    (check-type start integer)
    (let* ((packs (cons "ACL2"
                        (f-get-global 'packages-created-by-defpkg
                                      *the-live-state*)))
           (l (make-hash-table :test 'eq)))
      (loop for p in packs do
            (do-symbols (fn p)
              (unless
                  (or (not (memoizeable-function-p fn))
                      (memoizedp-raw fn)
                      (eq (symbol-package fn) *main-lisp-package*)
                      (member fn *never-profile-list*)
                      (gethash fn *profile-reject*)
                      (not (integerp (number-of-args fn)))
                      (not (integerp (number-of-return-values fn)))
                      (not (integerp (event-number fn)))
                      (< (event-number fn) start)
                      (and condition
                           (tree-occur 'hons-acons
                                       (body fn nil (w *the-live-state*)))))
                (setf (gethash fn l) t))))
      (memoize-here-come (hash-table-count l))
      (maphash
       (lambda (k v)
         (declare (ignore v))
         (profile k
                  :inline inline
                  :condition condition
                  :trace trace))
       l)
      (hash-table-count l))))

(defun min-profile-unmemoized (&key (start 0) (condition nil) (inline nil)
                                     (trace nil))

  "min-profile-unmemoized is a raw Lisp function
  (MIN-PROFILE-UNMEMOIZED :start fn) profiles most all ACL2 functions
  starting at the event named or numbered FN, which defaults to 0.  To
  reduce monitoring overhead, monitoring is limited to counting calls.
  MIN-PROFILE-UNMEMOIZED does not profile functions already memoized
  in any way, in *NEVER-PROFILE-LIST*, with an unknown number of
  arguments or number of return values, or in package COMMON-LISP."

  (let ((*record-bytes* nil)
        (*record-calls* t)
        (*record-hits* nil)
        (*record-hons-calls* nil)
        (*record-mht-calls* nil)
        (*record-pons-calls* nil)
        (*record-time* nil))
    (when (symbolp start) (setq start (event-number start)))
    (check-type start integer)
    (let* ((packs (cons "ACL2"
                        (f-get-global 'packages-created-by-defpkg
                                      *the-live-state*)))
           (l (make-hash-table :test 'eq)))
      (loop for p in packs do
            (do-symbols (fn p)
              (unless
                  (or (not (memoizeable-function-p fn))
                      (memoizedp-raw fn)
                      (eq (symbol-package fn) *main-lisp-package*)
                      (member fn *never-profile-list*)
                      (not (integerp (number-of-args fn)))
                      (not (integerp (number-of-return-values fn)))
                      (not (integerp (event-number fn)))
                      (< (event-number fn) start)
                      (and condition
                           (tree-occur 'hons-acons
                                       (body fn nil (w *the-live-state*)))))
                (setf (gethash fn l) t))))
      (memoize-here-come (hash-table-count l))
      (maphash
       (lambda (k v)
         (declare (ignore v))
         (profile k
                  :inline inline
                  :condition condition
                  :trace trace))
       l)
      (hash-table-count l))))

;  MEMOIZE-LET

; It might be a good enhancement to HT-LET to permit the carrying
; forward, with HOPY-CONS-CONSUME, of other honses.

(defun not-memoized-error (f)
  (ofe "~&; NOT-MEMOIZED-ERROR:  ** Error:  ~s is not memoized." f))

(defmacro memoize-let (fn form)
  (let ((fn-name (gensym "FN-NAME"))
        (tablevalue (gensym "TABLEVALUE"))
        (ponstablevalue (gensym "PONSTABLEVALUE"))
        (h (gensym "H"))
        (ht1 (gensym "HT1")))
    `(let* ((,fn-name ,fn)
            (,h (memoizedp-raw ,fn-name)))
       (unless ,h (not-memoized-error ,fn-name))
       (let* ((,tablevalue
               (symbol-value
                (access memoize-info-ht-entry ,h :tablename)))
              (,ponstablevalue
               (symbol-value
                (access memoize-info-ht-entry ,h :ponstablename)))
              (,ht1 (mht)))
         (unwind-protect
             (progn (setf (symbol-value
                           (access memoize-info-ht-entry ,h
                                   :tablename))
                          ,ht1)
                    (setf (symbol-value
                           (access memoize-info-ht-entry ,h
                                   :ponstablename))
                           (mht))
                    ,form)
           ;; During the evaluation of form, a change to a stobj may
           ;; result in tablename getting a new value, in which case
           ;; we may not restore its old value.  And a change in the
           ;; memoization status of fn would make a restoration
           ;; pointless.
           (let ((test (and (eq (symbol-value
                                 (access memoize-info-ht-entry
                                         ,h :tablename))
                                ,ht1)
                            (eq ,h (memoizedp-raw ,fn-name)))))
             (setf (symbol-value (access memoize-info-ht-entry
                                         ,h :tablename))
                   (and test ,tablevalue))
             (setf (symbol-value (access memoize-info-ht-entry
                                         ,h :ponstablename))
                   (and test ,ponstablevalue))))))))


; MEMOIZE-ON AND MEMOIZE-OFF

(defmacro memoize-on (fn x)
  `(let* ((,*mo-f* ,fn) (,*mo-h* (memoizedp-raw ,*mo-f*)))
     (unless ,*mo-h* (not-memoized-error ,*mo-f*))
     (let ((,*mo-o* (symbol-function (the symbol ,*mo-f*))))
       (unwind-protect
           (progn (setf (symbol-function (the symbol ,*mo-f*))
                        (access memoize-info-ht-entry ,*mo-h*
                                :memoized-fn))
                  ,x)
         (setf (symbol-function (the symbol ,*mo-f*)) ,*mo-o*)))))

(defmacro memoize-off (fn x)
  `(let* ((,*mo-f* ,fn) (,*mo-h* (memoizedp-raw ,*mo-f*)))
       (unless ,*mo-h* (not-memoized-error ,*mo-f*))
       (let ((,*mo-o* (symbol-function (the symbol ,*mo-f*))))
         (unwind-protect
             (progn (setf (symbol-function (the symbol ,*mo-f*))
                          (access memoize-info-ht-entry ,*mo-h*
                                  :old-fn))
                    ,x)
           (setf (symbol-function (the symbol ,*mo-f*)) ,*mo-o*)))))

(defn global-suppress-condition-nil-memoization ()
  (maphash
   (lambda (k l)
     (when (symbolp k)
       (when (null (access memoize-info-ht-entry l :condition))
         (setf (symbol-function k)
               (access memoize-info-ht-entry l :old-fn)))))
   *memoize-info-ht*))

(defn global-restore-memoize ()
  (maphash (lambda (k l)
             (when (symbolp k)
               (setf (symbol-function k)
                     (access memoize-info-ht-entry l :memoized-fn))))
           *memoize-info-ht*))

; STATISTICS GATHERING AND PRINTING ROUTINES
; STATISTICS GATHERING AND PRINTING ROUTINES

(defparameter *memoize-summary-order-list*
  '(total-time number-of-calls)

  "*MEMOIZE-SUMMARY-ORDER-LIST* is a raw Lisp variable.  It is a list
  of order functions that MEMOIZE-SUMMARY uses to sort all functions
  that are currently memoized in preparation for displaying
  information about them.  The order is lexicographical with the first
  order having the most weight.  Each order function must take one
  argument, a symbol, and return a rational.

  The default is '(total-time number-of-calls).

  Options for the functions include:

     bytes-allocated
     bytes-allocated/call
     event-number
     execution-order
     hits/calls
     hons-calls
     pons-calls
     number-of-calls
     number-of-hits
     number-of-mht-calls
     symbol-name-order
     time-for-non-hits/call
     time/call
     total-time.
  ")

(defg *memoize-summary-limit* 20

  "*MEMOIZE-SUMMARY-LIMIT* is a raw Lisp variable whose value is the
  maximum number of functions upon which MEMOIZE-SUMMARY reports.  A
  nil value means report on all.")

(defg *memoize-summary-order-reversed* nil

  "*MEMOIZE-SUMMARY-ORDER-REVERSED* is a raw Lisp variable.  When it
  is not NIL, then MEMOIZE-SUMMARY reports on functions in order from
  least to greatest.")

(defun print-alist (alist separation)
  (check-type separation (integer 0))
  (setq alist
        (loop for x in alist collect
              (progn
                (check-type x
                            (cons string
                                  (cons (or string (integer 0))
                                        null)))
                (list (car x)
                      (if (integerp (cadr x))
                          (ofnum (cadr x))
                        (cadr x))))))
  (let* ((max1 (loop for x in alist maximize (length (car x))))
         (max2 (loop for x in alist maximize (length (cadr x))))
         (width (max (or *print-right-margin* 70)
                     (+ separation max1 max2))))
    (loop for x in alist do
          (fresh-line)
          (princ (car x))
          (loop for i fixnum
                below (the fixnum
                        (- width (the fixnum (+ (length (car x))
                                                (length (cadr x))))))
                do (write-char #\Space))
          (princ (cadr x))))
  nil)

(defun hons-statistics ()

  "HONS-STATISTICS is a raw Lisp function (HONS-STATISTICS) prints
  detailed info about the hons system."

  (our-syntax-nice
   (oft "~&; Examining *hons-cdr-ht*:")
   (maphash (lambda (key value)
              (oft "~&; To ~s has been honsed:~%" key)
              (cond ((not (listp value))
                     (maphash (lambda (key v2) (declare (ignore v2))
                                (oft "~s, " key))
                              value))
                    (t (loop for pair in value do
                             (oft "~s, " (car pair))))))
            *hons-cdr-ht*)
   (oft "~&; End of *hons-cdr-ht* examination. ~
         ~%~%; Examining *hons-cdr-ht-eql*:")
   (maphash (lambda (key value)
              (oft "~%; To ~s has been honsed:~%" key)
              (cond ((not (listp value))
                     (maphash (lambda (key v2) (declare (ignore v2))
                                (oft "~s, " key))
                              value))
                    (t (loop for pair in value do
                             (oft "~s, " (car pair))))))
            *hons-cdr-ht-eql*)
   (oft "~%; End of *hons-cdr-ht-eql* examination. ~
         ~%~%; Examining *nil-ht*:~%")
   (oft "; To NIL has been honsed:~%")
   (maphash (lambda (key v2) (declare (ignore v2))
              (oft "~s, " key))
            *nil-ht*)
   (oft "~% End of *nil-ht* examination.")))

(defun hons-count ()
  (let ((n 0))
    (declare (fixnum n))
    (loop for tab in '(*hons-cdr-ht* *hons-cdr-ht-eql*) do
          (maphash (lambda (k v) (declare (ignore k))
                     (cond ((not (listp v))
                            (safe-incf n (hash-table-count v)
                                       hons-count))
                           (t (safe-incf n (length v) hons-count))))
                   (symbol-value tab)))
    (+ n (hash-table-count *nil-ht*))))

(defun hons-summary ()

; Documentation in acl2-mods.lisp.

  (our-syntax-nice
   (oft "(defun hons-summary~%")
   (let ((sssub 0) (nhonses 0) (nsubs 0))
     (declare (fixnum sssub nhonses nsubs))
     (loop for tab in '(*hons-cdr-ht* *hons-cdr-ht-eql*) do
           (maphash
            (lambda (k v) (declare (ignore k))
              (cond
               ((not (listp v))
                (safe-incf sssub (hash-table-size v) hons-summary)
                (safe-incf nhonses (hash-table-count v) hons-summary)
                (safe-incf nsubs 1 hons-summary))
               (t (safe-incf nhonses (length v) hons-summary))))
            (symbol-value tab)))
     (safe-incf nhonses (hash-table-count *nil-ht*) hons-summary)
     (print-alist
      `(,@(if (> *hons-call-counter* 0)
              `((" Hons hits/calls"
                 ,(let* ((c *hons-call-counter*)
                         (d (- c *hons-misses-counter*)))
                    (ofn "~,1e / ~,1e = ~,2f" d c (float (/ d c)))))))
          ,@(loop for tab in '(*hons-cdr-ht*
                               *hons-cdr-ht-eql*
                               *nil-ht*
                               *hons-str-ht*)
                  unless (eql 0 (hash-table-size (symbol-value tab)))
                  collect
                  (let* ((tabv (symbol-value tab))
                         (c (hash-table-count tabv))
                         (s (hash-table-size tabv)))
                    (list (ofn " ~a count/size"
                               (symbol-name tab))
                          (ofn "~,1e / ~,1e = ~,2f"
                               (ofnum c) (ofnum s) (float (/ c s))))))
          (" Number of sub tables" ,nsubs)
          (" Sum of sub table sizes" ,sssub)
          (" Number of honses" ,nhonses))
      5))
   (oft ")"))
  nil)

(defun pons-summary ()
  (our-syntax-nice
   (let ((sssub 0) (nponses 0) (nsubs 0) (nponstab 0))
     (declare (fixnum sssub nponses nsubs))
   (oft "(defun pons-summary~%")
   (maphash
    (lambda (k v)
      (cond ((symbolp k)
             (let ((tab (symbol-value (access memoize-info-ht-entry v :ponstablename))))
               (when tab
                 (safe-incf nponstab 1 pons-summary)
                 (maphash
                  (lambda (k v) (declare (ignore k))
                    (cond
                     ((not (listp v))
                      (safe-incf sssub (hash-table-size v) pons-summary)
                      (safe-incf nponses (hash-table-count v) pons-summary)
                      (safe-incf nsubs 1 pons-summary))
                     (t (safe-incf nponses (length v) pons-summary))))
                  tab))))))
    *memoize-info-ht*)
   (print-alist
    `((" Pons hits/calls"
       ,(let* ((c *pons-call-counter*)
               (d *pons-hit-counter*))
          (ofn "~,1e / ~,1e = ~,2f" d c (/ d (+ .0000001 c)))))
      (" Number of pons tables" ,nponstab)
      (" Number of pons sub tables" ,nsubs)
      (" Sum of pons sub table sizes" ,sssub)
      (" Number of ponses" ,nponses))
    5)
   (oft ")")
   nil)))

(defun memoize-statistics (&optional (fn (memoized-functions)))

  "(MEMOIZE-STATISTICS fn) prints all the memoized values for fn."

  (cond ((listp fn) (mapc #'memoize-statistics fn))
        ((not (memoizedp-raw fn))
         (oft "~%; Memoize-statistics:  ~s is not memoized." fn))
        (t (let ((tb (symbol-value
                      (access memoize-info-ht-entry
                              (gethash fn *memoize-info-ht*)
                              :tablename))))
             (cond ((and tb (not (eql 0 (hash-table-count tb))))
                    (oft "~%; Memoized values for ~s." fn)
                    (maphash (lambda (key v)
                               (format t "~%~s~%=>~%~s" key v))
                             tb))))))
  nil)

(defn print-call-stack ()

  "(PRINT-CALL-STACK) prints the stack of memoized function calls
  currently running and the time they have been running."

  (let (l (time (internal-real-time)))
    (declare (fixnum time))
    (maphash (lambda (k v)
               (cond ((symbolp k)
                      (let ((x (symbol-value
                                (the symbol
                                  (access memoize-info-ht-entry
                                          v :start-time)))))
                        (declare (fixnum x))
                        (when (> x 0)
                          (push (cons (symbol-name k) x) l))))))
             *memoize-info-ht*)
    (setq l (sort l #'< :key #'cdr))
    (setq l (loop for pair in l collect
                  (list (car pair)
                        (ofnum (/ (- time (cdr pair))
                                  *float-ticks/second*)))))
    (when l
      (terpri)
      (print-alist
       (cons '("Stack of memoized function calls."
               "Time since outermost call.")
             l)
       5))))

(defun hons-calls (x)

  "For a memoized function fn, (HONS-CALLS fn) is the number of times fn
   has called hons."

  (setq x (coerce-index x))
  (aref *watch-array*
        (the fixnum (+ *ma-hons-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

(defun pons-calls (x)

  "For a memoized function fn, (PONS-CALLS fn) is the number of times
   fn has called pons."

  (setq x (coerce-index x))
  (aref *watch-array*
        (the fixnum (+ *ma-pons-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

(defun bytes-allocated (x)

  "For a memoized function fn, (BYTES-ALLOCATED fn) is the number of
  bytes fn has caused to be allocated on the heap."

  (setq x (coerce-index x))
  (aref *watch-array*
        (the fixnum (+ *ma-bytes-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

(defun number-of-hits (x)

  "For a memoized function fn, (NUMBER-OF-HITS fn) is the number of
  times that a call of fn returned a remembered answer."

  (setq x (coerce-index x))
  (aref *watch-array*
        (the fixnum (+ *ma-hits-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

(defun number-of-mht-calls (x)

  "For a memoized function fn, (NUMBER-OF-MHT-CALLS fn) is the number
  of times that the memoize hash-table for fn was created."

  (setq x (coerce-index x))
  (aref *watch-array*
        (the fixnum (+ *ma-mht-index*
                       (the fixnum
                         (* *2max-memoize-fns*
                            (the fixnum x)))))))

(defun time-for-non-hits/call (x)
  (setq x (coerce-index x))
  (let ((n (- (number-of-calls x) (number-of-hits x))))
    (if (zerop n) 0 (/ (total-time x) n))))

(defun time/call (x)
  (setq x (coerce-index x))
  (let ((n (number-of-calls x)))
    (if (zerop n) 0 (/ (total-time x) n))))

(defun hits/calls (x)
  (setq x (coerce-index x))
  (let ((n (number-of-calls x)))
    (if (zerop n) 0 (/ (number-of-hits x) (float n)))))

(defun hons-acons-summary ()

  "(HONS-ACONS-SUMMARY) prints information about the existing fast
  hons alists."

  (let ((count 0) (size 0) (number 0) last-key alist)
    (declare (fixnum count size number))
    (maphash
     (lambda (key v)
       (cond ((typep v 'fixnum)
              (push (list
                     (if (setq last-key (cdr (last key)))
                         last-key number)
                     (len key)
                     (len key))
                    alist)
              (safe-incf number 1 hons-acons-summary)
              (safe-incf size (len key) hons-acons-summary)
              (safe-incf count (len key) hons-acons-summary))
             (t (push (list
                       (if (setq last-key (cdr (last key)))
                           last-key number)
                       (hash-table-size v)
                       (hash-table-count v))
                      alist)
                (safe-incf number 1 hons-acons-summary)
                (safe-incf size (hash-table-size v)
                           hons-acons-summary)
                (safe-incf count (hash-table-count v)
                           hons-acons-summary))))
     *hons-acons-ht*)
    (cond (alist
           (oft "~&~%Hons-acons statistics")
           (print-alist
            (list (list "Count/size"
                        (ofn "~,1e/~,1e = ~,2f"
                             (ofnum (hash-table-count
                                   *hons-acons-ht*))
                             (ofnum (hash-table-size
                                   *hons-acons-ht*))
                             (/ (hash-table-count *hons-acons-ht*)
                                (hash-table-size
                                 *hons-acons-ht*))))
                  (list "Total of counts" count)
                  (list "Total of sizes" size))
            5)
           (oft "~&~%For each HONS-ACONS entry~%(name size count)")
           (loop for x in alist do (print x))))))

(defun bytes-allocated/call (x)
  (setq x (coerce-index x))
  (let ((n (number-of-calls x)))
    (if (eql n 0)
        0
      (/ (bytes-allocated x) n))))

(defn char-list-fraction (l)
  (if (atom l) 0
    (+ (char-code (car l))
       (/ (char-list-fraction (cdr l))
          256))))

(defn symbol-name-order (s)

  "SYMBOL-NAME-ORDER maps symbols to rationals preserving
  lexicographic order."

  (unless (symbolp s) (setq s (fixnum-to-symbol s)))
  (- (char-list-fraction (coerce (symbol-name s) 'list))))

(defun execution-order (s)
  (unless (symbolp s) (setq s (fixnum-to-symbol s)))
  (the fixnum (symbol-value
               (the symbol
                 (access memoize-info-ht-entry
                         (gethash s *memoize-info-ht*)
                         :start-time)))))

(defn compute-calls-and-times ()
  (let ((ma *watch-array*)
        (2m *2max-memoize-fns*)
        (ca *compute-array*)
        (n (the fixnum (1+ *max-symbol-to-fixnum*))))
    (declare (type (simple-array fixnum (*)) ma)
             (type (simple-array t (*)) ca)
             (fixnum 2m n))
    (cond ((eql (length ca) n)
           (loop for i fixnum below n
                 do (setf (aref ca i) nil)))
          (t (setq *compute-array*
                   (make-array n :initial-element nil))
             (setq ca *compute-array*)))
    (loop for i fixnum below (the fixnum (* 2 n))
          do (setf (aref ma i) 0))
    (loop for i fixnum
          from *ma-initial-max-symbol-to-fixnum*
          to *max-symbol-to-fixnum* do
          (let ((2im (the fixnum (* i 2m))))
            (declare (fixnum 2im))
            (loop for j fixnum
                  from *ma-initial-max-symbol-to-fixnum*
                  to *max-symbol-to-fixnum* do
                  (let* ((2j (the fixnum (* 2 j)))
                         (index (the fixnum (+ 2j 2im))))
                    (declare (fixnum 2j index))
                    (let ((calls (the fixnum (aref ma index))))
                      (declare (fixnum calls))
                      (when (> calls 0)
                        (let ((time (aref ma (the fixnum
                                               (1+ index)))))
                          (declare (fixnum time))
                          (safe-incf (aref ma 2j)
                                     calls
                                     compute-calls-and-times)
                          (safe-incf (aref ma (the fixnum (1+ 2j)))
                                     time
                                     compute-calls-and-times)
                          (push i (aref ca j)))))))))))

(defun number-of-calls (x)
  (setq x (coerce-index x))

; One must call COMPUTE-CALLS-AND-TIMES before invoking
; NUMBER-OF-CALLS to get sensible results.

  (aref *watch-array*
        (the fixnum (* 2 (the fixnum x)))))

(defun total-time (x)

  (setq x (coerce-index x))

; One must call COMPUTE-CALLS-AND-TIMES before invoking
; TOTAL-TIME to get sensible results.

  (/ (aref *watch-array*
           (the fixnum (1+ (the fixnum (* 2 x)))))
     *float-ticks/second*))

(defun list-fast-fns (ticks)
  (let ((ma *watch-array*))
    (declare (type (simple-array fixnum (*)) ma))
    (compute-calls-and-times)
    (sort
     (loop for i fixnum from (1+ *ma-initial-max-symbol-to-fixnum*)
           to *max-symbol-to-fixnum*
           when (and
                 (let* ((2i (* 2 i))
                        (calls (aref ma 2i))
                        (time (aref ma (the fixnum (1+ 2i)))))
                   (declare (fixnum 2i calls time))
                   (and (>= calls 1000)
                        (< time (* calls ticks))))
                 (let ((l (gethash (gethash i *memoize-info-ht*)
                                   *memoize-info-ht*)))
                   (and l (null (access memoize-info-ht-entry l
                                        :condition)))))
           collect (fixnum-to-symbol i))
     #'>
     :key #'symbol-name-order)))

(defn lex-> (l1 l2)
  (cond ((or (atom l1)
             (atom l2))
         nil)
        ((> (car l1) (car l2)) t)
        ((< (car l1) (car l2)) nil)
        (t (lex-> (cdr l1) (cdr l2)))))

(defun memoize-summary-sort ()
  (let (pairs)
    (maphash
     (lambda (fn v)
       (when (symbolp fn)
       (let ((num (access memoize-info-ht-entry v :num)))
         (declare (fixnum num))
         (when (< 0 (number-of-calls num))
           (push (cons fn (loop for order
                                in *memoize-summary-order-list*
                                collect (funcall order fn)))
                 pairs)))))
     *memoize-info-ht*)
    (sort pairs
          (if *memoize-summary-order-reversed*
              (lambda (x y) (lex-> y x))
            #'lex->)
          :key #'cdr)))

(defun memoize-summary ()

; Documentation in acl2-mods.lisp.

  (compute-calls-and-times)
  (memoize-summary-after-compute-calls-and-times))

(defun memoize-summary-after-compute-calls-and-times ()
  
;  If COMPUTE-CALLS-AND-TIMES is not called shortly before this
;  function, MEMOIZE-SUMMARY-AFTER-COMPUTE-CALLS-AND-TIMES, is called,
;  the information reported may be quite untimely.

;  See the end of trace-emod.lisp for documentation.

 (let* ((fn-pairs (memoize-summary-sort))
        (orig-len-fn-pairs (length fn-pairs))
        (len-fn-pairs orig-len-fn-pairs)
        (ma *watch-array*))
   (declare (type (simple-array fixnum (*)) ma))
   (when (and *memoize-summary-limit*
              (> orig-len-fn-pairs *memoize-summary-limit*))
     (setq fn-pairs
           (loop for i from 1 to *memoize-summary-limit* as
                 x in fn-pairs collect x))
     (setq len-fn-pairs *memoize-summary-limit*))
   (oft "*memoize-summary-order-list* => ~s~
         ~%*memoize-summary-limit* => ~s~
         ~%| ~s functions memoized, ~s called, ~a~s listed |"
        *memoize-summary-order-list*
        *memoize-summary-limit*
        (length-memoized-functions)
        orig-len-fn-pairs
        (if (< len-fn-pairs orig-len-fn-pairs)
            "top "
          "")
        len-fn-pairs)
   (loop for pair in fn-pairs do
         (let* ((fn (car pair))
                (l (gethash fn *memoize-info-ht*))
                (tablename
                 (symbol-value (access memoize-info-ht-entry l
                                       :tablename)))
                (ponstablename (symbol-value
                                (access memoize-info-ht-entry
                                        l :ponstablename)))
                (start-time
                 (the fixnum
                   (symbol-value
                    (the symbol
                      (access memoize-info-ht-entry
                              l :start-time)))))
                (num (the fixnum
                       (access memoize-info-ht-entry l :num)))
                (nhits (the fixnum (number-of-hits num)))
                (nmht (the fixnum (number-of-mht-calls num)))
                (ncalls (the fixnum
                          (max (the fixnum (number-of-calls num))
                               1)))
                (hons-calls (the fixnum (hons-calls num)))
                (pons-calls (the fixnum (pons-calls num)))
                (heap-bytes-allocated (bytes-allocated num))
                (tt (total-time num))
                (t/c (time/call num))
                (tnh (time-for-non-hits/call num))
                (in-progress-str
                 (if (eql start-time -1) " " ", running, ")))
           (declare (fixnum start-time num nhits nmht ncalls
              hons-calls pons-calls heap-bytes-allocated))
           (print-alist
            `((,(ofn "(defun ~a~a~a"
                     (symbol-name fn)
                     (if (or (eql 0 nhits)
                             (not *report-hits*))
                         (ofn " call~a"
                              (if (eql nhits 1) "" "s"))
                       " hits/calls")
                     in-progress-str)
               ,(if (or *report-calls* *report-hits*)
                    (if (or (eql 0 nhits)
                            (not *report-hits*))
                        (ofn "~a" (ofnum ncalls))
                      (ofn "~a / ~a = ~a"
                           (ofnum nhits)
                           (ofnum ncalls)
                           (ofnum (/ nhits (float ncalls)))))
                  ""))
              ,@(if (and *report-mht-calls* (>= nmht 2))
                    `(("Number of calls to mht" ,(ofnum nmht))))
              ,@(if (and *report-time* (> tt 0))
                      `((" Time of all outermost calls"
                         ,(ofnum tt))
                        (" Time per outermost call"
                         ,(ofnum t/c))))
              ,@(if (and (> heap-bytes-allocated 0) *report-bytes*)
                `((" Heap bytes allocated"
                    ,(ofnum heap-bytes-allocated))
                   (" Heap bytes allocated per outermost call"
                    ,(ofnum (/ heap-bytes-allocated ncalls)))))
              ,@(if (and (> hons-calls 0) *report-hons-calls*)
               `((" Hons calls"
                       ,(ofnum hons-calls))))
              ,@(if (and (> pons-calls 0) *report-pons-calls*)
                `((" Pons calls"
                       ,(ofnum pons-calls))))
              ,@(if (and *report-hits* *report-time*
                         (not (eql 0 nhits)))
                    `((" Time per missed outermost call"
                       ,(ofnum tnh))))
              ,@(if *report-calls-from*
                    (let (l (2num (the fixnum (* 2 num))))
                      (declare (fixnum 2num))
                      (loop for callern fixnum
                            in (aref *compute-array* num) do
                            (let* ((call-loc
                                    (the fixnum
                                      (+ 2num
                                         (the fixnum
                                           (* callern
                                              *2max-memoize-fns*)))))
                                   (calls (aref ma call-loc))
                                   (time 0))
                             (declare (fixnum call-loc calls time))
                             (push
                              `((,(ofn " From ~a"
                                   (if (<= callern
                                    *ma-initial-max-symbol-to-fixnum*)
                                           "outside"
                                         (fixnum-to-symbol callern)))
                                 ,(ofn "~a call~a~a"
                                       (ofnum calls)
                                       (if (= calls 1) "" "s")
                                       (let ((time-loc
                                              (the fixnum
                                                (1+ call-loc))))
                                         (declare (fixnum time-loc))
                                         (if (> (setq time
                                                      (aref ma time-loc))
                                                0)
                                             (ofn " took ~a"
                                                  (ofnum
                                                   (/ time
                                                      *float-ticks/second*)))
                                           ""))))
                                . ,calls)
                              l)))
                      (setq l (sort l #'> :key #'cdr))
                      (strip-cars l)))
              .
              ,(if (not *report-on-hons-and-pons-tables*) nil
                 (let ((spsub 0) (nponses 0) (npsubs 0))
                   (declare (fixnum spsub nponses npsubs))
                   (and
                    ponstablename
                    (maphash
                     (lambda (key value)
                       (declare (ignore key))
                       (cond
                        ((not (listp value))
                         (safe-incf spsub (hash-table-size value)
                                    memoize-summary)
                         (safe-incf nponses
                                    (hash-table-count value)
                                    memoize-summary)
                         (safe-incf npsubs 1 memoize-summary))
                        (t (safe-incf nponses
                                      (length value)
                                      memoize-summary))))
                     ponstablename))
                   `(,@(and
                        tablename
                        `((,(ofn " (Memoize table count/size")
                           ,(ofn "~a / ~a = ~a)"
                                 (ofnum (hash-table-count tablename))
                                 (ofnum (hash-table-size tablename))
                                 (ofnum
                                  (/ (hash-table-count tablename)
                                     (+ .001 (hash-table-size
                                              tablename))))))))
                       ,@(and
                          ponstablename
                          `((" (Pons table count/size"
                             ,(ofn "~a / ~a = ~a)"
                                   (ofnum (hash-table-count
                                           ponstablename))
                                   (ofnum (hash-table-size
                                           ponstablename))
                                   (ofnum (/ (hash-table-count
                                              ponstablename)
                                             (+ .001
                                                (hash-table-size
                                                 ponstablename))))))
                            (" (Number of pons sub tables"
                             ,(ofn "~a)" npsubs))
                            (" (Sum of pons sub table sizes"
                             ,(ofn "~a)" spsub))
                            (" (Number of ponses"
                             ,(ofn "~a)" nponses))))))))
            5))
         (oft ")"))))

;  CLEARING FOR HONS AND MEMOIZE TABLES

(defun clear-hash-tables ()
  (check-no-extra-processes)
  (labels ((rehons (x)
                   (cond ((and (consp x)
                               (not (eq x (hons-copy1-consume x))))
                          (setq *rehons-culprit* x)
                          (ofe "~%; clear-hash-tables: Error: ** ~
                                failed to rehons *rehons-culprit*."))
                         (t (maybe-str-hash x)))))

; See also hons.lisp.

; Corrected definition, by Sol.

; Should this function save any honses from *hons-acons-ht*?
; Currently it saves not only persistent honses, but also honses that
; are keys of, or keys of keys of, the *hons-acons-ht*.  An optional
; argument could presumably be added to make these additional saves
; optional (but then what should the default be for this optional
; argument?).

; hons-let assumes that clrhash is not used here.

    (our-lock-unlock-memoize1
     (let (l)
       (maphash (lambda (k v)
                  (declare (ignore v))
                  (if (honsp k) (push k l)))
                *hons-acons-ht*)
       (setq *hons-cdr-ht* (mht ; :size *hons-cdr-ht-size*
                            :test #'eq
                            :weak :key))
       (setq *hons-cdr-ht-eql* (mht))
       (setq *nil-ht* (mht ; :size *nil-ht-size*
                       :weak :value))
       (loop for x in (table-alist 'persistent-hons-table
                                   (w *the-live-state*))
             when (car x)
             do (rehons (car x)))

       (loop for x in *hons-vars-alist* do (rehons (car x)))

       (mapc #'rehons l)
       (maphash (lambda (k v)
                  (cond ((and (consp k) (honsp k))
                         ;; all parts of k are already honsed.
                         nil)
                        ((integerp v)
                         (loop for tail on k while (consp tail) do
                               (rehons (caar tail))))
                        (t (maphash (lambda (k v)
                                      (declare (ignore v))
                                      (rehons k))
                                    v))))
                *hons-acons-ht*)))))
     
(defun empty-ht-p (x)
  (and (hash-table-p x)
       (eql 0 (hash-table-count x))))

(defn clear-one-memo-and-pons-hash (l)
  (setf (symbol-value
         (the symbol
           (access memoize-info-ht-entry l :tablename)))
        nil)
  (setf (symbol-value
         (the symbol
           (access memoize-info-ht-entry l :ponstablename)))
        nil))

(defun clear-memoize-table (k)

; See also hons.lisp.

  (when (symbolp k)
    (let ((l (gethash k *memoize-info-ht*)))
      (when l (clear-one-memo-and-pons-hash l)))))


(defun clear-memoize-tables ()

; See hons.lisp.

  (let (success)
    (unwind-protect
        (progn
          (maphash (lambda (k l)
                     (when (symbolp k)
                       (clear-one-memo-and-pons-hash l)))
                   *memoize-info-ht*)
          (setq success t))
      (or success (ofe "~%; clear-memoize-tables: Error. **"))))
  nil)

(defn clear-watch-array ()
  (let ((m *watch-array*))
    (declare (type (simple-array fixnum (*)) m))
    (loop for i fixnum below (length m)
          unless (eql (aref m i) 0)
          do (setf (aref m i) 0))))

(defun flush-hons-get-hash-table-link (x)

  ;  See hons.lisp.

  (unless (atom x) (remhash x *hons-acons-ht*))
  x)


; HONS READ

; Hash consing when reading is implemented via a change to the
; readtable for the characters open parenthesis, close parenthesis,
; and period, the consing dot.

; *** NOTE:  The following collection of functions is just that: a
;            collection.  Unless you understand everything about the
;            various read table macros, then don't touch this code!

; See matching comment below.

; Note: our implementation of the #=/## reader, which we built because
; some Lisps would not us get past #500 or so, does not comply with
; ANSI at least in this regard: it does not allow the entry of looping
; structures as in '(#0= (3 #0#)), which is no problem for ACL2 users.

; WARNING: Any call of READ using *hons-readtable* as *readtable*
; needs to worry about the possible top-level return of
; *CLOSE-PAREN-OBJ* and *DOT-OBJ*, which are simply part of the
; fiction whereby we read while honsing.  Those two objects should
; absolutely not be returned as the value of an ACL2 function.  See,
; for example, the definition of HONS-READ.

(defg *close-paren-obj* '(#\)))

(defg *dot-obj*         '(#\.))

(defg *hons-readtable* (copy-readtable *acl2-readtable*))
(declaim (readtable *acl2-readtable* *hons-readtable*))

(defg *compact-print-file-ht* (mht))

(defg *compact-read-file-ht* (mht))

(defg *compact-print-file-n* 0)
(declaim (fixnum *compact-print-file-n*))

(defparameter *space-owed* nil)
(declaim (type boolean *space-owed*))

(defg *compact-read-file-readtable*
  (copy-readtable *hons-readtable*))
(declaim (readtable *compact-read-file-readtable*))

(defg *compact-read-init-done* nil)

(defg *hons-readtable-init-done* nil)

(defun nonsense (x)
  (or (eq x *close-paren-obj*) (eq x *dot-obj*)))

(defun check-nonsense (x)
  (cond ((nonsense x)
         (hread-error "~&;  Illegal object: ~s." (car x)))))

(defun hread-error (string &rest r)
  (our-syntax-nice
   (let* ((stream *standard-input*)
          (*standard-output* *error-output*)
          (*standard-input* *debug-io*))
     (apply #'format *error-output* string r)
     (cond ((and (streamp stream) (file-position stream))
            (format *error-output*
                    "~&; near file-position ~s in stream ~s."
                    (file-position stream) stream)))
     (ofe "hread"))))

(defun illegal-error1 (x)
  (hread-error "~&; ** Error:  Illegal:  ~s." x))

(defun illegal-error2 (stream char)
  (declare (ignore stream))
  (illegal-error1 char))

(defun close-paren-read-macro (stream char)
  (declare (ignore char stream))
  (if *read-suppress* (illegal-error1 #\)))
  *close-paren-obj*)

(defun dot-read-macro (stream char)
  (declare (ignore char))
  (if *read-suppress* (illegal-error1 #\.))
  (let ((ch (peek-char nil stream t nil t)))
    (cond ((or (member ch '(#\( #\) #\' #\` #\, #\" #\;
                            #\Tab #\Space #\Newline))
               (eql 13 (char-code ch))
               (multiple-value-bind (fn nonterminating)
                   (get-macro-character ch)
                 (and fn (not nonterminating))))
           *dot-obj*)
          (t (let ((*readtable* *acl2-readtable*))
               (unread-char #\. stream)
               (read nil t nil t))))))

(defun hons-read-list ()

  ; Do not call this unless you are sure there is a superior lock of
  ; *hons-cdr-ht*.

  (let ((o (read nil t nil t)))
    (cond
     ((eq o *close-paren-obj*) nil)
     ((eq o *dot-obj*)
      (let ((lo (read nil t nil t))
            (lp (read nil t nil t)))
        (check-nonsense lo)
        (cond
         ((eq lp *close-paren-obj*) lo)
         (t (illegal-error1 #\.)))))
     (t (hons-normed (maybe-str-hash o) (hons-read-list))))))

(defun hons-read-list-top ()

  ; Do not call this unless you are sure there is a superior lock of
  ; *hons-cdr-ht*.

  (let ((o (read nil t nil t)))
    (cond
     ((eq o *close-paren-obj*) nil)
     (t (check-nonsense o)
        (hons-normed (maybe-str-hash o)
                     (hons-read-list))))))

(defun hons-read-reader (stream char)

  ; Do not call this unless you are sure there is a superior lock of
  ; *hons-cdr-ht*.


  (declare (ignore char))
  (cond (*read-suppress*
         (unread-char #\( stream)
         (let ((*readtable* *acl2-readtable*))
           (read nil t nil t)))
        (t (hons-read-list-top))))

(defun hons-read (&optional i (eep t) eofv rp)

  ; Do not call this unless you are sure there is a superior lock of
  ; *hons-cdr-ht*.

  (let* ((*readtable* *hons-readtable*)
         (*standard-input* (or i *standard-input*)))
    (let ((x (read nil eep eofv rp)))
      (check-nonsense x)
      x)))

(defun hons-read-object (channel state-state)
  (our-lock-unlock-honsmv1
   (let ((*acl2-readtable* *hons-readtable*))
     (mv-let
      (eofp obj state)
      (read-object channel state-state)
      (check-nonsense obj)
      (mv eofp obj state)))))

(defun hons-read-file (file-name)
  (our-lock-unlock-hons1
      (with-open-file (file file-name)
        (let (temp ans (eof (cons nil nil)))
          (declare (dynamic-extent eof))
          (loop (setq temp (hons-read file nil eof nil))
                (if (eq eof temp)
                    (return (hons-copy1-consume-top (nreverse ans))))
                (setq ans (cons temp ans)))))))


;  COMPACT PRINT AND READ

(defmacro space-if-necessary ()

; do not call

  '(when *space-owed* (write-char #\Space) (setq *space-owed* nil)))

(defun compact-print-file-scan (x)

; do not call

  (unless (or (and (symbolp x)
                   (let ((p (symbol-package x)))
                     (or (eq p *main-lisp-package*)
                         (eq p *package*)))
                   (<= (length (symbol-name x)) 4))
              (and (stringp x) (<= (length x) 2))
              (and (integerp x) (< -100 x 1000))
              (characterp x))
    (let ((g (gethash x *compact-print-file-ht*)))
      (unless (or (atom x) (eq g 'give-it-a-name))
        (compact-print-file-scan (car x))
        (compact-print-file-scan (cdr x)))
      (unless (eq g 'give-it-a-name)
        (setf (gethash x *compact-print-file-ht*)
              (if g 'give-it-a-name 'found-at-least-once))))))

(defun compact-print-file-help (x hash)

; do not call

  (cond ((typep hash 'fixnum)
         (space-if-necessary)
         (write-char #\#)
         (princ hash)
         (write-char #\#))
        (t (cond ((eq hash 'give-it-a-name)
                  (let ((n *compact-print-file-n*))
                    (declare (fixnum n))
                    (when (eql n most-positive-fixnum)
                      (ofe "~&; *compact-print-file-n* overflow."))
                    (setq n (the fixnum (+ 1 n)))
                    (setq *compact-print-file-n* n)
                    (setf (gethash x *compact-print-file-ht*) n)
                    (space-if-necessary)
                    (write-char #\#)
                    (princ n)
                    (write-char #\=))))
           (cond
            ((atom x)
             (space-if-necessary)
             (prin1 x)
             (setq *space-owed* t))
            (t (write-char #\()
               (setq *space-owed* nil)
               (loop (compact-print-file-help
                      (car x)
                      (gethash (car x) *compact-print-file-ht*))
                     (cond
                      ((null (cdr x))
                       (write-char #\))
                       (setq *space-owed* nil)
                       (return))
                      ((or (progn
                             (setq hash
                                   (gethash (cdr x)
                                            *compact-print-file-ht*))
                             (or (eq hash 'give-it-a-name)
                                 (typep hash 'fixnum)))
                           (atom (cdr x)))
                       (space-if-necessary)
                       (write-char #\.)
                       (setq *space-owed* t)
                       (compact-print-file-help (cdr x) hash)
                       (write-char #\))
                       (setq *space-owed* nil)
                       (return))
                      (t (pop x)))))))))

(defun compact-print-file (data file-name)
  (our-lock-unlock-hons1 
   (progn
     (setq *compact-print-file-ht* (mht))
     (setq *compact-print-file-n* 0)
     (setq *space-owed* nil)
     (with-open-file (*standard-output* file-name
                                        :direction :output
                                        :if-does-not-exist :create
                                        :if-exists :supersede)
       (our-syntax
        (compact-print-file-scan data)
        (compact-print-file-help
         data (gethash data *compact-print-file-ht*))
        (namestring (truename *standard-output*)))))))
  
(defun ns-=-reader (stream subchar arg)

; do not call

  (declare (ignore stream subchar))
  (when (gethash arg *compact-read-file-ht*)
    (hread-error
     "~&; ns-=-reader:  ** Error:  #~s= is already defined to be ~s."
     arg (gethash arg *compact-read-file-ht*)))
  (setf (gethash arg *compact-read-file-ht*) (read nil t nil t)))

(defun ns-ns-reader (stream subchar arg)

; do not call

  (declare (ignore stream subchar))
  (or (gethash arg *compact-read-file-ht*)
      (hread-error "~&; ns-ns-reader:  ** Error:  meaningless #~s#."
                   arg)))

(defun compact-read-file (fn)
  (our-lock-unlock-hons1
   (progn
     (setq *compact-read-file-ht* (mht))
     (with-open-file (*standard-input* fn)
       (our-syntax
        (let* ((*readtable* *compact-read-file-readtable*)
               (eof (cons nil nil))
               (p (read)))
          (check-nonsense p)
          (unless (eq (read nil nil eof) eof)
            (ofe "~%; compact-read-file: ** Error:  ~s has too many ~
                forms." fn))
          (hons-copy1-consume-top p)))))))


;  HONS READTABLE INIT

(defun hons-readtable-init ()
  (setq *hons-readtable* (copy-readtable *acl2-readtable*))
  (let ((*readtable* *hons-readtable*))
    (set-macro-character #\( #'hons-read-reader)
    (set-macro-character #\) #'close-paren-read-macro)
    (set-macro-character #\. #'dot-read-macro t))
  (setq *hons-readtable-init-done* t))

; COMPACT READ INIT

(defun compact-read-init ()
  (setq *compact-read-file-readtable*
        (copy-readtable *hons-readtable*))
  (let ((*readtable* *compact-read-file-readtable*))
    (set-dispatch-macro-character #\# #\# #'ns-ns-reader)
    (set-dispatch-macro-character #\# #\= #'ns-=-reader))
  (setq *compact-read-init-done* t))


; MEMOIZE INIT

(defun memoize-init ()

; Should only be repeated by unmemoize-fn.

  (check-no-extra-processes)
  (let (success)
    (unwind-protect
        (progn
          (setq *pons-call-counter* 0)
          (setq *pons-hit-counter* 0)
          (setq *memoize-info-ht* (mht))
          (setf (gethash *ma-initial-max-symbol-to-fixnum*
                         *memoize-info-ht*)
                "outside-caller")
          (setq *max-symbol-to-fixnum*
                *ma-initial-max-symbol-to-fixnum*)
          (setq *2max-memoize-fns*
                (* 2 *initial-max-memoize-fns*))
          (sync-watch-array)
          (setq success t))
      (if success (setq *memoize-init-done* t)
        (ofd "~%; memoize init: Error **")))))



;  HONS INIT

; The ACL2 persistent-hons-table, which is updated by DEFHONST, should
; be an alist whose keys are honses to be preserved as honses through
; any cleaning of hash tables, but not through an init-hash-tables.

(defun init-hons-acons-table ()

; See also hons.lisp.

  (setq *hons-acons-ht* (mht :test #'eq :weak :key)))


(defun init-hash-tables ()

; See also hons.lisp.

  ; there can be no processes that might possibly do honsing alive at
  ; this point other than this one

  (let (success)
    (unwind-protect
        (progn
          (setq *hons-call-counter* 0)
          (setq *hons-misses-counter* 0)
          (setq *hons-cdr-ht* (mht))
          (setq *hons-cdr-ht* (mht :size *hons-cdr-ht-size*
                                   :weak :key :test #'eq))
          (setq *nil-ht* (mht :size *nil-ht-size* :weak :value))
          (init-hons-acons-table)
          (setq *hons-str-ht* (mht :test #'equal :weak :value))
          (setq *hons-copy-aux-ht* (mht :test #'eq))
          (setq success t))
      (if success (setq *init-hash-tables-done* t)
        (ofe "~%; init-hash-tables:  Error **.")))))
  
(defvar *hons-init-hook*
  '(progn
     #+openmcl

; Some things that an OpenMCL user might want to put into
; ~/openmcl-init.lisp but may not know to do so:

     (progn

       (defun set-gc-threshold (bound)
         (when (< (ccl::lisp-heap-gc-threshold) bound)
           (ofg "~&; *hons-init-hook*:  Setting ~
                 (ccl::lisp-heap-gc-threshold) to ~:d bytes."
                bound)
           (ccl::set-lisp-heap-gc-threshold bound)
           (ccl::use-lisp-heap-gc-threshold))
         nil)

       (defun maybe-set-gc-threshold (&optional (fraction 1/32))
         (let (n)
           (ignore-errors (setq n (physical-memory)))
           (cond ((and (integerp n) (> n (* 2 (expt 10 9))))
                  (setq n (floor (* n fraction)))
                  (set-gc-threshold n)))))

; Try to determine whether *terminal-io* is a file.

       (when (fboundp 'live-terminal-p)
         (eval '(live-terminal-p)))

; It is important to watch what the gc is doing when using hons and
; memoize.

       (unless (equal '(t t)
                      (multiple-value-list (ccl::gc-verbose-p)))
         (ofv "~&; *hons-init-hook*:  Setting OpenMCL's gc verbose.")
         (ccl::gc-verbose t t))

; OpenMCL's ephemeral gc doesn't work well with honsing and memoizing,
; it seems.

       (when (ccl::egc-active-p)
         (ofv "~&; *hons-init-hook*:  Turning off OpenMCL's ~
                ephemeral gc.")
         (ccl::egc nil))

; Allocate heap space perhaps a little more generously than OpenMCL's
; default.

       (maybe-set-gc-threshold)

; Allow control-d to exit.

       (when (and (boundp 'ccl::*quit-on-eof*)
                  (not (eq t (symbol-value 'ccl::*quit-on-eof*))))
         (ofv "~&; *hons-init-hook*:  Control-d now exits OpenMCL.")
         (setf (symbol-value 'ccl::*quit-on-eof*) t))

       (ccl::add-gc-hook
        (lambda () (setq *gc-msg-for-our-gc* t)
          :post-gc))
       nil)

; Sol Sword's scheme to control GC in OpenMCL.  See long comment below.

     #+(and openmcl sol)
     (progn
       ;; Trigger GC after we've used all but (1/8, but not more than 1 GB)
       ;; of the physical memory.
       (defg *max-mem-usage*
         (let ((phys (physical-memory)))
           (max (floor (* 7/8 phys))
                (- phys (expt 2 30)))))

       (defg *gc-min-threshold* (expt 2 30))

       (defun set-and-reset-gc-thresholds ()
         (let ((n (max (- *max-mem-usage* (ccl::%usedbytes))
                       *gc-min-threshold*)))
           (cond ((not (eql n (ccl::lisp-heap-gc-threshold)))
                  (ccl::set-lisp-heap-gc-threshold n)
                  (ofg "~&; set-and-reset-gc-thresholds: Setting ~
                       (lisp-heap-gc-threshold) to ~:d bytes.~%"
                       n))))
         (ccl::use-lisp-heap-gc-threshold)
         (ofg "~&; set-and-reset-gc-thresholds: Calling ~
              ~%(use-lisp-heap-gc-threshold).")
         (cond ((not (eql *gc-min-threshold*
                          (ccl::lisp-heap-gc-threshold)))
                (ccl::set-lisp-heap-gc-threshold *gc-min-threshold*)
                (ofg "~&; set-and-reset-gc-thresholds: Setting ~
                       (lisp-heap-gc-threshold) to ~:d bytes.~%"
                     *gc-min-threshold*))))

       (ccl::add-gc-hook
        #'(lambda ()
            (ccl::process-interrupt
             (slot-value ccl:*application*
                         'ccl::initial-listener-process)
             #'set-and-reset-gc-thresholds))
        :post-gc)

       (set-and-reset-gc-thresholds)

       )

     ))

#||

         Sol Sword's scheme to control GC in OpenMCL

The goal is to get OpenMCL to perform a GC whenever we're using almost
all the physical memory, but not otherwise.

The usual way of controlling GC on OpenMCL is via
LISP-HEAP-GC-THRESHOLD.  This value is approximately amount of memory
that will be allocated immediately after GC.  This means that the next
GC will occur after LISP-HEAP-GC-THRESHOLD more bytes are used (by
consing or array allocation or whatever.)  But this means the total
memory used by the time the next GC comes around is the threshold plus
the amount that remained in use at the end of the previous GC.  This
is a problem because of the following scenario:

 - We set the LISP-HEAP-GC-THRESHOLD to 3GB since we'd like to be able
   to use most of the 4GB physical memory available.

 - A GC runs or we say USE-LISP-HEAP-GC-THRESHOLD to ensure that 3GB is
   available to us.

 - We run a computation until we've exhausted this 3GB, at which point
   a GC occurs.  

 - The GC reclaims 1.2 GB out of the 3GB used, so there is 1.8 GB
   still in use.

 - After GC, 3GB more is automatically allocated -- but this means we
   won't GC again until we have 4.8 GB in use, meaning we've gone to
   swap.

What we really want is, instead of allocating a constant additional
amount after each GC, to allocate up to a fixed total amount including
what's already in use.  To emulate that behavior, we use the hack
below.  This operates as follows (assuming the same 4GB total physical
memory as in the above example:)

1. We set the LISP-HEAP-GC-THRESHOLD to (3.5G - used bytes) and call
USE-LISP-HEAP-GC-THRESHOLD so that our next GC will occur when we've
used a total of 3.5G.

2. We set the threshold back to 1GB without calling
USE-LISP-HEAP-GC-THRESHOLD.

3. Run a computation until we use up the 3.5G and the GC is called.
Say the GC reclaims 1.2GB so there's 2.3GB in use.  1GB more (the
current LISP-HEAP-GC-THRESHOLD) is allocated so the ceiling is 3.3GB.)

4. A post-GC hook runs which again sets the threshold to (3.5G -
used bytes), calls USE-LISP-HEAP-GC-THRESHOLD to raise the ceiling to
3.5G, then sets the threshold back to 1GB, and the process repeats.

A subtlety about this scheme is that post-GC hooks runs in a separate
thread from the main execution.  A possible bug is that in step 4,
between checking the amount of memory in use and calling
USE-LISP-HEAP-GC-THRESHOLD, more memory might be used up by the main
execution, which would set the ceiling higher than we intended.  To
prevent this, we interrupt the main thread to run step 4.

||#

(defun hons-init ()

; HONS-INIT may be called more than once, so choose things to repeat
; carefully.

  (check-no-extra-processes)
  (eval *hons-init-hook*)
  (unless *init-hash-tables-done* (init-hash-tables))
  (unless *hons-readtable-init-done* (hons-readtable-init))
  (unless *memoize-init-done* (memoize-init))
  (unless *compact-read-init-done* (compact-read-init))
  (float-ticks/second-init)  

; Good idea to repeat float-ticks/second-init; speed varies from cpu
; to cpu.

  nil)

(defun all-module-names ()
  (loop for x in
        (sort (strip-cars (table-alist 'defm-table (w *the-live-state*)))
              (lambda (x y) (< (event-number x) (event-number y))))
        collect x))

(defun all-modules ()
  (loop for x in (all-module-names) collect (eval x)))

;;; SHORTER, OLDER NAMES

(defun memsum (&rest r)
  (apply #'memoize-summary r))
(defun hsum (&rest r)
  (apply #'hons-summary r))
(defun memstat (&rest r)
  (apply #'memoize-statistics r))
(defun hstat (&rest r)
  (apply #'hons-statistics r))
(defmacro memo-on (&rest r)
  `(memoize-on ,@r))
(defmacro memo-off (&rest r)
  `(memoize-off ,@r))
(defun clear-memo-tables (&rest r)
  (apply #'clear-memoize-tables r))
(defun clear-hash-and-memo-tables (&rest r)
  (apply #'clear-hash-and-memoize-tables r))

(defg *compiled-module-ht* (mht :test 'eq)
  
  "The hash table *COMPILED-MODULE-HT* maps a module name n to a cons
  of (1) the module named n with (2) the compiled version of module n,
  which is a compiled function, for primitive modules, and an array,
  for nonprimitive modules.  The car is EQ to the module that was the
  symbol value of n when the compilation was done.")
