/* **********************************************************
 * Copyright 2003 VMware, Inc.  All rights reserved.
 * -- VMware Confidential
 * **********************************************************/

/*
 * x86types.h --
 *
 *	Type definitions for the x86 architecture.
 */

#ifndef _X86TYPES_H_
#define _X86TYPES_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_VMMEXT
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMNIXMOD
#define INCLUDE_ALLOW_VMK_MODULE
#define INCLUDE_ALLOW_VMKERNEL
#define INCLUDE_ALLOW_DISTRIBUTE
#define INCLUDE_ALLOW_VMCORE
#define INCLUDE_ALLOW_VMMON
#include "includeCheck.h"

#include "vm_basic_types.h"
#include "vm_basic_defs.h"
#include "vmware_pack_init.h"

/*
 * Virtual, physical, machine address and page conversion macros
 *
 * XXX These should either go in vm_basic_types.h, where the types
 * are defined, or those definitions should be moved here.
 */

#define VA_2_VPN(_va)  ((_va) >> PAGE_SHIFT)
#define PTR_2_VPN(_ptr) VA_2_VPN((VA)(_ptr))
#define VPN_2_VA(_vpn) ((_vpn) << PAGE_SHIFT)

/*
 * Notice that we don't cast PA_2_PPN's argument to an unsigned type, because
 * we would lose compile-time checks for pointer operands and byte-sized
 * operands. If you use a signed quantity for _pa, ones may be shifted into the
 * high bits of your ppn.
 */

#define PA_2_PPN(_pa)  ((_pa) >> PAGE_SHIFT)
#define PPN_2_PA(_ppn)    ((PA)(_ppn) << PAGE_SHIFT)

static INLINE MA  MPN_2_MA(MPN mpn) { return (MA)mpn << PAGE_SHIFT; }
static INLINE MPN MA_2_MPN(MA  ma)  { return (MPN)(ma >> PAGE_SHIFT); }


/*
 * Types used for PL4 page table in x86_64
 */

typedef uint64 VM_L4E;
typedef uint64 VM_L3E;
typedef uint64 VM_L2E;
typedef uint64 VM_L1E;


/*
 * 4 Mb pages
 */

#define VM_LARGE_PAGE_SHIFT  22
#define VM_LARGE_PAGE_SIZE   (1 << VM_LARGE_PAGE_SHIFT)
#define VM_LARGE_PAGE_MASK   (VM_LARGE_PAGE_SIZE - 1)


/*
 * Page table
 */

typedef uint32 VM_PDE;
typedef uint32 VM_PTE;
typedef uint64 VM_PAE_PDE;
typedef uint64 VM_PAE_PTE;
typedef uint64 VM_PDPTE;


/*
 * Registers
 */

typedef  int8    Reg8;
typedef  int16   Reg16;
typedef  int32   Reg32;
typedef  int64   Reg64;

typedef uint8   UReg8;
typedef uint16  UReg16;
typedef uint32  UReg32;
typedef uint64  UReg64;

// only define these in the monitor where size is fixed
#if defined(VMM32) || defined(CQ32)
typedef  Reg32  Reg;
typedef UReg32 UReg;
#endif
#if defined(VMM64) || defined(CQ64)
typedef  Reg64  Reg;
typedef UReg64 UReg;
#endif

typedef union SharedReg64 {
   Reg8  reg8[2];
   Reg16 reg16;
   Reg32 reg32;
   Reg64 reg64;
} SharedReg64;

typedef union SharedUReg64 {
   UReg8  ureg8[2];
   UReg16 ureg16;
   UReg32 ureg32;
   UReg32 ureg32Pair[2];
   UReg64 ureg64;
} SharedUReg64;

typedef uint8 Instruction;

typedef uint16 Selector;

/*
 * Segment Descriptors.
 */

typedef struct Descriptor {
   unsigned   limit_lo  : 16;
   unsigned   base_lo   : 16;
   unsigned   base_mid  : 8;
   unsigned   type      : 4;
   unsigned   S         : 1;
   unsigned   DPL       : 2;
   unsigned   present   : 1;
   unsigned   limit_hi  : 4;
   unsigned   AVL       : 1;
   unsigned   longmode  : 1;
   unsigned   DB        : 1;
   unsigned   gran      : 1;
   unsigned   base_hi   : 8;
} Descriptor;

/*
 * 16-byte system descriptors for 64-bit mode.
 */

typedef 
#include "vmware_pack_begin.h"
struct Descriptor64 {
   unsigned   limit_lo  : 16;   // Limit bits 15-0.
   unsigned   base_lo   : 24;   // Base bits  23-0.
   unsigned   type      : 4;
   unsigned   S         : 1;
   unsigned   DPL       : 2;
   unsigned   present   : 1;
   unsigned   limit_hi  : 4;    // Limit bits 19-16.
   unsigned   AVL       : 1;
   unsigned   reserved0 : 2;
   unsigned   gran      : 1;
   unsigned   base_mid  : 8;    // Base bits 31-24.
   unsigned   base_hi   : 32;   // Base bits 63-32.
   unsigned   reserved1 : 8;
   unsigned   ext_attrs : 5;
   unsigned   reserved2 : 19;
}
#include "vmware_pack_end.h"
Descriptor64;

typedef struct CallGate {
   unsigned   offset_lo : 16;
   unsigned   selector  : 16;
   unsigned   parms     : 5;
   unsigned   zero0     : 3;
   unsigned   type      : 5;
   unsigned   DPL       : 2;
   unsigned   present   : 1;
   unsigned   offset_hi : 16;
} CallGate;

typedef union {
   Descriptor desc;
   CallGate   cg;
   uint32     word[2];
} DescriptorUnion;

typedef union {
   Descriptor64 desc;
   uint32       word[4];
   uint64       qword[2];
} Descriptor64Union;

#endif // ifndef _X86TYPES_H_
