/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib-object.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvd.h>

/* seqence_header.frame_rate_code values */
enum {
	FRAME_RATE_CODE_FORBIDDEN  = 0,
	FRAME_RATE_CODE_24000_1001 = 1,
	FRAME_RATE_CODE_24 = 2,
	FRAME_RATE_CODE_25 = 3,
	FRAME_RATE_CODE_30000_1001 = 4,
	FRAME_RATE_CODE_30 = 5,
	FRAME_RATE_CODE_50 = 6,
	FRAME_RATE_CODE_60000_1001 = 7,
	FRAME_RATE_CODE_60 = 8,
	FRAME_RATE_CODE_RESERVED = 9
	/* rest reserved */
};

const guint8 SEQUENCE_HEADER_BYTES[4] = {0x00, 0x00, 0x01, 0xb3};

static DvdDecoderClass *dvd_decoder_m2v_parent_class = NULL;

static void      dvd_decoder_m2v_class_init	(DvdDecoderM2vClass	 *class);
static void      dvd_decoder_m2v_instance_init	(GTypeInstance		 *instance,
						 gpointer		  g_class);
static void      dvd_decoder_m2v_dispose	(GObject		 *object);
static void	 dvd_decoder_m2v_read_startcode	(DvdDecoder		 *decoder,
						 guint8			**start,
						 guint8			 *end,
						 guint32		  pts);
static void	 dvd_decoder_m2v_read_header	(DvdDecoder		 *decoder,
						 guint8			**start,
						 guint8			 *end,
						 guint32		  pts);
static void	 dvd_decoder_m2v_read_frame	(DvdDecoder		 *decoder,
						 guint8			**start,
						 guint8			 *end,
						 guint32		  pts);

static void
dvd_decoder_m2v_class_init	(DvdDecoderM2vClass *class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	dvd_decoder_m2v_parent_class = g_type_class_ref (DVD_DECODER_TYPE);
	
	object_class->dispose = dvd_decoder_m2v_dispose;
}

static void
dvd_decoder_m2v_instance_init	(GTypeInstance	*instance,
				 gpointer	 g_class)
{
	DvdDecoder *decoder;
	DvdDecoderM2v *m2v_decoder;
	
	decoder = DVD_DECODER (instance);
	m2v_decoder = DVD_DECODER_M2V (instance);
	decoder->stream_type = DVD_STREAM_MPEG2_VIDEO;
	dvd_decoder_ensure_buffer_size (decoder, 1024);
	
	/* these functions called by parent DvdDecoder */
	decoder->read_startcode = dvd_decoder_m2v_read_startcode;
	decoder->read_header    = dvd_decoder_m2v_read_header;
	decoder->read_frame     = dvd_decoder_m2v_read_frame;
	
	m2v_decoder->first_sequence = FALSE;
	
	m2v_decoder->state = DVD_DECODER_M2V_STATE_SEEK_SEQ;
	m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEQ_HDR;
	
	m2v_decoder->header_byte = 0;
	
	m2v_decoder->start_found = FALSE;
	/*m2v_decoder->zero_bytes = 0;*/
	m2v_decoder->zero_bytes = 0xffff;
}

static void
dvd_decoder_m2v_dispose		(GObject	*object)
{
	DvdDecoderM2v *decoder;
	
	decoder = DVD_DECODER_M2V (object);
	
	G_OBJECT_CLASS (dvd_decoder_m2v_parent_class)->dispose (object);
}

/**
 * dvd_decoder_m2v_get_type
 * @return The GType for the DvdDecoderM2v class.
 */
GType
dvd_decoder_m2v_get_type	(void)
{
	static GType decoder_m2v_type = 0;

	if (decoder_m2v_type == 0) {
		GTypeInfo decoder_m2v_info = {
			sizeof (DvdDecoderM2vClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_decoder_m2v_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdDecoderM2v),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_decoder_m2v_instance_init
	    	};
		decoder_m2v_type = g_type_register_static (DVD_DECODER_TYPE,
						  	   "DvdDecoderM2v",
							   &decoder_m2v_info, 0);
	}
	return decoder_m2v_type;
}

/* Finds 4 start code bytes 0 0 1 X */
/* If TRUE **start now points to byte after 4 byte start code which may be end */
static gboolean
dvd_decoder_m2v_find_start_code	(DvdDecoderM2v	 *m2v_decoder,
				 guint8		**start,
				 guint8		 *end,
				 DvdM2vStartCode *start_code,
				 gboolean	  write_bytes)
{
	guint8 *iter;
	gboolean code_found = FALSE;
	
	iter = *start;
	
	if (m2v_decoder->start_found == TRUE) {
		/* last attempt missing code byte 0000 0000 0001 NULL */
		m2v_decoder->start_found = FALSE;
		if (write_bytes == TRUE) {
			dvd_decoder_write_frame_byte (DVD_DECODER (m2v_decoder), *iter);
		}
		*start_code = *iter;
		iter++;
		*start = iter;
		/*switch (*start_code) {
		case DVD_M2V_START_CODE_PICTURE:
			g_message ("found start code DVD_M2V_START_CODE_PICTURE");
			break;
		case DVD_M2V_START_CODE_SLICE_FIRST:
			g_message ("found start code DVD_M2V_START_CODE_SLICE_FIRST");
			break;
		case DVD_M2V_START_CODE_SLICE_LAST:
			g_message ("found start code DVD_M2V_START_CODE_SLICE_LAST");
			break;
		case DVD_M2V_START_CODE_USER_DATA:
			g_message ("found start code DVD_M2V_START_CODE_USER_DATA");
			break;
		case DVD_M2V_START_CODE_SEQUENCE_HDR:
			g_message ("found start code DVD_M2V_START_CODE_SEQUENCE_HDR");
			break;
		case DVD_M2V_START_CODE_SEQUENCE_ERR:
			g_message ("found start code DVD_M2V_START_CODE_SEQUENCE_ERR");
			break;
		case DVD_M2V_START_CODE_EXTENSION:
			g_message ("found start code DVD_M2V_START_CODE_EXTENSION");
			break;
		case DVD_M2V_START_CODE_SEQUENCE_END:
			g_message ("found start code DVD_M2V_START_CODE_SEQUENCE_END");
			break;
		case DVD_M2V_START_CODE_GROUP_OF_PIC:
			g_message ("found start code DVD_M2V_START_CODE_GROUP_OF_PIC");
			break;
		default:
			g_message ("found start code DVD_M2V_START_CODE_SLICE_0x%x", *start_code);
		}*/
		return TRUE;
	}
	while (iter < end) {
		if (write_bytes == TRUE) {
			dvd_decoder_write_frame_byte (DVD_DECODER (m2v_decoder), *iter);
		}

		if ((*iter == 0x01) &&
		    (m2v_decoder->zero_bytes == 0)) {
			m2v_decoder->zero_bytes = 0xffff;
			iter++;
			if (iter == end) {
				/* next byte will be code byte */
				m2v_decoder->start_found = TRUE;
			} else {
				if (write_bytes == TRUE) {
					dvd_decoder_write_frame_byte (DVD_DECODER (m2v_decoder), *iter);
				}
				code_found = TRUE;
				*start_code = *iter;
				iter++;
				/*switch (*start_code) {
				case DVD_M2V_START_CODE_PICTURE:
					g_message ("found start code DVD_M2V_START_CODE_PICTURE");
					break;
				case DVD_M2V_START_CODE_SLICE_FIRST:
					g_message ("found start code DVD_M2V_START_CODE_SLICE_FIRST");
					break;
				case DVD_M2V_START_CODE_SLICE_LAST:
					g_message ("found start code DVD_M2V_START_CODE_SLICE_LAST");
					break;
				case DVD_M2V_START_CODE_USER_DATA:
					g_message ("found start code DVD_M2V_START_CODE_USER_DATA");
					break;
				case DVD_M2V_START_CODE_SEQUENCE_HDR:
					g_message ("found start code DVD_M2V_START_CODE_SEQUENCE_HDR");
					break;
				case DVD_M2V_START_CODE_SEQUENCE_ERR:
					g_message ("found start code DVD_M2V_START_CODE_SEQUENCE_ERR");
					break;
				case DVD_M2V_START_CODE_EXTENSION:
					g_message ("found start code DVD_M2V_START_CODE_EXTENSION");
					break;
				case DVD_M2V_START_CODE_SEQUENCE_END:
					g_message ("found start code DVD_M2V_START_CODE_SEQUENCE_END");
					break;
				case DVD_M2V_START_CODE_GROUP_OF_PIC:
					g_message ("found start code DVD_M2V_START_CODE_GROUP_OF_PIC");
					break;
				default:
					g_message ("found start code DVD_M2V_START_CODE_SLICE_0x%x", *start_code);
				}*/
				
			}
			break;
		} else {
			m2v_decoder->zero_bytes = (m2v_decoder->zero_bytes << 8) + *iter;
		}
		iter++;
	}
	*start = iter;
	
	return code_found;
}

/* Search through stream until we find a sequence header */
static void
dvd_decoder_m2v_read_startcode	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	DvdDecoderM2v *m2v_decoder;

	m2v_decoder = DVD_DECODER_M2V (decoder);
	if ((m2v_decoder->state == DVD_DECODER_M2V_STATE_ERROR) ||
	    (m2v_decoder->state == DVD_DECODER_M2V_STATE_MPEG1)) {
		*start = end;
		return;
	}
	
	while (1) {
		DvdM2vStartCode start_code;
		guint8 *iter;
		
		iter = *start;
		if (dvd_decoder_m2v_find_start_code (m2v_decoder, &iter, end, &start_code, FALSE) == FALSE) {
			*start = iter;
			break;
		} else {
			*start = iter;
			
			switch (start_code) {
			case DVD_M2V_START_CODE_SEQUENCE_END:
			case DVD_M2V_START_CODE_PICTURE:
			case DVD_M2V_START_CODE_USER_DATA:
			case DVD_M2V_START_CODE_SEQUENCE_ERR:
			case DVD_M2V_START_CODE_EXTENSION:
			case DVD_M2V_START_CODE_GROUP_OF_PIC:
				break;
			case DVD_M2V_START_CODE_SEQUENCE_HDR:
				/* g_message ("found a sequence header"); */
				dvd_decoder_set_state (decoder, DVD_DECODER_STATE_HEADER);
				dvd_decoder_set_start_pts (decoder, pts);
				m2v_decoder->state = m2v_decoder->next_state;
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEEK_START;
				m2v_decoder->first_sequence = TRUE;
				dvd_decoder_write_frame_bytes (decoder, &SEQUENCE_HEADER_BYTES[0], 4);
				return;
			default:
				if ((start_code >= DVD_M2V_START_CODE_SLICE_FIRST) &&
				    (start_code <= DVD_M2V_START_CODE_SLICE_LAST)) {
					/* 176x slice codes */
					break;
				}
				/*g_warning ("Unknown mpeg stream header ident = 0x%x", start_code);*/
				/* ToDo - emit error signals */
				m2v_decoder->state = DVD_DECODER_M2V_STATE_ERROR;
				dvd_decoder_set_state (decoder, DVD_DECODER_STATE_HEADER);
				*start = end;
				break;
			}
		}
	}
}

/*
 * Reads a sequence header provided one byte at a time.
 *
 * Increments byte_no until header is complete.
 *
 * Returns TRUE and sets byte_no to zero when header is complete.
 */
static gboolean
read_seqence_header		(DvdM2vSequenceHdr 	 *header,
				 guint8			  data,
				 guint			 *byte_no)
{
	gboolean header_complete;
	
	header_complete = FALSE;
	
	switch (*byte_no) {
	case 0:
		/*g_message ("reading seqence header");*/
		memset (header, '0', sizeof (DvdM2vSequenceHdr));
		header->horizontal_size = data;				/* first 8 bits of 12 bit uint */
		header->horizontal_size = (header->horizontal_size << 4);
		break;
	case 1:
		header->horizontal_size += (data >> 4);			/* first 4 bits are last 4 bits of 12 bit uint */
		header->vertical_size = (data & 0x0f);			/* last 4 bits are first 4 bits of 12 bit uint */
		header->vertical_size = (header->vertical_size << 8);
		break;
	case 2:
		header->vertical_size += data;				/* last 8 bits of 12 bit uint */
		break;
	case 3:
		header->aspect = (data >> 4);				/* first 4 bits are 4 bits of 4 bit uint */
		header->frame_rate_code = (data & 0x0f);		/* last 4 bits are 4 bits of 4 bit uint */
		break;
	case 4:
		header->bit_rate = data;				/* first 8 bits of 18 bit uint */
		header->bit_rate = (header->bit_rate << 10);
		break;
	case 5:
		header->bit_rate += (((guint) data) << 2);		/* bits 10-16 of 18 bit uint */
		break;
	case 6:
		header->bit_rate += (data >> 6);			/* first 2 bits are last 2 bits of 18 bit uint */
		/* 1 bit discarded here */
		header->vbv_buffer_size	= (data & 0x1f);		/* last 5 bits are first 5 bits of 10 bit uint */
		header->vbv_buffer_size = (header->vbv_buffer_size << 5);
		break;
	case 7:
		header->vbv_buffer_size	+= (data >> 3);			/* first 5 bits are last 5 bits of 10 bit uint */
		header->cp_flag = ((data & 0x04) >> 2);			/* bit 6 */
		header->load_iq_matrix = ((data & 0x03) >> 1);		/* bit 7 */
		if (header->load_iq_matrix > 0) {
			/*g_message ("sequence header has iq_matrix");*/
			/* 64 bytes of data starting at last bit of this byte */
			header->iq_matrix[0] = (data << 7);
		} else {
			header->load_niq_matrix	= (data & 0x01);	/* bit 8 */
			/*if (header->load_niq_matrix > 0) {
				g_message ("sequence header has no iq_matrix, but has niq_matrix");
			}*/
		}
		/*g_message ("aspect - %u", header->aspect);
		g_message ("horizontal_size - %u", header->horizontal_size);
		g_message ("vertical_size - %u", header->vertical_size);
		g_message ("mpeg bitrate - %u", header->bit_rate);
		g_message ("vbv_buffer_size - %u", header->vbv_buffer_size);
		g_message ("frame rate - %u", header->frame_rate_code);*/
		break;
	default :
		/* byte_no >= 8 */
		if (header->load_iq_matrix > 0) {
			if (*byte_no < 71) {
				header->iq_matrix[*byte_no - 8] += (data >> 1);
				header->iq_matrix[*byte_no - 7] += (data << 7);
				/*g_message ("iq_matrix[%u]=0x%x", *byte_no - 7, data);*/
			} else if (*byte_no == 71) {
				header->iq_matrix[*byte_no - 8] += (data >> 1);
				header->load_niq_matrix = (data & 0x01);
				/*g_message ("iq_matrix[%u]=0x%x", *byte_no - 8, data);*/
				if (header->load_niq_matrix > 0) {
					/*g_message ("sequence header has iq_matrix and niq_matrix");*/
				}
			} else {
				/* byte_no >= 72 */
				if (header->load_niq_matrix > 0) {
					if (*byte_no < 135) {
						header->niq_matrix[*byte_no - 72] = data;
						/*g_message ("niq_matrix[%u]=0x%x", *byte_no - 72, data);*/
					} else {
						header->niq_matrix[*byte_no - 72] = data;
						/*g_message ("niq_matrix[%u]=0x%x", *byte_no - 72, data);*/
						header_complete = TRUE;
					}
				} else {
					header_complete = TRUE;
				}
			}
		} else if (header->load_niq_matrix > 0) {
			if (*byte_no < 71) {
				header->niq_matrix[*byte_no - 8] = data;
				/*g_message ("niq_matrix[%u]=0x%x", *byte_no - 8, data);*/
			} else {
				header->niq_matrix[*byte_no - 8] = data;
				/*g_message ("niq_matrix[%u]=0x%x", *byte_no - 8, data);*/
				header_complete = TRUE;
			}
		} else {
			header_complete = TRUE;
		}
		break;
	}
	
	if (header_complete == TRUE) {
		/*g_message ("Sequence header length including start code = %u", *byte_no + 5);*/
		
		/* must reset header byte for next header read */
		*byte_no = 0;
	} else {
		(*byte_no)++;
	}
	
	return header_complete;
}

/* start code then 6 bytes long */
static gboolean
read_seqence_extension_header	(DvdM2vSequenceHdrExt 	 *header_ext,
				 guint8			  data,
				 guint			 *byte_no)
{
	gboolean header_complete;
	
	header_complete = FALSE;
	
	switch (*byte_no) {
	case 0:
		/*g_message ("reading seqence extension header");*/
		g_assert ((data >> 4) == DVD_M2V_EXT_ID_SEQUENCE);	/* first 4 bits identify sequence extension header */
		header_ext->pl_indication = (data << 4);		/* last 4 bits are first 4 bits of 8 bit uint */
		break;
	case 1:
		header_ext->pl_indication += (data >> 4);		/* first 4 bits are last 4 bits of 8 bit uint */
		header_ext->progressive_sequence = ((data & 0x80) >> 3);/* bit 5 */
		header_ext->chroma_format = ((data & 0x06) >> 1);	/* bit 6 and 7 are 2 bit uint */
		header_ext->horizontal_size_ext = ((data & 0x01) << 1);	/* bit 8 is first bit of 2 bit uint */
		break;
	case 2:
		
		header_ext->horizontal_size_ext += (data >> 7);		/* first bit is last bit of 2 bit uint */
		header_ext->vertical_size_ext = ((data & 0x60) >> 5);	/* bit 2 and 3 are 2 bit uint */
		header_ext->bit_rate_extension = (data & 0x1f);		/* last 5 bits are first 5 bits of 12 bit uint */
		header_ext->bit_rate_extension = (header_ext->bit_rate_extension << 7);
		break;
	case 3:
		header_ext->bit_rate_extension += (data >> 1);		/* first 7 bits are last 7 bits of 12 bit uint */
		g_assert ((data & 0x01) == 1);				/* last bit is a marker bit */
		break;
	case 4:
		header_ext->vbv_buffer_size_ext = data;			/* 8 bits of 8 bit uint */
		break;
	case 5:
		header_ext->low_delay = (data >> 7);			/* first bit */
		header_ext->frame_rate_ext_n = ((data & 0x60) >> 5);	/* bit 2 and 3 are 2 bit uint */
		header_ext->frame_rate_ext_d = (data & 0x1f);		/* last 5 bits are 5 bits of 5 bit uint */
		header_complete = TRUE;
		break;
	default:
		g_assert_not_reached ();
	}
	if (header_complete == TRUE) {
		/*g_message ("Sequence header extension length including start code = %u", *byte_no + 5);*/
		
		/* must reset header byte for next header read */
		*byte_no = 0;
	} else {
		(*byte_no)++;
	}
	
	return header_complete;
}

static gboolean
read_seqence_display_header	(DvdM2vSequenceDispHdr 	 *seqence_display_header,
				 guint8			  data,
				 guint			 *byte_no)
{
	gboolean header_complete;
	
	header_complete = FALSE;
	
	switch (*byte_no) {
	case 0:
		/*g_message ("reading seqence display header");*/
		g_assert ((data >> 4) == DVD_M2V_EXT_ID_SEQUENCE_DISPLAY);		/* first 4 bits identify sequence display header */
		seqence_display_header->video_format = ((data & 0x0e) >> 1);		/* next 3 bits are 3 bit uint */
		seqence_display_header->colour_description = (data & 0x01);		/* last bit is 1 bit uint */
		break;
	case 1:
		if (seqence_display_header->colour_description > 0) {
			/*g_message ("seqence display header has colour description");*/
			seqence_display_header->colour_primaries = data;
		} else {
			seqence_display_header->display_horizontal_size = data;		/* first 8 bits of 14 bit uint */
			seqence_display_header->display_horizontal_size = (seqence_display_header->display_horizontal_size << 6);
		}
		break;
	case 2:
		if (seqence_display_header->colour_description > 0) {
			seqence_display_header->transfer_characteristics = data;
		} else {
			seqence_display_header->display_horizontal_size += (data >> 2);	/* first 6 bits are last of 14 bit uint */
			g_assert ((data & 0x02) > 0);					/* 1 marker bit */
			seqence_display_header->display_vertical_size = (data &0x01);	/* last bit is first bit of 14 bit uint */
			seqence_display_header->display_vertical_size = (seqence_display_header->display_vertical_size << 13);
		}
		break;
	case 3:
		if (seqence_display_header->colour_description > 0) {
			seqence_display_header->matrix_coefficients = data;
		} else {
			seqence_display_header->display_vertical_size += (((guint) data) << 5);	/* bit 2-9 of 14 bit uint */
		}
		break;
	case 4:
		if (seqence_display_header->colour_description > 0) {
			seqence_display_header->display_horizontal_size = data;		/* first 8 bits of 14 bit uint */
			seqence_display_header->display_horizontal_size = (seqence_display_header->display_horizontal_size << 6);
		} else {
			seqence_display_header->display_vertical_size += (data >> 3);	/* first 5 bits are last of 14 bit uint */
			/* 3 unused bits */
			header_complete = TRUE;
		}
		break;
	case 5:
		seqence_display_header->display_horizontal_size += (data >> 2);	/* first 6 bits are last of 14 bit uint */
		g_assert ((data & 0x02) > 0);					/* 1 marker bit */
		seqence_display_header->display_vertical_size = (data &0x01);	/* last bit is first bit of 14 bit uint */
		seqence_display_header->display_vertical_size = (seqence_display_header->display_vertical_size << 13);
		break;
	case 6:
		seqence_display_header->display_vertical_size += (((guint) data) << 5);	/* bit 2-9 of 14 bit uint */
		break;
	case 7:
		seqence_display_header->display_vertical_size += (data >> 3);	/* first 5 bits are last of 14 bit uint */
		/* 3 unused bits */
		header_complete = TRUE;
		break;
	default:
		g_assert_not_reached ();
	}
	if (header_complete == TRUE) {
		/*g_message ("Sequence display header length including start code = %u", *byte_no + 5);*/
		
		/* must reset header byte for next header read */
		*byte_no = 0;
	} else {
		(*byte_no)++;
	}
	
	return header_complete;
}

static gboolean
read_gop_header			(DvdM2vGOPHdr		 *header,
				 guint8			  data,
				 guint			 *byte_no)
{
	gboolean header_complete;
	
	header_complete = FALSE;
	
	switch (*byte_no) {
	case 0:
		/*g_message ("reading GOP header");*/
		header->time.drop_frame	= (data >> 7);		/* first bit */
		header->time.hours	= ((data >> 2) & 0x3f);	/* bit 2 - 6  are 6 bit uint */
		header->time.minutes	= ((data & 0x03) << 4);	/* last 2 bits are first 2 of 6 bit uint */
		break;
	case 1:
		header->time.minutes	+= (data >> 4);		/* first 4 bits are last 4 bits of 6 bit uint */
		/* 1 marker bit */
		header->time.seconds	= ((data & 0x07) << 3);	/* last 3 bits are first 3 bits of 6 bit uint */
		break;
	case 2:
		header->time.seconds	+= (data >> 5);		/* first 3 bits are last 3 bits of 6 bit uint */
		header->time.pictures	= ((data & 0x1f) << 1);	/* last 5 bits are first 5 bits of 6 bit uint */
		break;
	case 3:
		header->time.pictures	+= (data >> 7);		/* first bit is last bit of 6 bit uint */
		break;
	case 4:
		header->closed_gop	= ((data & 0x40) >> 6);	/* second bit */
		header->broken_link	= ((data & 0x20) >> 5);	/* 3rd bit */
		/* 5 x 0 bits */
		g_assert ((data & 0x1f) == 0);
		/*g_message ("GOP time: drop_frame=%u h=%u m=%u s=%u pictures=%u closed_gop=%u broken_link=%u",
			   header->time.drop_frame,
			   header->time.hours,
			   header->time.minutes,
			   header->time.seconds,
			   header->time.pictures,
			   header->closed_gop,
			   header->broken_link);*/
		header_complete = TRUE;
		break;
	default:
		g_assert_not_reached ();
	}
	if (header_complete == TRUE) {
		/*g_message ("GOP header length including start code = %u", *byte_no + 5);*/
		
		/* must reset header byte for next header read */
		*byte_no = 0;
	} else {
		(*byte_no)++;
	}
	
	return header_complete;
}

#if 0
video_sequence() {
    next_start_code()
    sequence_header()
    if ( nextbits() == extension_start_code ) {
          sequence_extension()
          do {
               extension_and_user_data( 0 )
               do {
                     if (nextbits() == group_start_code) {
                          group_of_pictures_header()
                          extension_and_user_data( 1 )
                     }
                     picture_header()
                     picture_coding_extension()
                     extensions_and_user_data( 2 )
                     picture_data()
               } while ( (nextbits() == picture_start_code) ||
                          (nextbits() == group_start_code) )
               if ( nextbits() != sequence_end_code ) {
                     sequence_header()
                     sequence_extension()
               }
          } while ( nextbits() != sequence_end_code )
    } else {
          /* ISO/IEC 11172-2 */
    }
    sequence_end_code
}
#endif /*0*/

static void
dvd_decoder_m2v_read_header	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	DvdDecoderM2v *m2v_decoder;
	guint8 *iter;
	gboolean header_read_done;
	DvdM2vStartCode start_code;;
	
	iter = *start;
	m2v_decoder = DVD_DECODER_M2V (decoder);

	switch (m2v_decoder->state) {
	case DVD_DECODER_M2V_STATE_SEEK_START:
		if (dvd_decoder_m2v_find_start_code (m2v_decoder, &iter, end, &start_code, TRUE) == FALSE) {
			/* need more bytes */
			*start = iter;
			break;
		}
		*start = iter;
		
		switch (m2v_decoder->next_state) {
		case DVD_DECODER_M2V_STATE_GOP_HDR:
			if (start_code != DVD_M2V_START_CODE_GROUP_OF_PIC) {
				if (m2v_decoder->first_sequence == TRUE) {
					g_warning ("DvdDecoderM2v - no GOP in first sequence, discarding the sequence");
					m2v_decoder->state = DVD_DECODER_M2V_STATE_SEEK_SEQ;
					m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEQ_HDR;
					dvd_decoder_discard_frame (decoder);
					dvd_decoder_set_state (decoder, DVD_DECODER_STATE_STARTCODE);
					break;
				}
				if (start_code == DVD_M2V_START_CODE_PICTURE) {
					
					/* No GOP present */
					m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PIC_HDR;
				} else if (start_code == DVD_M2V_START_CODE_SEQUENCE_END) {
					
					/* end of mpeg stream */
					m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEQ_END;
				} else {
					g_warning ("DvdDecoderM2v - unexpected header, broken bit stream");
					m2v_decoder->next_state = DVD_DECODER_M2V_STATE_ERROR;
				}
			} else if (m2v_decoder->first_sequence == TRUE) {
				/*g_message ("DvdDecoderM2v - Sequence with following GOP header found");*/
				m2v_decoder->first_sequence = FALSE;
			}
			break;
		case DVD_DECODER_M2V_STATE_SEQ_EXT_HDR:
			if (start_code == DVD_M2V_START_CODE_EXTENSION) {
				g_assert (((*iter) >> 4) == DVD_M2V_EXT_ID_SEQUENCE);
				/* MPEG2 */
				/*g_message ("DvdDecoderM2v - MPEG2 stream encountered by MPEG2 decoder");*/
			} else {
				/* MPEG1 */
				g_warning ("DvdDecoderM2v - no sequence extension header following extension header");
				g_warning ("DvdDecoderM2v - MPEG1 stream encountered by MPEG2 decoder");
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_MPEG1;
			}
			break;
		case DVD_DECODER_M2V_STATE_PIC_HDR:
			if (start_code == DVD_M2V_START_CODE_SEQUENCE_END) {
				/* end of mpeg stream */
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEQ_END;
			}
			break;
		case DVD_DECODER_M2V_STATE_EXTENSIONS_0:
			if (start_code == DVD_M2V_START_CODE_GROUP_OF_PIC) {
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_GOP_HDR;
				if (m2v_decoder->first_sequence == TRUE) {
					/*g_message ("DvdDecoderM2v - Sequence with following GOP header found");*/
					m2v_decoder->first_sequence = FALSE;
				}
			} else if (start_code == DVD_M2V_START_CODE_PICTURE) {
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PIC_HDR;
			} else if (start_code == DVD_M2V_START_CODE_USER_DATA) {
				g_error ("user data0");
			}
			break;
		case DVD_DECODER_M2V_STATE_EXTENSIONS_1:
			if (start_code == DVD_M2V_START_CODE_PICTURE) {
				/* end of extensions and user data */
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PIC_HDR;
			} else if (start_code == DVD_M2V_START_CODE_USER_DATA) {
				g_error ("user data1");
			} else if (start_code != DVD_M2V_START_CODE_EXTENSION) {
				g_assert_not_reached ();
			}
			break;
		case DVD_DECODER_M2V_STATE_EXTENSIONS_2:
			if ((start_code >= DVD_M2V_START_CODE_SLICE_FIRST) &&
			    (start_code <= DVD_M2V_START_CODE_SLICE_LAST)) {
				/* end of extensions and user data */
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PICTURE_DATA;
			} else if (start_code == DVD_M2V_START_CODE_USER_DATA) {
				g_error ("user data2");
			} else if (start_code != DVD_M2V_START_CODE_EXTENSION) {
				g_assert_not_reached ();
			}
			break;
		case DVD_DECODER_M2V_STATE_PICTURE_DATA:
			if ((start_code >= DVD_M2V_START_CODE_SLICE_FIRST) &&
			    (start_code <= DVD_M2V_START_CODE_SLICE_LAST)) {
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PICTURE_DATA;
			} else if (start_code == DVD_M2V_START_CODE_GROUP_OF_PIC) {
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_GOP_HDR;
			} else if (start_code == DVD_M2V_START_CODE_SEQUENCE_HDR) {
				/* g_message ("pictures in sequence = %u", m2v_decoder->sequence.picture_count); */
				dvd_decoder_set_frame_clocks (decoder, m2v_decoder->sequence.clocks);
				m2v_decoder->sequence.picture_count = 0;
				
				/* next sequence, need to rewind decoder buffer x 4, */
				/* output frame and then write sequence start code to decoder buffer */
				dvd_decoder_discard_bytes (decoder, 4);
				dvd_decoder_output_frame (decoder);
				dvd_decoder_write_frame_bytes (decoder, &SEQUENCE_HEADER_BYTES[0], 4);
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEQ_HDR;
			} else if (start_code == DVD_M2V_START_CODE_SEQUENCE_END) {
				/* end of mpeg stream? -not in europe! */
				g_message ("End of mpeg stream in Australia? - not in the rest of the world!");
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEQ_END;
			} else if (start_code == DVD_M2V_START_CODE_PICTURE) {
				/*g_warning ("START_CODE_PICTURE");*/
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PIC_HDR;
			} else if (start_code == DVD_M2V_START_CODE_USER_DATA) {
				g_warning ("m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PICTURE_DATA, "
					   "start_code = DVD_M2V_START_CODE_USER_DATA");
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PIC_HDR;
			} else {
				g_warning ("NEXT_STATE picture data start_code=%d", start_code);
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PIC_HDR;
			}
			break;
		default:
			break;
		}
		m2v_decoder->state = m2v_decoder->next_state;
		m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEEK_START;
		break;
	case DVD_DECODER_M2V_STATE_SEEK_SEQ:
		g_assert_not_reached ();
		break;
	case DVD_DECODER_M2V_STATE_SEQ_HDR:
		dvd_decoder_set_current_pts (decoder, pts);
		while (iter < end) {
			/* if the last byte/s of header are 0 they are counted in next start code! */
			/* or is this a bug in some encoding software? */
			m2v_decoder->zero_bytes = (m2v_decoder->zero_bytes << 8) + *iter;
			header_read_done = read_seqence_header (&m2v_decoder->seqence_header,
								*iter,
								&m2v_decoder->header_byte);
			dvd_decoder_write_frame_byte (decoder, *iter);
			if (header_read_done == TRUE) {
				m2v_decoder->state = m2v_decoder->next_state;
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEQ_EXT_HDR;
				iter++;
				break;
			}
			iter++;
		}
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_SEQ_EXT_HDR:
		while (iter < end) {
			/* if the last byte/s of header are 0 they are counted in next start code! */
			/* or is this a bug in some encoding software? */
			m2v_decoder->zero_bytes = (m2v_decoder->zero_bytes << 8) + *iter;
			header_read_done = read_seqence_extension_header (&m2v_decoder->seqence_header_extension,
									  *iter,
									  &m2v_decoder->header_byte);
			dvd_decoder_write_frame_byte (decoder, *iter);
			if (header_read_done == TRUE) {
				/* use seqence_header seqence_header_extension to build complete */
				/* values in decoders public sequence header */
				
				/* first 2 bits of horizontal_size are seqence_header_extension.horizontal_size_ext */
				/* last 12 bits are seqence_header.horizontal_size */
				m2v_decoder->sequence.horizontal_size = m2v_decoder->seqence_header_extension.horizontal_size_ext;
				m2v_decoder->sequence.horizontal_size = (m2v_decoder->sequence.horizontal_size << 12);
				m2v_decoder->sequence.horizontal_size += m2v_decoder->seqence_header.horizontal_size;
				
				/* vertical_size similar to horizontal size above */
				m2v_decoder->sequence.vertical_size = m2v_decoder->seqence_header_extension.vertical_size_ext;
				m2v_decoder->sequence.vertical_size = (m2v_decoder->sequence.vertical_size << 12);
				m2v_decoder->sequence.vertical_size += m2v_decoder->seqence_header.vertical_size;
				
				/* bit rate is 30 bits, first 12 m2v_decoder->seqence_header_extension.bit_rate_extension, */
				/* last 18 are m2v_decoder->seqence_header.bit_rate */
				m2v_decoder->sequence.bit_rate = m2v_decoder->seqence_header_extension.bit_rate_extension;
				m2v_decoder->sequence.bit_rate = (m2v_decoder->sequence.bit_rate << 18);
				m2v_decoder->sequence.bit_rate += m2v_decoder->seqence_header.bit_rate;
				m2v_decoder->sequence.bit_rate = m2v_decoder->sequence.bit_rate * 400;
				
				/* vbv_buffer_size is 18 bits, first 8 bits m2v_decoder->seqence_header_extension.vbv_buffer_size_ext, */
				/* last 10 are m2v_decoder->seqence_header.vbv_buffer_size */
				m2v_decoder->sequence.vbv_buffer_size = m2v_decoder->seqence_header_extension.vbv_buffer_size_ext;
				m2v_decoder->sequence.vbv_buffer_size = (m2v_decoder->sequence.vbv_buffer_size << 10);
				m2v_decoder->sequence.vbv_buffer_size += m2v_decoder->seqence_header.vbv_buffer_size;
				m2v_decoder->sequence.vbv_buffer_size = (m2v_decoder->sequence.vbv_buffer_size << 14);	/* bits = 16 * 1024 * vbv_buffer_size*/
				
				switch (m2v_decoder->seqence_header.frame_rate_code) {
				case FRAME_RATE_CODE_FORBIDDEN:
					g_warning ("video frame rate unspecified in sequence header - using 25FPS");
					m2v_decoder->clocks_per_picture = 1080000;
					/* ToDo calculate frame rate from sequence extension header */
					/* frame_rate = frame_rate_code * (frame_rate_ext_n + 1) ÷ (frame_rate_ext_d + 1) */
					break;
				case FRAME_RATE_CODE_24000_1001:
					m2v_decoder->clocks_per_picture = 1126125;
					break;
				case FRAME_RATE_CODE_24:
					m2v_decoder->clocks_per_picture = 1125000;
					break;
				case FRAME_RATE_CODE_25:
					m2v_decoder->clocks_per_picture = 1080000;
					break;
				case FRAME_RATE_CODE_30000_1001:
					m2v_decoder->clocks_per_picture = 900900;
					break;
				case FRAME_RATE_CODE_30:
					m2v_decoder->clocks_per_picture = 900000;
					break;
				case FRAME_RATE_CODE_50:
					m2v_decoder->clocks_per_picture = 540000;
					break;
				case FRAME_RATE_CODE_60000_1001:
					m2v_decoder->clocks_per_picture = 450450;
					break;
				case FRAME_RATE_CODE_60:
					m2v_decoder->clocks_per_picture = 450000;
					break;
				case FRAME_RATE_CODE_RESERVED:
					/* fall through */
				default:
					g_warning ("video frame rate unspecified in sequence header - using 25FPS");
					m2v_decoder->clocks_per_picture = 1080000;
					/* ToDo calculate frame rate from sequence extension header */
					/* frame_rate = frame_rate_code * (frame_rate_ext_n + 1) ÷ (frame_rate_ext_d + 1) */
					break;
				}
				/* g_message ("sequence.horizontal_size = %u", m2v_decoder->sequence.horizontal_size);
				g_message ("sequence.vertical_size = %u", m2v_decoder->sequence.vertical_size);
				g_message ("sequence.bit_rate = %llu bits per second, %llu bytes per second",
					   m2v_decoder->sequence.bit_rate,
					   m2v_decoder->sequence.bit_rate / 8);
				g_message ("sequence.vbv_buffer_size = %u bits, %u bytes",
					   m2v_decoder->sequence.vbv_buffer_size,
					   m2v_decoder->sequence.vbv_buffer_size / 8); */
				m2v_decoder->state = m2v_decoder->next_state;
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_EXTENSIONS_0;
				iter++;
				break;
			}
			iter++;
		}
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_EXTENSIONS_0:
		
		switch (*iter >> 4) {
		case DVD_M2V_EXT_ID_SEQUENCE_DISPLAY:
			/*g_message ("The extension header is a Sequence Display header");*/
			m2v_decoder->state = DVD_DECODER_M2V_STATE_SEQUENCE_DISPLAY;
			m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEEK_START;
			break;
		case DVD_M2V_EXT_ID_SEQUENCE_SCALABLE:
			/*g_message ("The extension header is a Sequence Scaleable header");*/
			m2v_decoder->state = DVD_DECODER_M2V_STATE_SEQUENCE_SCALABLE;
			m2v_decoder->next_state = DVD_DECODER_M2V_STATE_SEEK_START;
			break;
		default:
			g_assert_not_reached ();
		}
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_SEQUENCE_DISPLAY:
		while (iter < end) {
			/* if the last byte/s of header are 0 they are counted in next start code! */
			/* or is this a bug in some encoding software? */
			m2v_decoder->zero_bytes = (m2v_decoder->zero_bytes << 8) + *iter;
			header_read_done = read_seqence_display_header (&m2v_decoder->seqence_display_header,
									*iter,
									&m2v_decoder->header_byte);
			dvd_decoder_write_frame_byte (decoder, *iter);
			if (header_read_done == TRUE) {
				/* g_message ("display_horizontal_size = %u",
					   m2v_decoder->seqence_display_header.display_horizontal_size);
				g_message ("display_vertical_size = %u",
					   m2v_decoder->seqence_display_header.display_vertical_size);
				g_message ("video_format = %u",
					   m2v_decoder->seqence_display_header.video_format); */
				m2v_decoder->state = m2v_decoder->next_state;
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_EXTENSIONS_0;
				iter++;
				break;
			}
			iter++;
		}
		
		
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_SEQUENCE_SCALABLE:
		/*g_message ("read sequence scalable extension header");*/
		m2v_decoder->state = m2v_decoder->next_state;
		m2v_decoder->next_state = DVD_DECODER_M2V_STATE_EXTENSIONS_0;
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_GOP_HDR:
		while (iter < end) {
			/* if the last byte/s of header are 0 they are counted in next start code! */
			/* or is this a bug in some encoding software? */
			m2v_decoder->zero_bytes = (m2v_decoder->zero_bytes << 8) + *iter;
			header_read_done = read_gop_header (&m2v_decoder->gop_header,
							    *iter,
							    &m2v_decoder->header_byte);
			dvd_decoder_write_frame_byte (decoder, *iter);
			if (header_read_done == TRUE) {
				m2v_decoder->state = m2v_decoder->next_state;
				m2v_decoder->next_state = DVD_DECODER_M2V_STATE_EXTENSIONS_1;
				iter++;
				break;
			}
			iter++;
		}
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_EXTENSIONS_1:
		/*g_message ("read extension1 0x%x", (*iter >> 4));*/
		m2v_decoder->state = m2v_decoder->next_state;
		m2v_decoder->next_state = DVD_DECODER_M2V_STATE_EXTENSIONS_1;
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_PIC_HDR:
		/*g_message ("read pic header");*/
		m2v_decoder->sequence.picture_count++;
		m2v_decoder->sequence.clocks += m2v_decoder->clocks_per_picture;
		m2v_decoder->state = m2v_decoder->next_state;
		m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PIC_CODING_EXT;
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_PIC_CODING_EXT:
		/*g_message ("read pic coding extension 0x%x", (*iter >> 4));*/
		m2v_decoder->state = m2v_decoder->next_state;
		m2v_decoder->next_state = DVD_DECODER_M2V_STATE_EXTENSIONS_2;
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_EXTENSIONS_2:
		/*g_message ("read extension2 0x%x", (*iter >> 4));*/
		m2v_decoder->state = m2v_decoder->next_state;
		m2v_decoder->next_state = DVD_DECODER_M2V_STATE_EXTENSIONS_2;
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_PICTURE_DATA:
		m2v_decoder->state = m2v_decoder->next_state;
		m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PICTURE_DATA;
		/*g_assert_not_reached ();*/
		*start = iter;
		break;
	case DVD_DECODER_M2V_STATE_SEQ_END:
		/*g_message ("DvdDecoderM2v - seqence end found");*/
		m2v_decoder->state = m2v_decoder->next_state;
		m2v_decoder->next_state = DVD_DECODER_M2V_STATE_PICTURE_DATA;
		*start = iter;
		return;
	case DVD_DECODER_M2V_STATE_MPEG1:
	case DVD_DECODER_M2V_STATE_ERROR:
		*start = end;
		return;
	default:
		g_assert_not_reached ();
		return;
	}
}



/* writes any previous discarded 0 bytes or 0x000001 code and then up until the following: */
/* a) A sequence header is encountered and the start pointer is set to the sequence header start code 0xb3 or, */
/* b) The end of buffer is reached, discarding the following encountered data at the end of the buffer: */
/* 	1) 1 or 2 0x00 bytes at the end of the buffer are recorded in decoder->zero_bytes or */
/* 	2) if decoder->start_found = TRUE, 0x000001 was encountered at the end of the buffer and was discarded */

static void
dvd_decoder_m2v_read_frame	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	/* not used */
}
