# -*- coding: utf-8 -*-

# Copyright (c) 2008 - 2009 Lukas Hetzenecker <LuHe@gmx.at>

import copy
import re
from PyQt4.QtCore import *
from PyQt4.QtGui import *
import ui.ui_chat
import ui.ui_mobileNumberSelect
from lib.classes import *

class Chat(QMainWindow,  ui.ui_chat.Ui_Chat):
    def __init__(self, parent, main, contact):
        super(Chat,  self).__init__(parent)

        self.parent = parent
        self.main = main

        self.log = main.log
        self.connection = main.connection
        self.database = main.database
        self.settings = main.settings
        self.helper = main.helper

        # Destroy the window when it is closed
        # Otherwise all signals would be still connected - this causes wired problems
        self.setAttribute(Qt.WA_DeleteOnClose)

        # Only one chat window for each contact
        if contact.id() in main.activeChats:
            main.app.alert(main.activeChats[contact.id()])
            return

        self.setupUi(self)
        self.setWindowIcon(QIcon(":/phone"))

        self.connect(main, SIGNAL("messageNew"),  self.newMessage)
        self.connect(self.sendButton, SIGNAL("clicked()"),  self.sendButtonClicked)
        self.connect(self.messageText, SIGNAL("sendMessage"),  self.sendButton,  SLOT("animateClick()"))
        self.connect(self.messageText,  SIGNAL("textChanged()"),  self.textChanged)
        self.connect(self.connection, SIGNAL("messageSent"),  self.messageStateChanged)
        self.connect(self.connection, SIGNAL("messageQueued"),  self.messageStateChanged)
        self.connect(self.connection, SIGNAL("messageQueued"),  self.sendSignal)
        self.connect(self.connection, SIGNAL("connectionCompleted"),  self.checkSendButton)
        self.connect(self.connection, SIGNAL("connectionClosed"),  self.checkSendButton)
        self.connect(self.settings,  SIGNAL("reloadSettings"),  self.loadSettings)

        self.setWindowTitle(contact.name() + " - Series60-Remote")
        self.messageText.setFocus()

        windowSize = self.settings.setting("windows/chat/size")
        chatSplitter = self.settings.setting("windows/chat/splitter")

        if windowSize.isValid():
            self.resize(windowSize)

        if not chatSplitter.isNull():
            self.splitter.restoreState(chatSplitter)

        self.contact = copy.deepcopy(contact)
        # Create an entry in the list of open chats with the current window
        main.activeChats[contact.id()] = self

        if "phone" in contact.internalValues():
            phone = contact.internalValue("phone")
        elif len(contact.value("mobile_number")) > 1:
            self.log.info(QString("Contact %1 has more then one mobile number.").arg(contact.name()))
            phone = self.askMobileNumber(contact)
            if phone == None:
                self.close()
                return
        elif len(contact.value("mobile_number")) == 1:
            phone = contact.value("mobile_number")[0]
        elif re.match(r"^[+]{0,1}\d*$", contact.name()) != None:
            # name is a phone number
            phone = contact.name()
        else:
            self.close()
            return

        self.contact.addInternalValue("phone",  phone)

        self.queueMessages = 0
        self.maxMessagePriority = 0

        self.lastMessages()
        self.checkSendButton()
        self.loadSettings()
        self.show()
        
        self.main.emit(SIGNAL("chatOpened"),  contact)
    
    def loadSettings(self):
        self.messageText.setSendMessageOnReturn(self.settings.setting("general/sendMessageOnReturn"))

    def lastMessages(self):
        messageList = self.database.messagesLast(self.contact,  self.settings.setting("messages/displayLast"))
        
        for msg in self.main.unreadMessages:
            if msg.contact() == self.contact and msg not in messageList:
                messageList.append(msg)
        
        for msg in messageList:
            if msg.type() == MessageType.Incoming:
                name = msg.contact().name()
                color = QColor("orangered")
            elif msg.type() == MessageType.Outgoing:
                name = msg.device().name()
                color = QColor("dodgerblue")

            if msg in self.main.unreadMessages:
                msg.setPriority(MessagePriority.Medium)
                self.showMessage(msg,  name,  color)
                self.main.unreadMessages.remove(msg)
                self.connection.setRead(msg,  send=self.settings.setting("messages/markAsRead"))
            else:
                msg.setPriority(MessagePriority.Low)
                self.showMessage(msg,  name,  color.lighter(135),  self.historyBrowser.palette().color(QPalette.Text).lighter(425))
    
    def askMobileNumber(self,  contact):
        dlg = QDialog(self)
        dialog = ui.ui_mobileNumberSelect.Ui_MobileNumberSelectDialog()
        dialog.setupUi(dlg)
        self.main.setupButtonBox(dialog.buttonBox)
        dialog.contactLabel.setText(self.trUtf8("Please choose the telephone number for contact <i>%1</i>:").arg(contact.name()))
        for number in contact.value("mobile_number"):
            dialog.mobileBox.addItem(number)

        if not dlg.exec_():
            return None

        return str(dialog.mobileBox.currentText())

    def sendButtonClicked(self):
        message = unicode(self.messageText.toPlainText())
        msg = Message()
        msg.setType(MessageType.Outgoing)
        msg.setDevice(self.connection.device())
        msg.setContact(self.contact)
        msg.setDateTime(QDateTime.currentDateTime())
        msg.setMessage(message)

        self.connection.sendMessage(msg)
        self.messageText.clear()
        self.messageText.setFocus()

    def textChanged(self):
        len = int(self.messageText.toPlainText().length())
        chars,  messages = self.helper.countMessages(len)
        self.charLabel.setText(self.tr("%1 chars left; %n message(s)",  "",  messages).arg(chars))
        
        self.checkSendButton()

    def newMessage(self,  msg):
        if msg.contact() != self.contact:
            return
        
        self.connection.setRead(msg,  send=self.settings.setting("messages/markAsRead"))
        msg.setPriority(MessagePriority.Medium)
        self.showMessage(msg,  self.contact.name(),  QColor("orangered"))
        self.main.unreadMessages.remove(msg)

    def sendSignal(self,  msg):
        if msg.contact() != self.contact:
            return

        msg.setPriority(MessagePriority.Medium)
        name = msg.device().name()
        self.showMessage(msg,  name,  QColor("dodgerblue"))

    def showMessage(self,  msg, name,  contactColor,  textColor=None):
        message = msg.message()
        message = Qt.escape(message)
        message = message.replace("\n",  "<br />")

        if not textColor:
            textColor = self.historyBrowser.palette().color(QPalette.Text)

        if msg.dateTime().date() == QDate.currentDate():
            date = unicode(msg.dateTime().time().toString(Qt.DefaultLocaleLongDate))
        else:
            date = unicode(msg.dateTime().toString(Qt.DefaultLocaleShortDate))

        browser = self.historyBrowser

        # Do not overwrite the current selection
        browser.moveCursor(QTextCursor.End)

        browser.insertHtml('<table width=100%>' + \
        '<tr><td align="left"><font color="' + contactColor.name() + '"><b>' + Qt.escape(name) +  '</b></font></td>' + \
        '<font color="' + textColor.name() + '">' + \
        '<td align=right><font color="' + textColor.name() + '">' + date + '</font></td></tr>' + \
        '<tr><td colspan=2><font color="' + textColor.name() + '">' + message + '</font></td></tr>' + \
        '</table><br />')

        # Scroll to the End
        if msg.priority() >= self.maxMessagePriority:
            browser.moveCursor(QTextCursor.End)

    def messageStateChanged(self,  message):
        queue = self.connection.pendingMessages()
        
        anz = 0
        for message in queue:
            if message.contact() == self.contact:
                anz += 1

        time = QTime().currentTime().toString()
        if anz >= 1:
            self.statusBar().showMessage(self.tr("[%1] %n message(s) in queue",  "",  anz).arg(time))
        elif anz == 0 and self.queueMessages > 0:
            self.statusBar().showMessage(self.tr("[%1] All messages were sent").arg(time),  5000)

        self.queueMessages = anz

    def checkSendButton(self,  dummy=None):
        if self.messageText.toPlainText() and self.connection.connected():
            self.sendButton.setEnabled(True)
        else:
            self.sendButton.setEnabled(False)

    def closeEvent(self,  event):
        self.settings.setSetting("windows/chat/size", self.size())
        self.settings.setSetting("windows/chat/splitter", self.splitter.saveState())
        del self.main.activeChats[self.contact.id()]
        
        event.accept()
