/*
 * robconsole.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file robconsole.cpp
  * Implements all console specific stuff for RoboTour.
  * E.g. ConsoleFrontend, ConsoleBotPresenter, ...<br>
  * <b>Note:</b> This is the <i>only</i> file in which one should
  * directly use <tt>System::print()</tt>! All other files should
  * delegate output to some handler which is implemented in here!
  */
#include "robconsole.h"

#include "robbase.h"
#include "robvars.h"
#include "robstrings.h"
#include "robtrans.h"

#include <rtsystem.h>
#include <rtstring.h>
#include <rtfile.h>

using namespace lrt; 

namespace rt {

Array<int> getColWidths(const lrt::String& header, int numCols)
{
	Array<int> ret(numCols); 
	int i = 0, pos = header.indexOf('+'), oldPos = 0; 
	while(i < numCols && pos < header.length()) {
		oldPos = pos; 
		pos = header.indexOf('+', pos + 1);
		if(pos < 0) 
			pos = header.length();
		ret[i] = pos - oldPos; 
		i++; 
	}

	// default assignment if plusses are missing
	for(; i < numCols; i++)
		ret[i] = 5; 

	return ret; 
}



////////////// ConsoleFrontend


ConsoleFrontend::ConsoleFrontend() : verbose(5), printtime(0), debugtime(0)
{}
ConsoleFrontend::~ConsoleFrontend()
{}

void ConsoleFrontend::handleLoadError(const String& affectedFile, const String& message) const
{
	if(verbose >= 0) {
		System::println(String(_("load error: ")) + affectedFile + ": " + message);
#ifndef __UNIX__
		if(System::isInteractive())
			System::message(String(_("load error: ")) + affectedFile + ": " + message);
#endif
	}
}

void ConsoleFrontend::handleBotError(const Bot* affectedBot, ExecReturnType type) const
{
	if((verbose >= 7) && (type != failDie)) {
		System::println(String(_("robot killed: ")) +
			affectedBot->vars[botPosX] + "/" + affectedBot->vars[botPosY] + ": " +
			getFailMsg(type));
	}
}

void ConsoleFrontend::handleSystemError(int num, const String& message) const
{
#ifdef __SYMBIAN32__
	System::println(_("fatal error: ") + message);
	System::println(_("     [press any key to exit]"));
	System::read();
	System::exit();
#else
	System::exit(num, _("fatal error: ") + message);
#endif
}

void ConsoleFrontend::handleWarning(const String& message) const
{
	if(verbose >= 0) {
		System::println(_("warning: ") + message);
	}
}


void ConsoleFrontend::printVersion()
{
	System::println(String(_("Version ")) + getUserVersion()); 
	System::println(getAboutText()); 
	System::println(); 
}

void ConsoleFrontend::printHelp()
{
	printVersion(); 

	System::println(_("Usage: robotour [options] prog1 [prog2 [...]]"));
	System::println();
	System::println(_("Options:"));
	System::println(_("  -h, -help  Print this help text.")); 
#ifdef __NETWORK__
	System::println(_("  -tele      Switch to work group mode (internet simulation)"));
#endif
#ifdef __SYMBIAN32__
	System::println(_("  -cd F      Set current folder to F before running robotour."));
#endif
	System::println(_("  -n         Number of simulations to do (standard: 10)"));
	System::println(_("  -s         Switch to \"single\" mode: First against all"));
	System::println(_("  -c         Switch to \"charts\" mode: everyone against everyone (default)"));
	System::println(_("  -t n       Switch to \"top <n>\" mode: like RobServ"));
	System::println(_("  -T n       Same as -t, but old results are recognized"));
	System::println(_("     -cf F   The competition folder (bots & result files) (default: current)"));
	System::println(_("  -i         Simulate all bots in one field"));
	System::println(_("  -o optfile Specify location of the options file (default: ./robocom.rco)"));
	System::println(_("  -p x       print the field every x cycles (default: 0 = function inactive)"));
	System::println(_("  -debug p x Debug every x cycles (default: 0 = function inactive) the bots\n"
					  "             on field range p (coords start at 0; example for p: 0,0:1,1)"));
	System::println(_("  -batch     Non-interactive mode: Never ever pop up any message boxes"));
	System::println(_("  -r         No randomize => always the same result"));
	System::println(_("  -v n       Set verbose level (0-5) (default = 5; no output = 0)"));
	System::println(_("  progn      Any robot (.rob or .rbi) file, or file pattern (e.g. *.rob)."));

	// print plugin help
	for(Iterator<FrontendPlugin*> iter = plugins.begin(); iter.hasElement(); ++iter)
	{
		FrontendPlugin* plugin = iter.get();
		String name = plugin->getName();
		String text = plugin->getHelpText();
		if(text.length() == 0)
			continue;
//		System::println();
//		System::println(name + " Options:"); // makes the help grow too much and only confuses the user
		System::println(text);
	}
}

bool ConsoleFrontend::interpreteParamsImpl(const Array<String>& params, Array<bool>& used)
{
	if(params.length() < 1) {
		printHelp(); return false;
	}
	fightTypeParam = 20;

	for( int i = 0; i < params.length(); i++ )
	{
		if( used[i] ) continue;

		if( params[i][0] == '-' )
		{
			String arg = params[i].substring(1);

			// "n"umber of repeats
			if((arg == "n") || (arg == "N"))
			{
				i++; // look at next arg
				if(used[i])
					handleSystemError(1, _("Parameter missing for argument -") + arg);
				numRepeats = params[i].intValue(10);
			}

			// "p"rinttime
			else if((arg == "p") || (arg == "P"))
			{
				i++; // look at next arg
				if(used[i])
					handleSystemError(1, _("Parameter missing for argument -") + arg);
				printtime = params[i].intValue(0);
			}

			// "debug"time
			else if(!arg.compareIgnoreCase("debug"))
			{
				i+=2; // look at next args
				if(used[i-1] || used[i])
					handleSystemError(1, _("Parameter(s) missing for argument -") + arg);
				debugrange = params[i-1];
				debugtime = params[i].intValue(0);
			}

			// "s"ingle mode
			else if((arg == "s") || (arg == "S"))
			{
				fightType = ftSingle;
			}
			// "c"harts mode
			else if((arg == "c") || (arg == "C"))
			{
				fightType = ftCharts;
			}

			// "t"op mode
			else if((arg == "t") || (arg == "T"))
			{
				if(arg == "T") fightType = ftTopWithOld;
				else fightType = ftTop; // without old results

				i++; // look at next arg
				if(used[i])
					handleSystemError(1, _("Parameter missing for argument -") + arg);
				fightTypeParam = params[i].intValue(20);
			}

			// competition folder
			else if(arg == "cf")
			{
				i++; // look at next arg
				if(used[i])
					handleSystemError(1, _("Parameter missing for argument -") + arg);
				competitionFolder = params[i];
				// check that a path was given
				char c = competitionFolder[competitionFolder.length() - 1];
				// no path => append slash
				if(!(c == '/') && !(c == File::separatorChar))
					competitionFolder += File::separatorChar;
			}

			// all-"i"n-one mode
			else if((arg == "i") || (arg == "I"))
			{
				fightType = ftAllInOne;
			}

			// "v"erbose
			else if((arg == "v") || (arg == "V"))
			{
				i++; // look at next arg
				if(used[i])
					handleSystemError(1, _("Parameter missing for argument -") + arg);
				verbose = params[i].intValue(5);
			}

			// "o"ption file
			else if((arg == "o") || (arg == "O"))
			{
				i++; // look at next arg
				if(used[i])
					handleSystemError(1, _("Parameter missing for argument -") + arg);
				optionFile = params[i];
			}

			// no "r"andomize
			else if(arg == "r")
			{
				noRandom = true;
			}

			// "R"andomize!
			else if(arg == "R")
			{
				noRandom = false;
			}

			// "batch" mode (non-interactive)
			else if(!arg.compareIgnoreCase("batch"))
			{
				System::setInteractive(false);
			}

			// "h"elp or "help"
			else if((arg == "h") || (arg == "H") || !arg.compareIgnoreCase("help") || !arg.compareIgnoreCase("-help"))
			{
				printHelp();
				return false;
			}

			else if((arg == "version") || (arg == "-version")) 
			{
				printVersion();
				return false; 
			}

#ifdef __SYMBIAN32__
			// current folder
			else if(!arg.compareIgnoreCase("cd"))
			{
				i++; // look at next arg
				if(used[i])
					handleSystemError(1, _("Parameter missing for argument -") + arg);
				String newCurrentFolder = params[i];
				// check that a path was given
				char c = newCurrentFolder[newCurrentFolder.length() - 1];
				// no path => append slash
				if(!(c == '/') && !(c == File::separatorChar))
					newCurrentFolder += File::separatorChar;
				// get competition folder
				String compFolder = File(competitionFolder).getLocalName();
				// set current folder
				File::setCurrentFolder(File(newCurrentFolder));
				// set competition folder
				competitionFolder = File(compFolder).getName();
			}
#endif

			else
			{
				System::println(_("Warning: unknown parameter ") + params[i]);
			}
		}
		else // file or file pattern
		{
			int starPos = params[i].indexOf('*'); // has got a star?
			// resolve the param against the competition folder
			String fullParam = File(competitionFolder, params[i]).getName();
			if(starPos < 0) {   // no, so just add it
				if(File(fullParam).exists())
				  files += fullParam;
				else
				  System::println(_("warning: file not found: ") + fullParam);
			}
			else { // yes, so add all files matching the pattern
				Array<File> f(StarFilenameFilter::getFiles(fullParam));
				for(int fi = 0; fi < f.length(); fi++)
					files += f[fi].getName();
			}
		}
	}
	return true;
}

void ConsoleFrontend::fillOptions(SimOptions& options)
{
	// process standard stuff like noRandom, numRepeats, ...
	Frontend::fillOptions(options);

	ConsoleBotPresenter* cbp = 0;

	if(verbose >= 2) {
		bool verbosePresenter = (verbose >= 5);
		cbp = new ConsoleBotPresenter(verbosePresenter);
		options.supervisors += cbp;
	}
	if(verbose >= 0)
		options.tourDisps += new MyTourDisplayer(cbp);
	if(printtime > 0)
		options.supervisors += new FieldPrinter(printtime);
	if(debugtime > 0)
		options.supervisors += new DebugPrinter(debugtime, debugrange);
}


/////// ConsoleFrontend::MyTourDisplayer
ConsoleFrontend::MyTourDisplayer::MyTourDisplayer(ConsoleBotPresenter* cbp)
	: cbp(cbp)
{
}

void ConsoleFrontend::MyTourDisplayer::init(const TourInitInfo& info)
{
	starttime = Time::getCurrentTime();
	if(cbp)
		cbp->initTour(info);
}
TourStatusType ConsoleFrontend::MyTourDisplayer::update(const Array<TourResult>& status)
{ return tourContinue; }

String resizeString(const String& str, int length)
{
	if(str.length() == length) return String(str);
	if(str.length() > length) return str.substring(0, length);
	String ret(' ', length);
	for(int i = 0; i < str.length(); i++)
		ret[i] = str[i];
	return ret;
}

void ConsoleFrontend::MyTourDisplayer::exit(TourStatusType exitType, const Array<TourResult>& result)
{
	Array<TourResult> myResults(result);
	myResults.sort(TourResult::compareByPoints);

	Time duration = Time::getCurrentTime() - starttime;
	System::println(String::format(_("Simulation took %.3f seconds."), duration.sec + duration.msec / 1000.0));
	System::println();
	System::println(_("Tournament results:"));
	// NOTE FOR TRANSLATORS: The plus characters in the second line below are 
	// used to mark the table columns. They will not be shown on the screen. 
	System::println(_("Robot name                     Wins Losses Ties Points"));
	String header = _("+------------------------------+----+------+----+-----");
	Array<int> cols = getColWidths(header, 5); 
	header = header.replace('+', '-'); 
	System::println(header); 

	for(int i = 0; i < myResults.length(); i++)
		System::println(resizeString(myResults[i].headers["name"].value, cols[0]-1) + " " +
			String(myResults[i].wins, 10, cols[1]-1) + " " +
			String(myResults[i].looses, 10, cols[2]-1) + " " +
			String(myResults[i].ties, 10, cols[3]-1) + " " +
			String(int(myResults[i].points), 10, cols[4]-1));
}

////////////////// ConsoleBotPresenter //////////////////////

ConsoleBotPresenter::ConsoleBotPresenter(bool verbose) : verbose(verbose)
{
	retState.event = gameNothing;
}

ConsoleBotPresenter::~ConsoleBotPresenter() {}

SimSuperPosition ConsoleBotPresenter::getPreferredPosition()
{
	return supPosBotPresenter;
}

unsigned long ConsoleBotPresenter::getStepCycles()
{
	return 5000;
}

void ConsoleBotPresenter::initTour(const TourInitInfo& info)
{
	simNum = 0;
	simCount = info.numSimulations;
}

void ConsoleBotPresenter::initSim(Simulation* const sim)
{
	simNum++;
	System::println(String::format(_("Starting simulation %d of %d..."), (int) simNum, (int) simCount));
	for(Iterator<Bot*> iter = sim->bots.begin(); iter.hasElement(); ++iter)
	{
		Bot* bot = iter.get();
		Program* prog = bot->owner;
		System::println(String::format(_("%s (%d) has got %d banks, starts at position: %d/%d;%d"), 
			prog->headers["name"].value.cStr(), (int) prog->globalNum,
			(int) bot->banks.length(), (int) bot->vars[botPosX], (int) bot->vars[botPosY], (int) bot->tasks[0]->vars[taskDir]));
	}
}

SimSupervisor::GameState ConsoleBotPresenter::exec(Simulation* const sim)
{
	if(!verbose) return retState;
	for(int i = 0; i < sim->getNumPrograms(); i++)
		System::print(String(sim->getProgram(i)->vars[progMybots], 10, 5));
	System::println();
	return retState;
}

void ConsoleBotPresenter::exitSim(Simulation* const sim, const GameState& simResult)
{
	switch(simResult.event) {
		case gameAbort:
			System::println(_("The match was aborted and is considered as a tie."));
			break;
		case gameTie:
			System::println(_("The bots tied."));
			break;
		case gameFinished:
			System::println(String::format(_("%s won after %d cycles!"), 
				sim->getProgram(simResult.affectedProgram)->headers["name"].value.cStr(), 
				(int) sim->curCycle));
			break;
		default:
			break;
	}
}

/////////////////// FieldPrinter /////////////////////////////

FieldPrinter::FieldPrinter(int printtime) : printtime(printtime)
{
	retState.event = gameNothing;
}

FieldPrinter::~FieldPrinter() {}

SimSuperPosition FieldPrinter::getPreferredPosition()
{
	return supPosDontCare;
}

unsigned long FieldPrinter::getStepCycles()
{
	return printtime;
}

void FieldPrinter::initSim(Simulation* const sim)
{
	boundary = "+";
	for(int l = 0; l < (sim->fieldWidth * 2); l++)
		boundary += '-';
	boundary += '+';
}

SimSupervisor::GameState FieldPrinter::exec(Simulation* const sim)
{
	System::setPos(0,0);
	
	System::print(String::format(_("Cycle %5d: "), sim->curCycle));
	for(int p = 0; p < sim->getNumPrograms(); p++)
		System::print(String(sim->getProgram(p)->vars[progMybots], 10, 5));
	System::println();
#ifndef __SYMBIAN32__   // it doesn't fit on the Psion screen with fields=18 (now default) 
	System::println(boundary);
#endif
	for(int i = 0; i < sim->fieldHeight; i++)
	{
      String line;
	  line += '|';
      for(int j = 0; j < sim->fieldWidth; j++)
        {
          if(sim->field[j + i * sim->fieldWidth] == 0)
          {
              line += "  ";
          }
          else
          {
              char *sp = "    ";
              switch( sim->field[j + i * sim->fieldWidth]->owner->programNum % 5 )
              {
                case 0: sp="[]]]"; break; // [> <] v] ^]
                case 1: sp="####"; break; // #> <# v# ^#
                case 2: sp="{}}}"; break; // {> <} v} ^} 
                case 3: sp="||||"; break; // |> <| v| ^|
                case 4: sp="OOOO"; break; // O> <O vO ^O
              }

              switch( sim->field[j + i * sim->fieldWidth]->tasks[0]->vars[taskDir] )
              {
                case dirRight: line += sp[0]; line += '>'; break;
                case dirUp: line += '^'; line += sp[1]; break;
                case dirLeft: line += '<'; line += sp[2]; break;
                case dirDown: line += 'v'; line += sp[3]; break;
              }
          }
      }
      line += '|';
	  System::println(line);
	}
#ifndef __SYMBIAN32__   // it doesn't fit on the Psion screen with fields=18 (now default) 
	System::println(boundary);
#endif
	return retState;
}


/////////////////// DebugPrinter /////////////////////////////

DebugPrinter::DebugPrinter(int debugtime, String debugrange) : debugtime(debugtime)
{
	Array<String> parts = debugrange.split(":","");
	if(parts.length() != 2) System::exit(51, _("error: -debug: Format for p is startx,starty:endx,endy"));
	Array<String> start = parts[0].split(",","");
	Array<String> end = parts[1].split(",","");
	if((start.length() != 2) || (end.length() != 2))
		System::exit(51, _("error: -debug: Format for p is startx,starty:endx,endy"));
	sx = start[0].intValue(0);
	sy = start[1].intValue(0);
	ex = end[0].intValue(0);
	ey = end[1].intValue(0);
	retState.event = gameNothing;
}

DebugPrinter::~DebugPrinter() {}

SimSuperPosition DebugPrinter::getPreferredPosition()
{
	return supPosBotPresenter;
}

unsigned long DebugPrinter::getStepCycles()
{
	return debugtime;
}

void DebugPrinter::initSim(Simulation* const sim)
{
	for(int p = 0; p < sim->getNumPrograms(); p++)
		System::print(sim->getProgram(p)->print());

	if(sx >= sim->fieldWidth) sx = sim->fieldWidth - 1;
	if(ex >= sim->fieldWidth) ex = sim->fieldWidth - 1;
	if(sy >= sim->fieldHeight) sy = sim->fieldHeight - 1;
	if(ey >= sim->fieldHeight) ey = sim->fieldHeight - 1;
}

SimSupervisor::GameState DebugPrinter::exec(Simulation* const sim)
{
	System::setPos(0, sim->getNumPrograms());

	System::println(String::format(_("Cycle %d:      "), (int) sim->curCycle));
	int i = -1;
	for(int y = sy; y <= ey; y++) {
		for(int x = sx; x <= ex; x++)
		{
			i++;
			Bot* bot = sim->field[x + y * sim->fieldWidth];
			if(!bot) continue;
			String p = " Bot ";
			p += String(i, 10, 3);
			p += ": ";
			p += bot->owner->headers["name"].value.substring(0, 13);
			p += " Act:";
			p += String(bot->vars[botActive], 10, 5);
			p += ' ';
			p += bot->vars[botInstrSet];
			p += ',';
			p += String(bot->vars[botNumBanks], 10 , 2);
			p += ',';
			p += bot->vars[botMobile];
			p += " Gen:";
			p += String(bot->vars[botGeneration], 10, 2);
			p += " Instr: ";
			String instr = (bot->tasks[0]->curInstr ? bot->tasks[0]->curInstr->print() : "NULL");
			for(int l = instr.length(); l < 20; l++) instr += ' ';
			p += instr;
			p += "\n  ##:";
			for(int v = 1; v <= sim->glob.maxVars; v++)
			{
				p += String(bot->stdVars[v], 10, 2);
				p += ' ';
			}
			p += '\n';
			System::print(p);
		}
	}
	return retState;
}


} // namespace

