#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sockcheck.h"
#define SSLNEEDTIME
#include "sockets.h"

/* Sinit.c: this file contains interfaces to initialize and cleanup.
 *  Most o/s's don't need this, but some (WIN32, MSDOS) do.
 */
static int ssl_init= 0;

/* --------------------------------------------------------------------- */

/* Sinit: initializes Sockets */
void Sinit()
{
	#if defined(__WIN32__) || defined(_MSC_VER)
	if(ssl_init == 0) {
		static WORD    wVersionRequested;
		static WSADATA wsaData;
		int     err;
		char	buf[200]; 
		
		wVersionRequested= MAKEWORD(1,1);
		err = WSAStartup(wVersionRequested,&wsaData);
		mwFailIf(err, "unable to use Windows sockets, needs version 2.0 or better\n");
		if(LOBYTE(wsaData.wVersion) != 1 ||
		   HIBYTE(wsaData.wVersion) != 1) {
			int hiver;
			int lover;
			hiver= LOBYTE(wsaData.wVersion);
			lover= HIBYTE(wsaData.wVersion);
			WSACleanup();
			sprintf(buf, "wrong sockets version (wants 1.1, has %d.%d)\n",hiver,lover);
			mwFailIf(1, buf); 
		}
		atexit((void (*)(void)) WSACleanup);
	}
	#endif

	ssl_init= 1;
}


/* Smkskt.c: this function makes a SFSocket */
/* --------------------------------------------------------------------- */

/* makeSocket: this function allocates a SFSocket */
SFSocket *makeSocket(
  const char *hostname,
  const char *sktname,
  int   sktType)
{
	SFSocket *skt= NULL;


	/* allocate a SFSocket */
	skt= (SFSocket *) malloc(sizeof(SFSocket));
	mwFailIf(!skt, "makeSocket: while allocating a SFSocket");

	/* initialize SFSocket */
	skt->hostname= (char *) calloc((size_t) strlen(hostname)+1,sizeof(char));
	mwFailIf(!skt->hostname, "makeSocket: while allocating hostname string for socket");
	strcpy(skt->hostname,hostname);

	if(!sktname) sktname= "";
	skt->sktname= (char *) calloc((size_t) strlen(sktname)+1,sizeof(char));
	mwFailIf(!skt->sktname, "makeSocket: while allocating sktname string for socket");
	strcpy(skt->sktname,sktname);

	/* set port and skt to zero */
	skt->port= skt->skt= 0;
	skt->type= sktType;

	/* return SFSocket */
	return skt;
}

/* ------------------------------------------------------------------------ */

/* freeSocket: this function frees up a SFSocket */
void freeSocket(SFSocket *skt)
{

if(skt) {
	if(skt->hostname) free(skt->hostname);
	if(skt->sktname)  free(skt->sktname);
	skt->skt      = 0;
	skt->port     = 0;
	skt->type     = 0;
	skt->hostname = NULL;
	skt->sktname  = NULL;
	free((char *) skt);
	}

}


/* Sopen.c: these functions are responsible for opening Sockets
 *  Routines Include: <Sopen.c>
 *    SFSocket *Sopen( char *skthost, char *mode)
 *    SFSocket *Sopen_clientport( char *hostname, char *sktname, u_short port)
 *    static SFSocket *Sopen_server(char *sktname)
 *    static SFSocket *Sopen_client( char *hostname, char *sktname)
 */

/* ----------------------------------------------------------------------
 * Local Definitions:
 */
#define BUFSIZE		128

/* ---------------------------------------------------------------------- */

/* ---------------------------------------------------------------------- */

/* Sopen_server: this function opens a socket handle for the SSL socket
 *   function library
 */
SFSocket *Sopen_server(u_short port)
{
	SFSocket* skt; 
	struct sockaddr_in sin;	/* InterNet socket address structure */
	static char        hostname[BUFSIZE];
	char			   buf[200]; 
	int                length;
	int                status = 1;

	/* do any o/s-orientated initialization that is required */
	Sinit();

	/* make a SFSocket */
	if(gethostname(hostname,BUFSIZE) < 0) {
		return  (SFSocket *) NULL;
	}

	sprintf(buf, "%d", port); 
	skt=  makeSocket(hostname,buf, PM_SERVER);
	if(!skt)  {
		return  (SFSocket *) NULL;
	}

	/* create a socket */
	if((skt->skt=  socket(AF_INET, SOCK_STREAM, 0)) < 0 ) {
		freeSocket(skt);
		return (SFSocket *) NULL;
	}

	/*	initialize socket data structure
	 *  get port specified by sslport, defaults to any
	 *  memset: added by Steve Richards for AS/400
	 */
	memset(&sin,0x00,sizeof(struct sockaddr_in));
	sin.sin_family      = AF_INET;
	sin.sin_addr.s_addr = htonl(INADDR_ANY);
	sin.sin_port        = htons((u_short) port);

	/*	bind socket data structure to this socket     */
	if(bind(skt->skt,(struct sockaddr *) &sin,sizeof(sin))) {
		freeSocket(skt);
		return (SFSocket *) NULL;
	}

	/* getsockname fills in the socket structure with information,
	 * such as the port number assigned to this socket
	 */
	length = sizeof(sin);
	if(getsockname(skt->skt,(struct sockaddr *) &sin,&length)) {
		freeSocket(skt);
		return (SFSocket *) NULL;
		}
	skt->port= ntohs(sin.sin_port);

	/* --- Return to Server Initialization --- */

	/* prepare socket queue for (up to) PM_MAXREQUESTS connection
	 * requests
	 */
	listen(skt->skt,PM_MAXREQUESTS);

#ifndef SSLNOSETSOCKOPT
	/* turn off TCP's buffering algorithm so small packets don't get
	 * needlessly delayed
	 */
	if(setsockopt(skt->skt,IPPROTO_TCP,TCP_NODELAY,(char *) &status,sizeof(status)) < 0) {
		}
#endif	/* SSLNOSETSOCKOPT */

	/* return server socket */
	return skt;
}

/* ---------------------------------------------------------------------- */

/* Sopen_clientport: this function opens a client socket given a hostname
 * and port
 */
SFSocket *Sopen_client(
  const char   *hostname,
  u_short port)
{
	SFSocket          *skt        = NULL;
	int                status     = 1;
	static char        localhost[BUFSIZE];
	struct hostent    *hostentptr = NULL;
	struct sockaddr_in sin;				/* InterNet socket address structure */
	char               buf[200]; 


	/* do any o/s-orientated initialization that is required */
	Sinit();

	/* if hostname is null, then use current hostname */
	if(!hostname || !hostname[0]) {
		gethostname(localhost,BUFSIZE);
		hostname= localhost;
		}

	/* allocate a SFSocket */
	sprintf(buf, "%d", port); 
	skt= makeSocket(hostname,buf,PM_CLIENT);
	if(!skt) {
		return (SFSocket *) NULL;
	}
	skt->port= port;

	/*	open socket */

	skt->skt= socket(AF_INET,SOCK_STREAM,0);
	if(skt->skt < 0) {
		freeSocket(skt);
		return (SFSocket *) NULL;
	}

	/*	initialize socket data structure
	 *  memset: added by Steve Richards for AS/400
	 */
	memset(&sin,0x00,sizeof(struct sockaddr_in));
	sin.sin_family = AF_INET;
	hostentptr     = gethostbyname(hostname);


	if(!hostentptr) {
		freeSocket(skt);
		return (SFSocket *) NULL;
	}

	/* get  host address */
	sin.sin_addr = * ((struct in_addr *) hostentptr->h_addr);
	sin.sin_port = htons((u_short) port);

	/* send port number to server */

	/* connect to remote host */
	if(connect(skt->skt,(struct sockaddr *) &sin,sizeof(sin)) < 0){
		shutdown(skt->skt,2);
		close(skt->skt);
		freeSocket(skt);
		return (SFSocket *) NULL;
	}

#ifndef SSLNOSETSOCKOPT
	/* turn off TCP's buffering algorithm so small packets don't get
	 * needlessly delayed
	 */

	if(setsockopt(skt->skt,IPPROTO_TCP,TCP_NODELAY,(char *) &status,sizeof(status))
	  < 0) {
	}
#endif	/* SSLNOSETSOCKOPT */

	return skt;
}

/* undef local definitions from sopen.c */
#undef BUFSIZE


/* Saccept.c: this file contains the SFSocket accept support */
/* ------------------------------------------------------------------------- */

/* Saccept: this routine uses a server SFSocket to accept connections
 *  The accept() function clones a socket for use with a client connect.
 *  One may close the Saccept generated socket without affecting the
 *  server socket.
 */
SFSocket *Saccept(SFSocket *skt)
{
	int             addrlen;
#ifndef SSLNOSETSOCKOPT
	int             status=1;
#endif
	struct sockaddr addr;
	SFSocket         *acceptskt= NULL;


	/* sanity check */
	if(!skt) {
		return acceptskt;
	}

	/* allocate a SFSocket */
	acceptskt= makeSocket(skt->hostname,skt->sktname,PM_ACCEPT);
	if(!acceptskt) {
		return acceptskt;
	}

	/* accept a connection */
	addrlen       = sizeof (addr);
	acceptskt->skt= accept(skt->skt, &addr, &addrlen);
	if(acceptskt->skt <= 0) {	/* failure to accept */
		freeSocket(acceptskt);

		return (SFSocket *) NULL;
	}

	/* turn off TCP's buffering algorithm so small packets don't get delayed */
#ifndef SSLNOSETSOCKOPT

	if(setsockopt(skt->skt,IPPROTO_TCP,TCP_NODELAY,(char *) &status,sizeof(status)) < 0) {
	}
#endif	/* #ifndef SSLNOSETSOCKOPT ... */

	return acceptskt;
}


/* Sclose: this function closes a socket.  Closing a socket is easy, but
 * the PortMaster also needs to be informed
 */
void Sclose(SFSocket *skt)
{
	if(!skt) { /* return immediately */
		return;
	}

	shutdown(skt->skt, 2 /*SD_BOTH*/);
	close(skt->skt);
	freeSocket(skt);
}


/* ---------------------------------------------------------------------
 * Definitions:
 */
#define PEERBUF	65

/* --------------------------------------------------------------------- */

/* Speeraddr: get peer IP address */
unsigned long Speeraddr(SFSocket *skt)
{
	unsigned long          ret;
	static struct sockaddr sktname;
	static int             namelen  = sizeof(struct sockaddr);

	if(getpeername(skt->skt,&sktname,&namelen) == -1) {
		ret= 0;
	}
	else {
		union {
			unsigned char byte[4];
			unsigned long i;
		} u;
		u.byte[0]     = sktname.sa_data[2];
		u.byte[1]     = sktname.sa_data[3];
		u.byte[2]     = sktname.sa_data[4];
		u.byte[3]     = sktname.sa_data[5];
		ret           = u.i;
	}

	return ret;
}


/* Speername: get peername (a string) */
char *Speername(SFSocket *skt)
{
	static struct sockaddr sktname;
	static char            buf1[PEERBUF];
	static char            buf2[PEERBUF];
	static char            buf3[PEERBUF];
	static char           *buf    = buf1;
	static int             namelen= sizeof(struct sockaddr);

	if     (buf == buf1) buf= buf2;
	else if(buf == buf2) buf= buf3;
	else                 buf= buf1;

	if(getpeername(skt->skt,&sktname,&namelen) == -1) {
		strcpy(buf,"unknown");
	}
	else {
		struct hostent *host;
		struct in_addr inaddr;
		union {
			unsigned char byte[4];
			unsigned long i;
		} u;
		u.byte[0]     = sktname.sa_data[2];
		u.byte[1]     = sktname.sa_data[3];
		u.byte[2]     = sktname.sa_data[4];
		u.byte[3]     = sktname.sa_data[5];
		inaddr.s_addr = u.i;
		host          = gethostbyaddr((void *) &inaddr,sizeof(struct in_addr),AF_INET);
		if(host && host->h_name) strcpy(buf,host->h_name);
		else if(!host)           strcpy(buf,"unable to get host\n");
		else                     strcpy(buf,"unknown hostname");
	}

	return buf;
}

/* undefine: */
#undef PEERBUF


/* Stimeoutwait.c: this file contains code which allows one to wait until data
 *  is present at the Socket (this will block)
 *
 *  Returns: number of bytes of data awaiting perusal
 *	         or -1 if unable to "select" the socket
 *	         or -2 if timeout occurs
 */
int Stimeoutwait(SFSocket *skt,long seconds,long useconds)
{
	short          result;
	int            ret;
	fd_set         emask;
	fd_set         rmask;
	fd_set         wmask;
	struct timeval timeout;
	static char    buf[PM_BIGBUF];


	/* sanity check */
	if(!skt) {
		return -1;
	}

	FD_ZERO(&rmask);
	FD_SET(skt->skt,&rmask);
	FD_ZERO(&wmask);
	FD_ZERO(&emask);

	/* test if something is available for reading on the socket.  This form
	 * will block (sleep) until something arrives
	 */
	timeout.tv_sec = seconds;
	timeout.tv_usec= useconds;

#ifdef SSLNOPEEK
	result= select(skt->skt+1,
			rmask.fds_bits,wmask.fds_bits,emask.fds_bits, &timeout);
#else	/* #ifdef SSLNOPEEK  ... #else ... #endif */
	result= select(skt->skt+1, &rmask,&wmask,&emask, &timeout);
#endif	/* #ifdef SSLNOPEEK  ... #else ... #endif */

	/* socket error */
	if(result < 0) {
		return -1;
	}

	/* timed out */
	else if(result == 0) {
		return -2;
	}

	/* server sockets return the select result */
	if(skt->type == PM_SERVER) {
		return result;
	}

#ifdef SSLNOPEEK
	return 1;
#else	/* #ifdef SSLNOPEEK ... #else ... #endif */

	if(FD_ISSET(skt->skt,&rmask)) {
		ret= recv(skt->skt,buf,PM_BIGBUF-1,MSG_PEEK);
		if(result == 1 && ret == 0) ret= EOF;
		return ret;
	}

	/* socket is empty */
	return 0;
#endif	/* #ifdef SSLNOPEEK ... #else ... #endif */
}


/* Stest.c: this file contains code which allows one to test if data
 *  is present via the socket and not be blocked (not held, not made to sleep)
 *
 *  Returns: number of bytes of data awaiting perusal (which may be 0)
 *	         or -1 if unable to determine, or EOF if socket was closed by remote
 */
int Stest(SFSocket *skt)
{
	fd_set         emask;
	fd_set         rmask;
	fd_set         wmask;
	int            ret;
	short          result;
	static char    buf[PM_BIGBUF];
	struct timeval timeout;


	/* sanity check */
	if(!skt) {
		return -1;
	}

	FD_ZERO(&rmask);
	FD_SET(skt->skt,&rmask);
	FD_ZERO(&wmask);
	FD_ZERO(&emask);

	timeout.tv_sec = 0;
	timeout.tv_usec= 0;

#ifdef SSLNOPEEK
	result = select(skt->skt+1,rmask.fds_bits,wmask.fds_bits,emask.fds_bits,&timeout);
#else
	result = select(skt->skt+1,&rmask,&wmask,&emask,&timeout);
#endif	/* SSLNOPEEK */

	if(result < 0) {
		return EOF;
	}

	/* server sockets return the select result (client awaiting connection) */
	if(skt->type == PM_SERVER) {
		return result;
	}

#ifdef SSLNOPEEK

	if(FD_ISSET(skt->skt,&rmask)) {
		return 1;
	}

#else	/* #ifdef SSLNOPEEK ... #else ... #endif */

/* test if message available from socket, return qty bytes avail */

	if(FD_ISSET(skt->skt,&rmask)) {
		ret= recv(skt->skt,buf,PM_BIGBUF-1,MSG_PEEK);

		/* return error indication when select returned a result of 1
		* (indicating that *something* is on the socket)
		* and recv indicates that *nothing* is on the socket
		*/
		if(result == 1 && ret == 0) ret= EOF;
		return ret;
	}

#endif	/* #ifdef SSLNOPEEK ... #else ... #endif	*/

/* socket is empty */
	return 0;
}


/* Sread: this function performs a read from a Socket
 * It reads up to buflen bytes from the socket, but maybe less.
 * It returns the amount of bytes that were received, or zero if none.
 */
int Sread(
  SFSocket *skt,	/* socket handle			*/
  void     *buf,	/* socket character buffer	*/
  int       buflen)	/* max length of buffer		*/
{
	int cnt;

	/* sanity check */
	if(!skt) {
		return -1;
	}

	/* read bytes from Socket */
	cnt  = recv(skt->skt,(void *) buf,(unsigned) buflen,0);

	if(cnt > 0) {	/* "cnt" bytes received		*/
		return cnt;
	}

	/* error return */
	((char *) buf)[0]= '\0';
	return 0;
}


/* --------------------------------------------------------------------- */

/* Sreadbytes: this function performs a socket read.
 *  It reads repeatedly from the socket until it has
 *  got buflen bytes, while read is working.
 *  Otherwise, (read fails with a -1), the number
 *  of bytes received will be returned.
 */
int Sreadbytes(
  SFSocket *skt,	/* socket handle			*/
  void     *buf,	/* socket character buffer	*/
  int       buflen)	/* max length of buffer		*/
{
	int cnt;
	int rcnt;


	/* sanity check */
	if(!skt) {
		return -1;
		}

	/* get buflen bytes, no matter the wait */
	for(cnt= 0; cnt < buflen; cnt+= rcnt) {
		rcnt= recv(skt->skt,(void *) (((char *)buf)+cnt),(unsigned) (buflen-cnt),0);
		if(rcnt < 0) break;
		}

	return cnt;
}


/* Swrite: this function performs a socket write of a string.
 *  It will write out the requested bytes (up to a null byte),
 *  even if the socket buffer is full. (However that may take 
 *  some time.)
 *
 *  If the write fails, it will return the negative of the number
 *  of bytes actually transmitted.  Otherwise, it will return the
 *  number of bytes transmitted, which ought to be the same as buflen
 *  if life is just.
 */
int Swrite(
  SFSocket *skt,	/* socket handle			*/
  void     *buf,	/* socket character buffer	*/
  int       buflen)	/* length of buffer			*/
{
	int cnt;
	int wcnt;


	/* sanity check */
	if(!skt) {
		return -1;
	}

	/* write buflen bytes, no matter the wait */
	for(cnt= 0; cnt < buflen; cnt+= wcnt) {
		wcnt= send(skt->skt,(void *) (((char *) buf)+cnt),(unsigned) (buflen-cnt),0);
		if(wcnt < 0) {
			return cnt;
		}
	}

	return cnt;
}

