/*
 * rtsystem.stdlib.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

///// Stdlib implementation of System functions

#ifndef __LRT_SYSTEM_STDLIB__
#define __LRT_SYSTEM_STDLIB__

#include <stdlib.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/timeb.h>
#include <time.h>
#include <string.h>
#include <signal.h>

#include "rtsystem.h"
#include "rtfile.h"


namespace lrt {

Time::Time(const String& str) : sec(0), msec(0)
{
	Array<String> parts = str.split(" :", "");
	if(parts.length() != 7) // there should be 7 parts
		return;
	struct tm myTm;
	myTm.tm_isdst = -1;

	myTm.tm_mon = getMonthNum(parts[1]);
	myTm.tm_mday = parts[2].intValue(1);
	myTm.tm_hour = parts[3].intValue(0);
	myTm.tm_min = parts[4].intValue(0);
	myTm.tm_sec = parts[5].intValue(0);
	myTm.tm_year = parts[6].intValue(1970) - 1900;

	time_t myTime = mktime(&myTm);
	sec = (int)myTime;
}

Time Time::getCurrentTime()
{
	Time ret;
	struct timeb tp;
	ftime( &tp );
	ret.sec = tp.time;
	ret.msec = tp.millitm;
	return ret;
}

String Time::toString()
{
	time_t tm = (long)sec;
	return String(ctime(&tm));
}

void System::message(const String& str)
{
	println(str);
}

void System::exit(int errnum)
{
	finalize();
	::exit(errnum);
}


void System::exit(int errnum, const String &errmsg)
{
	finalize();
	::printf("%s\n", errmsg.cStr());
	::fflush(stdout);
	::exit(errnum);
}

// my signal handler - calls the user function instead
void mySignalHandler(int sig)
{
	if(!lrt::System::lastBreakFunction) // no user func installed
		return;
	lrt::System::BreakType type = lrt::System::CTRL_C;
	if(sig == SIGTERM)
		type = lrt::System::TERMINATE;
	// actually call user function
	lrt::System::lastBreakFunction(type);
}

void System::setOnBreak(System::BreakFunction func, bool catchKill)
{
	if(lastBreakFunction) { // remove my signal handler, unload the function
		lastBreakFunction = 0;
		signal(SIGINT, SIG_DFL);
		signal(SIGTERM, SIG_DFL);
	}
	if(func) { // add my new signal handler
		lastBreakFunction = func;
		signal(SIGINT, mySignalHandler);
		if(catchKill)
			signal(SIGTERM, mySignalHandler);
	}
}

void System::initialize()
{
	init = true;
}

void System::print(const char *cStr)
{
	if(!init) initialize();
	fputs(cStr, stdout);
}

void System::print(const String &str)
{
	print((const char *)str.cStr());
}

void System::println()
{
	if(!init) initialize();
	fputs("\n", stdout);
	fflush(stdout);
}

void System::println(const String &str)
{
	if(!init) initialize();
	fputs(str.cStr(), stdout);
	fputs("\n", stdout);
	fflush(stdout);
}

char System::read()
{
	if(!init) initialize();
	return (char) getchar();
}

String System::readLine(const String& query)
{
	if(!init) initialize();
	if(query.length() > 0) print(query);
	fflush(stdout);
	String ret;
	char ch = 0;
	while(true)
	{
	    fread(&ch, 1, 1, stdin);
	    if(ch == '\n') break;
	    ret += ch;
	}
	return ret;
}

void System::setPos(int x, int y)
{
#ifdef __UNIX__ // Using Unix, we can assume that terminal emulation is supported.
	::printf("\033[%d;%dH", y, x);
#endif
}

void System::copyMemory(const void* src, void* dest, int length)
{
#ifndef __NOCHECK__
	if(length < 0)
		System::exit(-1, "length < 0 in System::copyMemory()");
#endif
	::memcpy(dest, src, length);
}

void System::fillMemory(void* ptr, int length, char data)
{
#ifndef __NOCHECK__
	if(length < 0)
		System::exit(-1, "length < 0 in System::fillMemory()");
#endif
	::memset(ptr, data, length);
}

void System::finalize()
{
	File::finalize();
	init = false;
}

} // namespace


#endif
