/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-wav.h"
#include "ogmrip-mplayer.h"
#include "ogmrip-plugin.h"

#include "ogmjob-exec.h"

#include <stdio.h>
#include <glib/gi18n-lib.h>

#define OGMRIP_WAV_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_WAV, OGMRipWavPriv))

struct _OGMRipWavPriv
{
  gboolean header;
};

static gint ogmrip_wav_run (OGMJobSpawn *spawn);

G_DEFINE_TYPE (OGMRipWav, ogmrip_wav, OGMRIP_TYPE_AUDIO)

static gchar **
ogmrip_wav_command (OGMRipAudio *audio, gboolean header, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  argv = ogmrip_mplayer_wav_command (audio, FALSE, NULL, output);
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static void
ogmrip_wav_class_init (OGMRipWavClass *klass)
{
  OGMJobSpawnClass *spawn_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  spawn_class->run = ogmrip_wav_run;

  g_type_class_add_private (klass, sizeof (OGMRipWavPriv));
}

static void
ogmrip_wav_init (OGMRipWav *wav)
{
  wav->priv = OGMRIP_WAV_GET_PRIVATE (wav);
  wav->priv->header = TRUE;
}

static gint
ogmrip_wav_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gchar **argv;
  gint result;

  argv = ogmrip_wav_command (OGMRIP_AUDIO (spawn), 
      OGMRIP_WAV (spawn)->priv->header, NULL, NULL);
  if (!argv)
    return OGMJOB_RESULT_ERROR;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_mplayer_wav_watch, spawn, TRUE, FALSE, FALSE);
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
  g_object_unref (child);

  result = OGMJOB_SPAWN_CLASS (ogmrip_wav_parent_class)->run (spawn);

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);

  return result;
}

/**
 * ogmrip_wav_new:
 * @audio: An #OGMDvdAudioStream
 * @output: The output file
 *
 * Creates a new #OGMRipWav
 *
 * Returns: The new #OGMRipWav
 */
OGMJobSpawn *
ogmrip_wav_new (OGMDvdAudioStream *audio, const gchar *output)
{
  g_return_val_if_fail (audio != NULL, NULL);
  g_return_val_if_fail (output && *output, NULL);

  return g_object_new (OGMRIP_TYPE_WAV, "stream", audio, "output", output, NULL);
}

/**
 * ogmrip_wav_set_header:
 * @wav: An #OGMRipWav
 * @header: %TRUE to have the header
 *
 * Sets whether to include the wave header.
 */
void
ogmrip_wav_set_header (OGMRipWav *wav, gboolean header)
{
  g_return_if_fail (OGMRIP_IS_WAV (wav));

  wav->priv->header = header;
}

/**
 * ogmrip_wav_get_header:
 * @wav: An #OGMRipWav
 *
 * Gets whether to include the wave header.
 *
 * Returns: %TRUE to include the header
 */
gboolean
ogmrip_wav_get_header (OGMRipWav *wav)
{
  g_return_val_if_fail (OGMRIP_IS_WAV (wav), FALSE);

  return wav->priv->header;
}

static OGMRipPluginAudioCodec wav_plugin =
{
  NULL,
  G_TYPE_NONE,
  "wav",
  N_("Wave (uncompressed PCM)"),
  OGMRIP_FORMAT_PCM
};

OGMRipPluginAudioCodec *
ogmrip_init_plugin (void)
{
  wav_plugin.type = OGMRIP_TYPE_WAV;

  return &wav_plugin;
}

