/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/**
 * SECTION:ogmrip-video
 * @title: OGMRipVideo
 * @short_description: Base class for video codecs
 * @include: ogmrip-video.h
 */

#include "ogmrip-video.h"
#include "ogmrip-version.h"

#include <ogmjob-exec.h>
#include <string.h>
#include <stdio.h>

#define OGMRIP_VIDEO_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_VIDEO, OGMRipVideoPriv))

#define ROUND(x) ((gint) ((x) + 0.5) != (gint) (x) ? ((gint) ((x) + 0.5)) : ((gint) (x)))

#if MPLAYER_CHECK_VERSION(1,0,0,8)
#define CROP_FRAMES 12
#else
#define CROP_FRAMES 30
#endif /* MPLAYER_CHECK_VERSION */

#define MAX_FRAMES 100

struct _OGMRipVideoPriv
{
  gchar *log;
  gdouble bpp;
  gdouble quantizer;
  gint bitrate;
  guint angle;
  guint nframes;
  guint pass;
  guint threads;
  guint crop_x;
  guint crop_y;
  guint crop_width;
  guint crop_height;
  guint scale_width;
  guint scale_height;
  guint max_b_frames;
  gboolean grayscale;
  gboolean cartoon;
  gboolean denoise;
  gboolean trellis;
  gboolean deblock;
  gboolean dering;
  gboolean pullup;
  gboolean turbo;
  gboolean qpel;
  gboolean v4mv;
  OGMDvdAudioStream *astream;
  OGMRipQualityType quality;
  OGMRipScalerType scaler;
  OGMRipDeintType deint;
  OGMJobSpawn *crop_child;
  gboolean crop_canceled;
  guint crop_frames;
};

typedef struct
{
  gint val;
  gint ref;
} CropInfo;

static void ogmrip_video_dispose  (GObject     *gobject);
static void ogmrip_video_finalize (GObject     *gobject);
static void ogmrip_video_cancel   (OGMJobSpawn *spawn);

// static const gchar *deinterlacer[] = { "lb", "li", "ci", "md", "fd", "l5", "kerndeint", "yadif=3:1,mcdeint=2:1:10" };
static const gchar *deinterlacer[] = { "lb", "li", "ci", "md", "fd", "l5", "kerndeint", "yadif=0" };

static gint
g_ulist_compare (CropInfo *info, gint val)
{
  return info->val - val;
}

static GSList *
g_ulist_add (GSList *ulist, gint val)
{
  GSList *ulink;
  CropInfo *info;

  ulink = g_slist_find_custom (ulist, GINT_TO_POINTER (val), (GCompareFunc) g_ulist_compare);
  if (ulink)
  {
    info = ulink->data;
    info->ref ++;
  }
  else
  {
    info = g_new0 (CropInfo, 1);
    info->val = val;
    info->ref = 1;

    ulist = g_slist_append (ulist, info);
  }

  return ulist;
}

static gint
g_ulist_get_most_frequent (GSList *ulist)
{
  GSList *ulink;
  CropInfo *info, *umax;

  if (!ulist)
    return 0;

  umax = ulist->data;

  for (ulink = ulist; ulink; ulink = ulink->next)
  {
    info = ulink->data;

    if (info->ref > umax->ref)
      umax = info;
  }

  return umax->val;
}

static void
g_ulist_free (GSList *ulist)
{
  g_slist_foreach (ulist, (GFunc) g_free, NULL);
  g_slist_free (ulist);
}

static gdouble
ogmrip_video_crop_watch (OGMJobExec *exec, const gchar *buffer, OGMRipVideo *video)
{
  gchar *str;

  static guint frame = 0;
  static GSList *crop_x = NULL, *crop_y = NULL,
                *crop_w = NULL, *crop_h = NULL;

  str = strstr (buffer, "-vf crop=");
  if (str)
  {
    gint x, y, w, h;

    if (sscanf (str, "-vf crop=%d:%d:%d:%d", &w, &h, &x, &y) == 4)
    {
      crop_w = g_ulist_add (crop_w, w);
      crop_h = g_ulist_add (crop_h, h);
      crop_x = g_ulist_add (crop_x, x);
      crop_y = g_ulist_add (crop_y, y);
    }

    frame ++;
    if (frame == video->priv->nframes - 2)
    {
      gint x, y, w, h;

      w = g_ulist_get_most_frequent (crop_w);
      g_ulist_free (crop_w);
      crop_w = NULL;

      h = g_ulist_get_most_frequent (crop_h);
      g_ulist_free (crop_h);
      crop_h = NULL;

      x = g_ulist_get_most_frequent (crop_x);
      g_ulist_free (crop_x);
      crop_x = NULL;

      y = g_ulist_get_most_frequent (crop_y);
      g_ulist_free (crop_y);
      crop_y = NULL;

      ogmrip_video_set_crop_size (video, x, y, w, h);

      frame = 0;

      return 1.0;
    }

    return frame / (gdouble) (video->priv->nframes - 2);
  }
  else
  {
    gdouble d;

    if (sscanf (buffer, "V: %lf", &d))
    {
      video->priv->crop_frames ++;

      if (video->priv->crop_frames >= MAX_FRAMES)
        ogmjob_spawn_cancel (OGMJOB_SPAWN (exec));
    }
  }

  return -1.0;
}

static gchar **
ogmrip_video_crop_command (OGMRipVideo *video, const gchar *input, const gchar *output, gboolean step)
{
  OGMDvdTime time_;
  OGMDvdTitle *title;
  OGMRipDeintType deint;
  GPtrArray *argv;

  GString *filter;
  const gchar *device;
  gint vid;

#if MPLAYER_CHECK_VERSION(1,0,0,8)
  gint sstep;
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  argv = g_ptr_array_new ();

#if MPLAYER_CHECK_VERSION(1,0,0,8)
  g_ptr_array_add (argv, g_strdup ("mplayer"));
  g_ptr_array_add (argv, g_strdup ("-nolirc"));
  g_ptr_array_add (argv, g_strdup ("-vo"));
  g_ptr_array_add (argv, g_strdup ("null"));
#else /* MPLAYER_CHECK_VERSION(1,0,0,8) */
  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("lavc"));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup ("/dev/null"));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */

  g_ptr_array_add (argv, g_strdup ("-nosound"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));

  filter = g_string_new (NULL);

  deint = ogmrip_video_get_deinterlacer (video);
  if (deint != OGMRIP_DEINT_NONE)
  {
    if (deint == OGMRIP_DEINT_KERNEL || OGMRIP_DEINT_YADIF)
      g_string_append (filter, deinterlacer[deint - 1]);
    else
      g_string_append_printf (filter, "pp=%s", deinterlacer[deint - 1]);
  }

  if (filter->len > 0)
    g_string_append_c (filter, ',');
  g_string_append (filter, "cropdetect");

  g_ptr_array_add (argv, g_strdup ("-vf"));
  g_ptr_array_add (argv, g_string_free (filter, FALSE));

  ogmrip_codec_get_length (OGMRIP_CODEC (video), &time_);

#if MPLAYER_CHECK_VERSION(1,0,0,8)
  if (step)
  {
    sstep = (time_.hour * 3600 + time_.min * 60 + time_.sec) / (3 * video->priv->nframes) + 1;
    g_ptr_array_add (argv, g_strdup ("-sstep"));
    g_ptr_array_add (argv, g_strdup_printf ("%d", sstep));
  }
  else
  {
    g_ptr_array_add (argv, g_strdup ("-ss"));
    g_ptr_array_add (argv, g_strdup_printf ("%02u:%02u:%02u",
          time_.hour / 4, time_.min / 4 + (time_.hour % 4) * 15, time_.sec / 4 + (time_.min % 4) * 15));
  }
#else /* MPLAYER_CHECK_VERSION(1,0,0,8) */
  g_ptr_array_add (argv, g_strdup ("-ss"));
  g_ptr_array_add (argv, g_strdup_printf ("%02u:%02u:%02u",
        time_.hour / 4, time_.min / 4 + (time_.hour % 4) * 15, time_.sec / 4 + (time_.min % 4) * 15));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */

  g_ptr_array_add (argv, g_strdup ("-frames"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", video->priv->nframes));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_CHECK_VERSION(1,0,0,1)
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_CHECK_VERSION(1,0,0,1) */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,1) */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}
G_DEFINE_ABSTRACT_TYPE (OGMRipVideo, ogmrip_video, OGMRIP_TYPE_CODEC)

static void
ogmrip_video_class_init (OGMRipVideoClass *klass)
{
  GObjectClass *gobject_class;
  OGMJobSpawnClass *spawn_class;

  gobject_class = G_OBJECT_CLASS (klass);
  gobject_class->dispose = ogmrip_video_dispose;
  gobject_class->finalize = ogmrip_video_finalize;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);
  spawn_class->cancel = ogmrip_video_cancel;

  g_type_class_add_private (klass, sizeof (OGMRipVideoPriv));
}

static void
ogmrip_video_init (OGMRipVideo *video)
{
  video->priv = OGMRIP_VIDEO_GET_PRIVATE (video);
  video->priv->scaler = OGMRIP_SCALER_GAUSS;
  video->priv->astream = NULL;
  video->priv->bitrate = 800000;
  video->priv->quantizer = -1.0;
  video->priv->trellis = TRUE;
  video->priv->turbo = FALSE;
  video->priv->v4mv = TRUE;
  video->priv->max_b_frames = 2;
  video->priv->nframes = CROP_FRAMES;
  video->priv->angle = 1;
  video->priv->bpp = 0.25;
}

static void
ogmrip_video_dispose (GObject *gobject)
{
  OGMRipVideo *video;

  video = OGMRIP_VIDEO (gobject);

  if (video->priv->astream)
    ogmdvd_stream_unref (OGMDVD_STREAM (video->priv->astream));
  video->priv->astream = NULL;

  G_OBJECT_CLASS (ogmrip_video_parent_class)->dispose (gobject);
}

static void
ogmrip_video_finalize (GObject *gobject)
{
  OGMRipVideo *video;

  video = OGMRIP_VIDEO (gobject);

  g_free (video->priv->log);
  video->priv->log = NULL;

  G_OBJECT_CLASS (ogmrip_video_parent_class)->finalize (gobject);
}

static void
ogmrip_video_cancel (OGMJobSpawn *spawn)
{
  OGMRipVideo *video;

  video = OGMRIP_VIDEO (spawn);

  if (video->priv->crop_child)
  {
    ogmjob_spawn_cancel (video->priv->crop_child);
    video->priv->crop_canceled = TRUE;
  }

  OGMJOB_SPAWN_CLASS (ogmrip_video_parent_class)->cancel (spawn);
}

/**
 * ogmrip_video_get_ensure_sync:
 * @video: an #OGMRipVideo
 *
 * Gets the audio stream that will be encoded along with the video to ensure
 * the A/V synchronization.
 *
 * Returns: the #OGMDvdAudioStream, or NULL
 */
OGMDvdAudioStream * 
ogmrip_video_get_ensure_sync (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  return video->priv->astream;
}

/**
 * ogmrip_video_set_ensure_sync:
 * @video: an #OGMRipVideo
 * @stream: an #OGMDvdAudioStream
 *
 * Sets the audio stream that will be encoded along with the video to ensure
 * the A/V synchronization.
 */
void
ogmrip_video_set_ensure_sync (OGMRipVideo *video, OGMDvdAudioStream *stream)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  if (video->priv->astream != stream)
  {
    if (stream)
      ogmdvd_stream_ref (OGMDVD_STREAM (stream));

    if (video->priv->astream)
      ogmdvd_stream_unref (OGMDVD_STREAM (video->priv->astream));

    video->priv->astream = stream;
  }
}

/**
 * ogmrip_video_set_angle:
 * @video: an #OGMRipVideo
 * @angle: the angle
 *
 * Sets the angle to encode.
 */
void
ogmrip_video_set_angle (OGMRipVideo *video, guint angle)
{
  OGMDvdTitle *title;

  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));

  g_return_if_fail (angle > 0 && angle <= ogmdvd_title_get_n_angles (title));

  video->priv->angle = angle;
}

/**
 * ogmrip_video_get_angle:
 * @video: an #OGMRipVideo
 *
 * Gets the current angle.
 *
 * Returns: the angle, or -1
 */
gint
ogmrip_video_get_angle (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->angle;
}

/**
 * ogmrip_video_get_log:
 * @video: an #OGMRipVideo
 *
 * Gets the name of the file in which to dump first path information.
 *
 * Returns: the filename, or NULL 
 */
G_CONST_RETURN gchar *
ogmrip_video_get_log (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  return video->priv->log;
}

/**
 * ogmrip_video_set_log:
 * @video: an #OGMRipVideo
 * @log: the name of the log file
 *
 * Sets the name of the file in which to dump first path information.
 */
void
ogmrip_video_set_log (OGMRipVideo *video, const gchar *log)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  g_free (video->priv->log);
  video->priv->log = log ? g_strdup (log) : NULL;
}

/**
 * ogmrip_video_set_bitrate:
 * @video: an #OGMRipVideo
 * @bitrate: the video bitrate
 *
 * Sets the video bitrate to be used in bits/second, 4000 being the lowest and
 * 24000000 the highest available bitrates.
 */
void
ogmrip_video_set_bitrate (OGMRipVideo *video, guint bitrate)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->bitrate = CLAMP (bitrate, 4000, 24000000);
  video->priv->quantizer = -1.0;
}

/**
 * ogmrip_video_get_bitrate:
 * @video: an #OGMRipVideo
 *
 * Gets the video bitrate in bits/second.
 *
 * Returns: the video bitrate, or -1
 */
gint
ogmrip_video_get_bitrate (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->bitrate;
}

/**
 * ogmrip_video_set_quantizer:
 * @video: an #OGMRipVideo
 * @quantizer: the video quantizer
 *
 * Sets the video quantizer to be used, 1 being the lowest and 31 the highest
 * available quantizers.
 */
void
ogmrip_video_set_quantizer (OGMRipVideo *video, gdouble quantizer)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->quantizer = CLAMP (quantizer, 1, 31);
  video->priv->bitrate = -1;
}

/**
 * ogmrip_video_get_quantizer:
 * @video: an #OGMRipVideo
 *
 * Gets the video quantizer.
 *
 * Returns: the video quantizer, or -1
 */
gdouble
ogmrip_video_get_quantizer (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1.0);

  return video->priv->quantizer;
}

/**
 * ogmrip_video_set_bits_per_pixel:
 * @video: an #OGMRipVideo
 * @bpp: the number of bits per pixel
 *
 * Sets the number of bits per pixel to be used.
 */
void
ogmrip_video_set_bits_per_pixel (OGMRipVideo *video, gdouble bpp)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));
  g_return_if_fail (bpp > 0.0 && bpp <= 1.0);

  video->priv->bpp = bpp;
}

/**
 * ogmrip_video_get_bits_per_pixel:
 * @video: an #OGMRipVideo
 *
 * Gets the number of bits per pixel.
 *
 * Returns: the number of bits per pixel, or -1
 */
gdouble
ogmrip_video_get_bits_per_pixel (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1.0);

  return video->priv->bpp;
}

/**
 * ogmrip_video_set_pass:
 * @video: an #OGMRipVideo
 * @pass: the pass number
 *
 * Sets the pass number of the encoding.
 */
void
ogmrip_video_set_pass (OGMRipVideo *video, guint pass)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->pass = CLAMP (pass, 0, 3);
}

/**
 * ogmrip_video_get_pass:
 * @video: an #OGMRipVideo
 *
 * Gets the current pass number.
 *
 * Returns: the pass number, or -1
 */
gint
ogmrip_video_get_pass (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->pass;
}

/**
 * ogmrip_video_set_threads:
 * @video: an #OGMRipVideo
 * @threads: the number of threads
 *
 * Sets the number of threads to be used.
 */
void
ogmrip_video_set_threads (OGMRipVideo *video, guint threads)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->threads = MAX (threads, 0);
}

/**
 * ogmrip_video_get_threads:
 * @video: an #OGMRipVideo
 *
 * Gets the number of threads.
 *
 * Returns: the number of threads, or -1
 */
gint
ogmrip_video_get_threads (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->threads;
}

/**
 * ogmrip_video_set_scaler:
 * @video: an #OGMRipVideo
 * @scaler: an #OGMRipScalerType
 *
 * Sets the software scaler to be used.
 */
void
ogmrip_video_set_scaler (OGMRipVideo *video, OGMRipScalerType scaler)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->scaler = scaler;
}

/**
 * ogmrip_video_get_scaler:
 * @video: an #OGMRipVideo
 *
 * Gets the current software scaler.
 *
 * Returns: the software scaler, or -1
 */
gint
ogmrip_video_get_scaler (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->scaler;
}

/**
 * ogmrip_video_set_deinterlacer:
 * @video: an #OGMRipVideo
 * @deint: an #OGMRipDeintType
 *
 * Sets the deinterlacer to be used.
 */
void
ogmrip_video_set_deinterlacer (OGMRipVideo *video, OGMRipDeintType deint)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->deint = deint;
}

/**
 * ogmrip_video_get_deinterlacer:
 * @video: an #OGMRipVideo
 *
 * Gets the currnet deinterlacer.
 *
 * Returns: the deinterlacer, or -1
 */
gint
ogmrip_video_get_deinterlacer (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->deint;
}

/**
 * ogmrip_video_set_pullup:
 * @video: an #OGMRipVideo
 * @pullup: %TRUE to inverse telecine
 *
 * Sets whether an inverse telecine filter will be applied
 */
void
ogmrip_video_set_pullup (OGMRipVideo *video, gboolean pullup)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->pullup = pullup;
}

/**
 * ogmrip_video_get_pullup:
 * @video: an #OGMRipVideo
 *
 * Gets whether an inverse telecine filter will be applied
 *
 * Returns: %TRUE if inverse telecine
 */
gboolean
ogmrip_video_get_pullup (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->pullup;
}

/**
 * ogmrip_video_set_trellis:
 * @video: an #OGMRipVideo
 * @trellis: %TRUE to enable trellis quantization
 *
 * Sets whether trellis quantization will be enabled.
 */
void
ogmrip_video_set_trellis (OGMRipVideo *video, gboolean trellis)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->trellis = trellis;
}

/**
 * ogmrip_video_get_trellis:
 * @video: an #OGMRipVideo
 *
 * Gets whether trellis quantization is enabled.
 *
 * Returns: %TRUE if trellis quantization is enabled
 */
gboolean
ogmrip_video_get_trellis (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->trellis;
}

/**
 * ogmrip_video_set_4mv:
 * @video: an #OGMRipVideo
 * @v4mv: %TRUE to allow 4 motion vectors per macroblock
 *
 * Sets whether to allow 4 motion vectors per macroblock.
 */
void
ogmrip_video_set_4mv (OGMRipVideo *video, gboolean v4mv)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->v4mv = v4mv;
}

/**
 * ogmrip_video_get_4mv:
 * @video: an #OGMRipVideo
 *
 * Gets whether 4 motion vectors per macroblock are allowed.
 *
 * Returns: %TRUE if 4 motion vectors per macroblock are allowed
 */
gboolean
ogmrip_video_get_4mv (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->v4mv;
}

/**
 * ogmrip_video_set_qpel:
 * @video: an #OGMRipVideo
 * @qpel: %TRUE to use quarter pel motion compensation
 *
 * Sets whether to use quarter pel motion compensation.
 */
void
ogmrip_video_set_qpel (OGMRipVideo *video, gboolean qpel)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->qpel = qpel;
}

/**
 * ogmrip_video_get_qpel:
 * @video: an #OGMRipVideo
 *
 * Sets whether quarter pel motion compensation is used.
 *
 * Returns: %TRUE if quarter pel motion compensation is used
 */
gboolean
ogmrip_video_get_qpel (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->qpel;
}

/**
 * ogmrip_video_set_turbo:
 * @video: an #OGMRipVideo
 * @turbo: %TRUE to enable turbo
 *
 * Sets whether to enable turbo.
 */
void
ogmrip_video_set_turbo (OGMRipVideo *video, gboolean turbo)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->turbo = turbo;
}

/**
 * ogmrip_video_get_turbo:
 * @video: an #OGMRipVideo
 *
 * Gets whether turbo is enabled.
 *
 * Returns: %TRUE if turbo is enabled
 */
gboolean
ogmrip_video_get_turbo (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->turbo;
}

/**
 * ogmrip_video_set_grayscale:
 * @video: an #OGMRipVideo
 * @grayscale: %TRUE if movie is grayscale
 *
 * Sets whether the movie is grayscale.
 */
void
ogmrip_video_set_grayscale (OGMRipVideo *video, gboolean grayscale)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->grayscale = grayscale;
}

/**
 * ogmrip_video_get_grayscale:
 * @video: an #OGMRipVideo
 *
 * Gets whether the movie is grayscale.
 *
 * Returns: %TRUE if movie is grayscale
 */
gboolean
ogmrip_video_get_grayscale (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->grayscale;
}

/**
 * ogmrip_video_set_cartoon:
 * @video: an #OGMRipVideo
 * @cartoon: %TRUE if movie is a cartoon
 *
 * Sets whether the movie is a cartoon.
 */
void
ogmrip_video_set_cartoon (OGMRipVideo *video, gboolean cartoon)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->cartoon = cartoon;
}

/**
 * ogmrip_video_get_cartoon:
 * @video: an #OGMRipVideo
 *
 * Gets whether the movie is a cartoon.
 *
 * Returns: %TRUE if movie is a cartoon
 */
gboolean
ogmrip_video_get_cartoon (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->cartoon;
}

/**
 * ogmrip_video_set_denoise:
 * @video: an #OGMRipVideo
 * @denoise: %TRUE to reduce image noise
 *
 * Sets whether to reduce image noise.
 */
void
ogmrip_video_set_denoise (OGMRipVideo *video, gboolean denoise)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->denoise = denoise;
}

/**
 * ogmrip_video_get_denoise:
 * @video: an #OGMRipVideo
 *
 * Gets whether to reduce image noise.
 *
 * Returns: %TRUE to reduce image noise
 */
gboolean
ogmrip_video_get_denoise (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->denoise;
}

/**
 * ogmrip_video_set_max_b_frames:
 * @video: an #OGMRipVideo
 * @max_b_frames: the maximum number of B-frames
 *
 * Sets the maximum number of B-frames to put between I/P-frames.
 */
void
ogmrip_video_set_max_b_frames (OGMRipVideo *video, guint max_b_frames)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->max_b_frames = MIN (max_b_frames, 4);
}

/**
 * ogmrip_video_get_max_b_frames:
 * @video: an #OGMRipVideo
 *
 * Gets the maximum number of B-frames to put between I/P-frames.
 *
 * Returns: the maximum number of B-frames, or -1
 */
gint
ogmrip_video_get_max_b_frames (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->max_b_frames;
}

/**
 * ogmrip_video_set_quality:
 * @video: an #OGMRipVideo
 * @quality: the #OGMRipQualityType
 *
 * Sets the quality of the encoding.
 */
void
ogmrip_video_set_quality (OGMRipVideo *video, OGMRipQualityType quality)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->quality = MIN (quality, OGMRIP_QUALITY_FAST);
}

/**
 * ogmrip_video_get_quality:
 * @video: an #OGMRipVideo
 *
 * Gets the quality of the encoding.
 *
 * Returns: the #OGMRipQualityType, or -1
 */
gint
ogmrip_video_get_quality (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  return video->priv->quality;
}

/**
 * ogmrip_video_set_deblock:
 * @video: an #OGMRipVideo
 * @deblock: %TRUE to apply a deblocking filter
 *
 * Sets whether to apply a deblocking filter.
 */
void
ogmrip_video_set_deblock (OGMRipVideo *video, gboolean deblock)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->deblock = deblock;
}

/**
 * ogmrip_video_get_deblock:
 * @video: an #OGMRipVideo
 *
 * Gets whether a deblocking filter will be applied.
 *
 * Returns: %TRUE if a deblocking filter will be applied
 */
gboolean
ogmrip_video_get_deblock (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->deblock;
}

/**
 * ogmrip_video_set_dering:
 * @video: an #OGMRipVideo
 * @dering: %TRUE to apply a deringing filter
 *
 * Sets whether to apply a deringing filter.
 */
void
ogmrip_video_set_dering (OGMRipVideo *video, gboolean dering)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  video->priv->dering = dering;
}

/**
 * ogmrip_video_get_dering:
 * @video: an #OGMRipVideo
 *
 * Gets whether a deringing filter will be applied.
 *
 * Returns: %TRUE if a deringing filter will be applied
 */
gboolean
ogmrip_video_get_dering (OGMRipVideo *video)
{
  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  return video->priv->dering;
}

/**
 * ogmrip_video_get_start_delay:
 * @video: an #OGMRipVideo
 *
 * Gets the start delay that must be applied to audio streams when merging.
 *
 * Returns: the start delay, or -1
 */
gint
ogmrip_video_get_start_delay (OGMRipVideo *video)
{
  OGMRipVideoClass *klass;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), -1);

  klass = OGMRIP_VIDEO_GET_CLASS (video);

  if (klass->get_start_delay)
    return (* klass->get_start_delay) (video);

  return 0;
}

/**
 * ogmrip_video_get_raw_size:
 * @video: an #OGMRipVideo
 * @width: a pointer to store the width
 * @height: a pointer to store the height
 *
 * Gets the raw size of the video.
 */
void
ogmrip_video_get_raw_size (OGMRipVideo *video, guint *width, guint *height)
{
  OGMDvdTitle *title;

  g_return_if_fail (OGMRIP_IS_VIDEO (video));
  g_return_if_fail (width != NULL);
  g_return_if_fail (height != NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));

  g_return_if_fail (title != NULL);

  ogmdvd_title_get_size (title, width, height);
}

/**
 * ogmrip_video_get_crop_size:
 * @video: an #OGMRipVideo
 * @x: a pointer to store the cropped x position
 * @y: a pointer to store the cropped y position
 * @width: a pointer to store the cropped width
 * @height: a pointer to store the cropped height
 *
 * Gets whether the video will be cropped and the crop size.
 *
 * Returns: %TRUE if the video will be cropped
 */
gboolean
ogmrip_video_get_crop_size (OGMRipVideo *video, guint *x, guint *y, guint *width, guint *height)
{
  guint raw_width, raw_height;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);
  g_return_val_if_fail (x != NULL, FALSE);
  g_return_val_if_fail (y != NULL, FALSE);
  g_return_val_if_fail (width != NULL, FALSE);
  g_return_val_if_fail (height != NULL, FALSE);

  ogmrip_video_get_raw_size (video, &raw_width, &raw_height);

  *x = video->priv->crop_x;
  *y = video->priv->crop_y;
  *width = video->priv->crop_width;
  *height = video->priv->crop_height;

  if (*x == 0 && *y == 0 && *width == 0 && *height == 0)
  {
    *width = raw_width;
    *height = raw_height;
  }

  if (*x == 0 && *y == 0 && *width == raw_width && *height == raw_height)
    return FALSE;

  return TRUE;
}

/**
 * ogmrip_video_set_crop_size:
 * @video: an #OGMRipVideo
 * @x: the cropped x position
 * @y: the cropped y position
 * @width: the cropped width
 * @height: the cropped height
 *
 * Sets the crop size of the movie.
 */
void
ogmrip_video_set_crop_size (OGMRipVideo *video, guint x, guint y, guint width, guint height)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  if (width > 0 && height > 0)
  {
    video->priv->crop_x = x;
    video->priv->crop_y = y;
    video->priv->crop_width = (width / 16) * 16;
    video->priv->crop_height = (height / 16) * 16;
  }
}

/**
 * ogmrip_video_get_scale_size:
 * @video: an #OGMRipVideo
 * @width: a pointer to store the scaled width
 * @height: a pointer to store the scaled height
 *
 * Gets whether the video will be scaled and the scale size.
 *
 * Returns: %TRUE if the video will be scaled
 */
gboolean
ogmrip_video_get_scale_size (OGMRipVideo *video, guint *width, guint *height)
{
  guint raw_width, raw_height;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);
  g_return_val_if_fail (width != NULL, FALSE);
  g_return_val_if_fail (height != NULL, FALSE);

  ogmrip_video_get_raw_size (video, &raw_width, &raw_height);

  *width = video->priv->scale_width;
  *height = video->priv->scale_height;

  if (*width == 0 && *height == 0)
  {
    *width = raw_width;
    *height = raw_height;
  }

  if (*width == raw_width && *height == raw_height)
    return FALSE;

  return TRUE;
}

/**
 * ogmrip_video_set_scale_size:
 * @video: an #OGMRipVideo
 * @width: the scaled width
 * @height: the scaled height
 *
 * Sets the scaled size of the movie.
 */
void
ogmrip_video_set_scale_size (OGMRipVideo *video, guint width, guint height)
{
  g_return_if_fail (OGMRIP_IS_VIDEO (video));
  g_return_if_fail (height > 0);
  g_return_if_fail (width > 0);

  video->priv->scale_width = width;
  video->priv->scale_height = height;
}

/**
 * ogmrip_video_autoscale:
 * @video: an #OGMRipVideo
 *
 * Autodetects the scaling parameters.
 */
void
ogmrip_video_autoscale (OGMRipVideo *video)
{
  OGMDvdTitle *title;
  guint anumerator, adenominator;
  guint rnumerator, rdenominator;
  guint crop_width, crop_height;
  guint raw_width, raw_height;
  gfloat ratio, bpp;

  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_if_fail (title != NULL);

  ogmrip_video_get_raw_size (video, &raw_width, &raw_height);

  crop_width = video->priv->crop_width > 0 ? video->priv->crop_width : raw_width;
  crop_height = video->priv->crop_height > 0 ? video->priv->crop_height : raw_height;

  ogmdvd_title_get_aspect_ratio (title, &anumerator, &adenominator);
  ogmdvd_title_get_framerate (title, &rnumerator, &rdenominator);

  ratio = (crop_width * raw_height * anumerator) / (gdouble) (crop_height * raw_width * adenominator);

  for (video->priv->scale_width = 320; video->priv->scale_width <= 720; video->priv->scale_width += 16)
  {
    video->priv->scale_height = 16 * ROUND (video->priv->scale_width / ratio / 16);

    bpp = (video->priv->bitrate * rdenominator) / 
      (gdouble) (video->priv->scale_width * video->priv->scale_height * rnumerator);

    if (bpp < video->priv->bpp)
      break;
  }

  video->priv->scale_width = MIN (video->priv->scale_width, 720);
}

/**
 * ogmrip_video_autobitrate:
 * @video: an #OGMRipVideo
 * @nonvideo_size: the size of the non video streams
 * @overhead_size: the size of the overhead
 * @total_size: the total targetted size
 *
 * Autodetects the video bitrate.
 */
void
ogmrip_video_autobitrate (OGMRipVideo *video, guint64 nonvideo_size, guint64 overhead_size, guint64 total_size)
{
  OGMDvdTitle *title;
  gdouble video_size, length;

  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_if_fail (title != NULL);

  video_size = total_size - nonvideo_size - overhead_size;
  length = ogmrip_codec_get_length (OGMRIP_CODEC (video), NULL);

  ogmrip_video_set_bitrate (video, (video_size * 8.) / length);
}

/**
 * ogmrip_video_autocrop:
 * @video: an #OGMRipVideo
 * @nframes: the number of frames
 *
 * Autodetects the cropping parameters.
 *
 * Returns: %FALSE, on error or cancel
 */
gboolean
ogmrip_video_autocrop (OGMRipVideo *video, guint nframes)
{
  gchar **argv;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), FALSE);

  argv = ogmrip_video_crop_command (video, NULL, NULL, TRUE);

  if (!nframes)
    video->priv->nframes = CROP_FRAMES;
  else
    video->priv->nframes = nframes + 5;

  video->priv->crop_child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (video->priv->crop_child),
      (OGMJobWatch) ogmrip_video_crop_watch, video, TRUE, FALSE, TRUE);

  video->priv->crop_frames = 0;
  video->priv->crop_canceled = FALSE;
  ogmjob_spawn_run (video->priv->crop_child, NULL);

  g_object_unref (video->priv->crop_child);
  video->priv->crop_child = NULL;

  if (video->priv->crop_frames >= MAX_FRAMES)
  {
    argv = ogmrip_video_crop_command (video, NULL, NULL, FALSE);

    video->priv->crop_child = ogmjob_exec_newv (argv);
    ogmjob_exec_add_watch_full (OGMJOB_EXEC (video->priv->crop_child),
        (OGMJobWatch) ogmrip_video_crop_watch, video, TRUE, FALSE, TRUE);

    video->priv->crop_frames = 0;
    video->priv->crop_canceled = FALSE;
    ogmjob_spawn_run (video->priv->crop_child, NULL);

    g_object_unref (video->priv->crop_child);
    video->priv->crop_child = NULL;
  }

  if (video->priv->crop_canceled)
    return FALSE;

  return TRUE;
}

