/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-
 *
 * nautilus-burn-drive.c: easy to use cd burner software
 *
 * Copyright (C) 2002-2004 Red Hat, Inc.
 * Copyright (C) 2005 William Jon McCann <mccann@jhu.edu>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * Authors: Alexander Larsson <alexl@redhat.com>
 *          Bastien Nocera <hadess@hadess.net>
 *          William Jon McCann <mccann@jhu.edu>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef __linux__

#include <sys/ioctl.h>
#include <sys/types.h>
#include <linux/cdrom.h>
#include <scsi/scsi.h>
#include <scsi/sg.h>

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>

#include "nautilus-burn-drive.h"
#include "nautilus-burn-drive-common.h"

static char **
read_lines (char *filename)
{
  char *contents;
  gsize len;
  char *p, *n;
  GPtrArray *array;

  if (g_file_get_contents (filename, &contents, &len, NULL))
  {
    array = g_ptr_array_new ();

    p = contents;
    while ((n = memchr (p, '\n', len - (p - contents))) != NULL)
    {
      *n = 0;
      g_ptr_array_add (array, g_strdup (p));
      p = n + 1;
    }
    if ((gsize) (p - contents) < len)
      g_ptr_array_add (array, g_strndup (p, len - (p - contents)));

    g_ptr_array_add (array, NULL);

    g_free (contents);
    return (char **) g_ptr_array_free (array, FALSE);
  }
  return NULL;
}

struct scsi_unit
{
  char *vendor;
  char *model;
  char *rev;
  int bus;
  int id;
  int lun;
  int type;
};

struct drive_unit
{
  NautilusBurnDriveProtocolType protocol;
  char *device;
  char *display_name;
  int speed;
  gboolean can_write_cdr;
  gboolean can_write_cdrw;
  gboolean can_write_dvdr;
  gboolean can_write_dvdram;
  gboolean can_read_dvd;
};

static char *drive_get_name (struct drive_unit *drive, struct scsi_unit *scsi_units, int n_scsi_units);

static void
linux_add_whitelist (struct drive_unit *drive_s, struct scsi_unit *scsi_units, int n_scsi_units)
{
  NautilusBurnWhiteDrive recorder_whitelist[] =
  {
    { "IOMEGA - CDRW9602EXT-B", TRUE, TRUE,  FALSE, FALSE },
    { "SONY - CD-R CDU948S",    TRUE, FALSE, FALSE, FALSE },
  };

  guint i;

  for (i = 0; i < G_N_ELEMENTS (recorder_whitelist); i++)
  {
    if (drive_s->display_name == NULL)
      continue;

    if (!strcmp (drive_s->display_name, recorder_whitelist[i].name))
    {
      drive_s->can_write_cdr = recorder_whitelist[i].can_write_cdr;
      drive_s->can_write_cdrw = recorder_whitelist[i].can_write_cdrw;
      drive_s->can_write_dvdr = recorder_whitelist[i].can_write_dvdr;
      drive_s->can_write_dvdram = recorder_whitelist[i].can_write_dvdram;
    }
  }
}

static void
get_scsi_units (char **device_str, char **devices, struct scsi_unit *scsi_units)
{
  char vendor[9], model[17], rev[5];
  int host_no, access_count, queue_depth, device_busy, online, channel;
  int scsi_id, scsi_lun, scsi_type;
  int i, j;

  j = 0;

  for (i = 0; device_str[i] != NULL && devices[i] != NULL; i++)
  {
    if (strcmp (device_str[i], "<no active device>") == 0)
      continue;
    if (sscanf (devices[i], "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d", 
          &host_no, &channel, &scsi_id, &scsi_lun, &scsi_type, 
          &access_count, &queue_depth, &device_busy, &online) != 9)
    {
      g_warning ("Couldn't match line in /proc/scsi/sg/devices\n");
      continue;
    }
    if (scsi_type == 5)
    {                           /* TYPE_ROM (include/scsi/scsi.h) */
      if (sscanf (device_str[i], "%8c\t%16c\t%4c", vendor, model, rev) != 3)
      {
        g_warning ("Couldn't match line /proc/scsi/sg/device_strs\n");
        continue;
      }
      vendor[8] = '\0';
      model[16] = '\0';
      rev[4] = '\0';

      scsi_units[j].vendor = g_strdup (g_strstrip (vendor));
      scsi_units[j].model = g_strdup (g_strstrip (model));
      scsi_units[j].rev = g_strdup (g_strstrip (rev));
      scsi_units[j].bus = host_no;
      scsi_units[j].id = scsi_id;
      scsi_units[j].lun = scsi_lun;
      scsi_units[j].type = scsi_type;

      j++;
    }
  }
}

static int
count_strings (char *p)
{
  int n_strings;

  n_strings = 0;
  while (*p != 0)
  {
    n_strings++;
    while (*p != '\t' && *p != 0)
      p++;
    if (*p == '\t')
      p++;
  }
  return n_strings;
}

static int
get_cd_scsi_id (const char *dev, int *bus, int *id, int *lun)
{
  int fd;
  char *devfile;
  struct
  {
    long mux4;
    long hostUniqueId;
  } m_idlun;

  devfile = g_strdup_printf ("/dev/%s", dev);
  fd = g_open (devfile, O_RDWR | O_NONBLOCK, 0);
  if (fd < 0)
    fd = g_open (devfile, O_RDONLY | O_NONBLOCK, 0);

  g_free (devfile);

  /*
   * Avoid problems with Valgrind
   */
  memset (&m_idlun, 0, sizeof (m_idlun));
  *bus = *id = *lun = -1;

  if (fd < 0)
  {
    g_warning ("Failed to open cd device %s\n", dev);
    return 0;
  }

  if (ioctl (fd, SCSI_IOCTL_GET_BUS_NUMBER, bus) < 0 || *bus < 0)
  {
    g_warning ("Failed to get scsi bus nr\n");
    close (fd);
    return 0;
  }

  if (ioctl (fd, SCSI_IOCTL_GET_IDLUN, &m_idlun) < 0)
  {
    g_warning ("Failed to get scsi id and lun\n");
    close (fd);
    return 0;
  }
  *id = m_idlun.mux4 & 0xFF;
  *lun = (m_idlun.mux4 >> 8) & 0xFF;

  close (fd);
  return 1;
}

static struct scsi_unit *
lookup_scsi_unit (int bus, int id, int lun, struct scsi_unit *scsi_units, int n_scsi_units)
{
  int i;

  for (i = 0; i < n_scsi_units; i++)
    if (scsi_units[i].bus == bus && scsi_units[i].id == id && scsi_units[i].lun == lun)
      return &scsi_units[i];

  return NULL;
}

static char *
get_scsi_cd_name (int bus, int id, int lun, const char *dev, struct scsi_unit *scsi_units, int n_scsi_units)
{
  struct scsi_unit *scsi_unit;

  scsi_unit = lookup_scsi_unit (bus, id, lun, scsi_units, n_scsi_units);
  if (scsi_unit == NULL)
    return g_strdup_printf (_("Unnamed SCSI Drive (%s)"), dev);

  return g_strdup_printf ("%s - %s", scsi_unit->vendor, scsi_unit->model);
}

static char *
drive_get_name (struct drive_unit *drive, struct scsi_unit *scsi_units, int n_scsi_units)
{
  char *filename, *line, *retval;
  char stdname[4], devfsname[15];
  int bus, id, lun, i;

  g_return_val_if_fail (drive != NULL, FALSE);

  /*
   * clean up the string again if we have devfs 
   */
  i = sscanf (drive->device, "%4s %14s", stdname, devfsname);
  if (i < 1)
  {                             /* should never happen */
    g_warning ("drive_get_name: drive->device string broken!");
    return NULL;
  }
  if (i == 2)
  {
    g_free (drive->device);
    drive->device = g_strdup (devfsname);
  }
  stdname[3] = '\0';
  devfsname[14] = '\0';         /* just in case */

  if (drive->protocol == NAUTILUS_BURN_DRIVE_PROTOCOL_SCSI)
  {
    get_cd_scsi_id (drive->device, &bus, &id, &lun);
    retval = get_scsi_cd_name (bus, id, lun, drive->device, scsi_units, n_scsi_units);
  }
  else
  {
    filename = g_strdup_printf ("/proc/ide/%s/model", stdname);
    if (!g_file_get_contents (filename, &line, NULL, NULL) || line == NULL)
    {
      g_free (filename);
      return NULL;
    }
    g_free (filename);

    i = strlen (line);
    if (line[i - 1] != '\n')
      retval = g_strdup (line);
    else
      retval = g_strndup (line, i - 1);

    g_free (line);
  }

  return retval;
}

static GList *
add_linux_cd_recorder (GList * drives, gboolean recorder_only, struct drive_unit *drive_s, struct scsi_unit *scsi_units, int n_scsi_units)
{
  int bus, id, lun;
  NautilusBurnDrive *drive;

  drive = nautilus_burn_drive_new ();

  drive->type = NAUTILUS_BURN_DRIVE_TYPE_CD_DRIVE;
  drive->display_name = g_strdup (drive_s->display_name);

  if (drive_s->protocol == NAUTILUS_BURN_DRIVE_PROTOCOL_SCSI)
  {
    drive->protocol = NAUTILUS_BURN_DRIVE_PROTOCOL_SCSI;
    if (!get_cd_scsi_id (drive_s->device, &bus, &id, &lun))
    {
      g_free (drive->display_name);
      g_free (drive);
      return drives;
    }
    drive->cdrecord_id = g_strdup_printf ("%d,%d,%d", bus, id, lun);
  }
  else
  {
    drive->protocol = NAUTILUS_BURN_DRIVE_PROTOCOL_IDE;
    /*
     * kernel >=2.5 can write cd w/o ide-scsi 
     */
    drive->cdrecord_id = g_strdup_printf ("/dev/%s", drive_s->device);
  }

  if (recorder_only)
  {
    drive->max_speed_write = get_device_max_write_speed (drive->device);
    if (drive->max_speed_write == -1)
      drive->max_speed_write = drive_s->speed;
  }
  else
  {
    /*
     * Have a wild guess, the drive should actually correct us 
     */
    drive->max_speed_write = drive_s->speed;
  }

  drive->device = g_strdup_printf ("/dev/%s", drive_s->device);
  drive->max_speed_read = drive_s->speed;

  if (drive_s->can_write_dvdr)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_RW_RECORDER;
  if (drive_s->can_write_dvdram)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_RAM_RECORDER;
  if (drive_s->can_write_cdr)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_CD_RECORDER;
  if (drive_s->can_write_cdrw)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_CDRW_RECORDER;
  if (drive_s->can_read_dvd)
  {
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_DRIVE;
    nautilus_burn_drive_add_dvd_plus (drive);
  }

  return g_list_append (drives, drive);
}

static GList *
add_linux_cd_drive (GList * drives, struct drive_unit *drive_s, struct scsi_unit *scsi_units, int n_scsi_units)
{
  NautilusBurnDrive *drive;

  drive = nautilus_burn_drive_new ();
  drive->type = NAUTILUS_BURN_DRIVE_TYPE_CD_DRIVE;
  drive->cdrecord_id = NULL;
  drive->display_name = g_strdup (drive_s->display_name);
  drive->device = g_strdup_printf ("/dev/%s", drive_s->device);
  drive->max_speed_write = 0;   /* Can't write */
  drive->max_speed_read = drive_s->speed;
  if (drive_s->can_read_dvd)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_DRIVE;

  return g_list_append (drives, drive);
}

static char *
get_cd_device_file (const char *str)
{
  char *devname;

  if (str[0] == 's')
  {
    devname = g_strdup_printf ("/dev/scd%c", str[2]);
    if (g_file_test (devname, G_FILE_TEST_EXISTS))
    {
      g_free (devname);
      return g_strdup_printf ("scd%c", str[2]);
    }
    g_free (devname);
  }
  return g_strdup (str);
}

GList *
linux_scan (gboolean recorder_only)
{
  char **device_str, **devices;
  char **drive_info;
  struct scsi_unit *scsi_units;
  struct drive_unit *drives;
  char *p, *t;
  int n_drives, maj, min, i, j;
  int n_scsi_units;
  int fd;
  FILE *file;
  GList *drives_list;
  gboolean have_devfs;

  /*
   * devfs creates and populates the /dev/cdroms directory when its mounted
   * the 'old style names' are matched with devfs names below.
   * The cdroms.device string gets cleaned up again in drive_get_name()
   * we need the oldstyle name to get device->display_name for ide.
   */
  have_devfs = FALSE;
  if (g_file_test ("/dev/.devfsd", G_FILE_TEST_EXISTS))
    have_devfs = TRUE;

  drive_info = read_lines ("/proc/sys/dev/cdrom/info");
  if (drive_info == NULL || drive_info[0] == NULL || drive_info[1] == NULL)
  {
    g_warning ("Couldn't read /proc/sys/dev/cdrom/info");
    return NULL;
  }
  if (!g_str_has_prefix (drive_info[2], "drive name:\t"))
    return NULL;
  p = drive_info[2] + strlen ("drive name:\t");
  while (*p == '\t')
    p++;
  n_drives = count_strings (p);
  drives = g_new0 (struct drive_unit, n_drives);

  for (j = 0; j < n_drives; j++)
  {
    t = strchr (p, '\t');
    if (t != NULL)
      *t = 0;
    drives[j].device = get_cd_device_file (p);
    /*
     * Assume its an IDE device for now 
     */
    drives[j].protocol = NAUTILUS_BURN_DRIVE_PROTOCOL_IDE;
    if (t != NULL)
      p = t + 1;
  }

  /*
   * we only have to check the first char, since only ide or scsi   
   * devices are listed in /proc/sys/dev/cdrom/info. It will always
   * be 'h' or 's'
   */
  n_scsi_units = 0;
  for (i = 0; i < n_drives; i++)
  {
    if (drives[i].device[0] == 's')
    {
      drives[i].protocol = NAUTILUS_BURN_DRIVE_PROTOCOL_SCSI;
      n_scsi_units++;
    }
  }

  if (n_scsi_units > 0)
  {
    /*
     * open /dev/sg0 to force loading of the sg module if not loaded yet 
     */
    fd = g_open ("/dev/sg0", O_RDWR, 0);
    if (fd >= 0)
      close (fd);

    devices = read_lines ("/proc/scsi/sg/devices");
    device_str = read_lines ("/proc/scsi/sg/device_strs");
    if (device_str == NULL)
    {
      g_warning ("Can't read /proc/scsi/sg/device_strs");
      g_strfreev (devices);
      return NULL;
    }

    scsi_units = g_new0 (struct scsi_unit, n_scsi_units);
    get_scsi_units (device_str, devices, scsi_units);

    g_strfreev (device_str);
    g_strfreev (devices);
  }
  else
    scsi_units = NULL;

  for (i = 3; drive_info[i] != NULL; i++)
  {
    if (g_str_has_prefix (drive_info[i], "Can write CD-R:"))
    {
      p = drive_info[i] + strlen ("Can write CD-R:");
      while (*p == '\t')
        p++;
      for (j = 0; j < n_drives; j++)
      {
        drives[j].can_write_cdr = *p++ == '1';

        /*
         * Skip tab 
         */
        p++;
      }
    }
    if (g_str_has_prefix (drive_info[i], "Can write CD-RW:"))
    {
      p = drive_info[i] + strlen ("Can write CD-RW:");
      while (*p == '\t')
        p++;
      for (j = 0; j < n_drives; j++)
      {
        drives[j].can_write_cdrw = *p++ == '1';

        /*
         * Skip tab 
         */
        p++;
      }
    }
    if (g_str_has_prefix (drive_info[i], "Can write DVD-R:"))
    {
      p = drive_info[i] + strlen ("Can write DVD-R:");
      while (*p == '\t')
        p++;
      for (j = 0; j < n_drives; j++)
      {
        drives[j].can_write_dvdr = *p++ == '1';

        /*
         Skip tab 
         */
        p++;
      }
    }
    if (g_str_has_prefix (drive_info[i], "Can write DVD-RAM:"))
    {
      p = drive_info[i] + strlen ("Can write DVD-RAM:");
      while (*p == '\t')
        p++;
      for (j = 0; j < n_drives; j++)
      {
        drives[j].can_write_dvdram = *p++ == '1';

        /*
         * Skip tab 
         */
        p++;
      }
    }
    if (g_str_has_prefix (drive_info[i], "Can read DVD:"))
    {
      p = drive_info[i] + strlen ("Can read DVD:");
      while (*p == '\t')
        p++;
      for (j = 0; j < n_drives; j++)
      {
        drives[j].can_read_dvd = *p++ == '1';

        /*
         * Skip tab 
         */
        p++;
      }
    }
    if (g_str_has_prefix (drive_info[i], "drive speed:"))
    {
      p = drive_info[i] + strlen ("drive speed:");
      while (*p == '\t')
        p++;
      for (j = 0; j < n_drives; j++)
      {
        drives[j].speed = atoi (p);

        /*
         * Skip tab 
         */
        p++;
      }
    }
  }
  g_strfreev (drive_info);

  /*
   * get kernel major.minor version 
   */
  file = fopen ("/proc/sys/kernel/osrelease", "r");
  if (file == NULL || fscanf (file, "%d.%d", &maj, &min) != 2)
  {
    g_warning ("Could not get kernel version.");
    maj = min = 0;
  }
  fclose (file);

  drives_list = NULL;
  for (i = n_drives - 1, j = 0; i >= 0; i--, j++)
  {
    if (have_devfs)
    {
      char *s;
      s = g_strdup_printf ("%s cdroms/cdrom%d", drives[i].device, j);
      g_free (drives[i].device);
      drives[i].device = s;
    }
    drives[i].display_name = drive_get_name (&drives[i],
    scsi_units, n_scsi_units);
    linux_add_whitelist (&drives[i], scsi_units, n_scsi_units);

    if ((drives[i].can_write_cdr || drives[i].can_write_cdrw || drives[i].can_write_dvdr || drives[i].can_write_dvdram) && 
        (drives[i].protocol == NAUTILUS_BURN_DRIVE_PROTOCOL_SCSI || (maj > 2) || (maj == 2 && min >= 5)))
    {
      drives_list = add_linux_cd_recorder (drives_list,
      recorder_only, &drives[i],
      scsi_units, n_scsi_units);
    }
    else if (!recorder_only)
    {
      drives_list = add_linux_cd_drive (drives_list,
      &drives[i], scsi_units, n_scsi_units);
    }
  }

  for (i = n_drives - 1; i >= 0; i--)
  {
    g_free (drives[i].display_name);
    g_free (drives[i].device);
  }
  g_free (drives);

  for (i = n_scsi_units - 1; i >= 0; i--)
  {
    g_free (scsi_units[i].vendor);
    g_free (scsi_units[i].model);
    g_free (scsi_units[i].rev);
  }
  g_free (scsi_units);

  return drives_list;
}

#endif

