/* ============================================================
 * File        : streambrowser.cpp
 * Author      : Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date        : 2003-12-22
 * Description : stream handling,
 *               inspired by mythtv http://www.mythtv.org
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// development snapshot, many todo's

#include <iostream>

#include <qnetwork.h>
#include <qdatetime.h>
#include <qpainter.h>
#include <qdir.h>
#include <qprocess.h>
#include <qstringlist.h>
#include <qdict.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qapplication.h>
#include <qurl.h>

#include "streambrowser.h"


using namespace std;

static QString emptyStr = "";
static QString nullStr  = QString::null;

ViewerWindow::ViewerWindow(QWidget *parent, const char *name, WFlags f)
               : QWidget(parent, name, f)
{
  hide();
  setCaption(name);
  parentWidget = parent;

  QVBoxLayout *vbox = new QVBoxLayout(this, (int)(10));

  QFont fo("monospace");
  fo.setPointSize( font().pointSize() );


  textEdit = new QTextEdit(this, name);
  textEdit->setFont(fo);
  textEdit->setReadOnly(true);
  textEdit->setBackgroundOrigin(QWidget::WindowOrigin);
  textEdit->setFrameStyle(QFrame::Box + QFrame::Raised);
  vbox->addWidget(textEdit);
}

void ViewerWindow::setContent(QString& content)
{
   textEdit->setText(content);
   textEdit->setFocus();
   show();
}

void ViewerWindow::keyPressEvent( QKeyEvent *k )
{
   bool handled = false;
   
   switch (k->key())
   {
      case Key_Q:
        handled = true;
        close();
      break;
      case Key_End:
        handled = true;
        close();
      break;
      case Key_Escape:
        handled = true;
        close();
      break;
   }

   if (handled && parentWidget)
   {
      parentWidget->setFocus();
      parentWidget->show();
   }
   
   if (!handled) k->ignore();
}

// ----------------------------------------------------------------------


DumpWindow::DumpWindow(QWidget *parent, const char *name, WFlags f)
               : QWidget(parent, name, f)
{
  setCaption(name);
  
  QVBoxLayout *vbox = new QVBoxLayout(this, (int)(10));

  QFont myfont("monospace");
  myfont.setPointSize( font().pointSize() );

  QHBoxLayout *hbox = new QHBoxLayout(vbox, (int)(10));
  
  newUrl = new QLineEdit(this, "newurl");
  commit = new QPushButton("load", this);
  hbox->addWidget(newUrl);
  hbox->addWidget(commit);

  connect(commit, SIGNAL(clicked()), this, SLOT(slotCommitClicked()));
  
  textEdit = new QTextEdit(this, name);
  textEdit->setTextFormat(PlainText);
  textEdit->setFont(myfont);
  textEdit->setReadOnly(true);
  textEdit->setBackgroundOrigin(QWidget::WindowOrigin);
  textEdit->setFrameStyle(QFrame::Box + QFrame::Raised);
  vbox->addWidget(textEdit);
  resize( 600, 400 );
  show();
}

void DumpWindow::slotCommitClicked()
{
  emit manualUrlEntered( newUrl->text() );
}

void DumpWindow::setText(QString text)
{
   textEdit->setText(text);
}

void DumpWindow::clear()
{
   textEdit->setText("");
}

// ----------------------------------------------------------------------


StreamBrowser::StreamBrowser(QWidget *parent, QWidget *focusProxy, StreamStorage *storage)
              : QObject()
{
    userMessage = "";
    video = false;
    m_dumpwindow = 0;
    markedStorage = NULL;
    playListActive = false;
    
    setItemDisplayFolded( -1, false);
    setFolderDisplayFolded( -1, false );

    sampler   = 0;
    converter = 0;

    streamStatus    = new StreamStatus(this);
    streamHarvester = new StreamHarvester();

    oldStatus = StreamStatus::idle;
    
    QString downloadPath = ""; //getenv("HOME");
    downloadPath += "."SUBPATH"/downloads";
    downloader = new Downloader( downloadPath, storage );
    
    connect( streamStatus,
             SIGNAL(statusChange()),
             this,
             SLOT(streamStatusChanged()) );

    connect( streamStatus,
             SIGNAL(pollSignal()),
             this,
             SLOT(streamPollEvent()) );

    connect( streamHarvester,
             SIGNAL(fetchStatus(int, unsigned int)),
             this,
             SLOT(harvesterReady(int, unsigned int)) );

    connect( downloader,
             SIGNAL(downloadFinished(QString, bool, QString)),
             this,
             SLOT(slotDownloadFinished(QString, bool, QString)) );
    
    itemTreeLoaded     = streams;
    lastItemTreeLoaded = streams;

    itemTree = &databaseTree;

    if ( parent )
      videoContainer = new VideoContainer( parent, TARGET" Video", WStyle_Customize | WStyle_NoBorder );
    else
      videoContainer = new VideoContainer( parent, TARGET" Video", WStyle_Customize | WStyle_DialogBorder );

    if ( parent )
      viewerWindow = new ViewerWindow( parent, TARGET" Viewer", WStyle_Customize | WStyle_NoBorder );
    else
      viewerWindow = new ViewerWindow( parent, TARGET" Viewer", WStyle_Customize | WStyle_DialogBorder );
    
    videoContainer->installEventFilter( this );
    //viewerWindow->installEventFilter( this );
    
    streamStatus->setVideoContainer(videoContainer);
    this->focusProxy = focusProxy;

    streamStorage = storage;

    connect (
               streamStorage,
               SIGNAL( storageEvent(int, int, bool ) ),
               this,
               SLOT  ( slotStorageEvent(int, int, bool ) )
             );

    connect(
             streamStorage,
             SIGNAL(recordInserted(ChangedRecord*)),
             this,
             SLOT(slotRecordInserted(ChangedRecord*)) );

    connect(
             streamStorage,
             SIGNAL(recordUpdated(ChangedRecord*)),
             this,
             SLOT(slotRecordUpdated(ChangedRecord*)) );

    connect(
             streamStorage,
             SIGNAL(recordRemoved(ChangedRecord*)),
             this,
             SLOT(slotRecordRemoved(ChangedRecord*)) );

    // check custom properties to check on update,
    // set streamCustomParameterCount
    int i = -1;
    while ( streamStatus->checkCustomStreamInfoExists
            ( "StreamCustomEvent" + QString::number(++i) )
          ) {}

    streamCustomParameterCount = (unsigned int)i;

    //cout << "max: " << streamCustomParameterCount << endl;

    recorderManager = new RecorderManager(this, storage);

    connect(
             recorderManager,
             SIGNAL(recordingStopped(QString, RecordStopReason)),
             this,
             SLOT(slotRecordingStopped(QString, RecordStopReason)) );

    connect(
             recorderManager,
             SIGNAL(recordingStarted(QString)),
             this,
             SLOT(slotRecordingStarted(QString)) );

    connect(
             recorderManager,
             SIGNAL(recorderActive(bool)),
             this,
             SIGNAL(eventRecorderActive(bool)) );

    connect(
             recorderManager,
             SIGNAL(scheduleEvent(QString, QString, bool)),
             this,
             SLOT(slotScheduleEvent(QString, QString, bool)) );
    
}

StreamBrowser::~StreamBrowser()
{
    if (videoContainer) delete videoContainer;
    if (viewerWindow) delete viewerWindow;
    delete(recorderManager);
    delete(streamStatus);
    delete(streamHarvester);
    delete(downloader);
}

void StreamBrowser::destroyedDumpWindow()
{
   m_dumpwindow = 0;
}


// VideoContainer events
bool StreamBrowser::eventFilter( QObject *obj, QEvent *ev)
{
  // events for VideoContainer, already handled by mplayer
  //cout << "event passed: " << ev->type() << endl;
    
  if (obj && ev->type() == QEvent::KeyPress )
  {
    
    // pass key if not handled by mplayer (and how would I know it isn't?)
    QKeyEvent *kev = static_cast<QKeyEvent*>(ev);

    if ( kev->key() != Key_P )
      qApp->sendEvent(focusProxy, ev);
    return true;
  }

  if (obj && ev->type() == QEvent::Close )
  {
    // stop mplayer
    handlePressedKey(end);
    // prevent window closing
    return true;
  }

  return false;

}


void StreamBrowser::reportEvent(const QString& eventMsg, QString eventInfo)
{
   userMessage = eventMsg;
   userInfo    = eventInfo;
   emit eventUserMessage(userMessage, userInfo);
}


void StreamBrowser::setVideoRect(QRect& area)
{
  streamStatus->setVideoRect(area);
}

void StreamBrowser::setVideoMaxRect(QRect& area)
{
  streamStatus->setVideoMaxRect(area);
}

void StreamBrowser::setViewerRect(QRect& rect)
{
    m_viewerrect = rect;
    viewerWindow->setGeometry( m_viewerrect );
}

void StreamBrowser::setViewerMaxRect(QRect& rect)
{
   m_viewermaxrect = rect;
}


void StreamBrowser::hideVideo()
{
   if (video) videoContainer->hide();
}

void StreamBrowser::showVideo()
{
   if (video) videoContainer->show();
}

bool StreamBrowser::videoShown()
{
   return (video && videoContainer->isVisible());
}

void StreamBrowser::hideViewer()
{
   viewerWindow->hide();
}

void StreamBrowser::showViewer(QString& content)
{
   viewerWindow->setContent(content);
   //viewerWindow->show();
}

void StreamBrowser::setItemDisplayFolded(int offset, bool folded) // cursor offset, and loop items in list?
{
    m_itemdisplayfolded = folded;
    m_itemdisplayoffset = offset;
}

void StreamBrowser::setFolderDisplayFolded(int offset, bool folded) 
{
    m_folderdisplayfolded = folded;
    m_folderdisplayoffset = offset;
}

QString StreamBrowser::getHarvestedLink()
{
   return streamHarvester->getCurrentName();
}

void StreamBrowser::enableFFT(int displayPoints)
{
    delete converter;
    converter = new FFTConverter(displayPoints, 512);

    if (sampler)
      converter->loadSampler(sampler);

    connect(
            converter,
            SIGNAL(fftReady(SpectrumValues*)),
            this,
            SLOT(slotFFTReady(SpectrumValues*))
           );
}

void StreamBrowser::samplerActive(bool active)
{
   if (converter && active && !sampler)
   {
     sampler = new SampleObject();
     if (sampler)
       converter->loadSampler(sampler);
   }

   if (converter && !active && sampler)
   {
     converter->unloadSampler();
     delete(sampler);
     sampler = 0;
   }
}

void StreamBrowser::slotFFTReady(SpectrumValues* spectrum)
{
   emit eventFFTReady( (Spectrum*)spectrum );
}

// enable GUI context help in harvesting mode
StreamBrowser::Action StreamBrowser::getCurrentFolderAction()
{
   StreamObject* obj;
   obj = itemTree->getStreamFolder();

   // if folder has items take the action of the selected item
   if ( obj && obj->getObject() )
   {
      StreamFolder *folder = dynamic_cast<StreamFolder*>(obj);
      if ( folder ) obj = folder->getStreamItem();
      if ( !obj ) obj = folder;
   }

   if (obj)
     return Action(obj->getAction());
   else
     return none;
}

// enable GUI context help in harvesting mode
QString StreamBrowser::getCurrentFolderCaption()
{
   StreamObject *obj;
   StreamFolder *folder = NULL;
   
   obj = itemTree->getStreamFolder();
   if (obj) 
   {
      folder = dynamic_cast<StreamFolder*>(obj);
      return folder->caption;
   }
      else return "";
}

bool StreamBrowser::getCurrentObjectDetails(
             QString& title, QString& field1, QString& field2, bool preferFolder)
{
   StreamObject* obj;
   StreamFolder *folder = 0;
   
   obj = itemTree->getStreamFolder();
   if (obj) folder = dynamic_cast<StreamFolder*>(obj);
   
   if (folder) 
   {
     title  = folder->getName();
     field1 = folder->caption;
     field2 = folder->descr;
   }
     else return false;
     
   if (preferFolder) return true;
   
   // if folder has items take the action of the selected item
   if ( obj && obj->getObject() )
   {
      StreamFolder *folder = dynamic_cast<StreamFolder*>(obj);
      if ( folder ) 
      {
        StreamItem *item = folder->getStreamItem();
        if (item)
        {
          title  = item->getName();
          field1 = item->url;
          field2 = item->descr;      
        }
      }
   }   
   
   return true;   
}

bool StreamBrowser::getCurrentStreamObjectDetails( QString& folderName, QString& itemName,
                    QString& url, QString& descr, QString& handler, QStringList& meta)
{
   StreamObject* obj = 0;
   StreamFolder* folder = 0;
       
   obj = itemTree->getStreamFolder();
   if (obj) folder = dynamic_cast<StreamFolder*>(obj);
  
   if ( folder ) 
   {
        StreamItem *item = folder->getStreamItem();
        if (item)
        {
          folderName = folder->getName();
          itemName   = item->getName();
          url        = item->url;
          descr      = item->descr;      
          handler    = item->handler;
          meta       = item->meta;
        }
        return true;   
   } 
     else return false; 
}


int StreamBrowser::getDisplayFolderList(int rows, QStringList& list, bool& onTop, bool& atBottom)
{
    QString text;
    int cnt;

    list.clear();

    if ( itemTree->cursor < 0       ) itemTree->cursor = 0;
    if ( itemTree->cursor >= rows   ) itemTree->cursor = rows - 1;
    if ( m_folderdisplayoffset >= 0 ) itemTree->cursor = m_folderdisplayoffset;
    
    itemTree->resetDisplaySize(m_folderdisplayfolded, rows, itemTree->cursor, onTop, atBottom);

    for (int i = 0; i < rows; i++)
    {
      text = itemTree->getNextDisplayStr();      
      if ( !m_folderdisplayfolded && m_folderdisplayoffset > -1 && text == QString::null ) text = "";
      if (  m_folderdisplayfolded || text != QString::null ) list.append(text);
    }

    cnt = list.count();
    if ( itemTree->cursor > cnt - 1 ) itemTree->cursor = cnt - 1;

    return itemTree->cursor;
}

int StreamBrowser::getDisplayItemList(int rows, QStringList& list, bool& onTop, bool& atBottom)
{
    int cnt;

    list.clear();
    StreamFolder *folder = itemTree->getStreamFolder();

    if (folder)
    {
        if ( m_itemdisplayoffset >= 0 )
          folder->cursor = m_itemdisplayoffset;
        else
        {
          if ( folder->cursor < 0     ) folder->cursor = 0;
          if ( folder->cursor >= rows ) folder->cursor = rows - 1;
        }

        QString text;
        folder->resetDisplaySize(m_itemdisplayfolded, rows, folder->cursor, onTop, atBottom);

        for (int i = 0; i < rows; i++)
        {
          text = folder->getNextDisplayStr();
          if ( !m_itemdisplayfolded && m_itemdisplayoffset > -1 && text == QString::null ) text = "";
          if ( m_itemdisplayfolded || text != QString::null ) list.append(text);
        }

        cnt = list.count();
        if ( folder->cursor > cnt - 1 ) folder->cursor = cnt - 1;

        return folder->cursor;
    }
      else
        return 0;
}

void StreamBrowser::loadDumpWindow()
{
   if (m_dumpwindow != 0)
     m_dumpwindow->setText( streamStatus->getLastPlayedConsole() );
}

void StreamBrowser::streamStatusChanged()
{
    loadDumpWindow();

    if ( video != streamStatus->isVideo() )
    {
      video = streamStatus->isVideo();
      emit eventVideoActive ( video );
    }

    StreamStatus::Status& newStatus = streamStatus->getStatus();

    if ( newStatus == StreamStatus::nostream )
      streamUndetected();

    emit eventValuesUpdated( streamstatus );

    if ( newStatus == StreamStatus::playing && oldStatus != StreamStatus::paused )
    {
      streamStatus->issueCommand(StreamStatus::VolumeDn);
      streamStatus->issueCommand(StreamStatus::VolumeUp);
      if (playListActive)
         QTimer::singleShot( 5000, this, SLOT(timerSaysFullScreen()) );
    }
          
    samplerActive(newStatus == StreamStatus::playing);

    if ( newStatus == StreamStatus::stopped )
      playNextMarkedStream();

    oldStatus = newStatus;

}

unsigned int StreamBrowser::getStreamCustomParameterCount()
{
  return streamCustomParameterCount;
}

QString StreamBrowser::getStreamCustomParameter(unsigned int index, QString& area, QString& title)
{
   if ( index < streamCustomParameterCount)
   {
     area  = streamStatus->getCustomStreamArea("StreamCustomEvent" + QString::number(index, 10) );
     title = streamStatus->getCustomStreamTitle("StreamCustomEvent" + QString::number(index, 10) );
     return streamStatus->getCustomStreamInfo("StreamCustomEvent" + QString::number(index, 10) );
   }
   else
     return "";
}


QString StreamBrowser::getStreamParameter(Parameter parameter)
{
    QString text = "";
    int secs;
    
    switch ( parameter )
    {
      case streamName: text = streamStatus->getStreamName();
      break;
      case streamDescr: text = streamStatus->getStreamDescr();
      break;
      case streamUrl: text = streamStatus->getStreamUrl();
      break;
      case streamAudioFormat: text = streamStatus->getStreamInfo(StreamStatus::StreamAudioFormat);
      break;
      case streamAudioBps: text = streamStatus->getStreamInfo(StreamStatus::StreamBitrate);
      break;
      case streamAudioSps: text = streamStatus->getStreamInfo(StreamStatus::StreamRate);
      break;
      case streamAudioChannels: text = streamStatus->getStreamInfo(StreamStatus::StreamChannels);
      break;
      case streamStatusStr: text = streamStatus->getStatusString();
      break;
      case streamVideoFormat: text = streamStatus->getStreamInfo(StreamStatus::StreamVideoFormat);
      break;
      case streamVideoCodec: text = streamStatus->getStreamInfo(StreamStatus::StreamVideoCodec);
      break;
      case streamVideoFps: text = streamStatus->getStreamInfo(StreamStatus::StreamVideoFps);
      break;
      case streamVideoBps: text = streamStatus->getStreamInfo(StreamStatus::StreamVideoBitrate);
      break;
      case streamVideoLength: text = streamStatus->getStreamInfo(StreamStatus::StreamLength);
      break;
      case streamCache: text = streamStatus->getStreamInfo(StreamStatus::StreamCache);
      break;
      case streamTime: text = streamStatus->getStreamInfo(StreamStatus::StreamTime);
           secs = text.toInt();
           if (secs != 0)
           {
					text.sprintf("%02d:%02d", secs / 60, secs % 60);
           }
      break;
      case streamStability: text = streamStatus->getStreamInfo(StreamStatus::StreamStability);
      break;
      case streamVolume: text = streamStatus->getStreamInfo(StreamStatus::StreamVolume);
      break;
      default:
      break;
    }

    if (!text) text = "";
    
    return text;
}



void StreamBrowser::streamPollEvent()
{
   // StreamStatus pollSignal emitted every second if stream is not idle

   if ( ! streamStatus->isFullScreen() )
     emit eventValuesUpdated( streamplay );
   else
     emit eventValuesUpdated( fsstreamplay );
}

void StreamBrowser::streamUndetected()
{
   StreamFolder* folder = itemTree->getStreamFolder();

   if ( folder )
   {
      streamHarvester->setTagOnCurrent( folder->getIndex() );

      emit eventHarvesterBusy( true, "fetching data" );
      //cout << "fetch: " << streamStatus->getStreamUrl() << " " << streamStatus->getStreamName() << " " << streamStatus->getStreamHandler() << endl;
      streamHarvester->fetchData( streamStatus->getStreamUrl(),
                                  streamStatus->getStreamName(),
                                  streamStatus->getStreamDescr(),
                                  streamStatus->getStreamHandler() );
   }
}

void StreamBrowser::playNextMarkedStream()
{
    StreamFolder *folder = itemTree->getStreamFolder();
      
	if (folder && folder->nextMarkedObject())
	{
		folder->cursor = folder->getIndex();
		emit eventValuesUpdated( itemlist );
		playListActive = true;
		handlePressedKey(selectitem);
	}
	   else playListActive = false;
}

void StreamBrowser::fillHarvestList(bool filterStreams)
{
    StreamItem* item;
    QString prefix;

    StreamFolder *folder = 0;
    StreamObject *obj = harvestTree.findObject("harvest");

    if ( obj )
      folder = dynamic_cast<StreamFolder*>(obj);

    if (!folder)
    {
      cerr << TARGET": cannot find harvest folder" << endl;
      return;
    }

    folder->clearObjectList();

    for ( NewUrl *url = streamHarvester->urlList.first();
          url != 0;
          url = streamHarvester->urlList.next() )
    {
      // todo: do this the right way...
      prefix = "_";

      //if ( url->context == "relative" ) prefix1 = "";
      //if ( url->context == "absolute" ) prefix1 = "";
      if ( url->port    != -1         ) prefix = "-";
      if ( url->file    == "stream" )   prefix = "~"; // .ogg, .mp3 etc
      if ( url->proto   == "mms" )      prefix = "~";
      if ( url->proto   == "pnm" )      prefix = "~";
      if ( url->proto   == "rtsp" )     prefix = "~";

      if (prefix == "~" || !filterStreams)
      {
        item = new StreamItem(folder, url->name, url->url, url->descr, url->handler);
        item->handler = url->handler;
        item->meta = url->meta;

        // see if exists
        // STREAM_DL: url must match
        // FUZZY_DL : url or name of stream item in download folder must match

        if (item->handler == "STREAM_DL" || item->handler == "FUZZY_DL" )
        {
          item->setAction(download);
          prefix = "D"; // download icon
          Downloader::Availability availability;
          bool dummy;
          downloader->checkAvailable( url->url, availability, dummy, false, url->name, (item->handler == "FUZZY_DL") );

          switch (availability)
          {
            case Downloader::available:
               prefix = "#";
            break;
            case Downloader::pending:
               prefix = "R";
            break;
            default:
               prefix = "D";
            break; 
          }
        }
       
        item->setPrefix( prefix );
      }
    }
}

void StreamBrowser::checkAndSetIcon(StreamItem *item)
{
      // determine icon (throug caption prefix) for stream in browse mode
      // disabled for now, because the icons next to the buttons on the left
      // are somewhat confusing.
      // 
      // ... so is this comment
      QString prefix = ".";
      item->setPrefix( prefix );
      return;

      if (!item) return;

      prefix = "_"; // general url
      QUrl qurl(item->url);
      QString proto = qurl.protocol();
      int port      = qurl.port();
      QString file  = qurl.fileName() + qurl.query();

      if ( file.contains(".rm")  > 0 ||
           file.contains(".ram") > 0 ||
           file.contains(".asx") > 0 ||
           file.contains(".avi") > 0 ||
           file.contains(".mpg") > 0 ||
           file.contains(".pls") > 0 ||
           file.contains(".mp3") > 0 ||
           file.contains(".ogg") > 0 ||
           file.contains(".wmf") > 0 ||
           file.contains(".smi") > 0 ||
           file.contains(".m3u") > 0    )
        file = "stream";
      else
        file = "other";

      if ( port != -1          ) prefix = "-";
      if ( file    == "stream" ) prefix = "~";
      if ( proto   == "mms"    ) prefix = "~";
      if ( proto   == "pnm"    ) prefix = "~";
      if ( proto   == "rtsp"   ) prefix = "~";

      item->setPrefix( prefix );
}


void StreamBrowser::slotDownloadFinished(QString url, bool error, QString errorMessage)
{
   // is this url in current list?

   ItemTree::ObjectList folderList = itemTree->objectList;
   ItemTree::ObjectList itemList;
   StreamItem *streamItem;

   QString prefix = "#";  // downloaded
   bool changed = true;

   if (error)
      prefix = "D";  // download before play
   
    for (unsigned int i = 0; i < folderList.count(); i++)  // folders
    {
       itemList = folderList.at(i)->objectList;
       
       for (unsigned int j = 0; j < itemList.count(); j++)  // items
       {
          streamItem = dynamic_cast<StreamItem*>(itemList.at(j));
          if ( streamItem && url == streamItem->url )
          {
             streamItem->setPrefix(prefix);
             changed = true;
          }
          
       }
    }

    if (changed)
      emit eventValuesUpdated( itemlist );

   if (error)
      reportEvent(errorMessage, "");

}

void StreamBrowser::harvesterReady(int fstat, unsigned int count)
{
   switch (fstat)
   {
      case StreamHarvester::errorset:
        switch (count)
        {
          case StreamHarvester::invalidurl:
             emit eventHarvesterBusy(false, "invalid url");
          break;
          case StreamHarvester::nodata:
             emit eventHarvesterBusy(false, "no data received");
          break;
          case StreamHarvester::nourlindata:
             emit eventHarvesterBusy(false, "no url in data");
          break;
          case StreamHarvester::bogusparseroutput:
             emit eventHarvesterBusy(false, "parser problem, test parser on commandline");
          break;

        }
      break;

      case StreamHarvester::receiving:
       emit eventHarvesterBusy(true, "received " + QString::number(count) + " KB" );
      break;

      case StreamHarvester::parsing:
       emit eventHarvesterBusy(true, "url's found: " + QString::number(count));
      break;

      case StreamHarvester::ready:

          if (streamHarvester->urlList.count() == 0)
            emit eventHarvesterBusy( false, "no url's found" );
          else
            emit eventHarvesterBusy( false, "" );

            //cerr << "harvester ready, found " << streamHarvester->urlList.count() << endl;

          if ( streamHarvester->urlList.count() != count )
            cerr << TARGET": reported and actual urlcounts differ "
                  << count << ", "
                  << streamHarvester->urlList.count() << endl;

          NewUrl* url;

          // detect known stream meta file content
          if ( streamHarvester->urlList.count() == 2 )
          {
              NewUrl* url1 = streamHarvester->urlList.first();
              NewUrl* url2 = streamHarvester->urlList.next();

              // ram file, delete last (pnm) entry
              if ( url1->url.find("rtsp://", false) == 0 && url2->url.find("pnm://", false) == 0 )
              {
                 streamHarvester->urlList.removeLast(); // list length = 1, rtsp will be played
              }
              
          }
          
          // no other activity arrived at streamStatus?
          if ( streamHarvester->urlList.count() == 1 &&
                ( streamStatus->getStatus() == StreamStatus::nostream
                  || streamStatus->getStatus() == StreamStatus::idle 
                  || streamStatus->getStatus() == StreamStatus::stopping
                  || streamStatus->getStatus() == StreamStatus::stopped  ) ) // parser change
          {
              url = streamHarvester->urlList.first();

              // replace url only
              if ( url )
                streamStatus->initStream( url->url,
                                          streamHarvester->getCurrentName(),
                                          streamHarvester->getCurrentDescr(),
                                          url->handler );
//CHANGED                                          streamHarvester->getCurrentHandler() );
          }

          if ( streamHarvester->urlList.count() > 1 )
            setHarvestMode(true);
      break;
   }

}


void StreamBrowser::storeMarkedStreamsPrepare(QString storageName)
{
   ValueList values(9);
   bool loaded;
   
      
   ReposStorage *markedStorageList = new ReposStorage();
   if (markedStorage) delete(markedStorage);
   markedStorage = new StreamStorage();      
   connect (
             markedStorage,
             SIGNAL( storageEvent(int, int, bool ) ),
             this,
             SLOT  ( slotStorageEvent(int, int, bool ) )
           );

   connect (
             markedStorage,
             SIGNAL(recordInserted(ChangedRecord*)),
             this,
             SLOT(slotMarkedRecordInserted(ChangedRecord*)) 
           );
   markedStorageList->openRepository();
   loaded = markedStorageList->getStorageValuesByName(values, storageName);
   
   if (!loaded) cerr << "storage not found: " << storageName << endl;
   
   delete(markedStorageList);
   
   // copy marked streams to streamCopyList
   
   StreamFolder *folder = 0;
   StreamItem   *item   = 0;
   
   ValueList itemvalues(5);
   QString error = "";
        
   streamCopyList.setAutoDelete(false);
   streamCopyList.clear();   
   streamCopyResult.clear();
   
    for ( unsigned int i = 0; i < databaseTree.objectList.count(); i++)  // folders
    {
        folder = dynamic_cast<StreamFolder*>( databaseTree.objectList.at(i) );
        if (folder)
          for ( unsigned int j = 0; j < folder->objectList.count(); j++)  // streams
          {
            item = dynamic_cast<StreamItem*>( folder->objectList.at(j) );
            if (item && item->marked)
            {
                streamCopyList.append(item);
                //cout << item->getName() << endl;
            }
          }
    }
   
   streamCopyIndex = 0;
   // end copy marked streams
   
   if (loaded)
     loaded = markedStorage->selectStorage(110, values);
   
   if (!loaded)
   {
      reportEvent( "error: " + markedStorage->getLastError(), "" );
      delete(markedStorage);
      streamCopyList.clear();
      markedStorage = NULL;
   }
   // invokes storageEvent selected,
   // storageEvent calls loadList
   // loadList calls storageEvent loaded
   // then storeMarkedStreamsInsertNext() can be called
}

bool StreamBrowser::storeMarkedStreamsInsertNext(bool first)
{    
   ValueList itemvalues(5);
   StreamItem *item;
   QString error = "";
   bool inserted = false;
   
   if ( markedStorage && streamCopyIndex < 0 ) // timer signal
   {
      delete(markedStorage); // and flush storage buffer
      markedStorage = NULL;
      return false;
   }
      
   if (!first)
   {  
     item = streamCopyList.at(streamCopyIndex-1);
     if (item)
     {
       streamCopyResult.append("STORED  " + item->getName() );
       item->marked = false;
     }
   }      
   
   while ( streamCopyIndex>=0 && streamCopyIndex < (int)streamCopyList.count() && !inserted ) 
   {
      item = streamCopyList.at(streamCopyIndex);
      itemvalues[0] = item->getFolderName();
      itemvalues[1] = item->getName();
      itemvalues[2] = item->url;
      itemvalues[3] = item->descr;
      itemvalues[4] = item->handler;
      
      streamCopyIndex++;
      
      inserted = markedStorage->insertRecord(110, itemvalues, error);
      
      if ( !inserted ) 
      {
        streamCopyResult.append("SKIPPED " + itemvalues[1] + ": " + error);
      }
   }
      
   if (!inserted) // no pending recordinserted signals...
   {
      streamCopyIndex = -1;
      streamCopyList.clear();
      setMessageMode(true, &streamCopyResult);
      // don't delete(markedStorage) here, were in a slot signalled by markedStorage     
      // so set timer
      QTimer::singleShot( 100, this, SLOT(markedStreamsInsertReady()) );  // markedStorage deletion 
   }

   return inserted;
}


void StreamBrowser::setMessageMode(bool goMessageMode, QStringList* list)
{
    StreamFolder* folder1;
    StreamFolder* folder2;
    StreamItem* item;
    
    QString prefix;

    if ( goMessageMode )
    {
        messageTree.clearObjectList();

        folder1 = new StreamFolder("messages");
        folder1->caption = "Result of last action";
        folder1->descr = "The items in this folder represent the results of the last action performed\n";
        folder1->descr += "You cannot perform actions on the items presented.";
        messageTree.addObjectToList( folder1 );
        
        folder2 = new StreamFolder("return");
        folder2->caption = "Leave message list";
        folder2->descr = "After selecting this folder the player will return to the previous state";
        folder2->setAction(quitmessage);
        messageTree.addObjectToList( folder2 );

        for ( QStringList::Iterator i = list->begin();
              i != list->end();
              ++i
            )
        {
          item = new StreamItem( folder1, *i,
                                "",
                                "message", " " );
          item->setAction(none);
        }  
        
        itemTree = &messageTree;

        emit eventItemTreeSwitchedTo( messages );

        emit eventValuesUpdated( folderlist );  // implies itemlist update
        emit eventValuesUpdated( itemlist );    // ..but have to switch cursor to itemlist

    }
      else
    {
        itemTree = &databaseTree;

        emit eventItemTreeSwitchedTo( streams );
        emit eventValuesUpdated( folderlist );  // implies itemlist update
        if ( itemTreeLoaded == messages )
        {
          emit eventValuesUpdated( itemlist );
        }
    }

    if (goMessageMode) 
      itemTreeLoaded = messages; 
    else 
      itemTreeLoaded = streams; 
}

void StreamBrowser::setInfoMode(bool goInfoMode)
{
    StreamFolder* folder1;
    StreamFolder* folder2;
    StreamItem* item;
    
    QString prefix;


    QString title, folderName, itemName, url, descr, handler;
    QStringList meta, metaparts, displist;
  
    if ( goInfoMode && itemTreeLoaded == info )  // already in info mode
      goInfoMode = false;

    if ( !goInfoMode && itemTreeLoaded != info ) // not in info mode
      return;
      
    getCurrentStreamObjectDetails(folderName, itemName, url, descr, handler, meta);

    if (itemName   == "" ) itemName   = "-";
    title = itemName;
    if (folderName == "" ) folderName = "-";
    if (url == ""    ) url     = "-";
    if (descr == ""  ) descr   = "-";
    if (handler == "") handler = "default";

    folderName = "folder: " + folderName;
    itemName   = "name: "   + itemName;
    url     = "url: "     + url;
    descr   = "descr: "   + descr;
    handler = "handler: " + handler;        

    if ( goInfoMode )
    {
        infoTree.clearObjectList();

        folder1 = new StreamFolder("information");
        folder1->caption = "Information: " + title;
        folder1->descr = "The items in this folder represent ...\n";
        infoTree.addObjectToList( folder1 );
        
        folder2 = new StreamFolder("return");
        folder2->caption = "Leave information list";
        folder2->descr = "After selecting this folder the player will return to the previous state";
        folder2->setAction(quitinfo);
        infoTree.addObjectToList( folder2 );

        prefix = "I";
        
        item = new StreamItem( folder1, folderName, "", "info", "");
        item->setPrefix(prefix);
        item->setAction(none);
        //item = new StreamItem( folder1, itemName, "", "info", "");
        //item->setPrefix(prefix);
        //item->setAction(none);
        item = new StreamItem( folder1, url, "", "info", "");
        item->setPrefix(prefix);
        item->setAction(none);
        item = new StreamItem( folder1, descr, "", "info", "");
        item->setPrefix(prefix);
        item->setAction(none);
        item = new StreamItem( folder1, handler, "", "info", "");
        item->setPrefix(prefix);
        item->setAction(none);
        
        for (uint i = 0; i < meta.size(); i++)
        {
            metaparts = QStringList::split("(stsep)", *meta.at(i), true);
            prefix = "I";
            
            if (metaparts.size() < 3)
            {
              item = new StreamItem( folder1, "unknowm (meta data parsing error)",
                                    "",
                                    "meta data", "" );
              item->setPrefix(prefix);
              item->setAction(none);
            }
              else
            {
              QString inlineText = *metaparts.at(0);
                        
              if (*metaparts.at(1) == "html")   // open html viewer to view content
                prefix = "H";
              
              if (*metaparts.at(1) == "text")   // open text viewer to view (multiline text) content
                prefix = "H";
                
              if (*metaparts.at(1) == "inline") // display content inline
              {
                prefix = "I";
                inlineText += ": " + *metaparts.at(2);
              }
/*
        new viewer "url" could be used to provide feedback (e.g. mark nonfunctional)
        
        Example parser implementation:

        $meta = $doc->createElement('meta');
        $meta->appendChild( newNode('name'   , 'action: mark as nonfunctional') );
        $meta->appendChild( newNode('content', "wwmp/wwmp_act|marknf|$stationid") );
        $meta->appendChild( newNode('viewer' , 'url') );
        $item->appendChild( $meta );

*/
              item = new StreamItem( folder1, inlineText, "", "meta data", "" );
              item->meta.append(*metaparts.at(2)); // dump content in first position
              item->setPrefix(prefix);
    
              if (prefix != "I")
                item->setAction(htmlviewer);
            }
        }

        itemTree = &infoTree;

        emit eventItemTreeSwitchedTo( info );

        emit eventValuesUpdated( folderlist );  // implies itemlist update
        emit eventValuesUpdated( itemlist );    // ..but have to switch cursor to itemlist

    }
      else
    {
        switch (lastItemTreeLoaded)
        {
           case streams:
             //cout << "last loaded: streams" << endl;
             itemTree = &databaseTree;
           break;
           case harvester:
             //cout << "last loaded: harvester" << endl;
             itemTree = &harvestTree;
           break;
           default:
             //cout << "last loaded: other" << endl;
             itemTree = &databaseTree;
           break;
        }
        
        emit eventItemTreeSwitchedTo( lastItemTreeLoaded );
        emit eventValuesUpdated( folderlist );  // implies itemlist update
        if ( itemTreeLoaded == info )
        {
          emit eventValuesUpdated( itemlist );
        }
    }

    if (goInfoMode)
    {
      lastItemTreeLoaded = itemTreeLoaded;
      itemTreeLoaded     = info;
    }
    else
    { 
      itemTreeLoaded = lastItemTreeLoaded;
    }
}

void StreamBrowser::setStorageMode(bool goStorageMode)
{
    StreamFolder* folder1;
    StreamFolder* folder2;
    StreamItem* item;
    
    QString prefix;

    if ( goStorageMode )
    {
        storageTree.clearObjectList();

        folder1 = new StreamFolder("Store marked");
        folder1->caption = "Select storage to append marked streams to";
        folder1->descr = 
"Store stream urls marked with M in the selected storage\n\
Use this feature to copy stream urls between storages, or \
to save multiple harvested items to storage";
        storageTree.addObjectToList( folder1 );
        
        folder2 = new StreamFolder("Cancel");
        folder2->caption = "Quit storage mode";
        folder2->descr = "Return to browse mode without saving stream url's";
        folder2->setAction(quitstorage);
        storageTree.addObjectToList( folder2 );

        ValueList values(9);
        ReposStorage *storageList = new ReposStorage();
        bool loaded = storageList->openRepository();  // initialize...
        
        if (!loaded)
        {
           reportEvent("cannot load storage repository", "");
           return;
        }
        
        storageList->resetRecordList();
        while ( storageList->getNextRecord(values) ) 
        {
         item = new StreamItem( folder1, values[r_name],
                                "",
                                "select to store the marked stream urls in this folder", "");
         item->setAction(appendmarked);
        }  
        delete( storageList );
       
        // harvestTree.nextObject(false);

        itemTree = &storageTree;

        emit eventItemTreeSwitchedTo( storages );

        emit eventValuesUpdated( folderlist );  // implies itemlist update
        emit eventValuesUpdated( itemlist );    // ..but have to switch cursor to itemlist

    }
      else
    {
        itemTree = &databaseTree;

        emit eventItemTreeSwitchedTo( streams );
        emit eventValuesUpdated( folderlist );  // implies itemlist update
        if ( itemTreeLoaded == storages )
        {
          emit eventValuesUpdated( itemlist );
        }
    }

    if (goStorageMode) 
      itemTreeLoaded = storages; 
    else 
      itemTreeLoaded = streams; 
}
/*
      case StreamBrowser::stream:
        itemLabels[0]->setText( "harvesting: " + streamBrowser->getHarvestedLink() );
        break;
      case StreamBrowser::goback:
        itemLabels[0]->setText( "go back to previous list" );
        break;
      case StreamBrowser::quitharvest:
        itemLabels[0]->setText( "exit harvester" );
        break;
      case StreamBrowser::storestation:
        itemLabels[0]->setText( "store last played stream in the highlighted folder" );
        break;
      case StreamBrowser::storesite:
        itemLabels[0]->setText( "store the currently harvested site in the highlighted folder" );
        break;
      case StreamBrowser::storeitem:
        itemLabels[0]->setText( "store the selected item in the highlighted folder" );
        break;
      case StreamBrowser::filterlist:
        itemLabels[0]->setText( "filter items in the current harvester list" );
        break;
      default:
        itemLabels[0]->setText("harvesting sites"); // not used
        break;

*/

void StreamBrowser::setHarvestMode(bool goAndHarvest)
{
    StreamFolder* folder;
    StreamFolder* folder1;
    StreamFolder* folder2;
    StreamFolder* folder3;
    StreamItem* item;
    QString prefix;

    if ( goAndHarvest )
    {
        harvestTree.clearObjectList();

      // set harvested station and options as folders
        folder1 = new StreamFolder("Store list");
        folder1->caption = "Store the currently harvested site in the highlighted folder";
        folder1->descr = 
"The harvested site is de url that gave the results visible in the \
item list on the right (when folder 'harvest' is selected).";
        harvestTree.addObjectToList( folder1 );
        folder2 = new StreamFolder("Store link");
        folder2->caption = "Store the selected item in the highlighted folder";
        folder2->descr = "the selected item is the item under cursor in the harvest resultset.";
        harvestTree.addObjectToList( folder2 );
        folder3 = new StreamFolder("Store stream");
        folder3->caption = "Store last played stream in the highlighted folder";
        folder3->descr = 
"This action will save the now playing or previously stopped stream url \
in the folder under cursor";
        harvestTree.addObjectToList( folder3 );

        for (unsigned int i = 0; i < databaseTree.objectList.count(); i++)
        {
           item = new StreamItem( folder1, databaseTree.objectList.at(i)->getName(),
                                  "",
                                  "select this folder to store the harvested list", "" );
           item->setAction(storesite);
           item = new StreamItem( folder2, databaseTree.objectList.at(i)->getName(),
                                  "",
                                  "select this folder to store the current link in the list", "" );
           item->setAction(storeitem);
           item = new StreamItem( folder3, databaseTree.objectList.at(i)->getName(),
                                  "",
                                  "select this folder to store the last played station", "" );
           item->setAction(storestation);
        }

        folder = new StreamFolder("filter");
        folder->caption = "Filter items in the current harvester list";
        folder->descr = 
"This filter allows for two filters: \n\
 - show all urls\n -show urls with specific protocol or file extention\n\
protocols are e.g. pnm, rtsp; extentions are e.g. rm, ram, asx, mpg";
        harvestTree.addObjectToList( folder );
        item = new StreamItem( folder, "streams only",
                                "",
                                "filter the current harvester list", " ");
        item->setAction(filterlist);
        item = new StreamItem( folder, "all items",
                                "",
                                "filter the current harvester list", " " );
        item->setAction(filterlist);

        folder = new StreamFolder( "harvest" );
        folder->caption = "Harvesting: " + getHarvestedLink(); 
        folder->descr = 
"This folder contains the harvester resultset: stream and non-stream urls. \n\
Items in the resultset likely to be streams are marked:\n ~ : confirmed streams\n\
  - likely streams\nCheck the other folders for filtering or storing the harvester results";
   
        harvestTree.addObjectToList(folder);

        harvestTree.nextObject(false);
        harvestTree.nextObject(false);
        harvestTree.nextObject(false);
        harvestTree.nextObject(false);

/*
        for ( NewUrl *url = streamHarvester->urlList.first();
              url != 0;
              url = streamHarvester->urlList.next() )
        {
          // todo: do this the right way...
          prefix = "  ";

          //if ( url->context == "relative" ) prefix1 = "";
          //if ( url->context == "absolute" ) prefix1 = "";
          if ( url->port    != -1         ) prefix = "-";
          if ( url->file    == "stream" )   prefix = "~";
          if ( url->proto   == "mms" )      prefix = "~";
          if ( url->proto   == "pnm" )      prefix = "~";
          if ( url->proto   == "rtsp" )     prefix = "~";

          item = new StreamItem(folder, url->name, url->url, url->descr);
          item->setPrefix( prefix );

        }
*/
        fillHarvestList(false);

        // tryal
        folder->setIndex( streamHarvester->getCurrentTag() );

        folder = new StreamFolder("Go back");
        folder->caption = "Go back to previous list";
        folder->descr = 
"Take one step back. If your last step was done on a url that was a harvested \
url itself, the player will stay in harvester mode";
        folder->setAction(goback);
        harvestTree.addObjectToList(folder);

        folder = new StreamFolder("Exit");
        folder->caption = "Exit harvester";
        folder->descr = "";
        folder->setAction(quitharvest);
        harvestTree.addObjectToList(folder);

        itemTree = &harvestTree;

        emit eventItemTreeSwitchedTo( harvester );

        emit eventValuesUpdated( folderlist );  // implies itemlist update
        emit eventValuesUpdated( itemlist );    // ..but have to switch cursor to itemlist

    }
      else
    {
        itemTree = &databaseTree;

        emit eventItemTreeSwitchedTo( streams );
        emit eventValuesUpdated( folderlist );  // implies itemlist update
        if ( itemTreeLoaded == harvester )
        {
          emit eventValuesUpdated( itemlist );
        }
    }

    if (goAndHarvest) 
      itemTreeLoaded = harvester; 
    else 
      itemTreeLoaded = streams; 
}

void StreamBrowser::updateStreamItem(StreamItem* item, QString oldUrl)
{
    QString error;

    ValueList values(5);
    values[0] = item->getFolderName();
    values[1] = item->getName();
    values[2] = item->url;
    values[3] = item->descr;
    values[4] = item->handler;

    ValueList oldValues(5);
    oldValues[0] = item->getFolderName();
    oldValues[1] = item->getName();
    oldValues[2] = oldUrl;
    oldValues[3] = item->descr;
    oldValues[4] = item->handler;

  //if ( item && !
  streamStorage->updateRecord(ident_streambrowser, oldValues, values, error);
  //  cout << TARGET"storage error: " << error << endl;
}

QString urlenc(QString line)
{
   QUrl::encode(line);
   return line;
}

// start streamer
void StreamBrowser::initStream()
{
   QString line = "";
   QString oldUrl = "";
   QRegExp expr;
   bool hasParameter = false;
   bool cancelled = false;

   int index = 0;
   QString url;
   
   StreamFolder *folder = itemTree->getStreamFolder();
   if (folder)
   {
      StreamItem* pendingStation = folder->getStreamItem();
      if (pendingStation)
      {
          oldUrl = url = pendingStation->url;

          // check for <:name::value:> in url
          expr.setPattern("<:(.*)::(.*):>");
          expr.setMinimal(true);

          // ask user to edit/confirm value
          while ( (index = expr.search( url, index )) && index > -1 && !cancelled )
          {
            hasParameter = true;
            line = expr.cap(2);
            emit eventEditString(line, expr.cap(1), cancelled);
            url.replace(index, expr.matchedLength(), "<:" + expr.cap(1) + "::" + line + ":>");
            index++;
          }

          if (cancelled) return;
          
          // update url in stream storage, replace <:name::value:> in url with value
          if (hasParameter)
          {
             pendingStation->url = url;
             if (oldUrl != url) updateStreamItem(pendingStation, oldUrl);  // ignore (readonly) write error
             index = 0;
             while ( (index = expr.search( url, index )) && index > -1 )
               url.replace(index, expr.matchedLength(), urlenc(expr.cap(2)));
             //cout << "new url is: " << url << endl;
          }

          playListActive = pendingStation->marked;
          
          if (pendingStation->handler != "")  // parser change
          {
             streamStatus->stopStream();
             //streamUndetected();
				streamHarvester->setTagOnCurrent( folder->getIndex() );

				emit eventHarvesterBusy( true, "fetching data" );
				streamHarvester->fetchData(
											url,
											pendingStation->getName(),
											pendingStation->descr,
											pendingStation->handler
													);

				/*
				emit eventHarvesterBusy( true, "fetching data" );

				streamHarvester->fetchData( streamStatus->getStreamUrl(),
													streamStatus->getStreamName(),
													streamStatus->getStreamDescr(),
													streamStatus->getStreamHandler() );
				*/
          }
            else
          {
          
          streamStatus->initStream( 
                             url,
                             pendingStation->getName(),
                             pendingStation->descr,
                             pendingStation->handler 
          );
          }
            
      }
   }
}

// show html viewer
void StreamBrowser::showHtmlViewer()
{
   StreamFolder *folder = itemTree->getStreamFolder();
   if (folder)
   {
      StreamItem* pendingStation = folder->getStreamItem();
      if (pendingStation)
      {
         showViewer(pendingStation->meta[0]);
      }
   }
}

void StreamBrowser::startDownload()
{
   StreamFolder *folder = itemTree->getStreamFolder();
   if (folder)
   {
      StreamItem* pendingStation = folder->getStreamItem();
      if (pendingStation)
      {
         bool error, canplaynow, itemcreated;
         QString reason;
         QString file;
         
         file = downloader->download(pendingStation->url, pendingStation->getName(), canplaynow, itemcreated, 
                 error, reason, ( pendingStation->handler == "FUZZY_DL") );
         
         if (error)
           reportEvent(reason, "");
         else
           if (canplaynow)
             streamStatus->initStream( file,
                                       pendingStation->getName(),
                                       pendingStation->descr,
                                       pendingStation->handler );
           else
           {
             pendingStation->setPrefix("R"); // downloading

             emit eventValuesUpdated ( itemlist );

             if (!itemcreated)
               reason = "Could not create stream item in downloads folder, downloading anyway";
             else
               reason = "";
             
             reportEvent("A file download has started. Select the stream item again to play the (partially) downloaded file.", reason);
           }
      }
   }
}



void StreamBrowser::checkFolderCommand()
{
   StreamObject* obj;
   obj = itemTree->getStreamFolder();

   // if folder has items act on the selected item
   if ( obj && obj->getObject() )
   {
      StreamFolder *folder = dynamic_cast<StreamFolder*>(obj);
      if ( folder ) obj = folder->getStreamItem();
      // default to existing folder action
      if ( !obj ) obj = folder;
   }

   if ( obj )
     switch ( obj->getAction() )
     {
       case stream:
          initStream();
          break;
       case quitharvest:
          setHarvestMode(false);
          break;
       case quitstorage:
          setStorageMode(false);
          break;
       case quitmessage:
          if (markedStorage) 
          {
            delete(markedStorage);
            markedStorage = NULL;
          }
          setMessageMode(false, NULL);
          break;
       case quitinfo:
          setInfoMode(false);
          break;
       case appendmarked:
          storeMarkedStreamsPrepare( obj->getName() );
          break;
       case storestation:
          storeStationInFolder(obj->getName());
          break;
       case storesite:
          storeSiteInFolder(obj->getName());
          break;
       case storeitem:
          storeLinkInFolder(obj->getName());
          break;
       case filterlist:
          if (obj->getName() == "streams only")
            fillHarvestList(true);
          if (obj->getName() == "all items")
            fillHarvestList(false);
          harvestTree.nextObject(false);
          emit eventValuesUpdated( folderlist );  // implies itemlist update
          break;
       case goback:
          emit eventHarvesterBusy( true, "fetching data" );
          if ( !streamHarvester->goBack() )
            setHarvestMode(false);
          break;
       case htmlviewer:
          showHtmlViewer();
          break;
       case download:
          startDownload();
          break;
       default :
       break;
     }
}


void StreamBrowser::markStreamItem() 
{
   StreamObject* obj;
   obj = itemTree->getStreamFolder();

   // check folder's selected item
   if ( obj && obj->getObject() )
   {
      StreamFolder *folder = dynamic_cast<StreamFolder*>(obj);
      if ( folder ) obj = folder->getStreamItem();
      // quit if item under cusror is not a stream item from storage
      if ( !obj && obj->getAction() != stream ) return;
   }

   StreamItem *item = dynamic_cast<StreamItem*>(obj);
   if (item) 
   {
     item->toggleMarked();
     emit eventValuesUpdated( itemlist );  // itemlist update
   }
}


void StreamBrowser::initRecording(QString url, QString name, QString /*descr*/, QString /*handler*/)
{
   QString errorMsg, recName;

   recName = recorderManager->recordNow(url, name, 3600, errorMsg);

   if (recName == "")
     reportEvent(errorMsg, "");
}

void StreamBrowser::slotRecordingStopped(QString recName, RecordStopReason reason)
{

   // remove *-prefix from record StreamItem
   StreamObject *obj = databaseTree.findObject("recordings");
   if (obj)
   {
      StreamFolder *folder = dynamic_cast<StreamFolder*>(obj);
      obj = folder->findObject(recName);
      if (obj)
      {
         dynamic_cast<StreamItem*>(obj)->setPrefix("#");
         emit eventValuesUpdated(itemlist);
      }
   }

   switch (reason)
   {
     case command:
        reportEvent("Finished recording " + recName, "");
     break;
     case process:
        reportEvent("Could not start recorder for " + recName, "");
     break;
     case file:
        reportEvent("Could not open storage for recording " + recName, ""); // not used
     break;
     case recorder:
        reportEvent("The recorder exited. Try to press record when playing the stream", "");
     break;
   }
}

void StreamBrowser::slotRecordingStarted(QString recName)
{
   StreamObject *obj = databaseTree.findObject("recordings");
   if (obj)
   {
      StreamFolder *folder = dynamic_cast<StreamFolder*>(obj);
      obj = folder->findObject(recName);
      if (obj)
      {
         dynamic_cast<StreamItem*>(obj)->setPrefix("R");
         emit eventValuesUpdated(itemlist);
      }
   }

   reportEvent("Started recording " + recName, "");
}

void StreamBrowser::slotScheduleEvent(QString recName, QString errorMessage, bool /*isScheduled*/)
{

   StreamObject *obj = databaseTree.findObject("recordings");
   if (obj)
   {
      StreamFolder *folder = dynamic_cast<StreamFolder*>(obj);
      obj = folder->findObject(recName);
      if (obj)
      {
         StreamItem *item = dynamic_cast<StreamItem*>(obj);

          switch (recorderManager->getItemStatus(recName) )
          {
            case scheduled:
              item->setPrefix("S");
            break;
            case recording:
              item->setPrefix("R");
            break;
            case recorded:
              item->setPrefix("#");
            break;
          }

         emit eventValuesUpdated(itemlist);
      }
   }

   if (errorMessage != "") // old recordings
     reportEvent(errorMessage + " " + recName, "");

}

void StreamBrowser::checkRecordCommand()
{

    if ( streamStatus->getStatus() == StreamStatus::playing )
    {
      initRecording( streamStatus->getStreamUrl(),
                     streamStatus->getStreamName(),
                     streamStatus->getStreamDescr(),
                     streamStatus->getStreamHandler() );
      return;
    }

    StreamFolder *folder;
    StreamObject* obj;
    obj = itemTree->getStreamFolder();

    // if folder has items act on the selected item
    if ( obj && obj->getObject() )
    {
        StreamFolder *folder = dynamic_cast<StreamFolder*>(obj);
        if ( folder ) obj = folder->getStreamItem();
        // default to existing folder action
        if ( !obj ) obj = folder;
    }

    if ( obj )
      switch ( obj->getAction() )
      {
        case stream:
            folder = itemTree->getStreamFolder();
            if (folder)
            {
                StreamItem* pendingStation = folder->getStreamItem();
                if (pendingStation)
                  initRecording( pendingStation->url,
                                 pendingStation->getName(),
                                 pendingStation->descr,
                                 pendingStation->handler );
            }
            break;
        default :
        break;
      }
}

void StreamBrowser::stopRecording()
{
    StreamFolder *folder;
    StreamItem *item;

    folder = itemTree->getStreamFolder();

    if (folder->getName() == "recordings")
    {
       item = folder->getStreamItem();
       if (item)
         recorderManager->stopRecording( item->getName() );
    }
}

//------------------------ Storage manipulation ------------------------

void StreamBrowser::slotStorageEvent(int ident, int eventType, bool error)
{
  QString errorMsg;

  if (error) return;

  switch (eventType)
  {
    case StreamStorage::loaded:
       if (ident != 110)
         slotListLoaded();
       else
       {
         storeMarkedStreamsInsertNext(true);
       }
    break;
    case StreamStorage::selected:
          if ( ident == 110 && !markedStorage->loadList(110, errorMsg) )
          {
              cerr << TARGET": cannot read from storage"
              << markedStorage->getStorageDescription() << endl;
              reportEvent(errorMsg, "");
              delete( markedStorage );
          }
    default:
    break;
  }
}

void StreamBrowser::markedStreamsInsertReady()  //sent by timer after last insert
{
   storeMarkedStreamsInsertNext(false);
}

void StreamBrowser::timerSaysFullScreen()  //sent by timer after last insert
{
   streamStatus->toggleFullScreen(true);
}

void StreamBrowser::slotMarkedRecordInserted(ChangedRecord* /*cr*/) 
{
   storeMarkedStreamsInsertNext(false);
}


/*
    if ( !streamStorage->loadList(101) )
    {
       cerr << TARGET": cannot read from storage"
            << streamStorage->getStorageDescription() << endl;
       exit( -1 );
    }

*/


void StreamBrowser::slotListLoaded()
{
    StreamFolder *folder;
    QString previousFolder;

    folder = NULL;
    previousFolder = "";
    bool error = false;

    streamStorage->resetRecordList();
    databaseTree.clearObjectList();

    ValueList values(5);
    while ( streamStorage->getNextRecord(values) )
    {
        if ( values.count() != 5 )
        {
          cerr << "stream properties mismatch (received " << values.count()
               << " out of 5):" << endl;
          for (uint i = 0; i<values.count(); i++)
            cerr << "prop" << i << ": " << values[i] << endl;
          error = true;
        }
           else
        {
            if ( values[0] != previousFolder )
            {
              folder = new StreamFolder(values[0]);
              folder->caption = "Browsing storage " + streamStorage->getStorageName();
              databaseTree.addObjectToList(folder);
              previousFolder = values[0];
            }
    
            if (folder)
            {
              StreamItem *item = new StreamItem(folder, values[1], values[2], values[3], values[4]);
              checkAndSetIcon(item);
    
              // check: ineffective
              if (folder->getName() == "recordings")
                switch (recorderManager->getItemStatus(item->getName()) )
                {
                  case scheduled:
                    item->setPrefix("S");
                  break;
                  case recording:
                    item->setPrefix("R");
                  break;
                  case recorded:
                    item->setPrefix("#");
                  break;
                }
            }
        }
    }

    emit eventValuesUpdated( folderlist );
    
    if (error)
      reportEvent("Errors in stream storage. Run from command line to see details", "");
      
}

void StreamBrowser::slotRecordInserted(ChangedRecord* rec)
{
  StreamFolder *folder;
  StreamObject *obj = databaseTree.findObject(rec->values[0]);

  if ( rec->error )
    return;

  if (obj)
    folder = dynamic_cast<StreamFolder*>(obj);
  else
  {
    folder = new StreamFolder(rec->values[0]);
    databaseTree.addObjectToList(folder);
  }

  StreamItem *item = new StreamItem( folder, rec->values[1], rec->values[2], rec->values[3], rec->values[4]);
  checkAndSetIcon(item);
  // give *-prefix when item is inserted in the recording folder by recorder
  if ( folder->getName() == "recordings" )
    if ( rec->ident == ident_recorder )
      item->setPrefix("R");
    else
      item->setPrefix("#");

  emit eventValuesUpdated( folderlist );

  if ( rec->oldValues[1] != rec->values[1] )
    reportEvent( "Saved \"" + rec->oldValues[1] + "\" as \"" + rec->values[1] + "\"", "");
  else
    reportEvent( "Item \"" + rec->values[1] + "\" saved", "");
}

void StreamBrowser::slotRecordUpdated(ChangedRecord* rec)
{
  StreamFolder *folder = 0;
  StreamItem   *item = 0;
  StreamObject *obj = databaseTree.findObject(rec->oldValues[0]);

  if ( rec->error )
    return;

  // find item
  if (obj)
  {
    folder = dynamic_cast<StreamFolder*>( obj );
    item   = dynamic_cast<StreamItem*>( obj->findObject(rec->oldValues[1]) );
  }

  if (!item)
  {
     cerr << TARGET": updated item "
          << rec->oldValues[1] << " in folder "
          << rec->oldValues[0] << " not found" << endl;
     return;
  }


  if (rec->oldValues[0] != rec->values[0])
  {
   // manage folder change
    folder->removeObjectFromList(item);
    slotRecordInserted(rec);
  }
    else
  {
   // change properties
    item->setName( rec->values[1] );
    item->url   = rec->values[2];
    item->descr = rec->values[3];
    item->handler = rec->values[4];
  }

  emit eventValuesUpdated( folderlist );

  reportEvent( "Item \"" + rec->oldValues[1] + "\" updated", "");
}

void StreamBrowser::slotRecordRemoved(ChangedRecord* rec)
{
  StreamFolder *folder = 0;
  StreamItem   *item = 0;
  StreamObject *obj = databaseTree.findObject(rec->oldValues[0]);
  bool fileDeleted = false;

  if ( rec->error )
    return;

  // find item
  if (obj)
  {
    folder = dynamic_cast<StreamFolder*>( obj );
    item   = dynamic_cast<StreamItem*>( obj->findObject(rec->oldValues[1]) );
  }

  if (!item)
  {
     cerr << TARGET": removed item "
          << rec->oldValues[1] << " in folder "
          << rec->oldValues[0] << " not found" << endl;
     return;
  }

  if ( folder ) folder->removeObjectFromList(item);

  emit eventValuesUpdated( folderlist );

  if (!fileDeleted)
    reportEvent( "Item \"" + rec->values[1] + "\" removed", "");
  else
    reportEvent( "Item + file \"" + rec->values[1] + "\" removed", "");
}

void StreamBrowser::storeStationInFolder(QString folderName)
{
  QString& name  = streamStatus->getLastPlayedName();
  QString& url   = streamStatus->getLastPlayedUrl();
  QString& descr = streamStatus->getLastPlayedDescr();
  QString& handler = streamStatus->getLastPlayedHandler();

  storeItemInFolder(folderName, name, url, descr, handler);
}

void StreamBrowser::storeSiteInFolder(QString folderName)
{
  QString& name  = streamHarvester->getCurrentName();
  QString& url   = streamHarvester->getCurrentUrl();
  QString& descr = streamHarvester->getCurrentDescr();
  QString& handler = streamHarvester->getCurrentHandler();

  storeItemInFolder(folderName, name, url, descr, handler);
}

void StreamBrowser::storeLinkInFolder(QString folderName)
{
  StreamFolder *folder = 0;
  StreamItem   *item = 0;

  // get current item
  folder = itemTree->getStreamFolder();
  if (folder)
    item = folder->getStreamItem();

  if ( !item )
  {
    reportEvent( "No link to save", "");
    return;
  }

  QString name  = item->getName();
  QString url   = item->url;
  QString descr = item->descr;
  QString handler = item->handler;
  
  storeItemInFolder(folderName, name, url, descr, handler);
}

void StreamBrowser::storeItemInFolder(QString folderName, QString name, QString url, QString descr, QString handler)
{
  //cout << "store in folder " << folderName << endl;

  StreamFolder *folder = 0;
  StreamObject *obj = databaseTree.findObject(folderName);

  if ( obj )
    folder = dynamic_cast<StreamFolder*>(obj);

  if ( !folder )
  {
     cerr << TARGET": cannot find folder "
          << folderName << " to store item" << endl;
     return;
  }

  if ( url == "" )
  {
     reportEvent( "No url specified", "");
     return;
  }

  QString errorMsg;

  ValueList values(5);
  values[0] = folderName;
  values[1] = name;
  values[2] = url;
  values[3] = descr;
  values[4] = handler;

  bool res = streamStorage->insertRecord(101, values, errorMsg);
  if ( !res)
  {
      reportEvent( errorMsg, "");
      return;
  }

}

//------------------------ User Input handling ------------------------

bool StreamBrowser::handlePressedKey(int key)
{
    bool handled = true;
    QString error;
        
    if (key == prevfolder)
        folderPrev();
    else if (key == nextfolder)
        folderNext();
    else if (key == previtem)
        itemPrev(1);
    else if (key == nextitem)
        itemNext(1);
    else if (key == jmpprevitem)
        itemPrev(6);
    else if (key == jmpnextitem)
        itemNext(6);
    else if (key == selectitem)
          checkFolderCommand();
    else if (key == mark)
          markStreamItem();
    else if (key == storemarked)
          setStorageMode(true);
    else if (key == inspect)
          setInfoMode(true);
    else if (key == record)
          checkRecordCommand();
    else if (key == stoprecord)
          stopRecording();
    else if (key == stopallrecord)
          recorderManager->stopAllRecordings();
    else if (key == end)
        {
          streamStatus->stopStream();
          streamHarvester->stop(); // ... being busy
        }
    else if (key == pauseitem)
        streamStatus->issueCommand(StreamStatus::Pause);
    else if (key == mute)
        streamStatus->issueCommand(StreamStatus::Mute);
    else if (key == volup)
        streamStatus->issueCommand(StreamStatus::VolumeUp);
    else if (key == voldn)
        streamStatus->issueCommand(StreamStatus::VolumeDn);
    else if (key == avdec)
        streamStatus->issueCommand(StreamStatus::AVDec);
    else if (key == avinc)
        streamStatus->issueCommand(StreamStatus::AVInc);
    else if (key == forwardstream)
        streamStatus->issueCommand(StreamStatus::Forward);
    else if (key == rewindstream)
        streamStatus->issueCommand(StreamStatus::Rewind);
    else if (key == fullscreen) // fullscreen while video plays, debugwindow when no video
        streamStatus->toggleFullScreen(false);
    else if (key == dump)
        toggleDumpWindow();
    else if (key == escape)
      {
			if ( streamStatus->getStatus() != StreamStatus::idle &&
			     streamStatus->getStatus() != StreamStatus::stopped &&
			     streamStatus->getStatus() != StreamStatus::nostream &&
			     streamStatus->getStatus() != StreamStatus::failed      )
			  streamStatus->stopStream();
			else
			  handled = false;
      }
    else
        handled = false;

    return handled;
}


void StreamBrowser::folderPrev()
{
   itemTree->prevObject(m_folderdisplayfolded);
   itemTree->cursor--;
   emit eventValuesUpdated( folderlist );  // implies itemlist update
}

void StreamBrowser::itemPrev(uint cnt)
{
    StreamFolder *folder = itemTree->getStreamFolder();
    if (folder)
    {
      for (uint i = 0; i < cnt; i++)
      {
        folder->prevObject(m_itemdisplayfolded);
        folder->cursor--;
      }
    }
    emit eventValuesUpdated( itemlist );
}

void StreamBrowser::folderNext()
{
    itemTree->nextObject(m_folderdisplayfolded);
    itemTree->cursor++;
    emit eventValuesUpdated( folderlist );  // implies itemlist update
}

void StreamBrowser::itemNext(uint cnt)
{
    StreamFolder *folder = itemTree->getStreamFolder();
    if (folder)
    {
      for (uint i = 0; i < cnt; i++)
      {
        folder->nextObject(m_itemdisplayfolded);
        folder->cursor++;
      }
    }
    emit eventValuesUpdated( itemlist );
}

void StreamBrowser::toggleDumpWindow()
{
    if (m_dumpwindow == 0)
    {
      m_dumpwindow = new DumpWindow( 0, TARGET" Player dump",  WDestructiveClose | WStyle_Customize | WStyle_DialogBorder );
      connect( m_dumpwindow, SIGNAL(destroyed()), this, SLOT(destroyedDumpWindow()) );
      connect( m_dumpwindow, SIGNAL(manualUrlEntered(QString)), this, SLOT(handleExternalURI(QString)));

      loadDumpWindow();
    }
    else
    {
      m_dumpwindow->close(true);
      m_dumpwindow = 0;
    }
}


void StreamBrowser::handleExternalURI( QString uri )
{
   QString descr = "external url";
   QString handler = "";
   
   //cout << "received " << uri << endl;
   
   QUrl *qurl = new QUrl( uri );
   QString protocol = qurl->protocol();
   delete (qurl);

   if ( protocol == "file" )
   {
      // expecting playlist, no a/v-file
      QFile file( uri );

      if (!file.exists() || !file.open(IO_ReadOnly))
      {
         reportEvent("cannot open " + uri, "");
         return;
      }

      QTextStream stream( &file );
      QString line;
      // check/trust mime? Same as harvester, check yourself?
      while ( !stream.eof() && sizeof(line) < 20000 )
        line += stream.readLine() + " ";
      file.close();

      uri = line;

   }
   //   else
   //{
      //expecting stream or playlist
   //   streamStatus->initStream( uri, uri, descr);
  // }
   streamStatus->appendLastPlayedConsole("received external uri " + uri );
   streamStatus->initStream( uri, uri, descr, handler ); 
   //setHarvestMode( true );
   //streamHarvester->parseData( uri, uri, descr );
}

// (gui) init ready, do your thing...
void StreamBrowser::initReady()
{
// experimental
   if ( qApp->argc() > 1 )
          handleExternalURI( qApp->argv()[1] );
}
