/* ============================================================
 * File  : recorder.h
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2003-12-22
 * Description : av stream handling
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// recorder experimental

#ifndef RECORDER_H
#define RECORDER_H

#include <iostream>

#include <qnetwork.h>
#include <qdatetime.h>
#include <qpainter.h>
#include <qdir.h>
#include <qtimer.h>
#include <qprocess.h>
#include <qstringlist.h>
#include <qdict.h>
#include <qvbox.h>

#include <qdatetime.h>

#include "configelements.h"
#include "streamstorage.h"


class Recorder;

typedef QDict<Recorder> RecList;

enum RecordStopReason
{
   recorder = 0,
   file,
   command,
   process
};

enum ItemStatus
{
   scheduled = 0,
   recording,
   recorded
};

class RecorderManager : public QObject
{
   Q_OBJECT

   public:

     RecorderManager(QObject *parent, StreamStorage *storage);
     ~RecorderManager();

     // name = REC...YYYY-MM-DD....HH:MM....hh:mm.....
     //  or    REC...YYYYMMDD......HHMM.....hhmm.....
     //                date        start    stop
     // returns recordName (name of stream item entry in storage or empty string on error)

     QString recordNow(QString url, QString name, uint seconds, QString& errorMessage);

     ItemStatus getItemStatus(QString recName);

     void stopRecording(QString recName);
     void stopAllRecordings();

   private slots:
    void slotStorageEvent(int /*ident*/, int eventType, bool error);
    //void slotListLoaded();
    void slotRecordInserted(ChangedRecord* rec);
    void slotRecordUpdated(ChangedRecord* rec);
    void slotRecordRemoved(ChangedRecord* rec);
    void slotRecorderStarted(Recorder*);
    void slotRecorderStopped(Recorder*);
    void timerEvent();

   signals:
    void recordingStopped(QString recName, RecordStopReason reason);
    void recordingStarted(QString recName);
    void scheduleEvent(QString recName, QString message, bool isScheduled);
    void recorderActive(bool active);
    void recorderMessage(QString message);

   private:
     StreamStorage *streamStorage;
     RecList recList;
     bool kill;

     bool createStreamItem(QString name, QString url, QString descr, QString handler);
     bool deleteStreamItem(QString name, QString url, QString descr, QString handler);
     bool getUTime(QString name, QDateTime& start, QDateTime& stop);
     void handleNewRecord(ValueList &values, bool allowOverwrite);
     bool scheduleRecording(QString name, QString url, QString descr, QString handler,
                            QString& errorMessage, bool allowOverwrite);
     Recorder* assignRecorder(QString recName, QString url, QString fileName,
                              QDateTime startDt, QDateTime stopDt);
};


// recorder(, seconds, ): recording length seconds, 0=record until Recorder destruct

class Recorder : public QObject
{
  Q_OBJECT

  public:

    Recorder(QObject *parent, QString recName, QString url, QString fileName,
             QDateTime startDt, QDateTime stopDt);
    ~Recorder();

    bool startRecording(QString& errorMsg);
    void stopRecording();

    QString recName;
    QString fileName;
    QString handler;
    QString url;
    QDateTime startDt;
    QDateTime stopDt;
    bool isRecording;

    bool checkSchedule(QString& errorMsg);
    RecordStopReason getStopReason() { return reason; }

  private slots:
    void streamExited();
    void readFromStdout();
    void readFromStderr();

  signals:
    void recordingStarted(Recorder*);
    void recordingStopped(Recorder*);

  private:
    void startStream();
    void stopStream();

    void parsePlayerOutput(const QString msg);

    QObject *myParent;
    QProcess *proc;

    RecordStopReason reason;
    bool sawPlayerOutput;
};


#endif /* RECORDER_H */





