/* ============================================================
 * File  : harvester.h
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2004-03-08
 * Description : harvester scans url for streams
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include <iostream>
#include <qprocess.h>
#include <qthread.h>
#include <qprocess.h>
#include "qhttp.h"
#include <qdict.h>
#include <qmap.h>

#include "configelements.h"
#include "streamstorage.h"

using namespace std;

class StreamHarvester;

//------------------------------------------------------------------

class Downloader : public QObject
{
  Q_OBJECT
  
  public:
    enum Availability
    {
       available = 0,  // downloaded as filename
       pending,        // downloading in filename_temp
       notavailable    // not available
    };
    
    Downloader(QString directory, StreamStorage *storage);
    // returns absolute path of downloading file
    QString download(QString url, QString sName, bool& canplaynow, bool& error,
                     bool& itemcreated, QString& errorMessage, bool checkName = false);
    // check availability of file, and optionally create path in cache if not exists
    // use checkName=true for FUZZY_DL match
    QString checkAvailable(QString url, Availability& availability,
                           bool& errorWhileCreating, bool createPathIfNotExists,
                           QString name = "", bool checkName = false);
    //QString checkAvailable(QString url, bool& exists, bool& errorWhileCreating);
    QStringList& getDownloads() { return downloads; }    

  signals:
    void downloadFinished(QString url, bool error, QString errorMessage);
  
  private slots:
    void slotDownloadStopped();
    void slotRecordRemoved(ChangedRecord* rec);
  
  private:
    bool createStreamItem(QString name, QString url, QString descr, QString handler);
    bool updateStreamItemFilename(QString oldFile, QString newFile);
    bool deleteStreamItem(QString name, QString url, QString descr, QString handler);
    
    QString directory;
    QString home;
    QProcess *proc;
    QStringList downloads;

    StreamStorage *streamStorage;

};

//------------------------------------------------------------------

class CacheItem
{
   public:
     CacheItem(QString lastdate, QString filename);
     QString filename;
     QString lastdate;
};

class Cache
{
  public:
    typedef QDict<CacheItem> History;
     
     Cache();
     ~Cache();
     bool getCacheItem(QString url, QString& lastdate, QString& filename);
     bool copyCacheFile(QString source, QString dest);
     bool setCacheItem(QString url, QString lastdate, QString tempfilename);

  private:
     void loadCache();
     void saveCache();
     bool openCacheFile(bool overwrite);
     bool closeCacheFile();
     QString createNewFile();
     
     History history;
     uint    fileIndex;
     QString cachePath;
     QFile   myFile;
};

//------------------------------------------------------------------


class CookieBin
{
  public:
     typedef QMap<QString, QString> CookieMap;

     CookieBin();
     ~CookieBin();
     
     void    updateCookie(QString host, QString newCookie); // "set-cookie"-line from response header
     bool    hasCookies(QString host);                      // cookie(s) for host present?
     QString getCookieHdrLine(QString host);                // "cookie"-line to send with request header

     void    setReferer(QString host, QString referer);
     bool    hasReferer(QString host);
     QString getReferer(QString host);
     
  private:     
     CookieMap cookieMap;
     QString referer, refererHost;
};

//------------------------------------------------------------------

class FetchBuffer
{
   public:
     FetchBuffer(QString host, QString url, QString fileName, int requestId, QString& errorMessage);
     ~FetchBuffer();
     
     void writeData(QString& text);
     void setResponseDate(QString date) { responseDate = date; }
     void setRequestId(int id)          { requestId    = id;   } 
     void closeBuffer();
     
     QString getResponseDate() { return responseDate; } // use cache if empty string
     int     getRequestId()    { return requestId; }
     QString getUrl()          { return url; }
     QString getFileName()     { return fileName; }
     bool    isOpen()          { return file.isOpen(); }

     int urlEstimateCount;
         
   private:
     QFile file;
     QTextStream stream;
     int requestId;
     QString host, url, fileName, responseDate;
     bool useCache;
};

class Requester : public QHttpX
{
  Q_OBJECT

  public:

    Requester();
    ~Requester();

    bool fetchData(QString url, QString fileName, QString& errorMsg);
    void stop();

    CookieBin cookieBin; // cookie storage

    QString hHost, hUrl; // http header values
  
  public slots:
    void slotResponseHeaderReceived(const QHttpXResponseHeader& response);
    void slotRequestFinished(int id, bool error);
    void slotDataReadProgress(int read, int size);
    void slotReadyRead(const QHttpXResponseHeader &resp);

  signals:
    void fetchReady(bool error, QString& data);
    void fetchProgress(int read, int size);

  private:
    enum Request
    {
      idle = 0,
      sethost,
      getdata,
      notmodified,
      abort
    };
    Request status;

    QHttpX *http;
    Cache *cache;

    int port;
    QString host;
    QString url;   // source url
    
    QString fileName;
    int redirectCount;
    
    QFile destFile;
    QTextStream stream;
    FetchBuffer *fetchBuffer;

};

//------------------------------------------------------------------

class NewUrl : public QObject
{
  public:
    NewUrl(QString nUrl, QString nName, QString nDescr, QString nHandl);

    QString url;
    QString name;
    QString descr;
    QString proto;
    int port;
    QString file;     // {stream, other}
    QString context;  // {absolute,relative} (but all url's are MADE absolute using harvested url path)
    int tag;          // internal
    QString handler;  // STREAM_* command or name of parser to use when user activates this item.
                      //   STREAM_DL: download before play (podcast files) 
    QStringList meta; // additional information. 
                      //   Format:  name<viewer>content. Viewer resembles mime, can be: text, html
};

typedef QPtrList<NewUrl> UrlList;

//------------------------------------------------------------------

class StreamHarvester : public QObject
{
  Q_OBJECT

  public:
    enum Status
    {
      ready = 0,
      parsing,
      receiving,
      errorset
    };

    enum Error
    {
      invalidurl = 0,
      nodata,
      nourlindata,
      bogusparseroutput
    };

    StreamHarvester();
    ~StreamHarvester();

    void fetchData(QString& url, QString& name, QString& descr, QString& handler);
    void parseData(QString& data, QString& name, QString& descr, QString& handler);
    void stop();

    bool goBack();
    bool isBusy();

    QString& getCurrentUrl();
    QString& getCurrentName();
    QString& getCurrentDescr();
    QString& getCurrentHandler();
    int      getCurrentTag();

    void setTagOnCurrent(int tag);  // user tag on url

    QString checkHasParser(QString handler);
    void getParser(QString &parser, QString &path);
    
    UrlList urlList;
    QString parsersPath, parsersPath_install;
    
  protected:
    void customEvent(QCustomEvent *ev);

  signals:
    void fetchStatus(int, unsigned int cnt);

  public slots:
    void processExited();
    void parserExited(int);
    void slotFetchProgress(int read, int size);
    void slotfetchReady(bool error, QString&);

    void externalParserRead();
    void externalParserExited();
  
  private:
    void storeParserUrl();
    void externalParserStart(QString srcUrl, QString parser, QString name);
    void externalParserTerminate();
    bool externalParserLoadUrl(QString& xmlData);
    void send(QString type, int cnt);   
    NewUrl *eParserUrl;
    QString sourceUrl, baseUrl, rootUrl;
    int urlCount;

    QString defaultParser;
    
    void startFetch();
    void stopFetch();

    bool busy;
    bool pending;
    bool aborted;

    UrlList history;

    QString currentUrl;
    QString currentName;
    QString currentDescr;
    QString currentHandler;
    
    int     currentTag;
    int     rootTag;    // emptyhistory user tag storage

    QString data, parserOutput;
    QProcess* proc;
    QProcess *eParser;

    Requester *requester;
    QString lastHost;   
};


