/* LoaderSaverHelper.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;

import org.grinvin.graphs.GraphURI;
import org.grinvin.list.HasURI;
import org.grinvin.preferences.GrinvinPreferences;
import org.grinvin.util.IOHelper;
import org.grinvin.xml.XMLUtils;

import org.jdom.DocType;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.output.XMLOutputter;

/**
 *
 */
public class LoaderSaverHelper {

    private LoaderSaverHelper() {
    }

    //
    public static String prepareSessionURI(HasURI hasURI, String prefix, String extension) {
        URI uri = hasURI.getURI();
        if (!GraphURI.isSession(uri)) {
            throw new IllegalArgumentException("Can only save session objects into the workspace");
        }
        if (uri == null) {
            String name = prefix + "/" + IOHelper.uniqueFileName("GIO", "." + extension);
            try {
                uri = GraphURI.createSession(name);
                hasURI.setURI(uri);
            } catch (URISyntaxException e) {
                assert false : "Unexpected URISyntaxException";
            }
        }
        return uri.getSchemeSpecificPart();
    }

    public static Element loadElement(InputStream input) throws IOException {
        try {
            return XMLUtils.loadFromInputStream(input);
        } catch (JDOMException ex) {
            throw new IOFormatException("XML parse error", ex);
        }
    }
    
    public static Document loadDocument(InputStream input) throws IOException {
        try {
            return XMLUtils.loadDocumentFromInputStream(input);
        } catch (JDOMException ex) {
            throw new IOFormatException("XML parse error", ex);
        }
    }

    public static Element loadElementNonValidating(InputStream input, String rootElement) throws IOException {
        try {
            return checkElement(XMLUtils.loadFromInputStreamNonValidating(input), rootElement);
        } catch (JDOMException ex) {
            throw new IOFormatException("XML parse error", ex);
        }
    }
    
    public static Element checkElement(Element element, String rootElement) throws IOFormatException {
        if (!rootElement.equals(element.getName())) {
            throw new IOFormatException("Expected '" + rootElement + "' element");
        }
        return element;
    }
    //
    private static XMLOutputter outputter = null;

    //
    private static XMLOutputter getOutputter() {
        if (outputter == null) {
            outputter = new XMLOutputter(GrinvinPreferences.getInstance().getPreferredXMLFormat());
        }
        return outputter;
    }

    //
    public static void outputXML(Element element, DocType docType, OutputStream out) throws IOException {
        Document document = new Document(element);
        if (document != null) {
            document.setDocType(docType);
        }
        getOutputter().output(document, out);
    }

    public static void outputXML(Element element, OutputStream out) throws IOException {
        outputXML(element, null, out);
    }
}
