/* AbstractInvariantValueDelegate.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.values;

import javax.swing.JComponent;
import javax.swing.JLabel;

import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.invariants.InvariantType;
import org.grinvin.invariants.InvariantValue;
import org.grinvin.invariants.InvariantValueDelegate;
import org.grinvin.invariants.UnknownInvariantException;
import org.grinvin.invariants.UnknownInvariantTypeException;
import org.grinvin.io.IOFormatException;
import org.grinvin.list.graphs.GraphListElement;

import org.jdom.Element;

/**
 * 
 */
public abstract class AbstractInvariantValueDelegate<T extends InvariantValue> implements InvariantValueDelegate<T> {
    
    /**
     * Converts this value to a JDOM element. Delegates saving of the actual value to
     * {@link #saveValue}.
     */
    public Element toElement(T value) {
        Element element = new Element("invariantvalue").setAttribute("type", value.getType().getTypeName());
        element.addContent(new Element("invariant").setAttribute("id", value.getInvariant().getId()));
        element.addContent(new Element("computer").setAttribute("id", value.getComputerId())
            .setAttribute("version", value.getComputerVersion()));
        Element child = new Element("value");
        saveValue(child, value);
        return element.addContent(child);
    }
    
    
    /**
     * Should be overridden to save the actual value into a JDOM-element.
     * @param element JDOM element, which is of type 'value'
     */
    public abstract void saveValue (Element element, T value);
    
    // TODO: load (see InvariantValuesLoader)

    /**
     * Initialize the components of this value from a JDOM-element.
     * The element is known to be of type 'invariantvalue'.
     * Delegates to {@link #loadValue} to load the actual value.
     */
    public T load (Element element) throws 
            IOFormatException, UnknownInvariantException, UnknownInvariantTypeException {

        Element child = element.getChild ("invariant");
        if (child == null)
            throw new IOFormatException ("Missing 'invariant' element in 'invariantvalue'");
        Invariant invariant = InvariantManager.getInstance().getInvariant(child.getAttributeValue("id"));
        
        child = element.getChild ("computer");
        if (child == null)
            throw new IOFormatException ("Missing 'computer' element in 'invariantvalue'");
        String computerId = child.getAttributeValue("id");
        String computerVersion = child.getAttributeValue("version");
        
        child = element.getChild ("value");
        if (child == null)
            throw new IOFormatException ("Missing 'value' element in 'invariantvalue'");
        return loadValue (child, invariant, computerId, computerVersion);
        
    }

    /**
     * Initialize the components of this value from a JDOM-element.
     * The element is known to be of type 'invariantvalue'.
     * Delegates to {@link #loadValue} to load the actual value.
     * This method implements 1.0 compatibility.
     */
    public T load_1_0 (Element element) throws 
            IOFormatException, UnknownInvariantException, UnknownInvariantTypeException {

        Element child = element.getChild ("invariant");
        if (child == null)
            throw new IOFormatException ("Missing 'invariant' element in 'invariantvalue'");
        Invariant invariant = InvariantManager.getInstance().getInvariant(child.getTextTrim());
        
        child = element.getChild ("computer");
        if (child == null)
            throw new IOFormatException ("Missing 'computer' element in 'invariantvalue'");
        String computerId = child.getTextTrim();
        String computerVersion = child.getAttributeValue("version");
        
        child = element.getChild ("value");
        if (child == null)
            throw new IOFormatException ("Missing 'value' element in 'invariantvalue'");
        return loadValue (child, invariant, computerId, computerVersion);
        
    }
    
    /**
     * Should be overridden to load the actual value from a JDOM-element.
     * Clients should throw an IOFormatException when element is not
     * in the correct format.
     * @param element JDOM element, which is known to be of type
     * 'value'
     * @see #load
     */
    public abstract T loadValue (Element element, Invariant invariant, String computerId, String computerVersion) throws IOFormatException;

    
    /**
     * Convert a JDOM-element to an invariant value of the correct class.
     */
    public static InvariantValue fromElement (Element element) throws
            IOFormatException, UnknownInvariantException, UnknownInvariantTypeException {
        if (! "invariantvalue".equals (element.getName ()))
            throw new IOFormatException ("Expected <invariantvalue> element");
        String type = element.getAttributeValue ("type");
        if (type == null)
            throw new IOFormatException ("Missing type attribute for invariant value");
        
        return InvariantType.getEnumForType(type).getDelegate().load(element);
    }

    /**
     * Convert a JDOM-element to an invariant value of the correct class.
     * This method implements 1.0 compatibility.
     */
    public static InvariantValue fromElement_1_0(Element element) throws
            IOFormatException, UnknownInvariantException, UnknownInvariantTypeException {
        if (! "invariantvalue".equals (element.getName ()))
            throw new IOFormatException ("Expected <invariantvalue> element");
        String type = element.getAttributeValue ("type");
        if (type == null)
            throw new IOFormatException ("Missing type attribute for invariant value");
        
        return InvariantType.getEnumForType(type).getDelegate().load_1_0(element);
    }
    
    /**
     * Initialize a label which is used to display this invariant value.
     * The standard implementation simply renders the text returned by
     * {@link InvariantValue#toText} and leaves the icon blank.
     */
    public void initLabel(JLabel label, T value) {
        label.setIcon(null);
        label.setText(value.toText());
    }
    
    /**
     * Show a 'details window' for this component.
     * <p>The standard implementation does nothing.
     *
     * @param value The invariant value
     * @param element Graph list element for which this invariant value
     * was computed
     * @param component Component which invoked this method
     * @param x X coordinate of mouse pointer when this method was invoked
     * @param y Y coordinate of mouse pointer when this method was invoked
     */
    public void showDetails(T value, GraphListElement element, JComponent component, int x, int y) {
        // intentionally empty
    }
    
    /**
     * Is it possible to obtain a 'details window' for this component?
     * <p>The standard implementation always returns false.
     */
    public boolean hasDetails(T value) {
        return false;
    }

}
