/* InvariantFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants;

import org.grinvin.factories.Factory;
import org.grinvin.factories.FactoryParameterException;

/**
 * An object that satisfies this interface can be used to create an invariant.
 * To have a factory create an invariant, first
 * set its parameters using {@link #setParameterValues} and
 * then call {@link #createInvariant}.
 */
public interface InvariantFactory extends Factory, InvariantNode {
    
    /**
     * Creates and returns an invariant for the current parameter values.
     * @throws FactoryParameterException when the parameters
     * were not set in an appropriate manner before this method was called.
     */
    public InvariantExpressionNode createInvariant()
    throws FactoryParameterException;
    
    /**
     * Return the id of the invariant which will be created
     * from this factory when using the current parameter values. Every id
     * returned by this method has the same structure: it starts with the
     * generic id, as returned by {@link #getId} and is followed by a parameter
     * string, which is structured like a query string of a URI.
     */
    public String getInvariantId() throws FactoryParameterException;
    
    /**
     * Return the generic identifier of this factory. Should be a prefix
     * of every invariant id generated by this factory.
     * @see #getInvariantId
     */
    public String getId();
    
    /**
     * Return the localized name for this parametrized invariant.
     */
    public String getName();
    
    /**
     * Returns the class to which all values of invariants created by
     * this factory belong.
     * This class must be an extension of {@link InvariantValue}.
     */
    public Class<? extends InvariantValue> getType();
    
    /**
     * Should return the same string as {@link #getName}.
     */
    public String toString();
    
    /**
     * Always should return null. A factory cannot have other invariant nodes as
     * children.
     */
    public Iterable<InvariantNode> getChildren();
    
}
