/*
 * Copyright (c) 2000-2005 CyberFOX Software, Inc. All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Library General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the
 *  Free Software Foundation, Inc.
 *  59 Temple Place
 *  Suite 330
 *  Boston, MA 02111-1307
 *  USA
 */

import java.net.*;
import java.util.*;
import java.io.IOException;

public class CookieJar {
  private Map _cookies;
  private boolean m_ignore_redirect_cookies = true;
  private final static boolean do_uber_debug = false;

  public CookieJar() {
    _cookies = new TreeMap();
  }

  void ignoreCookiesInRedirects() {
    m_ignore_redirect_cookies = true;
  }

  void catchCookiesInRedirects() {
    m_ignore_redirect_cookies = false;
  }

  public Cookie getCookie(String keyName) {
    return (Cookie)_cookies.get(keyName);
  }

  /**
   * Get all the cookies and the page data itself, using post.
   *
   * @param pageName - The URL and CGI to retrieve (the CGI is seperated out and sent as the POST).
   *
   * @return - A StringBuffer containing the data at the provided URL, or returned from the POST operation.
   */
  public StringBuffer getAllCookiesAndPage(String pageName) throws CookieException {
    return getAllCookiesAndPage(pageName, null, true);
  }

  /**
   * Get all the cookies and the page data itself, using post.
   *
   * @param pageName - The URL and CGI to retrieve (the CGI is seperated out and sent as the POST).
   * @param referer - The URL that referred us to this page (can be null).
   * @return - A StringBuffer containing the data at the provided URL, or returned from the POST operation.
   */
  public StringBuffer getAllCookiesAndPage(String pageName, String referer) throws CookieException {
    return getAllCookiesAndPage(pageName, referer, true);
  }

  /**
   * Get all the cookies and the page data itself, using post or get.
   *
   * @param pageName - The URL and CGI to retrieve.
   * @param referer  - The URL that referred us to this page (can be null).
   * @param doPost   - Whether or not to use POST to send any CGI associated with the pageName.
   *
   * @return - A StringBuffer containing the data at the provided URL, or returned from the POST operation.
   */
  public StringBuffer getAllCookiesAndPage(String pageName, String referer, boolean doPost) throws CookieException {
    return getAllCookiesAndPage(pageName, referer, doPost, null);
  }

  public class CookieException extends Exception {
    public CookieException(String text, Throwable trigger) {
      super(text, trigger);
    }
  }

  public StringBuffer getAllCookiesAndPage(String pageName, String referer, boolean doPost, List pages) throws CookieException {
    URLConnection uc = getAllCookiesFromPage(pageName, referer, doPost, pages);
    if(uc == null) return null;

    StringBuffer sb = null;

    try {
      sb = Http.receivePage(uc);
    } catch(ConnectException ce) {
      logException(pageName, ce);
      if(ce.toString().indexOf("Connection refused") != -1) {
        throw new CookieException("Connection refused", ce);
      }
    } catch(IOException e) {
      logException(pageName, e);
      return null;
    }

    return sb;
  }

  private void logException(String pageName, Exception e) {
    String errmsg;
    int qLoc = pageName.indexOf('?');

    if(qLoc == -1) {
      errmsg = "Error loading page: " + pageName;
    } else {
      errmsg = "Error loading page: " + pageName.substring(0,qLoc);
    }
    ErrorManagement.handleException(errmsg, e);
  }

  /**
   * Retrieve any cookies from the provided page via GET or POST, but only return
   * the URLConnection letting the caller do what they want to with it.
   *
   * @param pageName - The page to load.
   * @param referer  - The page that referred us to this page, can be null.
   * @param post     - Use 'post' or 'get' (true == use 'post').
   * @return - A URLConnection connected to the response from the server for the given request.
   */
  public URLConnection getAllCookiesFromPage(String pageName, String referer, boolean post) {
    return getAllCookiesFromPage(pageName, referer, post, null);
  }
  public URLConnection getAllCookiesFromPage(String pageName, String referer, boolean post, List pages) {
    String sendRequest = pageName;

    if(pages != null) pages.add(pageName);
    String cgi = null;
    if(post) {
      int cgipos = pageName.indexOf("?");
      if(cgipos < 0) {
        post = false;
      } else {
        cgi = pageName.substring(cgipos + 1);
        sendRequest = pageName.substring(0, cgipos);
      }
    }

    if(do_uber_debug && JConfig.debugging) {
      int comIndex = sendRequest.indexOf(".com/");
      int netIndex = sendRequest.indexOf(".net/");

      if(netIndex == -1) netIndex = comIndex;
      if(comIndex == -1) comIndex = netIndex;
      comIndex = Math.min(netIndex, comIndex);
      if(comIndex == -1) comIndex = 0;
      else comIndex += 4;

      if(post) {
        System.err.println("POST " + sendRequest.substring(comIndex) + " HTTP/1.0");
        if(cgi != null) {
          if(!cgi.equals("")) {
            System.err.println("CGI: " + cgi);
          }
        }
      } else {
        System.err.println("GET " + sendRequest.substring(comIndex) + " HTTP/1.0");
      }
      if(_cookies.size() > 0) {
        System.err.println("internal-Cookie: " + this.toString());
      }
      System.err.println();
    }
    HttpURLConnection uc;

    if(_cookies.size() > 0) {
      if(post) {
        uc = (HttpURLConnection)Http.postFormPage(sendRequest, cgi, this.toString(), referer, m_ignore_redirect_cookies);
      } else {
        uc = (HttpURLConnection)Http.getPage(sendRequest, this.toString(), referer, m_ignore_redirect_cookies);
      }
    } else {
      if(post) {
        uc = (HttpURLConnection)Http.postFormPage(sendRequest, cgi, null, referer, m_ignore_redirect_cookies);
      } else {
        uc = (HttpURLConnection)Http.getPage(sendRequest, null, referer, m_ignore_redirect_cookies);
      }
    }

    if(uc != null) {
      int i = 1;
      String redirect = null;
      String nextKey;
      do {
        nextKey = uc.getHeaderFieldKey(i);
        if(nextKey != null) {
          if(do_uber_debug) {
            ErrorManagement.logDebug(nextKey+": " + uc.getHeaderField(i));
          }
          //  If we're redirected, shortcut to loading the new page.
          if(nextKey.startsWith("Location") ||
             nextKey.startsWith("location")) {
            redirect = uc.getHeaderField(i);
            int amploc = redirect.indexOf("&amp;");

            while(amploc != -1) {
              redirect = redirect.substring(0, amploc) + "&" +
                         redirect.substring(amploc+5);
              amploc = redirect.indexOf("&amp;");
            }
            if(!redirect.startsWith("http")) {
              String prefix;
              String slash = "";
              int serverEnd = pageName.indexOf(".com/");
              if(!redirect.startsWith("/")) {
                slash = "/";
              }
              if(serverEnd == -1) {
                prefix = pageName;
              } else {
                prefix = pageName.substring(0, serverEnd+4);
              }
              redirect = prefix + slash + redirect;
            }
          }

          if(nextKey.startsWith("Set-Cookie") ||
             nextKey.startsWith("Set-cookie")) {
            Cookie newCookie = new Cookie(uc.getHeaderField(i));
            _cookies.put(newCookie.getKey(), newCookie);
          }
        }
        i++;
      } while(nextKey != null);

      if(redirect != null) {
        if (do_uber_debug) {
          ErrorManagement.logDebug("Redirecting to: " + redirect);
        }
        return getAllCookiesFromPage(redirect, referer, post, pages);
      }
    }

    return uc;
  }

  public String toString() {
    boolean firstThrough = true;
    StringBuffer outBuf = null;
    Cookie stepper;

    Iterator stepOver = _cookies.values().iterator();
    while(stepOver.hasNext()) {
      stepper = (Cookie)stepOver.next();
      if(!stepper.getValue().equals("")) {
        if (!firstThrough) {
          outBuf.append("; ");
        } else {
          firstThrough = false;
          outBuf = new StringBuffer();
        }
        outBuf.append(stepper.getKey());
        outBuf.append("=");
        outBuf.append(stepper.getValue());
      }
    }

    if(outBuf != null) {
      return outBuf.toString();
    } else {
      return null;
    }
  }
}
