/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2000-2007 Sun Microsystems, Inc. All rights reserved. 
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License ("CDDL") (collectively, the "License").  You may
 * not use this file except in compliance with the License.  You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or mq/legal/LICENSE.txt.  See the License for the specific language
 * governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at mq/legal/LICENSE.txt.  Sun designates
 * this particular file as subject to the "Classpath" exception as provided by
 * Sun in the GPL Version 2 section of the License file that accompanied this
 * code.  If applicable, add the following below the License Header, with the
 * fields enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or  to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright holder. 
 */

/*
 * @(#)ReadWritePacket.java	1.28 06/27/07
 */ 

package com.sun.messaging.jmq.io;

import java.util.Hashtable;
import java.io.*;

/**
 * This class is an ecapsulation of a JMQ packet.
 */
public class ReadWritePacket extends ReadOnlyPacket {

    static int sequenceNumber = 0;

    // Dirty flags. These are true when the buffer is out-of-date
    // with respect to the state of this object.
    protected boolean headerBufferDirty = false;
    protected boolean ropBufferDirty    = false;

    // Message body set by setMessageBody()
    protected byte[] messageBody = null;
    protected int    messageBodyOffset = 0;
    protected int    messageBodyLength = 0;

    // true to automatically generate timestamps
    private   boolean genTimestamp      = true;

    // true to automatically generate sequence numbers
    private   boolean genSequenceNumber = true;

    public ReadWritePacket() {
	this.reset();
    }

    /**
     * Read packet from an InputStream. This method reads one packet
     * from the InputStream and sets the state of this object to
     * reflect the packet read.
     *
     * @param is        the InputStream to read the packet from
     */
    public synchronized void readPacket(InputStream is)
	throws IOException, EOFException {

	// Read packet into internal buffers
	super.readPacket(is);
    }


    /**
     * Write the packet to the specified OutputStream
     */
    public synchronized void writePacket(OutputStream os)
	throws IOException {

	if (genSequenceNumber) {
            updateSequenceNumber();
	}

	if (genTimestamp) {
            updateTimestamp();
	}

	updateBuffers();
	super.writePacket(os);
    }

    /**
     * Write the packet to the specified OutputStream, using a specified version
     */
    public synchronized void writePacket(OutputStream os, int new_version)
	throws IOException {

	setVersion(new_version);
	super.writePacket(os);
    }

    /**
     * Update the timestamp on the packet. If you do this
     * you should call generateTimestamp(false) before writing the
     * packet, otherwise the timestamp will be overwritten when
     * writePacket() is called.
     */
    public synchronized void updateTimestamp() {
        setTimestamp(System.currentTimeMillis());
    }

    /**
     * Update the sequence number on the packet. If you do this
     * you should call generateSequenceNumber(false) before writing the
     * packet, otherwise the sequence number will be overwritten when
     * writePacket() is called.
     */
    public synchronized void updateSequenceNumber() {
	synchronized(getClass()) {
            sequenceNumber++;
	    setSequence(sequenceNumber);
	}
    }

    /**
     * Write the packet to an OutputStream with some modified fields.
     *
     * @param os                The OutputStream to write the packet to
     * @param new_consumerID    New value for the consumerID
     * @param new_pauseFlow     New value for the pause flow flag (F bit)
     * @param new_redelivered   New value for the redelivered flag (R bit)
     * @param new_lastPkt       New value for the last message flag (L bit)
     *
     */
    public void writePacket(OutputStream os,
	long new_consumerID, 
	boolean new_pauseFlow,
	boolean new_redelivered,
	boolean new_lastPkt)
	throws IOException {

	writePacket(os, this.version, new_consumerID, new_pauseFlow,
		 new_redelivered, new_lastPkt);
    }

    /**
     * Write the packet to an OutputStream with some modified fields.
     *
     * @param os                The OutputStream to write the packet to
     * @param new_version       New packet version. VERSION1, VERSION2, VERSION3
     * @param new_consumerID    New value for the consumerID
     * @param new_pauseFlow     New value for the pause flow flag (F bit)
     * @param new_redelivered   New value for the redelivered flag (R bit)
     * @param new_lastPkt       New value for the last message flag (L bit)
     *
     */
    public void writePacket(OutputStream os,
	int new_version,
	long new_consumerID, 
	boolean new_pauseFlow,
	boolean new_redelivered,
	boolean new_lastPkt)
	throws IOException {

	setVersion(new_version);
	setRedelivered(new_redelivered);
	setConsumerID(new_consumerID);
	setIsLast(new_lastPkt);
	setFlowPaused(new_pauseFlow);

	writePacket(os);
    }

    /**
     * Update internal buffers to reflect currect state of object.
     * These are the buffers used when the packet is written.
     */
    public void updateBuffers()
	throws IOException {

	// You must update the ROP buffer first to correctly determine the
	// packet size -- so a dirty ropBuffer means a dirty headerBuffer
	// (since the size field must change)

	if (ropBuffer == null || ropBufferDirty) {
	    updateRopBuffer();
	    updateHeaderBuffer();
	} else if (headerBufferDirty) {
	    updateHeaderBuffer();
        }
    }

    /**
     * Update the header buffer to reflect the current state of the object.
     * The headerBuffer contains the fixed header portion of the packet.
     */
    private void updateHeaderBuffer()
	throws IOException {
	JMQByteArrayOutputStream bos =
			new JMQByteArrayOutputStream(headerBuffer);

	DataOutputStream dos = new DataOutputStream(bos);

	dos.writeInt(MAGIC);
	dos.writeShort(version);
	dos.writeShort(packetType);
	dos.writeInt(packetSize);

        // In VERSION1 transactionID was a 32bit value in the header
        if (version == VERSION1) {
	    dos.writeInt((int)transactionID);
        }

	dos.writeLong(expiration);

	// Writes timestamp, source IP addr, source port, and sequence number
	sysMessageID.writeID(dos);

	dos.writeInt(propertyOffset);
	dos.writeInt(propertySize);
	dos.writeByte(priority);
	dos.writeByte(encryption);

	dos.writeShort(bitFlags);

        // In VERSION1 consumerID is 32bits. In VERSION2 it is 64bits
        if (version == VERSION1) {
	    dos.writeInt((int)consumerID);
        } else {
	    dos.writeLong(consumerID);
        }

	dos.flush();

	bos.flush();

	// This should be a no-op since headerBuffer is the correct size
	// and JMQByteArrayOutputStream should not grow it (resulting in
	// a buffer reallocation). But just to be safe...
	headerBuffer = bos.getBuf();

        headerBufferDirty = false;
    }

    /**
     * Update the Rest Of Packet buffer (ropBuffer) to reflect current
     * state of the object. The ROP Buffer contains all of the packet
     * except for the fixed header.
     *
     * Also updates the propertyOffset, propertySize and packetSize fields
     * 
     */
    private void updateRopBuffer() 
	throws IOException{

	byte[] pad = new byte[4];	// Four nulls

	/*
	 * XXX PERF 01/27/00 dipol
	 * Make sure the properties and messageBody fields are current,
	 * otherwise we risk loosing them if we just read a packet and
	 * the user never accessed them. This could be optimized to
	 * reduce copying the data an extra time, but that would make
	 * reusing the ROP buffer tricky!
	 */
	try {
	    getProperties();
	} catch (ClassNotFoundException e) {
	}

	getMessageBody();

	if (ropBuffer == null) {
	    /*
	     * Allocate new rop buffer. We could be smarter when
	     * allocating the initial size, but ByteArrayOutputStream
	     * will grow it if we get it wrong.
	     */
	    int size = 1024;
	    if (messageBody != null) {
		size = size + messageBodyLength;
            }
	    ropBuffer = new byte[size];
	}

	JMQByteArrayOutputStream bos =
			new JMQByteArrayOutputStream(ropBuffer);

	DataOutputStream dos = new DataOutputStream(bos);

        // In VERSION2 the transaction ID is a 64 bit value in the
        // variable field portion of the packet.
	if (version >= VERSION2) {
            if (transactionID != 0) {
                writeLong(dos, PacketString.TRANSACTIONID, transactionID);
            }
	}

        if (producerID != 0) {
            writeLong(dos, PacketString.PRODUCERID, producerID);
        }

	// Write list of string items. writeString checks for null value
	writeString(dos, PacketString.DESTINATION, destination);
	writeString(dos, PacketString.DESTINATION_CLASS, destinationClass);
	writeString(dos, PacketString.MESSAGEID, messageID);
	writeString(dos, PacketString.CORRELATIONID, correlationID);
	writeString(dos, PacketString.REPLYTO, replyTo);
	writeString(dos, PacketString.REPLYTO_CLASS, replyToClass);
	writeString(dos, PacketString.TYPE, messageType);

	//Teminate list
	dos.writeShort(PacketString.NULL);
	dos.flush();

	// Pad to nearest 32 bit boundary
        int padding = 4 - (bos.getCount() % 4);
        bos.write(pad, 0, padding);
	bos.flush();

	propertyOffset = bos.getCount() + HEADER_SIZE;

	// Write property object
	if (properties != null) {
            if (version == VERSION3) {
  	        PacketProperties.write(properties, bos);
            } else {
  	        ObjectOutputStream oos = new ObjectOutputStream(bos);
	        oos.writeObject(properties);
	        oos.flush();
	        oos.close();
            }
	}

	propertySize = bos.getCount() - (propertyOffset - HEADER_SIZE);

	/*
	 * Copy message body to ropBuffer
	 * XXX PERF 01/27/00 dopol
	 * We could optimize this by tracking the message body buffer
	 * seperately and writing it directly to the socket. That would
	 * eliminate this copy. But it would make things inconsistent
	 * with ReadOnlyPacket, and therefore complicate the code.
	 */
	if (messageBody != null) {
	    bos.write(messageBody, messageBodyOffset, messageBodyLength);
	}
	ropLength = bos.getCount();
	packetSize = ropLength + HEADER_SIZE;

	bos.flush();

	// We update the ropBuffer reference in case JMQByteArrayOutputStream()
	// had to grow the buffer which would have resulted in a new buffer
	// being allocated.
	ropBuffer = bos.getBuf();

        ropBufferDirty = false;

	bos.close();

	return;
    }

    /**
     * Write a header string item to the specified output stream
     */
    private void writeString(DataOutputStream dos, int type, String value)
	throws IOException {
	if (value != null) {
	    dos.writeShort(type);
	    dos.writeUTF(value);
	}
    }

    /**
     * Write a long field to the variable portion of the packet
     */
    private void writeLong(DataOutputStream dos, int type, long value)
	throws IOException {
	dos.writeShort(type);
        dos.writeShort(8);
	dos.writeLong(value);
    }

    /**
     * Disable (and enable) sequence number generation. The JMQ packet 
     * specification defines a "sequence number" field that is defined
     * to be an increasing sequence number. By default ReadWritePacket
     * will automatically increment the sequence number and set it
     * on the packet every time writePacket() is called.
     * The sequence number is a class variable so all packets in a VM
     * share the same sequence.
     *
     * @param    generate    true to have the packet automatically generate
     *                       sequence numbers for you, false to not. Default
     *                       is "true".
     */
    public void generateSequenceNumber(boolean generate) {
        genSequenceNumber = generate;
    }

    /**
     * Disable (and enable) timestamp generation. The JMQ packet specification
     * specifies a "timestamp" field that is defined to be the time the
     * packet was sent. By default ReadWritePacket will automatically 
     * generate a timestamp and set it on the packet every time
     * writePacket() is called.
     *
     * @param    generate    true to have the packet automatically generate
     *                       a timestamp on each packet sent, false to not.
     *                       Default is "true".
     */
    public void generateTimestamp(boolean generate) {
	genTimestamp = generate;
    }

    /** 
     * Set the packet type.
     *
     * @param    new_packetType    The type of packet
     */
    public synchronized void setPacketType(int pType) {
	packetType = pType;
	headerBufferDirty = true;
    }

    public synchronized void setTimestamp(long t) {
	sysMessageID.timestamp = t;
	headerBufferDirty = true;
    }

    public synchronized void setExpiration(long e) {
	expiration = e;
	headerBufferDirty = true;
    }

    public synchronized void setPort(int p) {
	sysMessageID.port = p;
	headerBufferDirty = true;
    }

    public synchronized void setIP(byte[] ip) {
	sysMessageID.setIPAddress(ip);
	headerBufferDirty = true;
    }

    public synchronized void setIP(byte[] ip, byte[] mac) {
	sysMessageID.setIPAddress(ip, mac);
	headerBufferDirty = true;
    }

    public synchronized void setSequence(int n) {
	sysMessageID.sequence = n;
	headerBufferDirty = true;
    }

    // Version should be VERSION1, VERSION2 or VERSION3. Default is VERSION3
    public synchronized void setVersion(int n) {
	if (version != n) {
	    version = n;
	    headerBufferDirty = true;
	}
    }

    public synchronized void setTransactionID(long n) {
	transactionID = n;
	headerBufferDirty = true;
    }

    public synchronized void setEncryption(int e) {
	encryption = e;
	headerBufferDirty = true;
    }

    public synchronized void setPriority(int p) {
	priority = p;
	headerBufferDirty = true;
    }

    public void setConsumerID(long n) {
	consumerID = n;
	headerBufferDirty = true;
    }

    public void setInterestID(int n) {
        setConsumerID((long)n);
    }

    public void setPersistent(boolean b) {
	setFlag(PacketFlag.P_FLAG, b);
    }

    public void setRedelivered(boolean b) {
	setFlag(PacketFlag.R_FLAG, b);
    }

    public void setIsQueue(boolean b) {
	setFlag(PacketFlag.Q_FLAG, b);
    }

    public void setSelectorsProcessed(boolean b) {
	setFlag(PacketFlag.S_FLAG, b);
    }

    public void setSendAcknowledge(boolean b) {
	setFlag(PacketFlag.A_FLAG, b);
    }

    public void setIsLast(boolean b) {
	setFlag(PacketFlag.L_FLAG, b);
    }

    public void setFlowPaused(boolean b) {
	setFlag(PacketFlag.F_FLAG, b);
    }

    public void setIsTransacted(boolean b) {
	setFlag(PacketFlag.T_FLAG, b);
    }

    public void setConsumerFlow(boolean b) {
	setFlag(PacketFlag.C_FLAG, b);
    }

    public void setIndempotent(boolean b) {
	setFlag(PacketFlag.I_FLAG, b);
    }

    public synchronized void setFlag(int flag, boolean on) {
	if (on) {
	    bitFlags = bitFlags | flag;
	} else {
	    bitFlags = bitFlags & ~flag;
	}
	headerBufferDirty = true;
    }

    public synchronized void setProducerID(long l) {
	producerID = l;
	ropBufferDirty = true;
    }

    public synchronized void setDestination(String d) {
	destination = d;
	ropBufferDirty = true;
    }

    public synchronized void setDestinationClass(String d) {
	destinationClass = d;
	ropBufferDirty = true;
    }

    public synchronized void setMessageID(String id) {
	messageID = id;
	ropBufferDirty = true;
    }

    public synchronized void setCorrelationID(String id) {
	correlationID = id;
	ropBufferDirty = true;
    }

    public synchronized void setReplyTo(String r) {
	replyTo = r;
	ropBufferDirty = true;
    }

    public synchronized void setReplyToClass(String r) {
	replyToClass = r;
	ropBufferDirty = true;
    }

    public synchronized void setMessageType(String t) {
	messageType = t;
	ropBufferDirty = true;
    }

    /**
     * Set the message properties.
     * WARNING! The Hashtable is NOT copied.
     *
     * @param    body    The message body.
     */
    public synchronized void setProperties(Hashtable props) {
	properties = props;
	ropBufferDirty = true;
    }

    /**
     * Set the message body.
     * WARNING! The byte array is NOT copied.
     *
     * @param    body    The message body.
     */
    public synchronized void setMessageBody(byte[] body) {
	messageBody = body;
	messageBodyOffset = 0;
	if (body != null) {
	    messageBodyLength = body.length;
	} else {
	    messageBodyLength = 0;
	}
	ropBufferDirty = true;
    }

    /**
     * Set the message body. Specify offset and length of where to take
     * data from buffer.
     * WARNING! The byte array is NOT copied.
     *
     * @param    body    The message body.
     */
    public synchronized void setMessageBody(byte[] body, int off, int len) {
	messageBody = body;
	messageBodyOffset = off;
	messageBodyLength = len;
	ropBufferDirty = true;
    }

    /**
     * Get the length of the message body
     *
     * @return Legnth of the message body in bytes
     */
    public synchronized int getMessageBodyLength() {
        return messageBodyLength;
    }

    /**
     * Get the length of the message body. Same as getMessageBodyLength().
     * provided to protect against calling ReadOnlyPacket's getMessageBodySize()
     *
     * @return Legnth of the message body in bytes
     */
    public synchronized int getMessageBodySize() {
        return getMessageBodyLength();
    }


    /**
     * Get the offset into the message body buffer where the message
     * body data starts
     *
     * @return Byte offset into buffer returned by getMessageBody where
     *         message body data starts.
     *
     */
    public synchronized int getMessageBodyOffset() {
        return messageBodyOffset;
    }

    /**
     * Return the message body.
     * WARNING! This returns a reference to the message body, not a copy.
     * Also, if the body was set using setMessageBody(buf, off, len) then
     * you will get back the buffer that was passed to setMessageBody().
     * Therefore you may need to use getMessageBodyOffset() and
     * getMessageBodyLength() to determine the true location of the 
     * message body in the buffer.
     *
     * @return     A byte array containing the message body. 
     *		   null if no message body.
     */ 
    public synchronized byte[] getMessageBody() {

	if (messageBody == null) {
	    // No message body buffer. We must have just read a message.
            InputStream is = super.getMessageBodyStream();

	    if (is == null) {
		// No message body
	        return null;
	    }

	    try {
		// Read body into buffer
	        int length = is.available();
	        messageBody = new byte[length];
	        is.read(messageBody, 0, length);
                messageBodyOffset = 0;
                messageBodyLength = length;
	    } catch (IOException e) {
		// This should never happen since the InputStream is backed
		// by a buffer...
		System.err.println("Could not get message body: " + e);
		messageBody = null;
	    }
        }

	return messageBody;
    }

    /**
     * Make a shallow copy of this packet
     */
     public Object cloneShallow() {
	 return(super.cloneShallow());
     }

    /**
     * Make a deep copy of this packet
     */
    public Object clone() {
        // Make a deep copy of super class (ReadOnlyPacket)
        ReadWritePacket newPkt = (ReadWritePacket)super.clone();

	// Make copy of messageBody buffer
	if (this.messageBody != null) {
            byte[] b = new byte[this.messageBodyLength];
            System.arraycopy(this.messageBody, this.messageBodyOffset,
                                            b, 0,
                             this.messageBodyLength);
                              
	    newPkt.messageBody = b;
            newPkt.messageBodyOffset = 0;
            newPkt.messageBodyLength = b.length;
        }

        return (newPkt);
    }

    /**
     * Reset state of packet to initial values
     */
    public synchronized void reset() {
	super.reset();
        headerBufferDirty = false;
        ropBufferDirty    = false;
        messageBody = null;
        messageBodyOffset = 0;
        messageBodyLength = 0;
    }

    /* 
     * Dump the contents of the packet in human readable form to
     * the specified OutputStream.
     *
     * @param    os    OutputStream to write packet contents to
     */
    public void dump(PrintStream os) {
	super.dump(os);

	if (messageBody == null) {
            os.println("    messageBodyBuffer: null");
        } else {
	    os.println("    messageBodyBuffer: " + messageBody + " " +
			messageBodyLength + " bytes");
	}
	os.println("    headerBufferDirty: " + headerBufferDirty +
	"    ropBufferDirty: " + ropBufferDirty);

	os.println("         genTimestamp: " + genTimestamp + 
	"  genSequenceNumber: " + genSequenceNumber);
    }
}
