/*
 * Copyright (c) 2007-2010 by The Broad Institute, Inc. and the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * This software is licensed under the terms of the GNU Lesser General Public License (LGPL), Version 2.1 which
 * is available at http://www.opensource.org/licenses/lgpl-2.1.php.
 *
 * THE SOFTWARE IS PROVIDED "AS IS." THE BROAD AND MIT MAKE NO REPRESENTATIONS OR WARRANTIES OF
 * ANY KIND CONCERNING THE SOFTWARE, EXPRESS OR IMPLIED, INCLUDING, WITHOUT LIMITATION, WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, NONINFRINGEMENT, OR THE ABSENCE OF LATENT
 * OR OTHER DEFECTS, WHETHER OR NOT DISCOVERABLE.  IN NO EVENT SHALL THE BROAD OR MIT, OR THEIR
 * RESPECTIVE TRUSTEES, DIRECTORS, OFFICERS, EMPLOYEES, AND AFFILIATES BE LIABLE FOR ANY DAMAGES OF
 * ANY KIND, INCLUDING, WITHOUT LIMITATION, INCIDENTAL OR CONSEQUENTIAL DAMAGES, ECONOMIC
 * DAMAGES OR INJURY TO PROPERTY AND LOST PROFITS, REGARDLESS OF WHETHER THE BROAD OR MIT SHALL
 * BE ADVISED, SHALL HAVE OTHER REASON TO KNOW, OR IN FACT SHALL KNOW OF THE POSSIBILITY OF THE
 * FOREGOING.
 */
package org.broad.igv.sam;

import net.sf.samtools.util.CloseableIterator;
import org.apache.log4j.Logger;
import org.broad.igv.Globals;
import org.broad.igv.PreferenceManager;
import org.broad.igv.feature.Genome;
import org.broad.igv.sam.AlignmentTrack.SortOption;
import org.broad.igv.sam.reader.AlignmentQueryReader;
import org.broad.igv.sam.reader.SamListReader;
import org.broad.igv.sam.reader.SamQueryReaderFactory;
import org.broad.igv.session.ViewContext;
import org.broad.igv.track.MultiFileWrapper;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.util.LongRunningTask;
import org.broad.igv.util.NamedRunnable;
import org.broad.igv.util.ResourceLocator;

import javax.swing.*;
import java.io.IOException;
import java.util.*;

public class AlignmentDataManager {

    private static Logger log = Logger.getLogger(AlignmentDataManager.class);
    private AlignmentInterval loadedInterval = null;
    HashMap<String, String> chrMappings = new HashMap();
    boolean isLoading = false;
    AlignmentQueryReader reader;

    CoverageTrack coverageTrack;

    private static final int DEFAULT_DEPTH = 10;


    public static AlignmentDataManager getDataManager(ResourceLocator locator) {
        return new AlignmentDataManager(locator);
    }

    public AlignmentQueryReader getReader() {
        return reader;
    }

    private AlignmentDataManager(ResourceLocator locator) {

        if (locator.getPath().endsWith(".sam.list")) {
            MultiFileWrapper mfw = MultiFileWrapper.parse(locator);
            reader = new SamListReader(mfw.getLocators());
        } else {
            reader = SamQueryReaderFactory.getReader(locator);
        }
        initChrMap();
    }

    private void initChrMap() {
        Genome genome = ViewContext.getInstance().getGenome();
        if (genome != null) {
            Set<String> seqNames = reader.getSequenceNames();
            if (seqNames != null) {
                for (String chr : seqNames) {
                    String alias = genome.getChromosomeAlias(chr);
                    chrMappings.put(alias, chr);
                }

            }
        }
    }


    public boolean hasIndex() {
        return reader.hasIndex();
    }

    public int getMaxDepth() {
        return loadedInterval == null ? DEFAULT_DEPTH :
                (loadedInterval.getMaxCount() == 0 ? DEFAULT_DEPTH : loadedInterval.getMaxCount());
    }

    public void setCoverageTrack(CoverageTrack coverageTrack) {
        this.coverageTrack = coverageTrack;
    }

    public AlignmentInterval getLoadedInterval() {
        return loadedInterval;
    }


    /**
     * Sort alignment rows such that alignments that intersect from the
     * center appear left to right by start position
     */
    public void sortRows(SortOption option) {
        if (loadedInterval != null) {
            loadedInterval.sortRows(option);
        }

    }

    public void packAlignments() {
        if(loadedInterval == null) {
            return;
        }
        RowIterator iter = new RowIterator();
        final PreferenceManager.SAMPreferences prefs = PreferenceManager.getInstance().getSAMPreferences();
        final int maxLevels = prefs.getMaxLevels();
        final int qualityThreshold = prefs.getQualityThreshold();

        List<AlignmentInterval.Row> alignmentRows = AlignmentLoader.loadAndPackAlignments(
                iter,
                prefs.isShowDuplicates(),
                qualityThreshold,
                maxLevels,
                getLoadedInterval().getEnd());

        loadedInterval.setAlignmentRows(alignmentRows);
    }


    public synchronized List<AlignmentInterval.Row> getAlignmentRows(String genomeId, final String chr, final int start, final int end) {
        log.debug("Enter getAlignmentRows: " + chr + ":" + start + "-" + end);

        // If we've moved out of the loaded interval start a new load.
        if (loadedInterval == null || !loadedInterval.contains(genomeId, chr, start, end)) {
            loadAlignments(genomeId, chr, start, end);
        }

        // If there is any overlap in the loaded interval and the requested interval return it.
        if (loadedInterval != null && loadedInterval.overlaps(genomeId, chr, start, end)) {
            return loadedInterval.getAlignmentRows();
        } else {
            return null;
        }
    }

    public void reload() {
        if (loadedInterval != null) {
            String genomeId = loadedInterval.genomeId;
            String chr = loadedInterval.chr;
            int start = loadedInterval.getStart();
            int end = loadedInterval.getEnd();
            loadedInterval = null;
            loadAlignments(genomeId, chr, start, end);
        }
    }

    public void loadAlignments(final String genomeId, final String chr, final int start, final int end) {

        if (isLoading || chr.equals(Globals.CHR_ALL)) {
            return;
        }

        // If the requested interval is outside the currently loaded range load
        if (loadedInterval == null || !loadedInterval.contains(genomeId, chr, start, end)) {
            log.debug("Load alignments.  isLoading=" + isLoading);
            isLoading = true;
            NamedRunnable runnable = new NamedRunnable() {

                public String getName() {
                    return "preloadData";
                }

                public void run() {

                    log.debug("Preloading " + chr + ":" + start + "-" + end);
                    final PreferenceManager.SAMPreferences prefs = PreferenceManager.getInstance().getSAMPreferences();
                    final int maxLevels = prefs.getMaxLevels();
                    final int qualityThreshold = prefs.getQualityThreshold();


                    // Expand start and end to facilitate panning, but by no more than
                    // 1 screen or 8kb, whichever is less
                    // DON'T expand mitochondria

                    int expandLength = isMitochondria(chr) ? 0 : Math.min(8000, end - start) / 2;
                    int intervalStart = Math.max(0, start - expandLength);
                    int intervalEnd = end + expandLength;
                    CloseableIterator<Alignment> iter = null;
                    try {

                        String sequence = chrMappings.containsKey(chr) ? chrMappings.get(chr) : chr;

                        iter = reader.query(sequence, intervalStart, intervalEnd, false);

                        List<AlignmentInterval.Row> alignmentRows = AlignmentLoader.loadAndPackAlignments(iter,
                                prefs.isShowDuplicates(),
                                qualityThreshold,
                                maxLevels,
                                intervalEnd);

                        loadedInterval = new AlignmentInterval(genomeId, chr, intervalStart, intervalEnd, alignmentRows);

                        // Caclulate coverage.  TODO -- refactor/move this
                        for (AlignmentInterval.Row row : alignmentRows) {
                            for (Alignment a : row.alignments) {
                                getLoadedInterval().incCounts(a);
                            }
                        }


                        if (coverageTrack != null) {
                            coverageTrack.rescale();
                        }

                        IGVMainFrame.getInstance().doResizeTrackPanels();
                        IGVMainFrame.getInstance().repaintDataAndHeaderPanels();


                    } catch (Exception exception) {
                        log.error("Error loading alignments", exception);
                        JOptionPane.showMessageDialog(IGVMainFrame.getInstance(), "Error reading file: " + exception.getMessage());
                    } finally {
                        isLoading = false;
                        if (iter != null) {
                            iter.close();
                        }
                    }
                }
            };

            LongRunningTask.submit(runnable);

        }

    }


    private boolean isMitochondria(String chr) {
        return chr.equals("M") || chr.equals("chrM") ||
                chr.equals("MT") || chr.equals("chrMT");
    }

    /**
     * @return the alignmentRows
     */
    public List<AlignmentInterval.Row> getAlignmentRows() {
        return loadedInterval == null ? null : loadedInterval.getAlignmentRows();
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        if (reader != null) {
            try {
                reader.close();
            } catch (IOException ex) {
                log.error("Error closing AlignmentQueryReader. ", ex);
            }
        }

    }


    /**
     * An alignment iterator that iterates over packed rows.  Used for
     * "repacking".   Using the iterator avoids the need to copy alignments
     * from the rows
     */
    class RowIterator implements CloseableIterator<Alignment> {

        PriorityQueue<AlignmentInterval.Row> rows;
        Alignment nextAlignment;

        RowIterator() {
            rows = new PriorityQueue(5, new Comparator<AlignmentInterval.Row>() {

                public int compare(AlignmentInterval.Row o1, AlignmentInterval.Row o2) {
                    return o1.getNextStartPos() - o2.getNextStartPos();
                }
            });

            for (AlignmentInterval.Row r : loadedInterval.getAlignmentRows()) {
                r.resetIdx();
                rows.add(r);
            }

            advance();
        }

        public void close() {
            // Ignored
        }

        public boolean hasNext() {
            return nextAlignment != null;
        }

        public Alignment next() {
            Alignment tmp = nextAlignment;
            if (tmp != null) {
                advance();
            }
            return tmp;
        }

        private void advance() {

            nextAlignment = null;
            AlignmentInterval.Row nextRow = null;
            while (nextAlignment == null && !rows.isEmpty()) {
                while ((nextRow = rows.poll()) != null) {
                    if (nextRow.hasNext()) {
                        nextAlignment = nextRow.nextAlignment();
                        break;
                    }
                }
            }
            if (nextRow != null && nextAlignment != null) {
                rows.add(nextRow);
            }
        }

        public void remove() {
            // ignore
        }
    }

}

