/////////////////////////////////////////////////////////////////////////////
// Name:        DVDPropDlg.cpp
// Purpose:     DVD properties dialog
// Author:      Alex Thuering
// Created:     7.03.2005
// RCS-ID:      $Id: DVDPropDlg.cpp,v 1.27 2010/01/10 11:49:42 ntalex Exp $
// Copyright:   (c) Alex Thuering
// Licence:     GPL
/////////////////////////////////////////////////////////////////////////////

#include "DVDPropDlg.h"
#include "Config.h"

DVDPropDlg::DVDPropDlg(wxWindow *parent, DVD* dvd) :
	wxPropDlg(parent), m_dvd(dvd) {
	Create(true);
	SetSize(400, -1);
}

void DVDPropDlg::CreatePropPanel(wxSizer* sizer) {
	wxFlexGridSizer* grid = sizer ? new wxFlexGridSizer(2, 1, 16) : NULL;
	if (grid) {
		grid->AddGrowableCol(1);
		sizer->Add(grid, 0, wxEXPAND|wxALL, 6);
	}
	AddTextProp(grid, _("Disc label:"), grid ? m_dvd->GetLabel() : s_config.GetDefDiscLabel());

	wxArrayString labels = DVD::GetCapacityLabels();
	AddChoiceProp(grid, _("Disc capacity:"), labels[m_dvd->GetCapacity()], labels);

	// Video bitrate
	labels = DVD::GetVideoBitrateLabels();
	int b = m_dvd->GetVideoBitrate();
	int q = m_dvd->IsVideoBitrateAuto() ? 0 : (b%1000 == 0 && b>=2000 && b<=8000 ? 9-b/1000 : 8);
	wxSizer* vbSizer = AddChoiceCustomProp(grid, _("Video quality:"), labels[q], labels, 8, 100, false);
	if (vbSizer)
		vbSizer->AddSpacer(4);
	AddSpinProp(vbSizer, wxT(""), b == -1 ? 4500 : b, 500, 9000, 80, _("KBit/s"), false);
	if (vbSizer)
		SetLastControlCustom(GetLastControlIndex()-1, q == 8);

	// Video and audio format
	wxFlexGridSizer* fSizer = sizer ? new wxFlexGridSizer(3, 1, 16) : NULL;
	if (sizer) {
		fSizer->AddGrowableCol(0);
		fSizer->AddGrowableCol(1);
		fSizer->AddGrowableCol(2);
		sizer->Add(fSizer, 0, wxEXPAND|wxALL, 6);
	}
	// Video format
	wxSizer* grp = BeginGroup(fSizer, _("Video Format"));
	if (grp)
		grp->Add(4, 4);
	wxArrayString formats = DVD::GetVideoFormatLabels();
	AddRadioGroupProp(grp, formats, m_dvd->GetVideoFormat() - 2);
	if (grp)
		grp->Add(4, 4);
	// Aspect Ratio
	formats = DVD::GetAspectRatioLabels();
	grp = BeginGroup(fSizer, _("Aspect Ratio"));
	if (grp)
		grp->Add(4, 4);
	AddRadioGroupProp(grp, formats, m_dvd->GetAspectRatio() - 1);
	if (grp) {
		if (grp->GetMinSize().GetWidth() < 100)
			grp->SetMinSize(wxSize(100, -1));
		grp->Add(4, 4);
	}
	// Audio format
	grp = BeginGroup(fSizer, _("Audio Format"));
	if (grp)
		grp->Add(4, 4);
	formats = DVD::GetAudioFormatLabels();
	AddRadioGroupProp(grp, formats, m_dvd->GetAudioFormat() - 2);
	if (grp)
		grp->Add(4, 4);
	
	grid = sizer ? new wxFlexGridSizer(2, 1, 16) : NULL;
	if (grid) {
		grid->AddGrowableCol(1);
		sizer->Add(grid, 0, wxEXPAND|wxALL, 6);
	}
	AddComboProp(grid, _("First Play Commands:"), sizer ? m_dvd->GetVmgm().GetFpc() : wxT(""), GetFPCommands());

	AddCheckProp(sizer, _("Create jumppads"), sizer ? m_dvd->IsJumppadsEnabled() : true);
	AddCheckProp(sizer, _("Create empty menu/vmMenu if it doesn't exist"),
			sizer ? m_dvd->IsEmptyMenuEnabled() : true);
}

/**
 * Returns list of possible first play commands
 */
wxArrayString DVDPropDlg::GetFPCommands() {
	wxArrayString commands;
	for (int pgci=0; pgci<(int)m_dvd->GetVmgm().Count(); pgci++)
		commands.Add(wxString::Format(wxT("jump vmgm menu %d;"),pgci+1));
	int titleIdx = 0;
	for (int tsi=0; tsi<(int)m_dvd->GetTitlesets().Count(); tsi++) {
		Titleset* ts = m_dvd->GetTitlesets()[tsi];
		if (ts->GetMenus().Count()>0)
			commands.Add(wxString::Format(wxT("jump titleset %d menu;"), tsi+1));
		StringSet entrySet = ts->GetMenus().GetEntries();
		for (StringSet::const_iterator entry = entrySet.begin(); entry != entrySet.end(); entry++)
			commands.Add(wxString::Format(wxT("jump titleset %d menu entry %s;"), tsi+1, entry->c_str()));
		if (m_dvd->IsJumppadsEnabled()) {
			for (int pgci=1; pgci<(int)ts->GetMenus().Count(); pgci++)
				commands.Add(wxString::Format(wxT("jump titleset %d menu %d;"), tsi+1, pgci+1));
			for (int pgci=0; pgci<(int)ts->GetTitles().Count(); pgci++)
				commands.Add(wxString::Format(wxT("jump titleset %d title %d;"), tsi+1, pgci+1));
		} else {
			for (int pgci=0; pgci<(int)ts->GetTitles().Count(); pgci++) {
				commands.Add(wxString::Format(wxT("jump title %d;"), titleIdx+1));
				titleIdx++;
			}
		}
	}
	return commands;
}

bool DVDPropDlg::SetValues() {
	int n = 0;
	if (GetString(n).length() == 0) {
		wxMessageBox(_("Please enter the volume name."), this->GetTitle(), wxOK|wxICON_ERROR, this);
		return false;
	}
	if (GetString(n).length() > 32) {
		wxMessageBox(_("Volume name can only be a maximum of 32 characters."), this->GetTitle(),
				wxOK|wxICON_ERROR, this);
		return false;
	}
	m_dvd->SetLabel(GetString(n++));
	m_dvd->SetCapacity((DiscCapacity) GetInt(n++));
	int q = GetInt(n++);
	m_dvd->SetVideoBitrateAuto(q == 0);
	int b = GetInt(n++);
	if (q > 0)
		m_dvd->SetVideoBitrate(q < 8 ? (9-q)*1000 : b);
	m_dvd->SetVideoFormat((VideoFormat) (GetInt(n++) + 2));
	m_dvd->SetAspectRatio((AspectRatio) (GetInt(n++) + 1));
	m_dvd->SetAudioFormat((AudioFormat) (GetInt(n++) + 2));
	for (int tsi=-1; tsi<(int)m_dvd->GetTitlesets().GetCount(); tsi++) {
		PgcArray& pgcs = m_dvd->GetPgcArray(tsi, true);
		for (int pgci=0; pgci<(int)pgcs.GetCount(); pgci++) {
			Menu* menu = m_dvd->GetMenu(tsi, pgci);
			if (menu != NULL)
				menu->SetVideoFormat(m_dvd->GetVideoFormat());
		}
	}
	m_dvd->GetVmgm().SetFpc(GetString(n++));
	bool enableJumppad = GetBool(n++);
	if (m_dvd->IsJumppadsEnabled() != enableJumppad && !enableJumppad) { 
		// disabling jumppad => check all actions
		for (int tsi=-1; tsi<(int)m_dvd->GetTitlesets().GetCount(); tsi++) {
			PgcArray& pgcs = m_dvd->GetPgcArray(tsi, true);
			for (int pgci=0; pgci<(int)pgcs.GetCount(); pgci++) {
				Menu* menu = m_dvd->GetMenu(tsi, pgci);
				for (unsigned int objIdx = 0; objIdx < menu->GetObjectsCount(); objIdx++) {
					MenuObject* obj = menu->GetObject(objIdx);
					if (obj->IsButton()) {
						DVDAction& action = obj->GetAction();
						if (tsi == -1 && !action.IsCustom()) { // jump from vmMenu
							if (action.GetTsi() >= 0 && action.IsMenu() && action.GetPgci() > 0) {
								action.SetPgci(0);
								wxLogWarning(_("vmMenu %d %s: direct jump from vmMenu is possible only to the first menu in every titleset"),
										pgci + 1, obj->GetId(true).c_str());
							} else if (action.GetChapter() > 0) {
								action.SetChapter(0);
								wxLogWarning(_("vmMenu %d %s: direct jump from vmMenu to chapters is not possible, please use regular menu instead"),
										pgci + 1, obj->GetId(true).c_str());
							}
						} else if (tsi >= 0 && !action.IsCustom()) { // jump from titleset menu
							if (action.GetTsi() >= 0 && action.GetTsi() != tsi) {
								action.SetTsi(-2);
								action.SetPgci(-1);
								action.SetMenu(true);
								wxLogWarning(_("titleset %d menu %d %s: direct jump from regular menu to the menu or title from other titleset is not possible"),
										tsi + 1, pgci + 1, obj->GetId(true).c_str());
							}
						}
					}
				}
			}
		}
	}
	m_dvd->EnableJumppads(enableJumppad);
	m_dvd->EnableEmptyMenu(GetBool(n++));
	return true;
}
