# XMMS2tray - GNU/Linux systray integration for xmms2
# Copyright (C) 2006 Thomas Jollans
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import sys
import os
import time
import xmmsclient
import xmms2_glib
import gtk
import gtk.gdk
from StringIO import StringIO
import random
import menu_conf

HAVE_NOTIFY = True
try:
    import pynotify
except ImportError:
    HAVE_NOTIFY = False
    sys.stderr.write("pynotify not found. Notifications will not work.\n")

HAVE_IMAGING = True
try:
    from PIL import Image
except ImportError:
    HAVE_IMAGING = False
    sys.stderr.write("Python Imaging Library PIL not found. Cover art disabled.\n")


class XMMS2tray:
    def clicked(self, what, ev):
        if ev.button == 3: #right
            #for f,a in updaters: f(a)
            self.menu.popup(None, None, None, 3, ev.get_time())

    def menu_called(self, what, button, time):
        self.menu.popup(None, None, gtk.status_icon_position_menu,
                                                    button, time, what)

    def menuclicked(self, widget, item):
        if 'action' in item:
            #print item['action']
            if item['action'] == 'quit':
                self.quit()
            elif item['action'] == 'playpause':
                if 'check' in item and item['check']:
                    if widget.active: self.xmms.playback_start()
                    else: self.xmms.playback_pause()
                else:
                    self.xmms.playback_status(self.playpause)
            elif item['action'] == 'kill':
                self.xmms.quit()
            elif item['action'] == 'play':
                self.xmms.playback_start()
            elif item['action'] == 'pause':
                self.xmms.playback_pause()
            elif item['action'] == 'stop':
                self.xmms.playback_stop()
            elif item['action'] == 'next':
                self.xmms.playlist_set_next_rel(1)
                self.xmms.playback_tickle()
            elif item['action'] == 'prev':
                self.xmms.playlist_set_next_rel(-1)
                self.xmms.playback_tickle()
        elif 'command' in item:
            # since the command [cw]ould block our nice event look, fork away.
            if os.fork() == 0:
                os.system(item['command'])
                sys.exit()

    def playpause(self, res):
        if res.value() == xmmsclient.PLAYBACK_STATUS_PLAY:
            self.xmms.playback_pause()
        else:
            self.xmms.playback_start()

    def set_tip(self):
        if not len(self.curinfo) > 1:
            tip = self.pbstatus
        else:
            if 'artist' in self.curinfo and 'title' in self.curinfo:
                nowpl = '%(artist)s - %(title)s' % self.curinfo
            elif 'title' in self.curinfo: #ex. curl
                nowpl = self.curinfo['title']
            else:
                nowpl = self.curinfo['url'].split('/')[-1]
            tip = "%s: %s" % (self.pbstatus, nowpl)
        #self.tips.set_tip(self.eb, tip)
        self.icon.set_tooltip(tip)

    def newsong(self,res):
        if res.get_type() == xmmsclient.OBJECT_CMD_ARG_UINT32:
            self.xmms.medialib_get_info(res.value(),self.newsong)
        else:
            if res.get_type() == xmmsclient.OBJECT_CMD_ARG_BIN: # coverart
                info = self.curinfo
                coverimg = StringIO(res.value())
            else:
                info = res.value()
                coverimg = None
                self.curinfo = info

            img = os.getcwd() + '/data/xmms2_64.png'
            if HAVE_IMAGING and 'picture_front' in info:
                if info['picture_front'] == self.lastimg[0]:
                    #same image, just use last.
                    img = self.lastimg[1]
                else:
                    if self.lastimg[1]:
                        #there is an old image to delete.
                        os.remove(self.lastimg[1])
                        self.lastimg = ('', '')
                    if coverimg:
                        orig_pic = Image.open(coverimg)
                        if orig_pic.size[1] <= 64:
                            sm_pic = orig_pic
                        else:
                            sm_pic = orig_pic.resize((64,64), Image.BICUBIC)
                        fname = '/tmp/xmms2tray_cover%08X.png' \
                                    % random.randint(0, 2**32-1)
                        sm_pic.save(fname)
                        self.lastimg = (info['picture_front'], fname)
                        img = fname
                    else:
                        self.xmms.bindata_retrieve(info['picture_front'],
                                                    self.newsong)
                        return


            if not info: return #for "not playing"
            if 'artist' in info and 'title' in info:
                nowpl = '%(artist)s - <i>%(title)s</i>\n' % info
            elif 'title' in info: #ex. curl
                nowpl = info['title']
            else:
                nowpl = info['url'].split('/')[-1]
            self.set_tip()
            if HAVE_NOTIFY:
                n = pynotify.Notification('Now Playing', nowpl, img)
                n.set_property('status-icon', self.icon)
                n.show()

    def songchange(self, res):
        if self.curinfo['id'] == res.value():
            self.xmms.medialib_get_info(res.value(),self.newsong)

    def quit(self, gtk_q=True):
        if self.lastimg[1]:
            os.remove(self.lastimg[1])

        if gtk_q: gtk.main_quit()

    def pbstatus_cb(self, res):
        if res.value() is None: return
        status = res.value()
        if status == xmmsclient.PLAYBACK_STATUS_STOP:
            self.pbstatus = "Stopped"
        elif status == xmmsclient.PLAYBACK_STATUS_PAUSE:
            self.pbstatus = "Paused"
        else:
            self.pbstatus = "Playing"
        self.set_tip()

    def __init__(self):
        icon = gtk.StatusIcon()
        icon.set_from_file('data/xmms2_24.png')
        icon.connect('popup-menu', self.menu_called)
        icon.set_visible(True)
        self.icon = icon
        self.pbstatus = "Stopped"
        self.curinfo = {'id':0}
        self.lastimg = ('', '')

        self.xmms = xmmsclient.XMMS("XMMS2tray")
        try:
            self.connect()
        except IOError:
            mbox = gtk.MessageDialog(None, 0, gtk.MESSAGE_ERROR,
                gtk.BUTTONS_YES_NO, "I could not connect to the XMMS2 daemon. "
                + "It may not be running. Attempt to start it ?")
            resp = mbox.run()
            mbox.destroy()
            if resp == gtk.RESPONSE_YES:
                os.system('xmms2-launcher')
                self.connect()
            else: sys.exit(1)
        xmms2_glib.GLibConnector(self.xmms)

        menucfg = menu_conf.TOP
        menucfg.append(None)
        for cl in menu_conf.CLIENTS:
            if cl['command'][0] == '$':
                if os.access('/usr'+cl['command'][1:], os.R_OK | os.X_OK):
                    cl['command'] = '/usr'+cl['command'][1:]
                    if cl['iconfile'][0] == '$':
                        cl['iconfile'] = '/usr'+cl['iconfile'][1:]
                    menucfg.append(cl)
                elif os.access('/usr/local'+cl['command'][1:], os.R_OK|os.X_OK):
                    cl['command'] = '/usr/local'+cl['command'][1:]
                    if cl['iconfile'][0] == '$':
                        cl['iconfile'] = '/usr/local'+cl['iconfile'][1:]
                    menucfg.append(cl)
            elif os.access(cl['command'], os.R_OK | os.X_OK):
                menucfg.append(cl)
        menucfg.append(None)
        menucfg += menu_conf.BOTTOM

        self.menu = gtk.Menu()
        mi = None
        for item in menucfg:
            if item in ('separator','sep',None):
                mi = gtk.SeparatorMenuItem()
                self.menu.append(mi)
                mi.show()
                continue
            elif 'stockicon' in item:
                mi = gtk.ImageMenuItem(item['title'])
                img = gtk.Image()
                img.set_from_stock(getattr(gtk,'STOCK_'
                    +item['stockicon'].upper()),gtk.ICON_SIZE_MENU)
                mi.set_image(img)
            elif 'iconfile' in item:
                mi = gtk.ImageMenuItem(item['title'])
                if os.path.isfile(item['iconfile']):
                    img = gtk.Image()
                    pb = gtk.gdk.pixbuf_new_from_file(item['iconfile'])
                    # FIXME: hard-coded icon size !
                    img.set_from_pixbuf(pb.scale_simple(16,16,
                        gtk.gdk.INTERP_BILINEAR))
                    mi.set_image(img)
            elif 'check' in item and item['check']:
                mi = gtk.CheckMenuItem(item['title'])
                CheckWaiter(self.xmms, mi,item)
            else:
                mi = gtk.MenuItem(item['title'])
            self.menu.append(mi)
            mi.connect('activate', self.menuclicked, item)
            mi.show()
        self.menu.show()

        if HAVE_NOTIFY:
            pynotify.init("XMMS2tray")
        self.xmms.playback_status(self.pbstatus_cb)
        self.xmms.broadcast_playback_status(self.pbstatus_cb)
        self.xmms.playback_current_id(self.newsong)
        self.xmms.broadcast_playback_current_id(self.newsong)
        self.xmms.broadcast_medialib_entry_changed(self.songchange)

    def connect(self):
        self.xmms.connect(os.getenv('XMMS_PATH'), self.quit)

class CheckWaiter:
    def __init__(self, xmms, mi, item):
        self.item = item
        self.mi = mi
        if item['action'] == 'playpause':
            xmms.broadcast_playback_status(self.cb)
            xmms.playback_status(self.cb)
        self.ok = False
    def cb(self, res):
        if self.item['action'] == 'playpause':
            self.mi.set_active  (res.value() == xmmsclient.PLAYBACK_STATUS_PLAY)
        self.ok = True


def main():
    if not os.path.exists('data/xmms2_24.png'): # go somewhere else
        if os.path.exists('%s/share/xmms2tray/data/xmms2_24.png' % sys.prefix):
            os.chdir('%s/share/xmms2tray' % sys.prefix)
        else:
            raise RuntimeError('XMMS2tray has been installed in a non-standard way. Please run from the directory including data/')

    t = XMMS2tray()
    try:
        gtk.main()
    except KeyboardInterrupt:
        t.quit(False)


