/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "mainwindow.h"

#include <QAction>
#include <QCloseEvent>
#include <QApplication>
#include <QSettings>

#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "data/loginbatch.h"
#include "data/bookmark.h"
#include "commands/commandmanager.h"
#include "widgets/statusbar.h"
#include "dialogs/startpage.h"
#include "dialogs/logindialog.h"
#include "dialogs/settingsdialog.h"
#include "dialogs/connectioninfodialog.h"
#include "views/projectsview.h"
#include "xmlui/builder.h"
#include "viewmanager.h"
#include "connectionmanager.h"
#include "configdata.h"
#include "connectionmanager.h"
#include "iconloader.h"

using namespace WebIssues;

#if defined( Q_WS_WIN )
static const int TrayIconSize = 16;
#else
static const int TrayIconSize = 22;
#endif

MainWindow::MainWindow() :
    m_view( NULL )
{
    setWindowTitle( tr( "WebIssues" ) );

    QAction* action;

    action = new QAction( IconLoader::icon( "file-quit" ), tr( "&Quit" ), this );
    action->setShortcut( tr( "Ctrl+Q" ) );
    connect( action, SIGNAL( triggered() ), this, SLOT( quit() ) );
    setAction( "quit", action );

    action = new QAction( IconLoader::icon( "connection-open" ), tr( "&New Connection..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( newConnection() ) );
    setAction( "newConnection", action );

    action = new QAction( IconLoader::icon( "connection-close" ), tr( "&Close Connection" ), this );
    action->setVisible( false );
    connect( action, SIGNAL( triggered() ), this, SLOT( closeConnection() ) );
    setAction( "closeConnection", action );

    action = new QAction( IconLoader::icon( "status-info" ), tr( "Connection &Information" ), this );
    action->setVisible( false );
    connect( action, SIGNAL( triggered() ), this, SLOT( connectionInfo() ) );
    setAction( "connectionInfo", action );

    action = new QAction( IconLoader::icon( "configure" ), tr( "&Configure WebIssues..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( configure() ) );
    setAction( "configure", action );

    action = new QAction( IconLoader::icon( "help-contents" ), tr( "WebIssues &Manual" ), this );
    action->setShortcut( QKeySequence::HelpContents );
    connect( action, SIGNAL( triggered() ), qApp, SLOT( manual() ) );
    setAction( "manual", action );

    action = new QAction( IconLoader::icon( "webissues" ), tr( "&About WebIssues" ), this );
    connect( action, SIGNAL( triggered() ), qApp, SLOT( about() ) );
    setAction( "about", action );

    action = new QAction( IconLoader::icon( "about-qt" ), tr( "About &Qt" ), this );
    connect( action, SIGNAL( triggered() ), qApp, SLOT( aboutQt() ) );
    setAction( "aboutQt", action );

    setTitle( "menuWebIssues", tr( "&WebIssues" ) );
    setTitle( "menuTools", tr( "&Tools" ) );
    setTitle( "menuHelp", tr( "&Help" ) );
    setTitle( "toolbarMain", tr( "Main Toolbar" ) );

    loadXmlFile( ":/resources/mainwindow.xml" );

    m_builder = new XmlUi::Builder( this );
    m_builder->addClient( this );

    setStatusBar( new WebIssues::StatusBar( this ) );

    qRegisterMetaType<Bookmark>( "Bookmark" );

    showStartPage();

    QSettings* settings = configData->settings();
    settings->beginGroup( "MainWindow" );
    resize( settings->value( "Size", QSize( 350, 400 ) ).toSize() );
    move( settings->value( "Position", QSize( 100, 100 ) ).toPoint() );
    settings->endGroup();

    connect( configData, SIGNAL( settingsChanged() ), this, SLOT( settingsChanged() ) );

    m_trayIcon = new QSystemTrayIcon( this );
    m_trayIcon->setIcon( IconLoader::pixmap( "webissues", TrayIconSize ) );
    m_trayIcon->setToolTip( tr( "WebIssues - Not connected" ) );

    connect( m_trayIcon, SIGNAL( activated( QSystemTrayIcon::ActivationReason ) ), this, SLOT( trayIconActivated( QSystemTrayIcon::ActivationReason ) ) );

    connect( m_builder, SIGNAL( reset() ), this, SLOT( builderReset() ) );
    builderReset();

    settingsChanged();

    statusBar()->showInfo( tr( "Not connected to server" ) );
}

MainWindow::~MainWindow()
{
    QSettings* settings = configData->settings();
    settings->beginGroup( "MainWindow" );
    settings->setValue( "Size", size() );
    settings->setValue( "Position", pos() );
    settings->endGroup();
}

void MainWindow::closeEvent( QCloseEvent* e )
{
    if ( m_trayIcon->isVisible() )
        hide();
    else
        quit();

    e->ignore();
}

QMenu* MainWindow::createPopupMenu()
{
    return NULL;
}

void MainWindow::reconnect()
{
    BookmarkSettings settings;
    if ( settings.bookmarks().count() > 0 )
       openConnection( settings.bookmarks().at( 0 ) ); 
}

void MainWindow::quit()
{
    if ( viewManager && !viewManager->queryCloseViews() )
        return;

    qApp->quit();
}

void MainWindow::newConnection()
{
    LoginDialog dialog( this );
    if ( dialog.exec() != QDialog::Accepted )
        return;

    BookmarkSettings settings;
    settings.addBookmark( dialog.createBookmark() );

    initializeView();
}

void MainWindow::closeConnection()
{
    if ( !viewManager->queryCloseViews() )
        return;

    closeView();

    connectionManager->closeConnection();
}

void MainWindow::initializeView()
{
    m_view = new ProjectsView( this, this );

    connect( m_view, SIGNAL( captionChanged( const QString& ) ), this, SLOT( captionChanged( const QString& ) ) );

    connect( m_view, SIGNAL( showInfo( const QString& ) ), statusBar(), SLOT( showInfo( const QString& ) ) );
    connect( m_view, SIGNAL( showWarning( const QString& ) ), statusBar(), SLOT( showWarning( const QString& ) ) );
    connect( m_view, SIGNAL( showBusy( const QString& ) ), statusBar(), SLOT( showBusy( const QString& ) ) );

    connect( m_view, SIGNAL( showSummary( const QPixmap&, const QString& ) ), statusBar(), SLOT( showSummary( const QPixmap&, const QString& ) ) );

    delete m_startPage;
    m_startPage = NULL;

    setCentralWidget( m_view->mainWidget() );
    m_view->mainWidget()->show();

    action( "newConnection" )->setVisible( false );
    action( "closeConnection" )->setVisible( true );
    action( "connectionInfo" )->setVisible( true );

    m_builder->addClient( m_view );

    m_trayIcon->setIcon( IconLoader::overlayedPixmap( "webissues", "overlay-connected", TrayIconSize ) );
    m_trayIcon->setToolTip( tr( "WebIssues - Connected to %1" ).arg( connectionManager->connectionInfo()->serverName() ) );

    m_view->initialUpdate();
}

void MainWindow::closeView()
{
    action( "newConnection" )->setVisible( true );
    action( "newConnection" )->setEnabled( true );
    action( "closeConnection" )->setVisible( false );
    action( "connectionInfo" )->setVisible( false );

    m_builder->removeClient( m_view );

    m_trayIcon->setIcon( IconLoader::pixmap( "webissues", TrayIconSize ) );
    m_trayIcon->setToolTip( tr( "WebIssues - Not Connected" ) );

    delete m_view;
    m_view = NULL;

    showStartPage();

    setWindowTitle( tr( "WebIssues" ) );

    statusBar()->showInfo( tr( "Connection closed" ) );
    statusBar()->showSummary( QPixmap(), QString() );
}

void MainWindow::showStartPage()
{
    m_startPage = new StartPage( this );

    connect( m_startPage, SIGNAL( openConnection( const Bookmark& ) ), this, SLOT( openConnection( const Bookmark& ) ), Qt::QueuedConnection );
    connect( m_startPage, SIGNAL( editConnection( const Bookmark& ) ), this, SLOT( editConnection( const Bookmark& ) ), Qt::QueuedConnection );
    connect( m_startPage, SIGNAL( newConnection() ), this, SLOT( newConnection() ), Qt::QueuedConnection );
    connect( m_startPage, SIGNAL( abort() ), this, SLOT( abortLogin() ) );

    setCentralWidget( m_startPage );
    m_startPage->show();
}

void MainWindow::configure()
{
    SettingsDialog dialog( this );
    dialog.exec();
}

void MainWindow::settingsChanged()
{
    m_trayIcon->setVisible( configData->dockWindow() );
}

void MainWindow::openConnection( const Bookmark& bookmark )
{
    if ( bookmark.password().isEmpty() ) {
        showLoginDialog( bookmark, false );
        return;
    }

    m_bookmark = bookmark;

    m_batch = new LoginBatch( bookmark.login(), bookmark.password() );

    connect( m_batch, SIGNAL( completed( bool ) ), this, SLOT( loginCompleted( bool ) ) );

    connectionManager->createCommandManager();
    commandManager->setServerUrl( bookmark.url() ); 
    commandManager->execute( m_batch );

    statusBar()->showBusy( tr( "Connecting to server..." ) );

    m_startPage->setConnecting( true );

    action( "newConnection" )->setEnabled( false );    
}

void MainWindow::loginCompleted( bool successful )
{
    if ( successful ) {
        ConnectionInfo* info = m_batch->connectionInfo();
        connectionManager->openConnection( info );

        Bookmark bookmark( info->serverName(), info->serverUuid(),
            m_bookmark.url(), m_bookmark.login(), m_bookmark.password() );

        BookmarkSettings settings;
        settings.replaceBookmark( m_bookmark.serverUuid(), bookmark );

        initializeView();
    } else {
        QString message = commandManager->errorMessage( tr( "Connection failed" ) );

        connectionManager->closeConnection();

        statusBar()->showWarning( message );

        m_startPage->setConnecting( false );

        action( "newConnection" )->setEnabled( true );
    }
}

void MainWindow::abortLogin()
{
    commandManager->abort( m_batch );
}

void MainWindow::editConnection( const Bookmark& bookmark )
{
    showLoginDialog( bookmark, true );
}

void MainWindow::showLoginDialog( const Bookmark& bookmark, bool editable )
{
    LoginDialog dialog( this );
    dialog.loadBookmark( bookmark, editable );

    if ( dialog.exec() != QDialog::Accepted )
        return;

    BookmarkSettings settings;
    settings.replaceBookmark( bookmark.serverUuid(), dialog.createBookmark() );

    initializeView();
}

void MainWindow::connectionInfo()
{
    ConnectionInfoDialog dialog( this );
    dialog.exec();
}

void MainWindow::trayIconActivated( QSystemTrayIcon::ActivationReason reason )
{
    if ( reason == QSystemTrayIcon::Trigger ) {
        if (isMinimized()) {
            // hide needed when changing desktop
            hide();
            showNormal();
            activateWindow();
        } else if ( !isVisible() ) {
            show();
            activateWindow();
        } else {
            hide();
        }
    }
}

void MainWindow::captionChanged( const QString& caption )
{
    setWindowTitle( tr( "%1 - WebIssues" ).arg( caption ) );
}

void MainWindow::builderReset()
{
    m_trayIcon->setContextMenu( m_builder->contextMenu( "contextTray" ) );
}

#include "mainwindow.moc"
