/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef FILTERDIALOG_H
#define FILTERDIALOG_H

#include <QDialog>
#include <QList>

#include "attributes/definitioninfo.h"
#include "models/columncondition.h"

class QComboBox;
class QGridLayout;
class QPushButton;

namespace WebIssues
{

class AbstractValueEditor;

/**
* Dialog for editing a list of column conditions for filtering.
*/
class FilterDialog : public QDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param projectId Optional project identifier for attribute value context.
    *
    * @param parent The parent widget.
    */
    FilterDialog( int projectId, QWidget* parent );

    /**
    * Destructor.
    */
    ~FilterDialog();

public:
    /**
    * Set a list of available columns for filtering.
    */
    void setAvailableColumns( const QList<Column>& columns );

    /**
    * Initialize the dialog with a list of conditions.
    */
    void setConditions( const QList<ColumnCondition>& conditions );

    /**
    * Read the list of conditions from the dialog.
    */
    QList<ColumnCondition> conditions() const;

signals:
    /**
    * Emitted when the Ok or Apply button is clicked.
    */
    void settingsApplied();

public: // overrides
    void accept();

private slots:
    void addRow();
    void removeRow();
    void clear();

    void columnActivated( int index );
    void conditionActivated( int index );

private:
    struct FilterRow
    {
        QComboBox* m_columnCombo;
        QComboBox* m_conditionCombo;

        DefinitionInfo m_definitionInfo;

        QList<ColumnCondition::Condition> m_conditions;

        AbstractValueEditor* m_editor;
    };

private:
    void clearAllRows();

    void createRow( const ColumnCondition& condition );

    void selectColumn( FilterRow& row, Column column );
    void selectCondition( FilterRow& row, int index, ColumnCondition::Condition condition );

private:
    int m_typeId;
    int m_projectId;

    QWidget* m_gridPanel;
    QGridLayout* m_gridLayout;

    QPushButton* m_moreButton;
    QPushButton* m_fewerButton;
    QPushButton* m_clearButton;

    QList<Column> m_columns;

    QList<FilterRow> m_rows;
};

}

#endif
