/**************************************************************************
* Generic, native, relational database
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef TABLEITERATORS_H
#define TABLEITERATORS_H

#include "iteratorcore.h"
#include "tableindexes.h"

namespace RDB
{

/**
* General iterator for all rows in an index.
*
* Use this iterator to iterate over all rows of a table of an unknown type.
*
* @see IndexIterator
*/
class IndexIteratorBase
{
public:
    /**
    * Constructor.
    */
    IndexIteratorBase( UniqueIndexBase* index ) : m_core( index->data() ) { }

    /**
    * Constructor.
    */
    IndexIteratorBase( ForeignIndexBase* index ) : m_core( index->data() ) { }

    /**
    * Destructor.
    */
    ~IndexIteratorBase() { }

public:
    /**
    * Advance to the next row.
    *
    * @return @c true if successful or @c false if there are no more rows.
    */
    bool next() { return m_core.next(); }

    /**
    * Return the current row.
    */
    Row* get() const { return m_core.get(); }

protected:
    IndexIteratorCore m_core;
};

/**
* Type-safe iterator for all rows in an index.
*
* Use this iterator to iterate over all rows of a table.
*
* @code
* IndexIterator<MyRow> it( table.index() );
* while ( it.next() ) {
*     MyRow* row = it.get();
*     // ...
* }
* @endcode
*
* @see IndexConstIterator
*/
template<class ROW>
class IndexIterator : public IndexIteratorBase
{
public:
    /**
    * Constructor.
    */
    IndexIterator( UniqueIndex<ROW>* index ) : IndexIteratorBase( index ) { }

    /**
    * Constructor.
    */
    IndexIterator( ForeignIndex<ROW>* index ) : IndexIteratorBase( index ) { }

    /**
    * Destructor.
    */
    ~IndexIterator() { }

public:
    ROW* get() const { return (ROW*)m_core.get(); }
};

/**
* Iterator for all rows in a table.
*
* This is a convenience subclass of IndexIterator for initializing it with a table.
*/
template<class ROW>
class TableIterator : public IndexIterator<ROW>
{
public:
    /**
    * Constructor.
    */
    template<class TABLE>
    TableIterator( TABLE* table ) : IndexIterator<ROW>( table->index() ) { }

    /**
    * Destructor.
    */
    ~TableIterator() { }
};

/**
* General constant iterator for all rows in an index.
*
* Use this iterator to iterate over all rows of a constant table of an unknown type.
*
* @see IndexConstIterator
*/
class IndexConstIteratorBase
{
public:
    /**
    * Constructor.
    */
    IndexConstIteratorBase( const UniqueIndexBase* index ) : m_core( index->data() ) { }

    /**
    * Constructor.
    */
    IndexConstIteratorBase( const ForeignIndexBase* index ) : m_core( index->data() ) { }

    /**
    * Destructor.
    */
    ~IndexConstIteratorBase() { }

public:
    /**
    * Advance to the next row.
    *
    * @return @c true if successful or @c false if there are no more rows.
    */
    bool next() { return m_core.next(); }

    /**
    * Return the current row.
    */
    const Row* get() const { return m_core.get(); }

protected:
    IndexIteratorCore m_core;
};

/**
* Type-safe iterator for all rows in an index.
*
* Use this iterator to iterate over all rows of a constant table.
*
* @code
* IndexConstIterator<MyRow> it( table.index() );
* while ( it.next() ) {
*     const MyRow* row = it.get();
*     // ...
* }
* @endcode
*
* @see IndexIterator
*/
template<class ROW>
class IndexConstIterator : public IndexConstIteratorBase
{
public:
    /**
    * Constructor.
    */
    IndexConstIterator( const UniqueIndex<ROW>* index ) : IndexConstIteratorBase( index ) { }

    /**
    * Constructor.
    */
    IndexConstIterator( const ForeignIndex<ROW>* index ) : IndexConstIteratorBase( index ) { }

    /**
    * Destructor.
    */
    ~IndexConstIterator() { }

public:
    const ROW* get() const { return (const ROW*)m_core.get(); }
};

/**
* Iterator for all rows in a table.
*
* This is a convenience subclass of IndexConstIterator for initializing it with a table.
*/
template<class ROW>
class TableConstIterator : public IndexConstIterator<ROW>
{
public:
    /**
    * Constructor.
    */
    template<class TABLE>
    TableConstIterator( const TABLE* table ) : IndexConstIterator<ROW>( table->index() ) { }

    /**
    * Destructor.
    */
    ~TableConstIterator() { }
};

/**
* General iterator for rows matching a foreign key.
*
* Use this iterator to iterate over rows of a table of an unknown type.
*
* @see ForeignIterator
*/
class ForeignIteratorBase
{
public:
    /**
    * Constructor.
    */
    ForeignIteratorBase( ForeignIndexBase* index, int key ) : m_core( index->data(), key ) { }

    /**
    * Destructor.
    */
    ~ForeignIteratorBase() { }

public:
    /**
    * Advance to the next row.
    *
    * @return @c true if successful or @c false if there are no more rows.
    */
    bool next() { return m_core.next(); }

    /**
    * Return the current row.
    */
    Row* get() const { return m_core.get(); }

protected:
    ForeignIteratorCore m_core;
};

/**
* Type-safe iterator for rows matching a foreign key.
*
* Use this iterator to find all rows with the given value of a foreign key in a table.
*
* @code
* ForeignIterator<MyRow> it( table.parentIndex(), 100 );
* while ( it.next() ) {
*     MyRow* row = it.get();
*     // ...
* }
* @endcode
*
* @see ForeignConstIterator
*/
template<class ROW>
class ForeignIterator : public ForeignIteratorBase
{
public:
    /**
    * Constructor.
    */
    ForeignIterator( ForeignIndex<ROW>* index, int key ) : ForeignIteratorBase( index, key ) { }

    /**
    * Destructor.
    */
    ~ForeignIterator() { }

public:
    ROW* get() const { return (ROW*)m_core.get(); }
};

/**
* General const iterator for rows matching a foreign key.
*
* Use this iterator to iterate over rows of a constant table of an unknown type.
*
* @see ForeignConstIterator
*/
class ForeignConstIteratorBase
{
public:
    /**
    * Constructor.
    */
    ForeignConstIteratorBase( const ForeignIndexBase* index, int key ) : m_core( index->data(), key ) { }

    /**
    * Destructor.
    */
    ~ForeignConstIteratorBase() { }

public:
    /**
    * Advance to the next row.
    *
    * @return @c true if successful or @c false if there are no more rows.
    */
    bool next() { return m_core.next(); }

    /**
    * Return the current row.
    */
    const Row* get() const { return m_core.get(); }

protected:
    ForeignIteratorCore m_core;
};

/**
* Type-safe const iterator for rows matching a foreign key.
*
* Use this iterator to find all rows with the given value of a foreign key in a constant table.
*
* @code
* ForeignConstIterator<MyRow> it( table.parentIndex(), 100 );
* while ( it.next() ) {
*     const MyRow* row = it.get();
*     // ...
* }
* @endcode
*
* @see ForeignIterator
*/
template<class ROW>
class ForeignConstIterator : public ForeignConstIteratorBase
{
public:
    /**
    * Constructor.
    */
    ForeignConstIterator( const ForeignIndex<ROW>* index, int key ) : ForeignConstIteratorBase( index, key ) { }

    /**
    * Destructor.
    */
    ~ForeignConstIterator() { }

public:
    const ROW* get() const { return (const ROW*)m_core.get(); }
};

}

#endif
