/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef METADATADIALOGBASE_H
#define METADATADIALOGBASE_H

#include <QDialog>

#include "definitioninfo.h"

class QLabel;
class QBoxLayout;

namespace WebIssues
{

/**
* Base class of a dialog for editing attribute definition metadata.
*
* This is an abstract class. Use the MetadataDialogFactory to create
* metadata dialogs for given attribute types.
*
* Inherited classes must create widget using mainWidget() as their parent,
* put them into a layout and call setContentLayout() to initialize the
* dialog contents.
*
* Inherited classes must implement the updateWidgets() and updateMetadata()
* methods.
*/
class MetadataDialogBase : public QDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent widget.
    */
    MetadataDialogBase( QWidget* parent );

    /**
    * Destructor.
    */
    ~MetadataDialogBase();

public:
    /**
    * Initialize the dialog using given metadata.
    *
    * @param info Attribute definition to extract metadata from.
    */
    void setDefinitionInfo( const DefinitionInfo& info );

    /**
    * Return new metadata.
    *
    * Use this method after the dialog is closed with QDialog::Accepted status.
    *
    * @note Only the attribute type and custom metadata is returned. The required flag
    * and default value are not preserved.
    *
    * @return The attribute definition containing the new metadata.
    */
    const DefinitionInfo& definitionInfo() { return m_info; }

protected:
    /**
    * Initialize the dialog content.
    *
    * @param layout The layout to insert into the dialog.
    *
    * @param fixed If @c true, the dialog cannot be resized vertically.
    */
    void setContentLayout( QLayout* layout, bool fixed );

    /**
    * Show an information message in the status label.
    *
    * @param text Message to display.
    */
    void showInfo( const QString& text );

    /**
    * Show a warning message in the status label.
    *
    * @param text Message to display.
    */
    void showWarning( const QString& text );

protected:
    /**
    * Method implemented by inherited classes to update widgets from given metadata.
    *
    * @param info Definition to extract the metadata from.
    */
    virtual void updateWidgets( const DefinitionInfo& info ) = 0;

    /**
    * Method implemented by inherited classes to extract metadata from widgets.
    *
    * @param info Definition to put the metadata into.
    *
    * @return @c true if the data was successfully extracted.
    */
    virtual bool updateMetadata( DefinitionInfo& info ) = 0;

public: // overrides
    void accept();

private:
    DefinitionInfo m_info;
    DefinitionInfo::Type m_type;

    QLabel* m_statusPixmap;
    QLabel* m_statusLabel;

    QBoxLayout* m_mainLayout;
};

}

#endif
