// Constructs a validate accessibility object
function WebDeveloperValidateAccessibility()
{
    this.file              = null;
    this.formElement       = null;
    this.inputElement      = null;
    this.validationRequest = null;
}

// Cleans up
WebDeveloperValidateAccessibility.prototype.cleanUp = function()
{
    this.deleteFile();

    // If the validation request is set
    if(this.validationRequest)
    {
        this.validationRequest.abort();
    }
}

// Creates a source file
WebDeveloperValidateAccessibility.prototype.createSourceFile = function(uri)
{
    var temporaryDirectory = Components.classes["@mozilla.org/file/directory_service;1"].getService(Components.interfaces.nsIProperties).get("TmpD", Components.interfaces.nsIFile);

    // If the temporary directory exists, is a directory and is writable
    if(temporaryDirectory.exists() && temporaryDirectory.isDirectory() && temporaryDirectory.isWritable())
    {
        var fileName   = "";
        var sourceFile = Components.classes["@mozilla.org/file/local;1"].createInstance(Components.interfaces.nsILocalFile);

        // Try to get the host
        try
        {
            fileName = uri.host;
        }
        catch(exception)
        {
            // Do nothing
        }

        temporaryDirectory.append("webdeveloper-" + fileName + "-" + new Date().getTime() + ".html");
        sourceFile.initWithPath(temporaryDirectory.path);

        return sourceFile;
    }
    else
    {
        webdeveloper_error(document.getElementById("webdeveloper-string-bundle").getFormattedString("webdeveloper_tempDirectoryFailed", [temporaryDirectory.path]));

        return null;
    }
}

// Deletes the file
WebDeveloperValidateAccessibility.prototype.deleteFile = function()
{
    // If the file is set
    if(this.file)
    {
        // Try to delete the file
        try
        {
            this.file.remove(false);
        }
        catch(exception)
        {
            // Do nothing
        }

        this.file = null;
    }
}

// Returns the post data
WebDeveloperValidateAccessibility.prototype.getPostData = function()
{
    // Try to get the post data
    try
    {
        var sessionHistory = getWebNavigation().sessionHistory;
        var entry          = sessionHistory.getEntryAtIndex(sessionHistory.index, false).QueryInterface(Components.interfaces.nsISHEntry);

        return entry.postData;
    }
    catch(exception)
    {
        return null;
    }
}

// Parses the validation results by type
WebDeveloperValidateAccessibility.prototype.parseValidationResultsByType = function(type)
{
    var resultsHTML = this.validationRequest.responseText;
    var count       = resultsHTML.split(type + ".gif").length;

    // If the count is greater than 0
    if(count > 0)
    {
        return count - 1;
    }

    return 0;
}

// Retrieves the HTML
WebDeveloperValidateAccessibility.prototype.retrieveHTML = function()
{
    var converter        = Components.classes["@mozilla.org/intl/scriptableunicodeconverter"].createInstance(Components.interfaces.nsIScriptableUnicodeConverter);
    var htmlText         = null;
    var inputStream      = Components.classes["@mozilla.org/network/file-input-stream;1"].createInstance(Components.interfaces.nsIFileInputStream);
    var scriptableStream = Components.classes["@mozilla.org/scriptableinputstream;1"].createInstance(Components.interfaces.nsIScriptableInputStream);

    converter.charset = webdeveloper_getContentDocument().characterSet;

    inputStream.init(this.file, 0x01, 0444, null);
    scriptableStream.init(inputStream);

    htmlText = converter.ConvertToUnicode(scriptableStream.read(scriptableStream.available()));

    scriptableStream.close();
    inputStream.close();

    return htmlText;
}

// Saves the HTML
WebDeveloperValidateAccessibility.prototype.saveHTML = function(uri)
{
    var webBrowserPersistInterface = Components.interfaces.nsIWebBrowserPersist;
    var webBrowserPersist          = Components.classes["@mozilla.org/embedding/browser/nsWebBrowserPersist;1"].createInstance(webBrowserPersistInterface);

    webBrowserPersist.persistFlags     = webBrowserPersistInterface.PERSIST_FLAGS_AUTODETECT_APPLY_CONVERSION | webBrowserPersistInterface.PERSIST_FLAGS_FROM_CACHE | webBrowserPersistInterface.PERSIST_FLAGS_REPLACE_EXISTING_FILES;
    webBrowserPersist.progressListener = this;

    webBrowserPersist.saveURI(uri, null, uri, this.getPostData(), null, this.file);
}

// Submits the background request to validate the accessibility
WebDeveloperValidateAccessibility.prototype.submitBackgroundRequest = function()
{
    var requestBody = "CheckURL=1&URLTest=Your+HTML";

    // If the priority 1 preference is set
    if(webdeveloper_getBooleanPreference("webdeveloper.validate.local.accessibility.wai.priority1", true))
    {
        requestBody += "&p1=1";
    }

    // If the priority 2 preference is set
    if(webdeveloper_getBooleanPreference("webdeveloper.validate.local.accessibility.wai.priority2", true))
    {
        requestBody += "&p2=1";
    }

    // If the priority 3 preference is set
    if(webdeveloper_getBooleanPreference("webdeveloper.validate.local.accessibility.wai.priority3", true))
    {
        requestBody += "&p3=1";
    }

    // If the Section 508 preference is set
    if(webdeveloper_getBooleanPreference("webdeveloper.validate.local.accessibility.section508", true))
    {
        requestBody += "&s508=1";
    }

    requestBody                              += "&myHTML=" + encodeURIComponent(this.retrieveHTML());
    this.validationRequest.onreadystatechange = webdeveloper_updatePageAccessibilityValidationDetails;

    this.validationRequest.open("post", "http://www.hermish.com/check_this.cfm", true);

    // Try to set the request header
    try
    {
        this.validationRequest.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
        this.validationRequest.send(requestBody);
    }
    catch(exception)
    {
        // Reset the validation request
        this.validationRequest = new XMLHttpRequest();
    }

    this.deleteFile();
}

// Submits the form to validate the accessibility
WebDeveloperValidateAccessibility.prototype.submitForm = function()
{
    this.inputElement.value = this.retrieveHTML();

    this.deleteFile();
    this.formElement.submit();
}

// Validate the accessibility of the given URI in the background
WebDeveloperValidateAccessibility.prototype.validateBackgroundAccessibility = function(uri)
{
    this.file = this.createSourceFile(uri);

    // If the validation request is not set
    if(!this.validationRequest)
    {
        this.validationRequest = new XMLHttpRequest();
    }

    this.saveHTML(uri);
}

// Validate the accessibility of the given URI
WebDeveloperValidateAccessibility.prototype.validateAccessibility = function(uri)
{
    var oldTab            = getBrowser().selectedTab;
    var oldURL            = getBrowser().currentURI.spec;
    var generatedDocument = webdeveloper_generateDocument("");
    var bodyElement       = webdeveloper_getDocumentBodyElement(generatedDocument);
    var imageElement      = generatedDocument.createElement("img");
    var inputElement      = null;
    var pElement          = generatedDocument.createElement("p");
    var stringBundle      = document.getElementById("webdeveloper-string-bundle");

    generatedDocument.title = stringBundle.getString("webdeveloper_validateAccessibility");
    this.file               = this.createSourceFile(uri);
    this.formElement        = generatedDocument.createElement("form");

    webdeveloper_addGeneratedStyles(generatedDocument);

    imageElement.setAttribute("alt", "loading");
    imageElement.setAttribute("src", "chrome://webdeveloper/content/images/content/loading.gif");
    pElement.appendChild(imageElement);
    pElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_contactingValidator")));
    pElement.setAttribute("class", "loading");
    bodyElement.appendChild(pElement);

    this.formElement.setAttribute("action", "http://www.hermish.com/check_this.cfm");
    this.formElement.setAttribute("method", "post");
    this.formElement.setAttribute("style", "display: none");

    inputElement = generatedDocument.createElement("input");

    inputElement.setAttribute("name", "CheckURL");
    inputElement.setAttribute("type", "hidden");
    inputElement.setAttribute("value", "1");
    this.formElement.appendChild(inputElement);

    // If the priority 1 preference is set
    if(webdeveloper_getBooleanPreference("webdeveloper.validate.local.accessibility.wai.priority1", true))
    {
        inputElement = generatedDocument.createElement("input");

        inputElement.setAttribute("name", "p1");
        inputElement.setAttribute("type", "hidden");
        inputElement.setAttribute("value", "1");
        this.formElement.appendChild(inputElement);
    }

    // If the priority 2 preference is set
    if(webdeveloper_getBooleanPreference("webdeveloper.validate.local.accessibility.wai.priority2", true))
    {
        inputElement = generatedDocument.createElement("input");

        inputElement.setAttribute("name", "p2");
        inputElement.setAttribute("type", "hidden");
        inputElement.setAttribute("value", "1");
        this.formElement.appendChild(inputElement);
    }

    // If the priority 3 preference is set
    if(webdeveloper_getBooleanPreference("webdeveloper.validate.local.accessibility.wai.priority3", true))
    {
        inputElement = generatedDocument.createElement("input");

        inputElement.setAttribute("name", "p3");
        inputElement.setAttribute("type", "hidden");
        inputElement.setAttribute("value", "1");
        this.formElement.appendChild(inputElement);
    }

    // If the Section 508 preference is set
    if(webdeveloper_getBooleanPreference("webdeveloper.validate.local.accessibility.section508", true))
    {
        inputElement = generatedDocument.createElement("input");

        inputElement.setAttribute("name", "s508");
        inputElement.setAttribute("type", "hidden");
        inputElement.setAttribute("value", "1");
        this.formElement.appendChild(inputElement);
    }

    inputElement = generatedDocument.createElement("input");

    inputElement.setAttribute("name", "URLTest");
    inputElement.setAttribute("type", "hidden");
    inputElement.setAttribute("value", "Your HTML");
    this.formElement.appendChild(inputElement);

    this.inputElement = generatedDocument.createElement("input");

    this.inputElement.setAttribute("name", "myHTML");
    this.inputElement.setAttribute("type", "hidden");
    this.formElement.appendChild(this.inputElement);
    bodyElement.appendChild(this.formElement);

    // If the open tabs in background preference is set to true
    if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
    {
        getBrowser().selectedTab = oldTab;
    }

    this.saveHTML(uri);
}

// Called when the progress state changes
WebDeveloperValidateAccessibility.prototype.onStateChange = function(webProgress, request, stateFlags, status)
{
    // If the progress has stopped
    if(stateFlags & Components.interfaces.nsIWebProgressListener.STATE_STOP)
    {
        // If the file is set and exists
        if(this.file && this.file.exists())
        {
            // If the validation request is set
            if(this.validationRequest)
            {
                this.submitBackgroundRequest();
            }
            else
            {
                this.submitForm();
            }
        }
    }
}

// Indicates the interfaces this object supports
WebDeveloperValidateAccessibility.prototype.QueryInterface = function(id)
{
    // If the query is for a supported interface
    if(id.equals(Components.interfaces.nsISupports) || id.equals(Components.interfaces.nsIWebProgressListener))
    {
        return this;
    }

    throw Components.results.NS_NOINTERFACE;
}

// Dummy methods requiring implementations
WebDeveloperValidateAccessibility.prototype.onLocationChange = function(webProgress, request, location) {}
WebDeveloperValidateAccessibility.prototype.onProgressChange = function(webProgress, request, currentSelfProgress, maximumSelfProgress, currentTotalProgress, maximumTotalProgress) {}
WebDeveloperValidateAccessibility.prototype.onSecurityChange = function(webProgress, request, state) {}
WebDeveloperValidateAccessibility.prototype.onStatusChange   = function(webProgress, request, status, message) {}
