/**
 * @file geis_backend_multiplexor.h
 * @brief internal GEIS backend multiplexor interface
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#ifndef GEIS_BACKEND_MULTIPLEXOR_H_
#define GEIS_BACKEND_MULTIPLEXOR_H_

#include "geis/geis.h"


/**
 * Multiplexes back end events into a single notification file descriptor.
 *
 * The GEIS API presents a single file descriptor to the application to watch
 * for activity notification.  A back end may be monitoring activity on more
 * than one file descriptor.  The multiplexor combines these requirements.
 */
typedef struct _GeisBackendMultiplexor *GeisBackendMultiplexor;

/**
 * Indicates the type of fd event that occurred on a multiplexed descriptor.
 */
typedef enum _GeisBackendMultiplexorEvent
{
  GEIS_BE_MX_READ_AVAILABLE,
  GEIS_BE_MX_WRITE_AVAILABLE
} GeisBackendMultiplexorEvent;

/**
 * Handles events occurring on multiplexed file descriptors.
 *
 * Back ends must provide a callback with this signature to the multiplexor.
 */
typedef void (*GeisBackendFdEventCallback)(int                          fd,
                                           GeisBackendMultiplexorEvent  event,
                                           void                        *context);

/**
 * Constructs a new back end multiplexor.
 */
GeisBackendMultiplexor geis_backend_multiplexor_new();

/**
 * A reasonable default value for the max_events_per_pump parameter to
 * geis_backend_multiplexor_new().
 */
#define GEIS_BE_MX_DEFAULT_EVENTS_PER_PUMP  16

/**
 * Destroys a back end multiplexor.
 *
 * @param[in] mx  The back end multiplexor to destroy.
 */
void geis_backend_multiplexor_delete(GeisBackendMultiplexor mx);

/**
 * Adds a file descriptor to a back end multiplexor.
 *
 * @param[in] mx  The back end multiplexor.
 * @param[in] fd  The file descriptor to add.
 */
void geis_backend_multiplexor_add_fd(GeisBackendMultiplexor      mx,
                                     int                         fd,
                                     GeisBackendFdEventCallback  callback,
                                     void                       *context);

/**
 * Removes a file descriptor from a back end multiplexor.
 *
 * @param[in] mx  The back end multiplexor.
 * @param[in] fd  The file descriptor to remove.
 */
void geis_backend_multiplexor_remove_fd(GeisBackendMultiplexor mx,
                                        int                    fd);

/**
 * Gets the file descriptor of the back end multiplexor.
 *
 * @param[in] mx  The back end multiplexor.
 */
int geis_backend_multiplexor_fd(GeisBackendMultiplexor mx);

/**
 * Gets the maximum number of fd events per pump.
 *
 * @param[in] mx                   The back end multiplexor.
 */
int geis_backend_multiplexor_max_events_per_pump(GeisBackendMultiplexor mx);

/**
 * Sets the maximum fd events per pump of the multiplexor.
 *
 * @param[in] mx                   The back end multiplexor.
 * @param[in] max_events_per_pump  The maximum number of fd events pumped
 *                                 per call of geis_backend_multiplexor_pump().
 *
 * This value is tunable to accommodate different back ends and
 * application/toolkit requirements.
 */
void geis_backend_multiplexor_set_max_events_per_pump(GeisBackendMultiplexor mx,
                                                      int max_events_per_pump);

/**
 * Exercises the back end multiplexor.
 *
 * @param[in] mx  The back end multiplexor.
 *
 * Dispatches any events on multiplexed file descriptors to their assciated
 * handlers.
 */
GeisStatus geis_backend_multiplexor_pump(GeisBackendMultiplexor mx);

#endif /* GEIS_BACKEND_MULTIPLEXOR_H_ */
