/**
 * @file geis_filter.h
 * @brief internal uTouch Geis filter module private interface
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#ifndef GEIS_FILTER_H_
#define GEIS_FILTER_H_

#include "geis/geis.h"
#include "geis_filter_term.h"

/**
 * @defgroup utouch_geis_filter_container A Filter Container
 * @{
 */

/**
 * An unsorted container for holding filters.
 */
typedef struct _GeisFilterBag *GeisFilterBag;

/**
 * Creates a new filter bag,
 */
GeisFilterBag geis_filter_bag_new();

/**
 * Destroys a filter bag.
 *
 * @param[in] bag The filter bag,
 */
void geis_filter_bag_delete(GeisFilterBag bag);

/**
 * Gets the number of filters in the bag.
 *
 * @param[in] bag The filter bag,
 */
GeisSize geis_filter_bag_count(GeisFilterBag bag);

/**t
 * Gets an indicated filter from a bag.
 *
 * @param[in] bag   The filter bag.
 * @param[in] index The index.
 */
GeisFilter geis_filter_bag_filter(GeisFilterBag bag, GeisSize index);

/**
 * Gets an indicated filter from a bag.
 *
 * @param[in] bag   The filter bag.
 * @param[in] name  The name by which to find the filter.
 *
 * Returns a NULL pointer if the named filter is not found.
 */
GeisFilter geis_filter_bag_filter_by_name(GeisFilterBag bag, GeisString name);


/**
 * Inserts a filter in the bag.
 *
 * @param[in] bag    The filter bag.
 * @param[in] filter The filter to insert.
 */
GeisStatus geis_filter_bag_insert(GeisFilterBag bag, GeisFilter filter);

/**
 * Remoes a filter from the bag.
 *
 * @param[in] bag    The filter bag.
 * @param[in] filter The filter to remove.
 */
GeisStatus geis_filter_bag_remove(GeisFilterBag bag, GeisFilter filter);

/** @} */

/**
 * @defgroup utouch_geis_filter_internals Filter Internals
 * @{
 */

/**
 * Increments the reference count on a filter.
 *
 * @param[in] filter The filter.
 *
 * @returns the same GeisFilter instance with an incremented reference count.
 */
GeisFilter geis_filter_ref(GeisFilter filter);

/**
 * Decrements and returns the referemce count on a filter.
 *
 * @param[in] filter The filter.
 *
 * @returns the current reference count.
 */
GeisSize geis_filter_unref(GeisFilter filter);

/**
 * Gets the number of terms in the filter.
 *
 * @param[in] filter   The filter.
 */
GeisSize geis_filter_term_count(GeisFilter filter);

/**
 * Gets the indicated term in the filter.
 *
 * @param[in] filter   The filter.
 * @param[in] index    Indicates which term.
 */
GeisFilterTerm geis_filter_term(GeisFilter filter, GeisSize index);

/** @} */

#endif /* GEIS_FILTER_H_ */
