#!/usr/bin/python3
# -*- Mode: Python; indent-tabs-mode: nil; tab-width: 4; coding: utf-8 -*-

import os

import apt
import unittest

from UpdateManager.Core.UpdateList import UpdateList
from UpdateManager.Core.MyCache import MyCache

from mock import patch

CURDIR = os.path.dirname(os.path.abspath(__file__))


class UpdateListTestCase(unittest.TestCase):

    def setUp(self):
        # mangle the arch
        real_arch = apt.apt_pkg.config.find("APT::Architecture")
        apt.apt_pkg.config.set("APT::Architecture", "amd64")
        self.addCleanup(
            lambda: apt.apt_pkg.config.set("APT::Architecture", real_arch))

        self.aptroot = os.path.join(CURDIR,
                                    "aptroot-update-list-test")
        self.cache = MyCache(apt.progress.base.OpProgress(),
                             rootdir=self.aptroot)
        self.updates_list = UpdateList(parent=None)

    def assertUpdatesListLen(self, nr):
        origin = list(self.updates_list.pkgs.keys())[0]
        self.assertEqual(len(self.updates_list.pkgs[origin]), nr)

    def test_phased_percentage_not_included(self):
        """ Test that updates above the threshold are not included"""
        with patch.object(self.updates_list.random, "randint") as mock_randint:
            mock_randint.return_value = 100
            self.updates_list.update(self.cache)
            self.assertUpdatesListLen(1)

    def test_phased_percentage_included(self):
        """ Test that updates below the threshold are included"""
        with patch.object(self.updates_list.random, "randint") as mock_randint:
            mock_randint.return_value = 1
            self.updates_list.update(self.cache)
            self.assertUpdatesListLen(2)

    def test_phased_percentage_included_via_force(self):
        """ Test that the override config works """
        # set config to force override
        apt.apt_pkg.config.set(
            self.updates_list.ALWAYS_INCLUDE_PHASED_UPDATES, "1")
        self.addCleanup(lambda:
            apt.apt_pkg.config.set(
                self.updates_list.ALWAYS_INCLUDE_PHASED_UPDATES, "0"))
        # ensure its included even if its above the threshold
        with patch.object(self.updates_list.random, "randint") as mock_randint:
            mock_randint.return_value = 100
            self.updates_list.update(self.cache)
            self.assertUpdatesListLen(2)

    @patch('UpdateManager.Core.UpdateList.OriginsImportance')
    def test_phased_percentage_from_security(self, mock_origin_importance):
        """ Test that updates from the security node go in"""
        # pretend all updates come from security for the sake of this test
        mock_origin_importance.SECURITY = 0
        with patch.object(self.updates_list.random, "randint") as mock_randint:
            mock_randint.return_value = 100
            self.updates_list.update(self.cache)
            self.assertUpdatesListLen(2)


if __name__ == "__main__":
    unittest.main()
