/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alejandro J. Cura <alecu@canonical.com>
 *
 */

using Assertions;
using Gee;
using GLib;
using Ubuntuone.Webservice;


const string FAKE_URL = "http://fake/url";
const string FAKE_PASSWORD = "PezEspada";
const string FAKE_SKU = "fake_store:fake_album:id";
const string FAKE_TOKEN = "a fake token";

const string BROKEN_JSON = """
    {
        "a": "b",
    $!@$# BROKEN!
""";

const string FAKE_JSON_ACCOUNT = """
    {
        "username": "mrbean",
        "nickname": "Mr. Bean",
        "email": "mr@be.an"
    }
""";


const string FAKE_JSON_PAYMENT_METHOD = """
    {
        "meta": {
            "status": "Ok"
        },
        "payload": {
            "open_url": "",
            "user_email": "mr@be.an",
            "selected_payment_method": "Visa 1234"
        }
    }
""";


const string FAKE_JSON_AUTHENTICATION = """
    {
        "consumer_secret": "fake_secret", 
        "token": "fake_token", 
        "consumer_key": "fake_consumer_key", 
        "name": "FAKE_TOKEN_NAME", 
        "token_secret": "fake_token_secret"
    }
""";

const string FAKE_JSON_PURCHASE = """
    {
        "meta": {
            "status": "Ok"
        },
        "payload": {
            "order_id": "1111",
            "order_status": "OK",
            "order_detail": "TBD",
            "open_url": ""
        }
    }
""";


const string FAKE_JSON_PURCHASE_FAILURE = """
    {
        "meta": {
            "status": "Ok"
        },
        "payload": {
            "order_id": "1111",
            "order_status": "PAYMENT_FAILURE",
            "order_detail": "TBD",
            "open_url": "http://slashdot.org/"
        }
    }
""";


public class Main
{
  public static int main (string[] args)
  {
    int result = run_tests (args);
    // irl_test ();
    return result;
  }

  static int run_tests (string[] args)
  {
    Test.init (ref args);

    Test.add_data_func ("/Unit/PurchaseChecker/ParseAccount", test_parse_account);
    Test.add_data_func ("/Unit/PurchaseChecker/ParseBrokenAccount", test_parse_broken_account);
    Test.add_data_func ("/Unit/PurchaseChecker/ParsePaymentMethod", test_parse_payment_method);
    Test.add_data_func ("/Unit/PurchaseChecker/ParseAuthentication", test_parse_authentication);
    Test.add_data_func ("/Unit/PurchaseChecker/ReadyToPurchase", test_ready_to_purchase);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchPaymentInvalidated", test_fetch_payment_invalidated);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchPaymentCached", test_fetch_payment_cached);
    Test.add_data_func ("/Unit/PurchaseChecker/ReFetchPayment", test_refetch_payment);
    Test.add_data_func ("/Unit/PurchaseChecker/ReFetchPaymentSetsCache", test_refetch_payment_sets_cache);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchCredentials", test_fetch_credentials);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchCredentialsFails", test_fetch_credentials_fails);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchCredentialsFailsExtra", test_fetch_credentials_fails_extra);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchAccount", test_fetch_account);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchAccountFails", test_fetch_account_http_failure);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchAccountBrokenJson", test_fetch_account_wrong_json);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchPayment", test_fetch_payment);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchPaymentFails", test_fetch_payment_http_failure);
    Test.add_data_func ("/Unit/PurchaseChecker/FetchPaymentBrokenJson", test_fetch_payment_wrong_json);
    Test.add_data_func ("/Unit/PurchaseChecker/Purchase", test_purchase);
    Test.add_data_func ("/Unit/PurchaseChecker/GetPurchaseToken", test_get_purchase_token);
    Test.add_data_func ("/Unit/PurchaseChecker/GetPurchaseTokenBrokenJson", test_get_purchase_token_broken_json);
    Test.add_data_func ("/Unit/PurchaseChecker/SsoWebcall", test_authenticated_sso_webcall);
    Test.add_data_func ("/Unit/PurchaseChecker/SsoWebcallFails", test_authenticated_sso_webcall_fails);
    Test.add_data_func ("/Unit/PurchaseChecker/SsoWebcallWrongPassword", test_authenticated_sso_webcall_wrong_password);
    Test.add_data_func ("/Unit/PurchaseChecker/PurchaseDefaultPayment", test_purchase_with_default_payment);
    Test.add_data_func ("/Unit/PurchaseChecker/PurchaseDefaultPaymentBroken", test_purchase_with_default_payment_broken_json);
    Test.add_data_func ("/Unit/PurchaseChecker/PurchaseConnectionFails", test_purchase_with_default_payment_connection_fails);
    Test.add_data_func ("/Unit/PurchaseChecker/PurchaseDefaultPaymentFails", test_purchase_with_default_payment_fails);

    return Test.run ();
  }

  public static bool run_with_timeout (MainLoop ml, uint timeout_ms)
  {
    bool timeout_reached = false;
    var t_id = Timeout.add (timeout_ms, () => {
      timeout_reached = true;
      debug ("Timeout reached");
      ml.quit ();
      return false;
    });

    ml.run ();

    if (!timeout_reached) Source.remove (t_id);

    return !timeout_reached;
  }


  async void irl_test_async ()
  {
    var purchase_service = new PurchaseService();
    var purchase_sku = "1";
    try {
      yield purchase_service.fetch_payment_info (purchase_sku);
      debug ("data was available: %s %s %s", purchase_service.nickname, purchase_service.email, purchase_service.selected_payment_method);
      var real_user_password = "****fake_password_in_bzr***";
      purchase_service.purchase (purchase_sku, real_user_password);
      debug ("purchase completed.");
    } catch (PurchaseError e) {
      debug ("got purchase error: %s", e.message);
    }
  }

  static void irl_test ()
  {
    var loop = new GLib.MainLoop ();
    var m = new Main();
    Idle.add (() => {
      m.irl_test_async.begin ((obj, res) => {
        m.irl_test_async.end (res);
        loop.quit ();
      });
      return false;
    });
    assert (run_with_timeout (loop, 10000));
  }

  class FakeCredentialsManagement : GLib.Object, CredentialsManagement
  {
    private HashTable <string, string> fake_credentials;

    construct
    {
      fake_credentials = new HashTable <string, string> (str_hash, str_equal);
      fake_credentials["token"] = "fake_token";
    }

    public void find_credentials () throws IOError
    {
      Idle.add (() => {
        credentials_found (fake_credentials);
        return false;
      });
    }
  }

  class FailingCredentialsManagement : GLib.Object, CredentialsManagement
  {
    public void find_credentials () throws IOError
    {
      Idle.add (() => {
        credentials_error ();
        return false;
      });
    }
  }

  class ExtraFailingCredentialsManagement : GLib.Object, CredentialsManagement
  {
    public void find_credentials () throws IOError
    {
      throw new IOError.INVALID_DATA ("Miscellanea Errora");
    }
  }

  class BaseTestPurchaseService : PurchaseService
  {
    internal override CredentialsManagement build_credentials_management ()
    {
      return new FakeCredentialsManagement ();
    }
  }

  class FailingCredentialsPurchaseService : PurchaseService
  {
    internal override CredentialsManagement build_credentials_management ()
    {
      return new FailingCredentialsManagement ();
    }
  }

  class ExtraFailingCredentialsPurchaseService : PurchaseService
  {
    internal override CredentialsManagement build_credentials_management ()
    {
      return new ExtraFailingCredentialsManagement ();
    }
  }

  private static void test_parse_account ()
  {
    var purchase_service = new BaseTestPurchaseService ();
    try {
        purchase_service.parse_account_json (FAKE_JSON_ACCOUNT);
    } catch (GLib.Error e)
    {
        assert_not_reached ();
    }
    assert_cmpstr (purchase_service.nickname, OperatorType.EQUAL, "Mr. Bean");
    assert_cmpstr (purchase_service.email, OperatorType.EQUAL, "mr@be.an");
  }

  private static void test_parse_broken_account ()
  {
    var purchase_service = new BaseTestPurchaseService ();
    try
    {
        purchase_service.parse_account_json (BROKEN_JSON);
        assert_not_reached ();
    } catch (GLib.Error e)
    {
        // This is *just* the error we are expecting, so do nothing!
    }
  }

  private static void test_parse_payment_method ()
  {
    var purchase_service = new BaseTestPurchaseService ();
    try
    {
        purchase_service.parse_payment_method_json (FAKE_JSON_PAYMENT_METHOD);
    } catch (GLib.Error e)
    {
        assert_not_reached ();
    }
    assert_cmpstr (purchase_service.selected_payment_method, OperatorType.EQUAL, "Visa 1234");
  }

  private static void test_parse_authentication ()
  {
    var purchase_service = new BaseTestPurchaseService ();

    try
    {
        purchase_service.parse_authentication_json (FAKE_JSON_AUTHENTICATION);
    } catch (GLib.Error e)
    {
        assert_not_reached ();
    }
    assert_cmpstr (purchase_service.consumer_key, OperatorType.EQUAL, "fake_consumer_key");
    assert_cmpstr (purchase_service.token, OperatorType.EQUAL, "fake_token");
  }

  private static void test_ready_to_purchase ()
  {
    var purchase_service = new BaseTestPurchaseService ();
    assert (purchase_service.ready_to_purchase == false);
    purchase_service.selected_payment_method = "visa 1234";
    assert (purchase_service.ready_to_purchase == true);
  }

  class TestFetchPurchaseService : BaseTestPurchaseService {
    internal bool fetch_called = false;
    internal override async void refetch_payment_info (string purchase_sku)
    {
      fetch_called = true;
    }
  }

  private static void test_fetch_payment_invalidated ()
  {
    var purchase_service = new TestFetchPurchaseService ();
    purchase_service._selected_payment_method_expiry = new DateTime.now_utc ();
    purchase_service.fetch_payment_info ("fake_sku");
    assert (purchase_service.fetch_called);
  }

  private static void test_fetch_payment_cached ()
  {
    var purchase_service = new TestFetchPurchaseService ();
    purchase_service._selected_payment_method_expiry = new DateTime.now_utc ().add_seconds (60);
    purchase_service.fetch_payment_info ("fake_sku");
    assert (purchase_service.fetch_called == false);
  }

  class TestRefetchPurchaseService : BaseTestPurchaseService {
    internal bool credentials_fetched = false;
    internal bool account_fetched = false;
    internal bool payment_method_fetched = false;

    internal override async void fetch_credentials ()
    {
      Idle.add (() => {
        credentials_fetched = true;
        fetch_credentials.callback ();
        return false;
      });
      yield;
    }
    internal override async void fetch_account ()
    {
      Idle.add (() => {
        account_fetched = true;
        fetch_account.callback ();
        return false;
      });
      yield;
    }
    internal override async void fetch_payment_method (string purchase_sku)
    {
      Idle.add (() => {
        payment_method_fetched = true;
        fetch_payment_method.callback ();
        return false;
      });
      yield;
    }
  }

  private static void test_refetch_payment ()
  {
    var purchase_service = new TestRefetchPurchaseService ();

    MainLoop mainloop = new MainLoop ();
    purchase_service.refetch_payment_info.begin("fake_sku", (obj, res) => {
        mainloop.quit ();
        try {
            purchase_service.refetch_payment_info.end (res);
        } catch (PurchaseError e) {
            error ("Can't fetch payment info: %s", e.message);
        }
    });
    assert (run_with_timeout (mainloop, 1000));

    assert (purchase_service.credentials_fetched);
    assert (purchase_service.account_fetched);
    assert (purchase_service.payment_method_fetched);
  }

  private static void test_refetch_payment_sets_cache ()
  {
    var purchase_service = new TestRefetchPurchaseService ();

    assert (true == purchase_service.expired_payment_method_cache ());

    MainLoop mainloop = new MainLoop ();
    purchase_service.refetch_payment_info.begin("fake_sku", (obj, res) => {
        mainloop.quit ();
        try {
            purchase_service.refetch_payment_info.end (res);
        } catch (PurchaseError e) {
            error ("Can't fetch payment info: %s", e.message);
        }
    });
    assert (run_with_timeout (mainloop, 1000));

    assert (false == purchase_service.expired_payment_method_cache ());
  }

  private static void test_fetch_credentials ()
  {
    var purchase_service = new BaseTestPurchaseService ();

    MainLoop mainloop = new MainLoop ();
    purchase_service.fetch_credentials.begin((obj, res) => {
        mainloop.quit ();
        try {
            purchase_service.fetch_credentials.end (res);
        } catch (PurchaseError e) {
            error ("Can't fetch credentials: %s", e.message);
        }
    });
    assert (run_with_timeout (mainloop, 1000));

    assert (purchase_service._ubuntuone_credentials != null);
  }

  private static void test_fetch_credentials_fails ()
  {
    bool failed = false;
    var purchase_service = new FailingCredentialsPurchaseService ();

    MainLoop mainloop = new MainLoop ();
    purchase_service.fetch_credentials.begin((obj, res) => {
        mainloop.quit ();
        try {
            purchase_service.fetch_credentials.end (res);
        } catch (PurchaseError e) {
            failed = true;
        }
    });
    assert (run_with_timeout (mainloop, 1000));

    assert (failed);
  }

  private static void test_fetch_credentials_fails_extra ()
  {
    bool failed = false;
    var purchase_service = new ExtraFailingCredentialsPurchaseService ();

    MainLoop mainloop = new MainLoop ();
    purchase_service.fetch_credentials.begin((obj, res) => {
        mainloop.quit ();
        try {
            purchase_service.fetch_credentials.end (res);
        } catch (PurchaseError e) {
            failed = true;
        }
    });
    assert (run_with_timeout (mainloop, 1000));

    assert (failed);
  }

  class FakeWebcallPurchaseService : BaseTestPurchaseService
  {
    internal string used_uri = null;
    internal int status_code;
    internal string found_json;

    internal FakeWebcallPurchaseService (int _status_code, string _found_json) {
        status_code = _status_code;
        found_json = _found_json;
    }

    internal override async PurchaseError call_api (string method, string uri, out string response)
    {
        PurchaseError error = null;
        Idle.add (() => {
            call_api.callback ();
            return false;
        });
        yield;
        response = found_json;
        if (status_code != Soup.KnownStatusCode.OK) {
            error = new PurchaseError.PURCHASE_ERROR ("fake error");
        }
        return error;
    }

    internal override void _do_sso_webcall (Soup.Message message, string password)
    {
        message.response_body.append (Soup.MemoryUse.STATIC, found_json.data);
        message.status_code = status_code;
    }

    internal override Soup.Message send_signed_webservice_call (string method, string uri)
    {
        var message = new Soup.Message (method, uri);
        message.response_body.append (Soup.MemoryUse.STATIC, found_json.data);
        message.status_code = status_code;
        return message;
    }
  }

  private static FakeWebcallPurchaseService _test_fetch_account (int _status_code, string _found_json) throws PurchaseError
  {
    var purchase_service = new FakeWebcallPurchaseService (_status_code, _found_json);
    purchase_service._ubuntuone_credentials = new HashTable <string, string> (str_hash, str_equal);

    PurchaseError error = null;
    MainLoop mainloop = new MainLoop ();
    purchase_service.fetch_account.begin((obj, res) => {
        mainloop.quit ();
        try {
            purchase_service.fetch_account.end (res);
        } catch (PurchaseError e) {
            error = e;
        }
    });
    assert (run_with_timeout (mainloop, 1000));
    if (error != null) {
        throw error;
    }
    return purchase_service;
  }

  private static void test_fetch_account ()
  {
    try {
        var purchase_service = _test_fetch_account (Soup.KnownStatusCode.OK, FAKE_JSON_ACCOUNT);
        assert_cmpstr (purchase_service.nickname, OperatorType.EQUAL, "Mr. Bean");
        assert_cmpstr (purchase_service.email, OperatorType.EQUAL, "mr@be.an");
    } catch (PurchaseError e) {
        assert_not_reached ();
    }
  }

  private static void test_fetch_account_http_failure ()
  {
    try {
        _test_fetch_account (Soup.KnownStatusCode.INTERNAL_SERVER_ERROR, FAKE_JSON_ACCOUNT);
        assert_not_reached ();
    } catch (PurchaseError e) {
        // Expected error
    }
  }

  private static void test_fetch_account_wrong_json ()
  {
    try {
        _test_fetch_account (Soup.KnownStatusCode.OK, BROKEN_JSON);
        assert_not_reached ();
    } catch (PurchaseError e) {
        // Expected error
    }
  }

  private static FakeWebcallPurchaseService _test_fetch_payment (int _status_code, string _found_json) throws PurchaseError
  {
    var purchase_service = new FakeWebcallPurchaseService (_status_code, _found_json);
    purchase_service._ubuntuone_credentials = new HashTable <string, string> (str_hash, str_equal);

    PurchaseError error = null;
    MainLoop mainloop = new MainLoop ();
    purchase_service.fetch_payment_method.begin("fake_sku", (obj, res) => {
        mainloop.quit ();
        try {
            purchase_service.fetch_payment_method.end (res);
        } catch (PurchaseError e) {
            error = e;
        }
    });
    assert (run_with_timeout (mainloop, 1000));
    if (error != null) {
        throw error;
    }
    return purchase_service;
  }

  private static void test_fetch_payment ()
  {
    try {
        var purchase_service = _test_fetch_payment (Soup.KnownStatusCode.OK, FAKE_JSON_PAYMENT_METHOD);
        assert_cmpstr (purchase_service.selected_payment_method, OperatorType.EQUAL, "Visa 1234");
    } catch (PurchaseError e) {
        assert_not_reached ();
    }
  }

  private static void test_fetch_payment_http_failure ()
  {
    try {
        _test_fetch_payment (Soup.KnownStatusCode.INTERNAL_SERVER_ERROR, FAKE_JSON_PAYMENT_METHOD);
        assert_not_reached ();
    } catch (PurchaseError e) {
        // Expected error
    }
  }

  private static void test_fetch_payment_wrong_json ()
  {
    try {
        _test_fetch_payment (Soup.KnownStatusCode.OK, BROKEN_JSON);
        assert_not_reached ();
    } catch (PurchaseError e) {
        // Expected error
    }
  }

  class TestPurchaseService : BaseTestPurchaseService
  {
    internal string password;
    internal string sku;
    internal string purchase_token;

    internal override string get_purchase_token (string password)
    {
        this.password = password;
        return FAKE_TOKEN;
    }
    internal override void purchase_with_default_payment (string sku, string purchase_token)
    {
        this.sku = sku;
        this.purchase_token = purchase_token;
    }
  }

  private static void test_purchase ()
  {
    try {
        var purchase_service = new TestPurchaseService ();
        purchase_service.purchase (FAKE_SKU, FAKE_PASSWORD);
        assert_cmpstr (purchase_service.password, OperatorType.EQUAL, FAKE_PASSWORD);
        assert_cmpstr (purchase_service.sku, OperatorType.EQUAL, FAKE_SKU);
        assert_cmpstr (purchase_service.purchase_token, OperatorType.EQUAL, FAKE_TOKEN);
    } catch (Error e) {
        assert_not_reached ();
    }
  }

  class FakePurchaseTokenService : BaseTestPurchaseService
  {
    string found_json;

    internal FakePurchaseTokenService (string _found_json)
    {
        found_json = _found_json;
    }

    internal override string authenticated_sso_webcall (string method, string uri, string operation, string password)
        throws PurchaseError
    {
      return found_json;
    }
  }

  private static void test_get_purchase_token ()
  {
    var purchase_service = new FakePurchaseTokenService (FAKE_JSON_AUTHENTICATION);
    var expected = "fake_consumer_key:fake_token";
    try {
      var result = purchase_service.get_purchase_token (FAKE_PASSWORD);
      assert_cmpstr (expected, OperatorType.EQUAL, result);
    } catch (PurchaseError e) {
      assert_not_reached ();
    }
  }

  private static void test_get_purchase_token_broken_json ()
  {
    var purchase_service = new FakePurchaseTokenService (BROKEN_JSON);
    try {
      purchase_service.get_purchase_token (FAKE_PASSWORD);
      assert_not_reached ();
    } catch (PurchaseError e) {
      assert (e is PurchaseError.PURCHASE_ERROR);
    }
  }

  private static void test_authenticated_sso_webcall ()
  {
    var purchase_service = new FakeWebcallPurchaseService (Soup.KnownStatusCode.OK, FAKE_JSON_AUTHENTICATION);
    try {
      var result = purchase_service.authenticated_sso_webcall ("FAKE", "http://some/where", "operation=wolf", FAKE_PASSWORD);
      assert_cmpstr (FAKE_JSON_AUTHENTICATION, OperatorType.EQUAL, result);
    } catch (PurchaseError e) {
      assert_not_reached ();
    }
  }

  private static void test_authenticated_sso_webcall_fails ()
  {
    var purchase_service = new FakeWebcallPurchaseService (Soup.KnownStatusCode.INTERNAL_SERVER_ERROR, FAKE_JSON_AUTHENTICATION);
    try {
      purchase_service.authenticated_sso_webcall ("FAKE", "http://some/where", "operation=wolf", FAKE_PASSWORD);
      assert_not_reached ();
    } catch (Error e) {
        assert (e is PurchaseError.PURCHASE_ERROR);
    }
  }

  private static void test_authenticated_sso_webcall_wrong_password ()
  {
    var purchase_service = new FakeWebcallPurchaseService (Soup.KnownStatusCode.UNAUTHORIZED, FAKE_JSON_AUTHENTICATION);
    try {
      purchase_service.authenticated_sso_webcall ("FAKE", "http://some/where", "operation=wolf", FAKE_PASSWORD);
      assert_not_reached ();
    } catch (Error e) {
        assert (e is PurchaseError.WRONG_PASSWORD_ERROR);
    }
  }

  private static void test_purchase_with_default_payment ()
  {
    var purchase_service = new FakeWebcallPurchaseService (Soup.KnownStatusCode.OK, FAKE_JSON_PURCHASE);
    try {
      purchase_service.purchase_with_default_payment (FAKE_SKU, FAKE_TOKEN);
    } catch (PurchaseError e) {
      assert_not_reached ();
    }
  }

  private static void test_purchase_with_default_payment_broken_json ()
  {
    var purchase_service = new FakeWebcallPurchaseService (Soup.KnownStatusCode.OK, BROKEN_JSON);
    try {
      purchase_service.purchase_with_default_payment (FAKE_SKU, FAKE_TOKEN);
      assert_not_reached ();
    } catch (PurchaseError e) {
      assert (e is PurchaseError.PURCHASE_ERROR);
    }
  }

  private static void test_purchase_with_default_payment_fails ()
  {
    var purchase_service = new FakeWebcallPurchaseService (Soup.KnownStatusCode.OK, FAKE_JSON_PURCHASE_FAILURE);
    try {
      purchase_service.purchase_with_default_payment (FAKE_SKU, FAKE_TOKEN);
      assert_not_reached ();
    } catch (PurchaseError e) {
      assert (e is PurchaseError.PURCHASE_ERROR);
    }
  }

  private static void test_purchase_with_default_payment_connection_fails ()
  {
    var purchase_service = new FakeWebcallPurchaseService (Soup.KnownStatusCode.INTERNAL_SERVER_ERROR, FAKE_JSON_PURCHASE_FAILURE);
    try {
      purchase_service.purchase_with_default_payment (FAKE_SKU, FAKE_TOKEN);
      assert_not_reached ();
    } catch (PurchaseError e) {
      assert (e is PurchaseError.PURCHASE_ERROR);
    }
  }
}
