/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using GLib;

namespace Unity.MusicLens {
  
  public class MusicStoreScopeProxy : SimpleScope
  {
    private MusicStoreCollection collection;
    private PreviewPlayer preview_player;
    private Unity.MusicPreview? music_preview;
    private PreferencesManager preferences = PreferencesManager.get_default ();

    public MusicStoreScopeProxy ()
    {
      base ();

      scope = new Unity.Scope ("/com/canonical/unity/scope/musicstore");
      scope.search_in_global = false;
      scope.provides_personal_content = false;
      scope.activate_uri.connect (activate);
      scope.preview_uri.connect (preview);

      base.initialize ();

      collection = new MusicStoreCollection ();

      preferences.notify["remote-content-search"].connect((obj, pspec) => { scope.queue_search_changed(SearchType.DEFAULT); });

      try {
	scope.export ();
      } catch (GLib.IOError e) {
	stdout.printf (e.message);
      }
    }

    protected override int num_results_without_search { get { return 100; } }
    protected override int num_results_global_search { get { return 20; } }
    protected override int num_results_lens_search { get { return 50; } }

    public Unity.ActivationResponse activate (string uri)
    {
      /* launch the music store streaming client  or the webstore or whatevz */
	try {
	  AppInfo.launch_default_for_uri (uri, null);
	} catch (GLib.Error e) {
	  warning ("Failed to open uri %s. %s", uri, e.message);
	}
	return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);  
    }

    private void on_progress_changed (string uri, Unity.MusicPreview.TrackState state, double progress)
    {
      if (music_preview != null)
      {
        music_preview.current_track_uri = uri;
        music_preview.current_track_state = state;
        music_preview.current_progress = (float)progress;
      }
    }

    public void closed (Unity.Preview preview)
    {
      if (preview_player != null)
      {
        try
        {
          preview_player.close ();
        }
        catch (Error e)
        {
          warning ("Failed to close preview player: %s", e.message);
        }
      }
    }

    public void play (Unity.Preview preview, string uri)
    {
      debug ("play request: '%s'", uri);
      
      try
      {
        if (preview_player == null)
        {
          preview_player = new PreviewPlayer ();
          preview_player.progress.connect (on_progress_changed);
          preview_player.connect_to ();
        }

        // we will receive state back in on_progress_changed, but set it now so that it's immediately reflected in the dash
        music_preview.current_track_uri = uri;
        music_preview.current_progress = 0.0f;
        music_preview.current_track_state = Unity.MusicPreview.TrackState.PLAYING;

        preview_player.play (uri);
      }
      catch (Error e)
      {
        warning ("Failed to play '%s': %s", uri, e.message);
      }
    }

    public void pause (Unity.Preview preview, string uri)
    {
      debug ("pause request: '%s'", uri);

      try
      {
        if (preview_player != null)
        {
          // we will receive state back in on_progress_changed, but set it now so that it's immediately reflected in the dash
          music_preview.current_track_uri = uri;
          music_preview.current_track_state = Unity.MusicPreview.TrackState.PAUSED;

          preview_player.pause ();
        }
      }
      catch (Error e)
      {
        warning ("Failed to pause '%s': %s", uri, e.message);
      }
    }

    public Unity.ActivationResponse download_album (string uri)
    {
      return activate (uri);
    }

    public Unity.Preview preview (string uri)
    {
      music_preview = null;
      Album album = null;
      SList<Track> tracks = null;
      collection.get_album_details (uri, out album, out tracks);
      
      if (album != null)
      {
        File cover_file = File.new_for_uri (album.artwork_path); //artwork path is a remote uri
        var cover = new FileIcon (cover_file);
        
        int i = 1;
        music_preview = new Unity.MusicPreview (album.title, album.artist, cover);
        music_preview.play.connect (play);
        music_preview.pause.connect (pause);
        music_preview.closed.connect (closed);

        if (tracks != null)
        {
          foreach (Track track in tracks)
          {
            TrackMetadata tm = new TrackMetadata ();
            tm.uri = track.uri;
            tm.track_no = i++; //FIXME: u1ms search doesn't provide track numbers *yet*, this will change soon
            tm.title = track.title;
            tm.length = track.duration;
            music_preview.add_track (tm);
          }
        }

        GLib.Icon? icon = new GLib.FileIcon (File.new_for_path (Config.DATADIR + "/icons/unity-icon-theme/places/svg/service-u1.svg"));
        var download_action = new Unity.PreviewAction ("download_album", _("Download"), icon);
        if (album.formatted_price != null)
            download_action.extra_text = album.formatted_price;
        download_action.activated.connect (download_album);
        music_preview.add_action (download_action);
      }
      return music_preview;
    }

    public override async void perform_search (LensSearch search,
                                               SearchType search_type,
                                               owned List<FilterParser> filters,
                                               int max_results = -1,
                                               Cancellable? cancellable = null)
    {
      if (is_search_empty (search))
        return;

      /**
       * only perform the request if the user has not disabled
       * online/commercial suggestions. That will hide the category as well.
       */
      if (preferences.remote_content_search != Unity.PreferencesManager.RemoteContent.ALL)
      {
        search.results_model.clear ();
        search.finished ();
        return;
      }
      
      try {
	debug ("model has %u rows before search", search.results_model.get_n_rows ());
	yield collection.search (search, search_type, (owned) filters, max_results, cancellable);
	debug ("model has %u rows after search", search.results_model.get_n_rows ());
      } catch (IOError e) {
	warning ("Failed to search for '%s': %s", search.search_string, e.message);
      }
    }

  }
}
