/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using Gee;
using GLib;
using Sqlite;

namespace Unity.MusicLens {

  public errordomain DatabaseError {
    FAILED_TO_OPEN
  }
	
  /**
   * Abstracts talking to the banshee collection database
   */
  public class BansheeCollection : Object
  {
    private const int MAX_RESULTS = 100;
    
    private Database db;

    public BansheeCollection () throws DatabaseError
    {
      int rc = Database.open ("%s/banshee-1/banshee.db".printf (Environment.get_user_config_dir ()), out db);
      
      if (rc != Sqlite.OK) {
	printerr ("failed to open db, %d, %s\n", rc, db.errmsg ());
	throw new DatabaseError.FAILED_TO_OPEN ("Failed to open banshee database");
      }	
    }

    /**
     * Performs a search on the banshee db
     */
    public void search (LensSearch? search, 
			Dee.Model results_model, 
			GLib.List<FilterParser>? filters = null, 
			int max_results = -1)
    {	
      const int TRACK_TITLE = 0;
      const int TRACK_URI = 1;
      const int TRACK_MIMETYPE = 2;
      const int ALBUM_TITLE = 3;
      const int ALBUM_ARTWORKID = 4;
      const int ARTIST_NAME = 5;

      int rc = 0;
      Statement stmt;
      string album_art_dir = "%s/media-art/".printf (Environment.get_user_cache_dir ());

      // use a tree set to ensure we don't duplicate albums
      TreeSet<string> albums = new TreeSet<string> ();
      string filters_sql = build_sql_from_filters (filters);

      // BUILD SQL STATEMENT
      string sql = """SELECT CoreTracks.Title, CoreTracks.Uri, CoreTracks.MimeType, CoreAlbums.Title, CoreAlbums.ArtworkID, CoreArtists.Name                      FROM CoreTracks 
                      CROSS JOIN CoreArtists, CoreAlbums 
                      WHERE CoreArtists.ArtistID = CoreTracks.ArtistID 
                                                   AND CoreAlbums.AlbumID = CoreTracks.AlbumID 
                                                   AND CoreTracks.PrimarySourceID = 1 
						   AND ((CoreArtists.NameLowered LIKE '%%%s%%' ESCAPE '\' 
						  	 AND CoreArtists.NameLowered IS NOT NULL) 
						        OR (CoreAlbums.TitleLowered LIKE '%%%s%%' ESCAPE '\' 
						  	    AND CoreAlbums.TitleLowered IS NOT NULL) 
						  	OR (CoreTracks.TitleLowered LIKE '%%%s%%' ESCAPE '\' 
						  	    AND CoreTracks.TitleLowered IS NOT NULL)
                                                       )
                                 %s
                      ORDER BY CoreTracks.Score DESC
                      LIMIT 0, %d;""".printf (search.search_string,
                                              search.search_string,
                                              search.search_string,
                                              filters_sql,
                                              max_results == -1 ? MAX_RESULTS : max_results);
			
      rc = execute_sql (sql, out stmt);
      if (stmt == null)
	return;
      
      do {
	rc = stmt.step ();
	switch (rc) {
	case Sqlite.DONE:
	  break;
	case Sqlite.ROW:
	  // translate the raw sql row into something we can use
	  string artwork_path = "%s/%s.jpg".printf (album_art_dir, stmt.column_text (ALBUM_ARTWORKID));
          File artwork_file = File.new_for_path (artwork_path);
          if (!artwork_file.query_exists ())
            artwork_path = "audio-x-generic";

          Track track = new Track ();
	  track.title = stmt.column_text (TRACK_TITLE);
	  track.artist = stmt.column_text (ARTIST_NAME);
	  track.uri = stmt.column_text (TRACK_URI);
	  track.mime_type = stmt.column_text (TRACK_MIMETYPE);          
	  track.artwork_path = artwork_path;
	  
	  Album album = new Album ();
	  album.title = stmt.column_text (ALBUM_TITLE);
	  album.artist = stmt.column_text (ARTIST_NAME);
	  album.uri = "album://%s/%s".printf (album.artist, album.title);
	  album.artwork_path = artwork_path;
	  
	  results_model.append (track.uri, track.artwork_path, 0, track.mime_type, track.title, track.artist, track.uri);
	  
	  if (albums.add (album.artist + album.title)) {
	    StringBuilder uri_list_builder = new StringBuilder ();
	    foreach (string uri in get_track_uris (album)) {
	      uri_list_builder.append ("'");
	      uri_list_builder.append (uri);
	      uri_list_builder.append ("' ");
	    }
	    
	    results_model.append (album.uri, album.artwork_path, 1, "audio/mp3", album.title, album.artist, uri_list_builder.str);
	  }
	  
	  break;
	default:
	  break;
	}
      } while (rc == Sqlite.ROW);
    }

    /**
     * returns an array like {uri://, uri://, ...}
     */
    public string[] get_track_uris (Album album)
    {
      const int URI_COLUMN = 0;
      
      int rc;
      Statement stmt;
      
      string sql = "SELECT CoreTracks.Uri 
                    FROM CoreTracks 
                    CROSS JOIN CoreAlbums, CoreArtists 
                    WHERE CoreArtists.ArtistID = CoreTracks.ArtistID 
                          AND CoreAlbums.AlbumID = CoreTracks.AlbumID 
                          AND CoreAlbums.Title IS '%s'
                          AND CoreArtists.Name IS '%s'
                          AND CoreTracks.URI IS NOT NULL
                    ORDER BY CoreTracks.TrackNumber ASC".printf (album.title, album.artist);

      rc = execute_sql (sql, out stmt);
      
      ArrayList<string> uris = new ArrayList<string> ();
      
      do {
	rc = stmt.step ();
	switch (rc) {
	case Sqlite.DONE:
	  break;
	case Sqlite.ROW:
	  uris.add (stmt.column_text (URI_COLUMN));
	  break;
	default:
	  break;
	}
      } while (rc == Sqlite.ROW);
      
      return uris.to_array ();
    }
    
    /**
     * returns a string like "AND (Table.Column IS filter OR Table.OtherCol IS filter2) 
     * AND (Table.OtherColAgain IS AnotherFilter)" 
     */
    private string build_sql_from_filters (GLib.List<FilterParser> filters)
    {
      if (filters == null || filters.length () == 0)
	return "";
      
      var builder = new StringBuilder ();
      
      foreach (FilterParser parser in filters)
      {
	BansheeFilterParser bparser;
	if (parser is GenreFilterParser)
	  bparser = new BansheeGenreFilterParser (parser as GenreFilterParser);
	else if (parser is DecadeFilterParser)
	  bparser = new BansheeDecadeFilterParser (parser as DecadeFilterParser);
	else
	  {
	    warning ("Recieved an unimplemented filter type");
	    continue;	    
	  }

        string parsed = bparser.parse ();

        if (parsed == null || parsed == "")
          continue;
	
        builder.append (" AND ");
        builder.append (parsed);
      }

      builder.append (" ");
      return builder.str;
    }

    private int execute_sql (string sql, out Statement stmt)
    {
      int rc;
      debug ("preparing to execute sql %s\n", sql);
      
      if ((rc = db.prepare_v2 (sql, -1, out stmt, null)) == 1)
	{
	  warning ("SQL Error: %d, %s\n", rc, db.errmsg ());
        }
      
      return rc;
    }
  }
}
